/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%           RRRR    EEEEE   SSSSS   OOO   U   U  RRRR    CCCC  EEEEE          %
%           R   R   E       SS     O   O  U   U  R   R  C      E              %
%           RRRR    EEE      SSS   O   O  U   U  RRRR   C      EEE            %
%           R R     E          SS  O   O  U   U  R R    C      E              %
%           R  R    EEEEE   SSSSS   OOO    UUU   R  R    CCCC  EEEEE          %
%                                                                             %
%                                                                             %
%                        Get/Set ImageMagick Resources.                       %
%                                                                             %
%                                                                             %
%                              Software Design                                %
%                                John Cristy                                  %
%                               September 2002                                %
%                                                                             %
%                                                                             %
%  Copyright (C) 2003 ImageMagick Studio, a non-profit organization dedicated %
%  to making software imaging solutions freely available.                     %
%                                                                             %
%  Permission is hereby granted, free of charge, to any person obtaining a    %
%  copy of this software and associated documentation files ("ImageMagick"),  %
%  to deal in ImageMagick without restriction, including without limitation   %
%  the rights to use, copy, modify, merge, publish, distribute, sublicense,   %
%  and/or sell copies of ImageMagick, and to permit persons to whom the       %
%  ImageMagick is furnished to do so, subject to the following conditions:    %
%                                                                             %
%  The above copyright notice and this permission notice shall be included in %
%  all copies or substantial portions of ImageMagick.                         %
%                                                                             %
%  The software is provided "as is", without warranty of any kind, express or %
%  implied, including but not limited to the warranties of merchantability,   %
%  fitness for a particular purpose and noninfringement.  In no event shall   %
%  ImageMagick Studio be liable for any claim, damages or other liability,    %
%  whether in an action of contract, tort or otherwise, arising from, out of  %
%  or in connection with ImageMagick or the use or other dealings in          %
%  ImageMagick.                                                               %
%                                                                             %
%  Except as contained in this notice, the name of the ImageMagick Studio     %
%  shall not be used in advertising or otherwise to promote the sale, use or  %
%  other dealings in ImageMagick without prior written authorization from the %
%  ImageMagick Studio.                                                        %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%
*/

/*
  Include declarations.
*/
#include "studio.h"
#include "log.h"
#include "resource.h"
#include "utility.h"

/*
  Define  declarations.
*/
#define ResourceInfinity  (~0UL)
#define ResourceToMegabytes(value) ((long double) (value)*1024.0*1024.0)
#define MegabytesToResource(value) ((unsigned long) ((value)/1024.0/1024.0))
#define GigabytesToResource(value) \
  ((unsigned long) ((value)/1024.0/1024.0/1024.0))

/*
  Typedef declarations.
*/
typedef struct _ResourceInfo
{
  long double
    file,
    memory,
    map,
    disk;

  unsigned long
    file_limit,
    memory_limit,
    map_limit,
    disk_limit;
} ResourceInfo;

/*
  Global declarations.
*/
static SemaphoreInfo
  *resource_semaphore = (SemaphoreInfo *) NULL;

static ResourceInfo
  resource_info =
  {
    0, 0, 0, 0, 256, 1024, 4096, ResourceInfinity
  };

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   A c q u i r e M a g i c k R e s o u r c e                                 %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  AcquireMagickResource() acquires resources of the specified type.  True is
%  returned if the specified resource is available otherwise False.
%
%  The format of the AcquireMagickResource() method is:
%
%      unsigned int AcquireMagickResource(const ResourceType type,
%        const ExtendedSignedIntegralType size)
%
%  A description of each parameter follows:
%
%    o type: The type of resource.
%
%    o size: The number of bytes needed from for this resource.
%
%
*/
MagickExport unsigned int AcquireMagickResource(const ResourceType type,
  const ExtendedSignedIntegralType size)
{
  char
    message[MaxTextExtent];

  unsigned int
    status;

  status=True;
  AcquireSemaphoreInfo(&resource_semaphore);
  switch (type)
  {
    case FileResource:
    {
      resource_info.file+=size;
      if (resource_info.file_limit == ResourceInfinity)
        break;
      status=resource_info.file <= resource_info.file_limit;
      FormatString(message,"file +%lu/%lu/%lu",(unsigned long) size,
        (unsigned long) resource_info.file,resource_info.file_limit);
      (void) LogMagickEvent(ResourceEvent,GetMagickModule(),message);
      break;
    }
    case MemoryResource:
    {
      resource_info.memory+=size;
      if (resource_info.memory_limit == ResourceInfinity)
        break;
      status=resource_info.memory <=
        ResourceToMegabytes(resource_info.memory_limit);
      FormatString(message,"memory +%lumb/%lumb/%lumb",
        MegabytesToResource(size),MegabytesToResource(resource_info.memory),
        resource_info.memory_limit);
      (void) LogMagickEvent(ResourceEvent,GetMagickModule(),message);
      break;
    }
    case MapResource:
    {
      resource_info.map+=size;
      if (resource_info.map_limit == ResourceInfinity)
        break;
      status=resource_info.disk <=
        ResourceToMegabytes(resource_info.map_limit);
      FormatString(message,"map +%lumb/%lumb/%lumb",MegabytesToResource(size),
        MegabytesToResource(resource_info.map),resource_info.map_limit);
      (void) LogMagickEvent(ResourceEvent,GetMagickModule(),message);
      break;
    }
    case DiskResource:
    {
      resource_info.disk+=size;
      if (resource_info.disk_limit == ResourceInfinity)
        break;
      status=resource_info.disk <=
        ResourceToMegabytes(resource_info.disk_limit);
      FormatString(message,"disk +%lumb/%lugb/%lugb",MegabytesToResource(size),
        GigabytesToResource(resource_info.disk),resource_info.disk_limit/1024);
      (void) LogMagickEvent(ResourceEvent,GetMagickModule(),message);
      break;
    }
    default:
      break;
  }
  LiberateSemaphoreInfo(&resource_semaphore);
  return(status);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   D e s t r o y M a g i c k R e s o u r c e s                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  DestroyMagickResources() destroys the resource environment.
%
%  The format of the DestroyMagickResources() method is:
%
%      DestroyMagickResources(void)
%
%
*/
MagickExport void DestroyMagickResources(void)
{
  AcquireSemaphoreInfo(&resource_semaphore);
  DestroySemaphoreInfo(&resource_semaphore);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t M a g i c k R e s o u r c e                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetMagickResource() returns the the specified resource in megabytes.
%
%  The format of the GetMagickResource() method is:
%
%      unsigned long GetMagickResource(const ResourceType type)
%
%  A description of each parameter follows:
%
%    o type: The type of resource.
%
%
*/
MagickExport unsigned long GetMagickResource(const ResourceType type)
{
  unsigned long
    resource;

  resource=0;
  AcquireSemaphoreInfo(&resource_semaphore);
  switch (type)
  {
    case FileResource:
    {
      resource=(unsigned long) resource_info.file;
      break;
    }
    case MemoryResource:
    {
      resource=MegabytesToResource(resource_info.memory);
      break;
    }
    case MapResource:
    {
      resource=MegabytesToResource(resource_info.map);
      break;
    }
    case DiskResource:
    {
      resource=MegabytesToResource(resource_info.disk);
      break;
    }
    default:
      break;
  }
  LiberateSemaphoreInfo(&resource_semaphore);
  return(resource);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   I n i t i a l i z e M a g i c k R e s o u r c e s                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  InitializeMagickResources() initializes the resource environment.
%
%  The format of the InitializeMagickResources() method is:
%
%      InitializeMagickResources(void)
%
%
*/
MagickExport void InitializeMagickResources(void)
{
  long
    files,
    limit,
    pagesize,
    pages;

  /*
    Set Magick resource limits.
  */
  files=0;
#if defined(HAVE_SYSCONF) && defined(_SC_OPEN_MAX)
  files=sysconf(_SC_OPEN_MAX);
#endif
  pagesize=0;
#if defined(HAVE_SYSCONF) && defined(_SC_PAGE_SIZE)
  pagesize=sysconf(_SC_PAGE_SIZE);
#endif
#if defined(HAVE_GETPAGESIZE)
  pagesize=getpagesize();
#endif
  pages=0;
#if defined(HAVE_SYSCONF) && defined(_SC_PHYS_PAGES)
  pages=sysconf(_SC_PHYS_PAGES);
#endif
  limit=((pages+512)/1024)*((pagesize+512)/1024);
#if defined(PixelCacheThreshold)
  limit=PixelCacheThreshold;
#endif
  SetMagickResourceLimit(FileResource,files > 0 ? files/2 : 256);
  SetMagickResourceLimit(MemoryResource,limit > 0 ? 2*limit : 1024);
  SetMagickResourceLimit(MapResource,limit > 0 ? 8*limit : 4096);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   L i b e r a t e M a g i c k R e s o u r c e                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  LiberateMagickResource() liberates resources of the specified type.
%
%  The format of the LiberateMagickResource() method is:
%
%      void LiberateMagickResource(const ResourceType type,
%        const ExtendedSignedIntegralType size)
%
%  A description of each parameter follows:
%
%    o type: The type of resource.
%
%    o size: The size of the resource.
%
%
*/
MagickExport void LiberateMagickResource(const ResourceType type,
  const ExtendedSignedIntegralType size)
{
  char
    message[MaxTextExtent];

  AcquireSemaphoreInfo(&resource_semaphore);
  switch (type)
  {
    case FileResource:
    {
      resource_info.file-=size;
      FormatString(message,"file -%lu/%lu/%lu",(unsigned long) size,
        (unsigned long) resource_info.file,resource_info.file_limit);
      (void) LogMagickEvent(ResourceEvent,GetMagickModule(),message);
      break;
    }
    case MemoryResource:
    {
      resource_info.memory-=size;
      FormatString(message,"memory -%lumb/%lumb/%lumb",
        MegabytesToResource(size),MegabytesToResource(resource_info.memory),
        resource_info.memory_limit);
      (void) LogMagickEvent(ResourceEvent,GetMagickModule(),message);
      break;
    }
    case MapResource:
    {
      resource_info.map-=size;
      FormatString(message,"map -%lumb/%lumb/%lumb",MegabytesToResource(size),
        MegabytesToResource(resource_info.map),resource_info.map_limit);
      (void) LogMagickEvent(ResourceEvent,GetMagickModule(),message);
      break;
    }
    case DiskResource:
    {
      resource_info.disk-=size;
      FormatString(message,"disk -%lumb/%lugb/%lugb",MegabytesToResource(size),
        GigabytesToResource(resource_info.disk),resource_info.disk_limit/1024);
      (void) LogMagickEvent(ResourceEvent,GetMagickModule(),message);
      break;
    }
    default:
      break;
  }
  LiberateSemaphoreInfo(&resource_semaphore);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%  L i s t M a g i c k R e s o u r c e I n f o                                %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Method ListMagickResourceInfo lists the resource info to a file.
%
%  The format of the ListMagickResourceInfo method is:
%
%      unsigned int ListMagickResourceInfo(FILE *file,ExceptionInfo *exception)
%
%  A description of each parameter follows.
%
%    o file:  An pointer to a FILE.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport unsigned int ListMagickResourceInfo(FILE *file,
  ExceptionInfo *exception)
{
  if (file == (const FILE *) NULL)
    file=stdout;
  AcquireSemaphoreInfo(&resource_semaphore);
  (void) fprintf(file,"File    Memory       Map       Disk\n");
  (void) fprintf(file,"-----------------------------------\n");
  (void) fprintf(file,"%4lu  %6lumb  %6lumb  %6lugb\n",
    resource_info.file_limit,resource_info.memory_limit,resource_info.map_limit,
    resource_info.disk_limit/1024);
  (void) fflush(file);
  LiberateSemaphoreInfo(&resource_semaphore);
  return(True);
}
/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   S e t M a g i c k R e s o u r c e L i m i t                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  SetMagickResourceLimit() sets the limit for a particular resource in
%  megabytes.
%
%  The format of the SetMagickResourceLimit() method is:
%
%      void SetMagickResourceLimit(const ResourceType type,
%        const unsigned long limit)
%
%  A description of each parameter follows:
%
%    o type: The type of resource.
%
%    o limit: The maximum limit for the resource.
%
%
*/
MagickExport void SetMagickResourceLimit(const ResourceType type,
  const unsigned long limit)
{
  AcquireSemaphoreInfo(&resource_semaphore);
  switch (type)
  {
    case FileResource:
    {
      resource_info.file_limit=limit;
      break;
    }
    case MemoryResource:
    {
      resource_info.memory_limit=limit;
      break;
    }
    case MapResource:
    {
      resource_info.map_limit=limit;
      break;
    }
    case DiskResource:
    {
      resource_info.disk_limit=limit;
      break;
    }
    default:
      break;
  }
  LiberateSemaphoreInfo(&resource_semaphore);
}
