/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%               DDDD   IIIII  SSSSS  TTTTT   OOO   RRRR   TTTTT               %
%               D   D    I    SS       T    O   O  R   R    T                 %
%               D   D    I     SSS     T    O   O  RRRR     T                 %
%               D   D    I       SS    T    O   O  R R      T                 %
%               DDDD   IIIII  SSSSS    T     OOO   R  R     T                 %
%                                                                             %
%                                                                             %
%                     ImageMagick Image Distortion Methods.                   %
%                                                                             %
%                              Software Design                                %
%                                John Cristy                                  %
%                              Anthony Thyssen                                %
%                                 June 2007                                   %
%                                                                             %
%                                                                             %
%  Copyright 1999-2008 ImageMagick Studio LLC, a non-profit organization      %
%  dedicated to making software imaging solutions freely available.           %
%                                                                             %
%  You may not use this file except in compliance with the License.  You may  %
%  obtain a copy of the License at                                            %
%                                                                             %
%    http://www.imagemagick.org/script/license.php                            %
%                                                                             %
%  Unless required by applicable law or agreed to in writing, software        %
%  distributed under the License is distributed on an "AS IS" BASIS,          %
%  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   %
%  See the License for the specific language governing permissions and        %
%  limitations under the License.                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/artifact.h"
#include "magick/cache-view.h"
#include "magick/colorspace-private.h"
#include "magick/composite-private.h"
#include "magick/distort.h"
#include "magick/exception.h"
#include "magick/exception-private.h"
#include "magick/gem.h"
#include "magick/hashmap.h"
#include "magick/image.h"
#include "magick/list.h"
#include "magick/matrix.h"
#include "magick/memory_.h"
#include "magick/monitor-private.h"
#include "magick/pixel.h"
#include "magick/pixel-private.h"
#include "magick/resample.h"
#include "magick/registry.h"
#include "magick/semaphore.h"
#include "magick/string_.h"

/*
 * Numerious internal routines for ImageDistortions
 */
static inline double MagickMin(const double x,const double y)
{
  return( x < y ? x : y);
}
static inline double MagickMax(const double x,const double y)
{
  return( x > y ? x : y);
}

static inline void AffineArgsToCoefficients(double *affine)
{
  /* map  external sx,ry,rx,sy,tx,ty  to  internal c0,c2,c4,c1,c3,c5 */
  double tmp[4];  /* note indexes  0 and 5 remain unchanged */
  tmp[0]=affine[1]; tmp[1]=affine[2]; tmp[2]=affine[3]; tmp[3]=affine[4];
  affine[3]=tmp[0]; affine[1]=tmp[1]; affine[4]=tmp[2]; affine[2]=tmp[3];
}
static inline void CoefficientsToAffineArgs(double *coeff)
{
  /* map  internal c0,c1,c2,c3,c4,c5  to  external sx,ry,rx,sy,tx,ty */
  double tmp[4];  /* note indexes 0 and 5 remain unchanged */
  tmp[0]=coeff[3]; tmp[1]=coeff[1]; tmp[2]=coeff[4]; tmp[3]=coeff[2];
  coeff[1]=tmp[0]; coeff[2]=tmp[1]; coeff[3]=tmp[2]; coeff[4]=tmp[3];
}
static void InvertAffineCoefficients(const double *coeff,double *inverse)
{
  /* From "Digital Image Warping" by George Wolberg, page 50 */
  double determinant;

  determinant=1.0/(coeff[0]*coeff[4]-coeff[1]*coeff[3]);
  inverse[0]=determinant*coeff[4];
  inverse[1]=determinant*(-coeff[1]);
  inverse[2]=determinant*(coeff[1]*coeff[5]-coeff[2]*coeff[4]);
  inverse[3]=determinant*(-coeff[3]);
  inverse[4]=determinant*coeff[0];
  inverse[5]=determinant*(coeff[2]*coeff[3]-coeff[0]*coeff[5]);
}

static void InvertPerspectiveCoefficients(const double *coefficients,
  double *inverse)
{
  /* From "Digital Image Warping" by George Wolberg, page 53 */
  double determinant;

  determinant=1.0/(coefficients[0]*coefficients[4]-coefficients[3]*coefficients[1]);
  inverse[0]=determinant*(coefficients[4]-coefficients[7]*coefficients[5]);
  inverse[1]=determinant*(coefficients[7]*coefficients[2]-coefficients[1]);
  inverse[2]=determinant*(coefficients[1]*coefficients[5]-coefficients[4]*coefficients[2]);
  inverse[3]=determinant*(coefficients[6]*coefficients[5]-coefficients[3]);
  inverse[4]=determinant*(coefficients[0]-coefficients[6]*coefficients[2]);
  inverse[5]=determinant*(coefficients[3]*coefficients[2]-coefficients[0]*coefficients[5]);
  inverse[6]=determinant*(coefficients[3]*coefficients[7]-coefficients[6]*coefficients[4]);
  inverse[7]=determinant*(coefficients[6]*coefficients[1]-coefficients[0]*coefficients[7]);
}

static inline double MagickRound(double x)
{
  /* round the fraction to nearest integer */
  if (x >= 0.0)
    return((double) ((long) (x+0.5)));
  return((double) ((long) (x-0.5)));
}

static unsigned long poly_number_terms(double order)
{
  /* Return the number of terms for a 2d polynomial
     Order must either be an integer, or 1.5 to produce
     the 2 number_valuesal polyminal function...
        affine     1 (3)      u = c0 + c1*x + c2*y
        bilinear  1.5 (4)     u = '' + c3*x*y
        quadratic  2 (6)      u = '' + c4*x*x + c5*y*y
        cubic      3 (10)     u = '' + c6*x^3 + c7*x*x*y + c8*x*y*y + c9*y^3
        quartic    4 (15)     u = '' + c10*x^4 + ... + c14*y^4
        quintic    5 (21)     u = '' + c15*x^5 + ... + c20*y^5
     number in parenthesis minimum number of points needed.
     Anything beyond quintic, has not been implemented until
     a more automated way of determined terms is found.
   */
  if ( order < 1 || order > 5 ||
       ( order != floor(order) && (order-1.5) > MagickEpsilon) )
    return 0; /* invalid polynomial order */
  return ((unsigned long) floor((order+1)*(order+2)/2));
}

static double poly_basis_fn(long n, double x, double y)
{
  /* return the result for this polynomial term */
  switch(n) {
    case  0:  return( 1.0 ); /* constant */
    case  1:  return(  x  );
    case  2:  return(  y  ); /* affine      order = 1   terms = 3 */
    case  3:  return( x*y ); /* bilinear    order = 1.5 terms = 4 */
    case  4:  return( x*x );
    case  5:  return( y*y ); /* quadratic   order = 2   terms = 6 */
    case  6:  return( x*x*x );
    case  7:  return( x*x*y );
    case  8:  return( x*y*y );
    case  9:  return( y*y*y ); /* cubic       order = 3   terms = 10 */
    case 10:  return( x*x*x*x );
    case 11:  return( x*x*x*y );
    case 12:  return( x*x*y*y );
    case 13:  return( x*y*y*y );
    case 14:  return( y*y*y*y ); /* quartic     order = 4   terms = 15 */
    case 15:  return( x*x*x*x*x );
    case 16:  return( x*x*x*x*y );
    case 17:  return( x*x*x*y*y );
    case 18:  return( x*x*y*y*y );
    case 19:  return( x*y*y*y*y );
    case 20:  return( y*y*y*y*y ); /* quintic     order = 5   terms = 21 */
  }
  return( 0 ); /* should never happen */
}
static const char *poly_basis_str(long n)
{
  /* return the result for this polynomial term */
  switch(n) {
    case  0:  return(""); /* constant */
    case  1:  return("*ii");
    case  2:  return("*jj"); /* affiine      order = 1   terms = 3 */
    case  3:  return("*ii*jj"); /* biiliinear    order = 1.5 terms = 4 */
    case  4:  return("*ii*ii");
    case  5:  return("*jj*jj"); /* quadratiic   order = 2   terms = 6 */
    case  6:  return("*ii*ii*ii");
    case  7:  return("*ii*ii*jj");
    case  8:  return("*ii*jj*jj");
    case  9:  return("*jj*jj*jj"); /* cubiic       order = 3   terms = 10 */
    case 10:  return("*ii*ii*ii*ii");
    case 11:  return("*ii*ii*ii*jj");
    case 12:  return("*ii*ii*jj*jj");
    case 13:  return("*ii*jj*jj*jj");
    case 14:  return("*jj*jj*jj*jj"); /* quartiic     order = 4   terms = 15 */
    case 15:  return("*ii*ii*ii*ii*ii");
    case 16:  return("*ii*ii*ii*ii*jj");
    case 17:  return("*ii*ii*ii*jj*jj");
    case 18:  return("*ii*ii*jj*jj*jj");
    case 19:  return("*ii*jj*jj*jj*jj");
    case 20:  return("*jj*jj*jj*jj*jj"); /* quiintiic     order = 5   terms = 21 */
  }
  return( "UNKNOWN" ); /* should never happen */
}
static double poly_basis_dx(long n, double x, double y)
{
  /* polynomial term for x derivative */
  switch(n) {
    case  0:  return( 0.0 ); /* constant */
    case  1:  return( 1.0 );
    case  2:  return( 0.0 ); /* affine      order = 1   terms = 3 */
    case  3:  return(  y  ); /* bilinear    order = 1.5 terms = 4 */
    case  4:  return(  x  );
    case  5:  return( 0.0 ); /* quadratic   order = 2   terms = 6 */
    case  6:  return( x*x );
    case  7:  return( x*y );
    case  8:  return( y*y );
    case  9:  return( 0.0 ); /* cubic       order = 3   terms = 10 */
    case 10:  return( x*x*x );
    case 11:  return( x*x*y );
    case 12:  return( x*y*y );
    case 13:  return( y*y*y );
    case 14:  return( 0.0 ); /* quartic     order = 4   terms = 15 */
    case 15:  return( x*x*x*x );
    case 16:  return( x*x*x*y );
    case 17:  return( x*x*y*y );
    case 18:  return( x*y*y*y );
    case 19:  return( y*y*y*y );
    case 20:  return( 0.0 ); /* quintic     order = 5   terms = 21 */
  }
  return( 0.0 ); /* should never happen */
}
static double poly_basis_dy(long n, double x, double y)
{
  /* polynomial term for y derivative */
  switch(n) {
    case  0:  return( 0.0 ); /* constant */
    case  1:  return( 0.0 );
    case  2:  return( 1.0 ); /* affine      order = 1   terms = 3 */
    case  3:  return(  x  ); /* bilinear    order = 1.5 terms = 4 */
    case  4:  return( 0.0 );
    case  5:  return(  y  ); /* quadratic   order = 2   terms = 6 */
    default:  return( poly_basis_dx(n-1,x,y) ); /* weird but true */
  }
  /* NOTE: the only reason that last is not true for 'quadtratic'
     is due to the re-arrangement of terms to allow for 'bilinear'
  */
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
-   G e n e r a t e C o e f f i c i e n t s                                   %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GenerateCoefficients() takes user provided input arguments and generates
%  the coefficients, needed to apply the specific distortion for either
%  distorting images (generally using control points) or generating a color
%  gradient from sparsely separated color points.
%
%  The format of the GenerateCoefficients() method is:
%
%    Image *GenerateCoefficients(const Image *image, DistortImageMethod method,
%        const unsigned long number_arguments,const double *arguments,
%        unsigned long number_values, ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image: the image to be distorted.
%
%    o method: the method of image distortion/ sparse gradient
%
%    o number_arguments: the number of arguments given.
%
%    o arguments: the arguments for this distortion method.
%
%    o number_values: the style and format of given control points, (caller type)
%         0: 2 dimensional mapping of control points (Distort)
%            Format:  u,v,x,y  where u,v is the 'source' of the
%            the color to be plotted, for DistortImage()
%         N: Interpolation of control points with N values (usally r,g,b)
%            Format: x,y,r,g,b    mapping x,y to color values r,g,b
%            IN future, varible number of values may be given (1 to N)
%
%    o exception: Return any errors or warnings in this structure
%
%  Note that the returned array of double values must be freed by the
%  calling method using RelinquishMagickMemory().  This however may change in
%  the future to require a more 'method' specific method.
%
%  Because of this this method should not be classed as stable or used
%  outside other MagickCore library methods.
*/

static double *GenerateCoefficients(const Image *image,
  DistortImageMethod *method,const unsigned long number_arguments,
  const double *arguments,unsigned long number_values,ExceptionInfo *exception)
{
  double
    *coefficients;

  register unsigned long
    i;

  unsigned long
    number_coefficients, /* number of coefficients needed (array size) */
    cp_size,             /* number floating point numbers per control point */
    cp_x,cp_y,           /* the x,y indexes for control point */
    cp_values;           /* index of values for this control point */

  if ( number_values == 0 ) {
    /* Image distortion using control points (or other distortion)
       That is generate a mapping so that   x,y->u,v   given  u,v,x,y
    */
    number_values = 2;   /* Really two values of  u,v */
    cp_values = 0;       /* values are BEFORE the CP x,y */
    cp_x = 2;            /* location of x,y in input control values */
    cp_y = 3;
    /* NOTE: cp_values, also used for later 'reverse map distort' tests */
  }
  else {
    cp_x = 0;            /* location of x,y in input control values */
    cp_y = 1;
    cp_values = 2;       /* and the other values are after x,y */
  }
  cp_size = number_values+2; /* each CP defintion involves this many numbers */


  /* If not enough control point pairs are found for specific distortions
    fall back to Affine distortion (allowing 0 to 3 point pairs)
  */
  if ( number_arguments < 4*cp_size &&
       (  *method == BilinearDistortion
       || *method == PerspectiveDistortion
       ) )
    *method = AffineDistortion;

  switch (*method) {
    case AffineDistortion:
      number_coefficients=3*number_values;
      break;
    case BilinearDistortion:
      number_coefficients=4*number_values;
      break;
    case PolynomialDistortion:
      /* number of coefficents depend on the given polynomal 'order' */
      if ( number_arguments <= 1 && (number_arguments-1)%cp_size != 0)
      {
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
                   "InvalidArgument",
                   "%s : Invalid number of args: order [%ld CP values]... ",
                   "Polynomial", cp_size);
        return((double *) NULL);
      }
      i = poly_number_terms(arguments[0]);
      number_coefficients = 2 + i*number_values;
      if ( i == 0 ) {
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
                   "InvalidArgument","%s : '%s'","Polynomial",
                   "Invalid order, should be 1 to 5, or 1.5");
        return((double *) NULL);
      }
      if ( number_arguments < 1+i*cp_size ) {
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
               "InvalidArgument",
               "%s : 'require at least %ld CPs of %ld values'",
               "Polynomial", i, cp_size);
        return((double *) NULL);
      }
      break;
    /* The rest are constants as they are only used for image distorts */
    case ShepardsDistortion:
    case VoronoiColorInterpolate:
      number_coefficients=1;   /* may not be used, but provide anyway */
      break;
    case ArcDistortion:
      number_coefficients=5;
      break;
    case ScaleRotateTranslateDistortion:
    case AffineProjectionDistortion:
      number_coefficients=6;
      break;
    case BarrelDistortion:
      number_coefficients=7;
      break;
    case PolarDistortion:
    case DePolarDistortion:
      number_coefficients=8;
      number_coefficients=8;
      break;
    case PerspectiveDistortion:
    case PerspectiveProjectionDistortion:
      number_coefficients=9;
      break;
    case UndefinedDistortion:
    default:
      assert(! "Unknown Method Given"); /* just fail assertion */
  }

  /* allocate the array of coefficients needed */
  coefficients = (double *) AcquireQuantumMemory(number_coefficients,
                        sizeof(*coefficients));
  if (coefficients == (double *) NULL) {
    (void) ThrowMagickException(exception,GetMagickModule(),
                  ResourceLimitError,"MemoryAllocationFailed",
                  "%s", "GenerateCoefficients");
    return((double *) NULL);
  }

  /* zero out coeffiecents array */
  for (i=0; i < number_coefficients; i++)
    coefficients[i] = 0.0;

  switch (*method)
  {
    case AffineDistortion:
    {
      /* Affine Distortion
           v =  c0*x + c1*y + c2
         for each 'value' given

         Input Arguments are sets of control points...
         For Distort Images    u,v, x,y  ...
         For Sparse Gradients  x,y, r,g,b  ...
      */
      if ( number_arguments%cp_size != 0 ||
           number_arguments < cp_size ) {
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
               "InvalidArgument",
               "%s : 'require at least %ld CPs of %ld values'",
               "Affine", 1L, cp_size);
        coefficients=(double *) RelinquishMagickMemory(coefficients);
        return((double *) NULL);
      }
      /* handle special cases of not enough arguments */
      if ( number_arguments == cp_size ) {
        /* Only 1 CP Set Given */
        if ( cp_values == 0 ) {
          /* image distortion - translate the image */
          coefficients[0] = 1.0;
          coefficients[2] = arguments[0] - arguments[2];
          coefficients[4] = 1.0;
          coefficients[5] = arguments[1] - arguments[3];
        }
        else {
          /* sparse gradient - use the values directly */
          for (i=0; i<number_values; i++)
            coefficients[i*3+2] = arguments[cp_values+i];
        }
      }
      else {
        /* 2 or more points (usally 3) given.
           Solve a least squares simultanious equation for coefficients.
        */
        double
          **matrix,
          **vectors,
          terms[3];

        MagickBooleanType
          status;

        /* create matrix, and a fake vectors matrix */
        matrix = AcquireMagickMatrix(3UL,3UL);
        vectors = (double **) AcquireQuantumMemory(number_values,sizeof(*vectors));
        if (matrix == (double **) NULL || vectors == (double **) NULL)
        {
          matrix = RelinquishMagickMatrix(matrix, 3UL);
          vectors = (double **) RelinquishMagickMemory(vectors);
          coefficients = (double *) RelinquishMagickMemory(coefficients);
          (void) ThrowMagickException(exception,GetMagickModule(),
                  ResourceLimitError,"MemoryAllocationFailed",
                  "%s", "DistortCoefficients");
          return((double *) NULL);
        }
        /* fake a number_values x3 vectors matrix from coefficients array */
        for (i=0; i < number_values; i++)
          vectors[i] = &(coefficients[i*3]);
        /* Add given control point pairs for least squares solving */
        for (i=0; i < number_arguments; i+=cp_size) {
          terms[0] = arguments[i+cp_x];  /* x */
          terms[1] = arguments[i+cp_y];  /* y */
          terms[2] = 1;                  /* 1 */
          LeastSquaresAddTerms(matrix,vectors,terms,
                   &(arguments[i+cp_values]),3UL,number_values);
        }
        if ( number_arguments == 2*cp_size ) {
          /* Only two pairs were given, but we need 3 to solve the affine.
             Fake extra coordinates by rotating p1 around p0 by 90 degrees.
               x2 = x0 - (y1-y0)   y2 = y0 + (x1-x0)
           */
          terms[0] = arguments[cp_x]
                   - ( arguments[cp_size+cp_y] - arguments[cp_y] ); /* x2 */
          terms[1] = arguments[cp_y] +
                   + ( arguments[cp_size+cp_x] - arguments[cp_x] ); /* y2 */
          terms[2] = 1;                                             /* 1 */
          if ( cp_values == 0 ) {
            /* Image Distortion - rotate the u,v coordients too */
            double
              uv2[2];
            uv2[0] = arguments[0] - arguments[5] + arguments[1];   /* u2 */
            uv2[1] = arguments[1] + arguments[4] - arguments[0];   /* v2 */
            LeastSquaresAddTerms(matrix,vectors,terms,uv2,3UL,2UL);
          }
          else {
            /* Sparse Gradient - use values of p0 for linear gradient */
            LeastSquaresAddTerms(matrix,vectors,terms,
                  &(arguments[cp_values]),3UL,number_values);
          }
        }
        /* Solve for LeastSquares Coefficients */
        status=GaussJordanElimination(matrix,vectors,3UL,number_values);
        matrix = RelinquishMagickMatrix(matrix, 3UL);
        vectors = (double **) RelinquishMagickMemory(vectors);
        if ( status == MagickFalse ) {
          coefficients = (double *) RelinquishMagickMemory(coefficients);
          (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
                      "InvalidArgument","%s : '%s'","Affine",
                      "Unsolvable Matrix");
          return((double *) NULL);
        }
      }
      return(coefficients);
    }
    case AffineProjectionDistortion:
    {
      /*
        Arguments: Affine Matrix (forward mapping)
        Arguments  sx, rx, ry, sy, tx, ty
        Where      u = sx*x + ry*y + tx
                   v = rx*x + sy*y + ty

         AffineProjection Distortion Notes...
           + Will only work with a 2 number_valuesal Image Distortion
           + Can not be used for generating a sparse gradient (interpolation)
      */
      double inverse[8];
      if (number_arguments != 6) {
        coefficients = (double *) RelinquishMagickMemory(coefficients);
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
                     "InvalidArgument","%s : '%s'","AffineProjection",
                     "Needs 6 coefficient values");
        return((double *) NULL);
      }
      for(i=0; i<6UL; i++ )
        inverse[i] = arguments[i];
      AffineArgsToCoefficients(inverse); /* map into coefficents */
      InvertAffineCoefficients(inverse, coefficients); /* invert */
      *method = AffineDistortion;

      return(coefficients);
    }
    case ScaleRotateTranslateDistortion:
    {
      /* Scale, Rotate and Translate Distortion
         An alturnative Affine Distortion
         Argument options, by number of arguments given:
           7: x,y, sx,sy, a, nx,ny
           6: x,y,   s,   a, nx,ny
           5: x,y, sx,sy, a
           4: x,y,   s,   a
           3: x,y,        a
           2:        s,   a
           1:             a
         Where actions are (in order of application)
            x,y     'center' of transforms     (default = image center)
            sx,sy   scale image by this amount (default = 1)
            a       angle of rotation          (argument required)
            nx,ny   move 'center' here         (default = no movement)
         And convert to affine mapping coefficients

         ScaleRotateTranslate Distortion Notes...
           + Does not use a set of CPs in any normal way
           + Will only work with a 2 number_valuesal Image Distortion
           + Can not be used for generating a sparse gradient (interpolation)
      */
      double
        cosine, sine,
        x,y,sx,sy,a,nx,ny;

      /* set default center, and default scale */
      x = nx = (double)(image->columns-1.0)/2.0 + image->page.x;
      y = ny = (double)(image->rows-1.0)/2.0    + image->page.y;
      sx = sy = 1.0;
      switch ( number_arguments ) {
      case 0:
        coefficients = (double *) RelinquishMagickMemory(coefficients);
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
                     "InvalidArgument","%s : '%s'", "ScaleTranslateRotate",
                     "Needs at least 1 argument");
        return((double *) NULL);
      case 1:
        a = arguments[0];
        break;
      case 2:
        sx = sy = arguments[0];
        a = arguments[1];
        break;
      default:
        x = nx = arguments[0];
        y = ny = arguments[1];
        switch ( number_arguments ) {
        case 3:
          a = arguments[2];
          break;
        case 4:
          sx = sy = arguments[2];
          a = arguments[3];
          break;
        case 5:
          sx = arguments[2];
          sy = arguments[3];
          a = arguments[4];
          break;
        case 6:
          sx = sy = arguments[2];
          a = arguments[3];
          nx = arguments[4];
          ny = arguments[5];
          break;
        case 7:
          sx = arguments[2];
          sy = arguments[3];
          a = arguments[4];
          nx = arguments[5];
          ny = arguments[6];
          break;
        default:
          coefficients = (double *) RelinquishMagickMemory(coefficients);
          (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
                     "InvalidArgument","%s : '%s'", "ScaleTranslateRotate",
                     "Too Many Arguments (7 or less)");
          return((double *) NULL);
        }
        break;
      }
      /* FUTURE: trap if sx or sy == 0 -- image is scaled out of existance! */
      if ( fabs(sx) < MagickEpsilon || fabs(sy) < MagickEpsilon ) {
        coefficients = (double *) RelinquishMagickMemory(coefficients);
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
                     "InvalidArgument","%s : '%s'", "ScaleTranslateRotate",
                     "Zero Scale Given");
        return((double *) NULL);
      }
      /* Save the given arguments as an affine distortion */
      a=DegreesToRadians(a); cosine=cos(a); sine=sin(a);

      *method = AffineDistortion;
      coefficients[0]=cosine/sx;
      coefficients[1]=sine/sx;
      coefficients[2]=x-nx*coefficients[0]-ny*coefficients[1];
      coefficients[3]=(-sine)/sy;
      coefficients[4]=cosine/sy;
      coefficients[5]=y-nx*coefficients[3]-ny*coefficients[4];
      return(coefficients);
    }
    case PerspectiveDistortion:
    { /*
         Perspective Distortion (a ratio of affine distortions)

                p(x,y)    c0*x + c1*y + c2
            u = ------ = ------------------
                r(x,y)    c6*x + c7*y + 1

                q(x,y)    c3*x + c4*y + c5
            v = ------ = ------------------
                r(x,y)    c6*x + c7*y + 1

           c8 = Sign of 'r', or the denominator affine, for the actual image.
                This determines what part of the distorted image is 'ground'
                side of the horizon, the other part is 'sky' or invalid.
                Valid values are  +1.0  or  -1.0  only.

         Input Arguments are sets of control points...
         For Distort Images    u,v, x,y  ...
         For Sparse Gradients  x,y, r,g,b  ...

         Perspective Distortion Notes...
           + Can be thought of as ratio of  3 affine transformations
           + Not separatable: r() or c6 and c7 are used by both equations
           + All 8 coefficients must be determined simultaniously
           + Will only work with a 2 number_valuesal Image Distortion
           + Can not be used for generating a sparse gradient (interpolation)
           + It is not linear, but is simple to generate an inverse
           + All lines within an image remain lines.
           + but distances between points may vary.
      */
      double
        **matrix,
        *vectors[1],
        terms[8];

      unsigned long
        cp_u = cp_values,
        cp_v = cp_values+1;

      MagickBooleanType
        status;

      /* fake 1x8 vectors matrix directly using the coefficients array */
      vectors[0] = &(coefficients[0]);
      /* 8x8 least-squares matrix (zeroed) */
      matrix = AcquireMagickMatrix(8UL,8UL);
      if (matrix == (double **) NULL) {
        (void) ThrowMagickException(exception,GetMagickModule(),
                  ResourceLimitError,"MemoryAllocationFailed",
                  "%s", "DistortCoefficients");
        return((double *) NULL);
      }
      /* Add control points for least squares solving */
      for (i=0; i < number_arguments; i+=4) {
        terms[0]=arguments[i+cp_x];            /*   c0*x   */
        terms[1]=arguments[i+cp_y];            /*   c1*y   */
        terms[2]=1.0;                          /*   c2*1   */
        terms[3]=0.0;
        terms[4]=0.0;
        terms[5]=0.0;
        terms[6]=-terms[0]*arguments[i+cp_u];  /* 1/(c6*x) */
        terms[7]=-terms[1]*arguments[i+cp_u];  /* 1/(c7*y) */
        LeastSquaresAddTerms(matrix,vectors,terms,&(arguments[i+cp_u]),
            8UL,1UL);

        terms[0]=0.0;
        terms[1]=0.0;
        terms[2]=0.0;
        terms[3]=arguments[i+cp_x];           /*   c3*x   */
        terms[4]=arguments[i+cp_y];           /*   c4*y   */
        terms[5]=1.0;                         /*   c5*1   */
        terms[6]=-terms[3]*arguments[i+cp_v]; /* 1/(c6*x) */
        terms[7]=-terms[4]*arguments[i+cp_v]; /* 1/(c7*y) */
        LeastSquaresAddTerms(matrix,vectors,terms,&(arguments[i+cp_v]),
            8UL,1UL);
      }
      /* Solve for LeastSquares Coefficients */
      status=GaussJordanElimination(matrix,vectors,8UL,1UL);
      matrix = RelinquishMagickMatrix(matrix, 8UL);
      if ( status == MagickFalse ) {
        coefficients = (double *) RelinquishMagickMemory(coefficients);
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
                    "InvalidArgument","%s : '%s'","Perspective",
                    "Unsolvable Matrix");
        return((double *) NULL);
      }
      /*
        Calculate 9'th coefficient! The ground-sky determination.
        What is sign of the 'ground' in r() denominator affine function?
        Just use any valid image coordinate (first control point) in
        destination for determination of what part of view is 'ground'.
      */
      coefficients[8] = coefficients[6]*arguments[cp_x]
                      + coefficients[7]*arguments[cp_y] + 1.0;
      coefficients[8] = (coefficients[8] < 0.0) ? -1.0 : +1.0;

      return(coefficients);
    }
    case PerspectiveProjectionDistortion:
    {
      /*
        Arguments: Perspective Coefficents (forward mapping)
      */
      if (number_arguments != 8) {
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
                     "InvalidArgument","%s : '%s'","PerspectiveProjection",
                     "Needs 8 coefficient values");
        return((double *) NULL);
      }
      InvertPerspectiveCoefficients(arguments, coefficients);
      /*
        Calculate 9'th coefficient! The ground-sky determination.
        What is sign of the 'ground' in r() denominator affine function?
        Just use any valid image cocodinate in destination for determination.
        For a forward mapped perspective the images 0,0 coord will map to
        c2,c5 in the distorted image, so set the sign of denominator of that.
      */
      coefficients[8] = coefficients[6]*arguments[2]
                           + coefficients[7]*arguments[5] + 1.0;
      coefficients[8] = (coefficients[8] < 0.0) ? -1.0 : +1.0;
      *method = PerspectiveDistortion;

      return(coefficients);
    }
    case BilinearDistortion:
    {
      /* Bilinear Distortion
            v = c0*x + c1*y + c2*x*y + c3;
         for each 'value' given

         Input Arguments are sets of control points...
         For Distort Images    u,v, x,y  ...
         For Sparse Gradients  x,y, r,g,b  ...

         Bilinear Distortion Notes...
           + This a reversed mapped distortion
           + Can map source orthogonal rectangles to non-planar quadraterials
           + may be used for generating forward mapped 'grid' interpolation
      */
      double
        **matrix,
        **vectors,
        terms[4];

      MagickBooleanType
        status;

      /* create matrix, and a fake vectors matrix */
      matrix = AcquireMagickMatrix(4UL,4UL);
      vectors = (double **) AcquireQuantumMemory(number_values,sizeof(*vectors));
      if (matrix == (double **) NULL || vectors == (double **) NULL)
      {
        matrix       = RelinquishMagickMatrix(matrix, 4UL);
        vectors      = (double **) RelinquishMagickMemory(vectors);
        coefficients = (double *) RelinquishMagickMemory(coefficients);
        (void) ThrowMagickException(exception,GetMagickModule(),
                ResourceLimitError,"MemoryAllocationFailed",
                "%s", "DistortCoefficients");
        return((double *) NULL);
      }
      /* fake a number_values x3 vectors matrix from coefficients array */
      for (i=0; i < number_values; i++)
        vectors[i] = &(coefficients[i*4]);
      /* Add given control point pairs for least squares solving */
      for (i=0; i < number_arguments; i+=cp_size) {
        terms[0] = arguments[i+cp_x];   /*  x  */
        terms[1] = arguments[i+cp_y];   /*  y  */
        terms[2] = terms[0]*terms[1];   /* x*y */
        terms[3] = 1;                   /*  1  */
        LeastSquaresAddTerms(matrix,vectors,terms,
             &(arguments[i+cp_values]),4UL,number_values);
      }
      /* Solve for LeastSquares Coefficients */
      status=GaussJordanElimination(matrix,vectors,4UL,number_values);
      matrix  = RelinquishMagickMatrix(matrix, 4UL);
      vectors = (double **) RelinquishMagickMemory(vectors);
      if ( status == MagickFalse ) {
        coefficients = (double *) RelinquishMagickMemory(coefficients);
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
                     "InvalidArgument","%s : '%s'","Bilinear",
                     "Unsolvable Matrix");
        return((double *) NULL);
      }
      return(coefficients);
    }
    case PolynomialDistortion:
    {
      /* Polynomial Distortion

         First two coefficents are used to hole global polynomal information
           c0 = Order of the polynimial being created
           c1 = number_of_terms in one polynomial equation

         Rest of the coefficients map to the equations....
            v = c0 + c1*x + c2*y + c3*x*y + c4*x^2 + c5*y^2 + c6*x^3 + ...
         for each 'value' (number_values of them) given.
         As such total coefficients =  2 + number_terms * number_values

         Input Arguments are sets of control points...
         For Distort Images    order  [u,v, x,y] ...
         For Sparse Gradients  order  [x,y, r,g,b] ...

         Polynomial Distortion Notes...
           + UNDER DEVELOPMENT -- Do not expect this to remain as is.
           + Currently polynomial is a reversed mapped distortion.
           + Should be used to generate a 'grid' of bilinear distortions
             so that it will be 'forward' mapped.
           + Order 1.5 is fudged to map into a bilinear distortion.
      */
      double
        **matrix,
        **vectors,
        *terms;

      unsigned long
        nterms;   /* number of polynomial terms per number_values */

      register long
        j;

      MagickBooleanType
        status;

      /* first two coefficients hold polynomial order information */
      coefficients[0] = arguments[0];
      coefficients[1] = (double) poly_number_terms(arguments[0]);
      nterms = (unsigned long) coefficients[1];

      /* create matrix, a fake vectors matrix, and least sqs terms */
      matrix = AcquireMagickMatrix(nterms,nterms);
      vectors = (double **) AcquireQuantumMemory(number_values,sizeof(*vectors));
      terms = (double *) AcquireQuantumMemory(nterms, sizeof(*terms));
      if (matrix  == (double **) NULL ||
          vectors == (double **) NULL ||
          terms   == (double *) NULL )
      {
        matrix       = RelinquishMagickMatrix(matrix, nterms);
        vectors      = (double **) RelinquishMagickMemory(vectors);
        terms        = (double *) RelinquishMagickMemory(terms);
        coefficients = (double *) RelinquishMagickMemory(coefficients);
        (void) ThrowMagickException(exception,GetMagickModule(),
                ResourceLimitError,"MemoryAllocationFailed",
                "%s", "DistortCoefficients");
        return((double *) NULL);
      }
      /* fake a number_values x3 vectors matrix from coefficients array */
      for (i=0; i < number_values; i++)
        vectors[i] = &(coefficients[2+i*nterms]);
      /* Add given control point pairs for least squares solving */
      for (i=0; i < number_arguments; i+=cp_size) {
        for (j=0; j < (long) nterms; j++)
          terms[j] = poly_basis_fn(j,arguments[i+cp_x],arguments[i+cp_y]);
        LeastSquaresAddTerms(matrix,vectors,terms,
             &(arguments[i+cp_values]),nterms,number_values);
      }
      terms = (double *) RelinquishMagickMemory(terms);
      /* Solve for LeastSquares Coefficients */
      status=GaussJordanElimination(matrix,vectors,nterms,number_values);
      matrix  = RelinquishMagickMatrix(matrix, nterms);
      vectors = (double **) RelinquishMagickMemory(vectors);
      if ( status == MagickFalse ) {
        coefficients = (double *) RelinquishMagickMemory(coefficients);
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
                     "InvalidArgument","%s : '%s'","Polynomial",
                     "Unsolvable Matrix");
        return((double *) NULL);
      }
      return(coefficients);
    }
    case ArcDistortion:
    {
      /* Arc Distortion
         Args: arc_width  rotate  top_edge_radius  bottom_edge_radius
         All but first argument are optional
            arc_width      The angle over which to arc the image side-to-side
            rotate         Angle to rotate image from vertical center
            top_radius     Set top edge of source image at this radius
            bottom_radius  Set bootom edge to this radius (radial scaling)

         By default, if the radii arguments are nor provided the image radius
         is calculated so the horizontal center-line is fits the given arc
         without scaling.

         The output image size is ALWAYS adjusted to contain the whole image,
         and an offset is given to position image relative to the 0,0 point of
         the origin, allowing users to use relative positioning onto larger
         background (via -flatten).

         The arguments are converted to these coefficients
            c0: angle for center of source image
            c1: angle scale for mapping to source image
            c2: radius for top of source image
            c3: radius scale for mapping source image
            c4: centerline of arc within source image

         Note the coefficients use a center angle, so asymptotic join is
         furthest from both sides of the source image. This also means that
         for arc angles greater than 360 the sides of the image will be
         trimmed equally.

         Arc Distortion Notes...
           + Does not use a set of CPs
           + Will only work with Image Distortion
           + Can not be used for generating a sparse gradient (interpolation)
      */
      if ( number_arguments >= 1 && arguments[0] < MagickEpsilon ) {
        coefficients = (double *) RelinquishMagickMemory(coefficients);
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
                     "InvalidArgument","%s : '%s'", "Arc",
                     "Arc Angle Too Small");
        return((double *) NULL);
      }
      if ( number_arguments >= 3 && arguments[2] < MagickEpsilon ) {
        coefficients = (double *) RelinquishMagickMemory(coefficients);
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
                     "InvalidArgument","%s : '%s'", "Arc",
                     "Outer Radius Too Small");
        return((double *) NULL);
      }
      coefficients[0] = -MagickPI2;   /* -90, place at top! */
      if ( number_arguments >= 1 )
        coefficients[1] = DegreesToRadians(arguments[0]);
      else
        coefficients[1] = MagickPI2;   /* zero arguments - center is at top */
      if ( number_arguments >= 2 )
        coefficients[0] += DegreesToRadians(arguments[1]);
      coefficients[0] /= Magick2PI;  /* normalize radians */
      coefficients[0] -= MagickRound(coefficients[0]);
      coefficients[0] *= Magick2PI;  /* de-normalize back to radians */
      coefficients[3] = (double)image->rows-1;
      coefficients[2] = (double)image->columns/coefficients[1] + coefficients[3]/2.0;
      if ( number_arguments >= 3 ) {
        if ( number_arguments >= 4 )
          coefficients[3] = arguments[2] - arguments[3];
        else
          coefficients[3] *= arguments[2]/coefficients[2];
        coefficients[2] = arguments[2];
      }
      coefficients[4] = ((double)image->columns-1.0)/2.0;

      return(coefficients);
    }
    case PolarDistortion:
    case DePolarDistortion:
    {
      /* (De)Polar Distortion   (same set of arguments)
         Args:  Rmax, Rmin,  Xcenter,Ycenter,  Afrom,Ato
         DePolar can also have the extra arguments of Width, Height

         Coefficients 0 to 5 is the sanatized version first 6 input args
         Coefficient 6  is the angle to coord ratio  and visa-versa
         Coefficient 7  is the radius to coord ratio and visa-versa

         WARNING: It is posible for  Radius max<min  and/or  Angle from>to
      */
      if ( number_arguments == 3 || number_arguments == 5
          || ( number_arguments > 6 && *method == PolarDistortion )
          || number_arguments > 8 ) {
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
               "InvalidArgument", "%s : number of arguments",
               *method == PolarDistortion ? "Polar" : "DePolar");
        coefficients=(double *) RelinquishMagickMemory(coefficients);
        return((double *) NULL);
      }
      /* Rmin  - usally 0 */
      coefficients[1] = number_arguments >= 2 ? arguments[1] : 0.0;
      /* Rmax -  if 0 calculate appropriate value */
      if ( number_arguments >= 1 )
        coefficients[0] = arguments[0];
      else
        coefficients[0] = 0.0;
      /* Center X,Y */
      if ( number_arguments >= 4 ) {
        coefficients[2] = arguments[2];
        coefficients[3] = arguments[3];
      }
      else { /* center of actual image */
        coefficients[2] = (double)(image->columns-1)/2.0+image->page.x;
        coefficients[3] = (double)(image->rows-1)/2.0+image->page.y;
      }
      /* Angle from,to - about polar center 0 is downward */
      if ( number_arguments == 6 ) {
        coefficients[4] = DegreesToRadians(arguments[4]);
        coefficients[5] = DegreesToRadians(arguments[5]);
        if ( fabs(coefficients[4]-coefficients[5]) < MagickEpsilon )
          coefficients[5] += Magick2PI; /* angle 0 to 0 is a full circle */
      }
      else {
        coefficients[4] = -MagickPI;
        coefficients[5] = MagickPI;
      }
      /* if radius is not set... */
      if ( coefficients[0] < MagickEpsilon ) {
        /* Use closest edge */
        if ( fabs(coefficients[0]) < MagickEpsilon ) {
          coefficients[0]=MagickMin(fabs(coefficients[2]-image->page.x),
                                    fabs(coefficients[3]-image->page.y));
          coefficients[0]=MagickMin(coefficients[0],
                       fabs(coefficients[2]-image->page.x-image->columns));
          coefficients[0]=MagickMin(coefficients[0],
                       fabs(coefficients[3]-image->page.y-image->rows));
        }
        /* furthest diagonal */
        if ( fabs(-1.0-coefficients[0]) < MagickEpsilon ) {
          double rx,ry;
          rx = coefficients[2]-image->page.x;
          ry = coefficients[3]-image->page.y;
          coefficients[0] = rx*rx+ry*ry;
          ry = coefficients[3]-image->page.y-image->rows;
          coefficients[0] = MagickMax(coefficients[0],rx*rx+ry*ry);
          rx = coefficients[2]-image->page.x-image->columns;
          coefficients[0] = MagickMax(coefficients[0],rx*rx+ry*ry);
          ry = coefficients[3]-image->page.y;
          coefficients[0] = MagickMax(coefficients[0],rx*rx+ry*ry);
          coefficients[0] = sqrt(coefficients[0]);
        }
      }
      /* if radius is still 0 or negative, error */
      if ( coefficients[0] < MagickEpsilon ||
              coefficients[1] < -MagickEpsilon ) {
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
               "InvalidArgument", "%s : Invalid Radius",
               *method == PolarDistortion ? "Polar" : "DePolar");
        coefficients=(double *) RelinquishMagickMemory(coefficients);
        return((double *) NULL);
      }
      /* converstion ratios */
      if ( *method == PolarDistortion ) {
        coefficients[6]=(double) image->columns/(coefficients[5]-coefficients[4]);
        coefficients[7]=(double) image->rows/(coefficients[0]-coefficients[1]);
      }
      else { /* *method == DePolarDistortion */
        coefficients[6]=(coefficients[5]-coefficients[4])/image->columns;
        coefficients[7]=(coefficients[0]-coefficients[1])/image->rows;
      }
      return(coefficients);
    }
    case BarrelDistortion:
    {
      /* Barrel Distortion
           Rs=(A*Rd^3 + B*Rd^2 + C*Rd + D)*Rd

        Where Rd is the normalized radius from corner to middle of image
        Input Arguments:  A, B, C, [ D ]

        Returns
          A, B, C, D Xcenter, Ycenter, Rscale
      */
      if ( number_arguments != 3 && number_arguments != 4 &&
           number_arguments != 6 ) {
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
               "InvalidArgument",
               "%s : 'require A,B,C,D coefficients + optional x,y center'",
               "Barrel" );
        coefficients=(double *) RelinquishMagickMemory(coefficients);
        return((double *) NULL);
      }
      coefficients[0] = arguments[0];
      coefficients[1] = arguments[1];
      coefficients[2] = arguments[2];
      if ( number_arguments == 3 )
        coefficients[3] = 1.0 - arguments[0] - arguments[1] - arguments[2];
      else
        coefficients[3] = arguments[3];
      if ( number_arguments == 6 ) {
        coefficients[4] = arguments[4];
        coefficients[5] = arguments[5];
      }
      else {
        coefficients[4] = ((double)image->columns-1)/2.0 + image->page.x;
        coefficients[5] = ((double)image->rows-1)/2.0    + image->page.y;
      }
      coefficients[6] = 2.0/MagickMin((double) image->columns,(double) image->rows);
      return(coefficients);
    }
    case ShepardsDistortion:
    case VoronoiColorInterpolate:
    {
      /* Shepards Distortion  input arguments are the coefficents!
         Args:  power  u1,v1, x1,y1, ...
      */
      if ( number_arguments%cp_size != 0 ||
           number_arguments < cp_size ) {
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
               "InvalidArgument",
               "%s : 'require at least %ld CPs of %ld values'",
               "Shepards", 1UL, cp_size);
        coefficients=(double *) RelinquishMagickMemory(coefficients);
        return((double *) NULL);
      }
      return(coefficients);
    }
    default:
      break;
  }
  /* you should never reach this point */
  assert(! "No Method Handler"); /* just fail assertion */
  return((double *) NULL);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   D i s t o r t I m a g e                                                   %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  DistortImage() distorts an image using various distortion methods, by
%  mapping color lookups of the source image to a new destination image
%  usally of the same size as the source image, unless 'bestfit' is set to
%  true.
%
%  If 'bestfit' is enabled, and distortion allows it, the destination image is
%  adjusted to ensure the whole source 'image' will just fit within the final
%  destination image, which will be sized and offset accordingly.  Also in
%  many cases the virtual offset of the source image will be taken into
%  account in the mapping.
%
%  If the '-verbose' control option has been set print to standard error the
%  equicelent '-fx' formula with coefficients for the function, if practical.
%
%  The format of the DistortImage() method is:
%
%      Image *DistortImage(const Image *image,const DistortImageMethod method,
%        const unsigned long number_arguments,const double *arguments,
%        MagickBooleanType bestfit, ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image: the image to be distorted.
%
%    o method: the method of image distortion.
%
%    o number_arguments: the number of arguments given.
%
%    o arguments: the arguments for this distortion method.
%
%    o bestfit: Attempt to 'bestfit' the size of the resulting image.
%         This also forces the resulting image to be a 'layered' virtual
%         canvas image.  Can be overridden using 'distort:viewport' setting.
%
%    o exception: Return any errors or warnings in this structure
%
% Specific Distortion notes...
%
%  ArcDistortion will always ignore source image offset, and always 'bestfit'
%  the destination image with the top left corner offset relative to the polar
%  mapping center.
%
%  Bilinear has no simple inverse mapping so will not allow 'bestfit' style
%  of image distortion.
%
%  Affine, Perspective, and Bilinear, will do least squares fitting of the
%  distrotion when more than the minimum number of control point pairs are
%  provided.
%
%  Perspective, and Bilinear, will fall back to a Affine distortion when less
%  that 4 control point pairs are provided. While Affine distortions will let
%  you use any number of control point pairs, that is Zero pairs is a No-Op
%  (viewport only) distrotion, one pair is a translation and two pairs of
%  control points will do a scale-rotate-translate, without any shearing.
%
*/

static ResampleFilter **DestroyResampleFilterThreadSet(ResampleFilter **filter)
{
  register long
    i;

  assert(filter != (ResampleFilter **) NULL);
  for (i=0; i < (long) GetCacheViewMaximumThreads(); i++)
    if (filter[i] != (ResampleFilter *) NULL)
      filter[i]=DestroyResampleFilter(filter[i]);
  return((ResampleFilter **) RelinquishMagickMemory(filter));
}

static ResampleFilter **AcquireResampleFilterThreadSet(const Image *image,
  ExceptionInfo *exception)
{
  register long
    i;

  ResampleFilter
    **filter;

  filter=(ResampleFilter **) AcquireQuantumMemory(GetCacheViewMaximumThreads(),
    sizeof(*filter));
  if (filter == (ResampleFilter **) NULL)
    return((ResampleFilter **) NULL);
  (void) ResetMagickMemory(filter,0,GetCacheViewMaximumThreads()*
    sizeof(*filter));
  for (i=0; i < (long) GetCacheViewMaximumThreads(); i++)
  {
    filter[i]=AcquireResampleFilter(image,exception);
    if (filter[i] == (ResampleFilter *) NULL)
      return(DestroyResampleFilterThreadSet(filter));
  }
  return(filter);
}

MagickExport Image *DistortImage(const Image *image,DistortImageMethod method,
  const unsigned long number_arguments,const double *arguments,
  MagickBooleanType bestfit,ExceptionInfo *exception)
{
#define DistortImageTag  "Distort/Image"

  double
    *coefficients,
    output_scaling;

  Image
    *distort_image;

  RectangleInfo
    geometry;  /* geometry of the distorted space viewport */

  MagickBooleanType
    viewport_given;

  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"%s",image->filename);
  assert(exception != (ExceptionInfo *) NULL);
  assert(exception->signature == MagickSignature);

  /*
    Convert input arguments (usally as control points for reverse mapping)
    into mapping coefficients to apply the distortion.

    Note that some distortions are mapped to other distortions,
    and as such do not require specific code after this point.
  */
  coefficients = GenerateCoefficients(image, &method, number_arguments,
      arguments, 0, exception);
  if ( coefficients == (double *) NULL )
    return((Image *) NULL);

  /*
    Determine the size and offset for a 'bestfit' destination.
    Usally the four corners of the source image is enough.
  */

  /* default output image bounds, when no 'bestfit' is requested */
  geometry.width=image->columns;
  geometry.height=image->rows;
  geometry.x=0;
  geometry.y=0;

  if ( method == ArcDistortion ) {
    /* always use the 'best fit' viewport */
    bestfit = MagickTrue;
  }

  /* Work out the 'best fit', (required for ArcDistortion) */
  if ( bestfit ) {
    PointInfo
      s,d,min,max;

    s.x=s.y=min.x=max.x=min.y=max.y=0.0;   /* keep compiler happy */

/* defines to figure out the bounds of the distorted image */
#define InitalBounds(p) \
{ \
  /* printf("%lg,%lg -> %lg,%lg\n", s.x,s.y, d.x,d.y); */ \
  min.x = max.x = p.x; \
  min.y = max.y = p.y; \
}
#define ExpandBounds(p) \
{ \
  /* printf("%lg,%lg -> %lg,%lg\n", s.x,s.y, d.x,d.y); */ \
  min.x = MagickMin(min.x,p.x); \
  max.x = MagickMax(max.x,p.x); \
  min.y = MagickMin(min.y,p.y); \
  max.y = MagickMax(max.y,p.y); \
}

    switch (method)
    {
      case AffineDistortion:
      { double inverse[6];
        InvertAffineCoefficients(coefficients, inverse);
        s.x = (double) image->page.x;
        s.y = (double) image->page.y;
        d.x = inverse[0]*s.x+inverse[1]*s.y+inverse[2];
        d.y = inverse[3]*s.x+inverse[4]*s.y+inverse[5];
        InitalBounds(d);
        s.x = (double) image->page.x+image->columns-1;
        s.y = (double) image->page.y;
        d.x = inverse[0]*s.x+inverse[1]*s.y+inverse[2];
        d.y = inverse[3]*s.x+inverse[4]*s.y+inverse[5];
        ExpandBounds(d);
        s.x = (double) image->page.x;
        s.y = (double) image->page.y+image->rows-1;
        d.x = inverse[0]*s.x+inverse[1]*s.y+inverse[2];
        d.y = inverse[3]*s.x+inverse[4]*s.y+inverse[5];
        ExpandBounds(d);
        s.x = (double) image->page.x+image->columns-1;
        s.y = (double) image->page.y+image->rows-1;
        d.x = inverse[0]*s.x+inverse[1]*s.y+inverse[2];
        d.y = inverse[3]*s.x+inverse[4]*s.y+inverse[5];
        ExpandBounds(d);
        break;
      }
      case PerspectiveDistortion:
      { double inverse[8], scale;
        InvertPerspectiveCoefficients(coefficients, inverse);
        s.x = (double) image->page.x;
        s.y = (double) image->page.y;
        scale=inverse[6]*s.x+inverse[7]*s.y+1.0;
        scale=1.0/(  (fabs(scale) <= MagickEpsilon) ? 1.0 : scale );
        d.x = scale*(inverse[0]*s.x+inverse[1]*s.y+inverse[2]);
        d.y = scale*(inverse[3]*s.x+inverse[4]*s.y+inverse[5]);
        InitalBounds(d);
        s.x = (double) image->page.x+image->columns-1;
        s.y = (double) image->page.y;
        scale=inverse[6]*s.x+inverse[7]*s.y+1.0;
        scale=1.0/(  (fabs(scale) <= MagickEpsilon) ? 1.0 : scale );
        d.x = scale*(inverse[0]*s.x+inverse[1]*s.y+inverse[2]);
        d.y = scale*(inverse[3]*s.x+inverse[4]*s.y+inverse[5]);
        ExpandBounds(d);
        s.x = (double) image->page.x;
        s.y = (double) image->page.y+image->rows-1;
        scale=inverse[6]*s.x+inverse[7]*s.y+1.0;
        scale=1.0/(  (fabs(scale) <= MagickEpsilon) ? 1.0 : scale );
        d.x = scale*(inverse[0]*s.x+inverse[1]*s.y+inverse[2]);
        d.y = scale*(inverse[3]*s.x+inverse[4]*s.y+inverse[5]);
        ExpandBounds(d);
        s.x = (double) image->page.x+image->columns-1;
        s.y = (double) image->page.y+image->rows-1;
        scale=inverse[6]*s.x+inverse[7]*s.y+1.0;
        scale=1.0/(  (fabs(scale) <= MagickEpsilon) ? 1.0 : scale );
        d.x = scale*(inverse[0]*s.x+inverse[1]*s.y+inverse[2]);
        d.y = scale*(inverse[3]*s.x+inverse[4]*s.y+inverse[5]);
        ExpandBounds(d);
        break;
      }
      case ArcDistortion:
      { double a, ca, sa;
        /* Forward Map Corners */
        a = coefficients[0]-coefficients[1]/2; ca = cos(a); sa = sin(a);
        d.x = coefficients[2]*ca;
        d.y = coefficients[2]*sa;
        InitalBounds(d);
        d.x = (coefficients[2]-coefficients[3])*ca;
        d.y = (coefficients[2]-coefficients[3])*sa;
        ExpandBounds(d);
        a = coefficients[0]+coefficients[1]/2; ca = cos(a); sa = sin(a);
        d.x = coefficients[2]*ca;
        d.y = coefficients[2]*sa;
        ExpandBounds(d);
        d.x = (coefficients[2]-coefficients[3])*ca;
        d.y = (coefficients[2]-coefficients[3])*sa;
        ExpandBounds(d);
        /* Orthogonal points along top of arc */
        for( a=ceil((coefficients[0]-coefficients[1]/2.0)/MagickPI2)*MagickPI2;
               a<(coefficients[0]+coefficients[1]/2.0); a+=MagickPI2 ) {
          ca = cos(a); sa = sin(a);
          d.x = coefficients[2]*ca;
          d.y = coefficients[2]*sa;
          ExpandBounds(d);
        }
        /*
          Convert the angle_to_width and radius_to_height
          to appropriate scaling factors, to allow faster processing
          in the mapping function.
        */
        coefficients[1] = Magick2PI*image->columns/coefficients[1];
        coefficients[3] = (double)image->rows/coefficients[3];
        break;
      }
      case PolarDistortion:
      {
        if (number_arguments < 2)
          coefficients[2] = coefficients[3] = 0.0;
        min.x = coefficients[2]-coefficients[0];
        max.x = coefficients[2]+coefficients[0];
        min.y = coefficients[3]-coefficients[0];
        max.y = coefficients[3]+coefficients[0];
        coefficients[7] = 1.0;
        break;
      }
      case DePolarDistortion:
      {
        min.x = min.y = 0;
        max.y = coefficients[0]-coefficients[1];
        max.x = (coefficients[0]+coefficients[1])*MagickPI;
        /* conversion ratios calculated later */
        break;
      }
      case ShepardsDistortion:
      case BilinearDistortion:
      case PolynomialDistortion:
      case BarrelDistortion:
      default:
        /* no bestfit available for this distortion */
        bestfit = MagickFalse;
        break;
    }
    /* Set the output image geometry to calculated 'bestfit' */
    if ( bestfit ) {
      geometry.x = (long) floor(min.x-MagickEpsilon);
      geometry.y = (long) floor(min.y-MagickEpsilon);
      geometry.width=(unsigned long) ceil(max.x-geometry.x+MagickEpsilon);
      geometry.height=(unsigned long) ceil(max.y-geometry.y+MagickEpsilon);
    }
    /* now that we have a new size lets fit distortion to it exactly */
    if ( method == DePolarDistortion ) {
      coefficients[6]=(coefficients[5]-coefficients[4])/geometry.width;
      coefficients[7]=(coefficients[0]-coefficients[1])/geometry.height;
    }
  }

  /* The user provided a 'viewport' expert option which may
     overrides some parts of the current output image geometry.
     For ArcDistortion, this also overrides its default 'bestfit' setting.
  */
  { const char *artifact=GetImageArtifact(image,"distort:viewport");
    viewport_given = MagickFalse;
    if ( artifact != (const char *) NULL ) {
      (void) ParseAbsoluteGeometry(artifact,&geometry);
      viewport_given = MagickTrue;
    }
  }

  /* Verbose output */
  if ( GetImageArtifact(image,"verbose") != (const char *) NULL ) {
    register long
       i;
    char image_gen[255];
    const char *lookup = "p{xx-page.x,yy-page.x}";
    image_gen[0] = '\0';

    /* Set destination image size and virtual offset */
    if ( bestfit || viewport_given
         || PolarDistortion || DePolarDistortion ) {
      (void) sprintf(image_gen, "  -size %lux%lu -page %+ld%+ld xc: +insert \\\n",
            geometry.width, geometry.height, geometry.x, geometry.y );
      lookup = "v.p{xx-v.page.x,yy-v.page.x}";
    }

    switch (method) {
      case AffineDistortion:
      {
        double *inverse;

        inverse = (double *) AcquireQuantumMemory(6,sizeof(*inverse));
        if (inverse == (double *) NULL) {
          coefficients = (double *) RelinquishMagickMemory(coefficients);
          (void) ThrowMagickException(exception,GetMagickModule(),
                  ResourceLimitError,"MemoryAllocationFailed",
                  "%s", "DistortImages");
          return((Image *) NULL);
        }
        InvertAffineCoefficients(coefficients, inverse);
        CoefficientsToAffineArgs(inverse);
        fprintf(stderr, "Affine Projection:\n");
        fprintf(stderr, "  -distort AffineProjection \\\n      '");
        for (i=0; i<5; i++)
          fprintf(stderr, "%lg,", inverse[i]);
        fprintf(stderr, "%lg'\n", inverse[5]);
        inverse = (double *) RelinquishMagickMemory(inverse);

        fprintf(stderr, "Affine Distort, FX Equivelent:\n");
        fprintf(stderr, "%s", image_gen);
        fprintf(stderr, "  -fx 'ii=i+page.x; jj=j+page.y;\n");
        fprintf(stderr, "       xx=%+lf*ii %+lf*jj %+lf;\n",
            coefficients[0], coefficients[1], coefficients[2]);
        fprintf(stderr, "       yy=%+lf*ii %+lf*jj %+lf;\n",
            coefficients[3], coefficients[4], coefficients[5]);
        fprintf(stderr, "       %s'\n", lookup);

        break;
      }

      case PerspectiveDistortion:
      {
        double *inverse;

        inverse = (double *) AcquireQuantumMemory(8,sizeof(*inverse));
        if (inverse == (double *) NULL) {
          coefficients = (double *) RelinquishMagickMemory(coefficients);
          (void) ThrowMagickException(exception,GetMagickModule(),
                  ResourceLimitError,"MemoryAllocationFailed",
                  "%s", "DistortCoefficients");
          return((Image *) NULL);
        }
        InvertPerspectiveCoefficients(coefficients, inverse);
        fprintf(stderr, "Perspective Projection:\n");
        fprintf(stderr, "  -distort PerspectiveProjection \\\n      '");
        for (i=0; i<4; i++)
          fprintf(stderr, "%lg,", inverse[i]);
        fprintf(stderr, "\n       ");
        for (; i<7; i++)
          fprintf(stderr, "%lg,", inverse[i]);
        fprintf(stderr, "%lg'\n", inverse[7]);
        inverse = (double *) RelinquishMagickMemory(inverse);

        fprintf(stderr, "Perspective Distort, FX Equivelent:\n");
        fprintf(stderr, "%s", image_gen);
        fprintf(stderr, "  -fx 'ii=i+page.x; jj=j+page.y;\n");
        fprintf(stderr, "       rr=%+lf*ii %+lf*jj + 1;\n",
            coefficients[6], coefficients[7]);
        fprintf(stderr, "       xx=(%+lf*ii %+lf*jj %+lf)/rr;\n",
            coefficients[0], coefficients[1], coefficients[2]);
        fprintf(stderr, "       yy=(%+lf*ii %+lf*jj %+lf)/rr;\n",
            coefficients[3], coefficients[4], coefficients[5]);
        fprintf(stderr, "       rr%s0 ? %s : blue'\n",
            coefficients[8] < 0 ? "<" : ">", lookup);
        break;
      }

      case BilinearDistortion:
        fprintf(stderr, "Bilinear Distort, FX Equivelent:\n");
        fprintf(stderr, "%s", image_gen);
        fprintf(stderr, "  -fx 'ii=i+page.x; jj=j+page.y;\n");
        fprintf(stderr, "       xx=%+lf*ii %+lf*jj %+lf*ii*jj %+lf;\n",
            coefficients[0], coefficients[1], coefficients[2], coefficients[3]);
        fprintf(stderr, "       yy=%+lf*ii %+lf*jj %+lf*ii*jj %+lf;\n",
            coefficients[4], coefficients[5], coefficients[6], coefficients[7]);
        fprintf(stderr, "       %s'\n", lookup);
        break;

      case PolynomialDistortion:
      {
        unsigned long nterms = (unsigned long) coefficients[1];
        fprintf(stderr, "Polynomial (order %lg, terms %lu), FX Equivelent\n",
                       coefficients[0], nterms);
        fprintf(stderr, "%s", image_gen);
        fprintf(stderr, "  -fx 'ii=i+page.x; jj=j+page.y;\n");
        fprintf(stderr, "       xx =");
        for (i=0; i<(long) nterms; i++) {
          if ( i != 0 && i%4 == 0 ) fprintf(stderr, "\n         ");
          fprintf(stderr, " %+lf%s", coefficients[2+i],
               poly_basis_str(i));
        }
        fprintf(stderr, ";\n       yy =");
        for (i=0; i<(long) nterms; i++) {
          if ( i != 0 && i%4 == 0 ) fprintf(stderr, "\n         ");
          fprintf(stderr, " %+lf%s", coefficients[2+i+nterms],
               poly_basis_str(i));
        }
        fprintf(stderr, ";\n       %s'\n", lookup);
        break;
      }
      case ArcDistortion:
      {
        fprintf(stderr, "Arc Distort, Internal Coefficients:\n");
        for ( i=0; i<5; i++ )
          fprintf(stderr, "  c%ld=%-8lg\n", i, coefficients[i]);
        fprintf(stderr, "Arc Distort, FX Equivelent:\n");
        fprintf(stderr, "%s", image_gen);
        fprintf(stderr, "  -fx 'ii=i+page.x; jj=j+page.y;\n");
        fprintf(stderr, "       xx=(atan2(jj,ii)%+lg)/(2*pi);\n",
                                  -coefficients[0]);
        fprintf(stderr, "       xx=xx-round(xx);\n");
        fprintf(stderr, "       xx=xx*%lg %+lg;\n",
                            coefficients[1], coefficients[4]);
        fprintf(stderr, "       yy=hypot(ii,jj);\n");
        fprintf(stderr, "       yy=(%lg - yy) * %lg;\n",
                            coefficients[2], coefficients[3]);
        fprintf(stderr, "       v.p{xx,yy}'\n");
        break;
      }
/*
      case PolarDistortion:
      {
        fprintf(stderr, "Polar Distort, Internal Coefficents\n");
        for ( i=0; i<8; i++ )
          fprintf(stderr, "  c%lu=%-8lg\n", i, coefficients[i]);
        fprintf(stderr, "Polar Distort, FX Equivelent:\n");
        fprintf(stderr, "%s", image_gen);
        fprintf(stderr, "  -fx 'ii=i+page.x%+lg; jj=j+page.y%+lg;\n",
                         -coefficients[2], -coefficients[3]);
        fprintf(stderr, "       xx=(atan2(ii,jj)%+lg)/(2*pi);\n",
                         (coefficients[4]+coefficients[5])/2 );
        fprintf(stderr, "       xx=xx-round(xx);\n");
        fprintf(stderr, "       xx=xx*2*pi*%lg + (v.w-1)/2;\n",
                         coefficients[10] );
        fprintf(stderr, "       yy=(hypot(ii,jj)%+lg)*%lg;\n",
                         -coefficients[1], coefficients[11] );
        fprintf(stderr, "       v.p{xx,yy}'\n");
        break;
      }
*/
      case DePolarDistortion:
      {
        fprintf(stderr, "DePolar Distort, Internal Coefficents\n");
        for ( i=0; i<8; i++ )
          fprintf(stderr, "  c%ld=%-8lg\n", i, coefficients[i]);
        fprintf(stderr, "DePolar Distort, FX Equivelent:\n");
        fprintf(stderr, "%s", image_gen);
        fprintf(stderr, "  -fx 'xx=sin((i+page.x)*%lg %+lg) %+lg;\n",
                         coefficients[6], -coefficients[4], coefficients[2] );
        fprintf(stderr, "       jj=cos((j+page.y)*%lg %+lg) %+lg;\n",
                         coefficients[7], -coefficients[1], coefficients[3] );
        fprintf(stderr, "       v.p{xx,yy}'\n");
        break;
      }
      case BarrelDistortion:
      { double xc,yc;
        xc = ((double)image->columns-1.0)/2.0 + image->page.x;
        yc = ((double)image->rows-1.0)/2.0    + image->page.y;
        fprintf(stderr, "Barrel Distort, FX Equivelent:\n");
        fprintf(stderr, "%s", image_gen);
        if ( fabs(coefficients[4]-xc) < 0.1 && fabs(coefficients[5]-yc) < 0.1 )
          fprintf(stderr, "  -fx 'xc=(w-1)/2;  yc=(h-1)/2;\n");
        else
          fprintf(stderr, "  -fx 'xc=%lg;  yc=%lg;\n",
               coefficients[4], coefficients[5]);
        fprintf(stderr,
             "       ii=i-xc;  jj=j-yc;  rr=hypot(ii,jj)*2/min(w,h);\n");
        fprintf(stderr, "       fd=%lg*rr*rr*rr %+lg*rr*rr %+lg*rr %+lg;\n",
             coefficients[0], coefficients[1],
             coefficients[2], coefficients[3]);
        fprintf(stderr, "       p{fd*ii+xc,fd*jj+yc}'\n");
      }
      default:
        break;

    }
  }

  /* The user provided a 'scale' expert option will scale the
     output image size, by the factor given allowing for super-sampling
     of the distorted image space.  Any scaling factors must naturally
     be halved as a result.
  */
  { const char *artifact;
    artifact=GetImageArtifact(image,"distort:scale");
    output_scaling = 1.0;
    if (artifact != (const char *) NULL) {
      output_scaling = fabs(atof(artifact));
      geometry.width  *= output_scaling;
      geometry.height *= output_scaling;
      geometry.x      *= output_scaling;
      geometry.y      *= output_scaling;
      if ( output_scaling < 0.1 ) {
        coefficients = (double *) RelinquishMagickMemory(coefficients);
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
                "InvalidArgument","%s", "-set option:distort:scale" );
        return((Image *) NULL);
      }
      output_scaling = 1/output_scaling;
    }
  }
#define ScaleFilter(F,A,B,C,D) \
    ScaleResampleFilter( (F), \
      output_scaling*(A), output_scaling*(B), \
      output_scaling*(C), output_scaling*(D) )

  /*
    Initialize the distort image attributes.
  */
  distort_image=CloneImage(image,geometry.width,geometry.height,MagickTrue,
    exception);
  if (distort_image == (Image *) NULL)
    return((Image *) NULL);
  if (SetImageStorageClass(distort_image,DirectClass) == MagickFalse)
    { /* if image is ColorMapped - change it to DirectClass */
      InheritException(exception,&distort_image->exception);
      distort_image=DestroyImage(distort_image);
      return((Image *) NULL);
    }
  distort_image->page.x=geometry.x;
  distort_image->page.y=geometry.y;
  if (distort_image->background_color.opacity != OpaqueOpacity)
    distort_image->matte=MagickTrue;

  { /* ----- MAIN CODE -----
       Sample the source image to each pixel in the distort image.
     */
    long
      j,
      progress;

    MagickBooleanType
      status;

    ResampleFilter
      **resample_filter;

    ViewInfo
      **distort_view;

    status=MagickTrue;
    progress=0;
    resample_filter=AcquireResampleFilterThreadSet(image,exception);
    distort_view=AcquireCacheViewThreadSet(distort_image);
    #pragma omp parallel for
    for (j=0; j < (long) distort_image->rows; j++)
    {
      MagickPixelPacket
        pixel,    /* pixel color to assign to distorted image */
        invalid;  /* the color to assign when distort result is invalid */

      PointInfo
        d,s;  /* transform destination image x,y  to source image x,y */
      double
        validity;  /* how mathematically valid is this the mapping */

      register IndexPacket
        *indexes;

      register long
        i,
        id;

      register PixelPacket
        *q;


      id=GetCacheViewThreadId();
      q=SetCacheViewPixels(distort_view[id],0,j,distort_image->columns,1);
      if (q == (PixelPacket *) NULL)
        {
          status=MagickFalse;
          continue;
        }
      indexes=GetCacheViewIndexes(distort_view[id]);

      GetMagickPixelPacket(distort_image,&pixel);

      /* Define constant scaling vectors for Affine Distortions
        Other methods are either variable, or use interpolated lookup
      */
      switch (method)
      {
        case AffineDistortion:
          ScaleFilter( resample_filter[id],
            coefficients[0], coefficients[1],
            coefficients[3], coefficients[4] );
          break;
        default:
          break;
      }

      /* Initialize default pixel validity
      *    negative:         pixel is invalid  output 'matte_color'
      *    0.0 to 1.0:       antialiased, mix with resample output
      *    1.0 or greater:   use resampled output.
      */
      validity = 1.0;

      GetMagickPixelPacket(distort_image,&invalid);
      SetMagickPixelPacket(distort_image,&distort_image->matte_color,
        (IndexPacket *) NULL, &invalid);
      if (distort_image->colorspace == CMYKColorspace)
        ConvertRGBToCMYK(&invalid);   /* what about other color spaces? */

      for (i=0; i < (long) distort_image->columns; i++)
      {
        /* map pixel coordinate to distortion space coordinate */
        d.x = (double) (geometry.x+i+0.5)*output_scaling;
        d.y = (double) (geometry.y+j+0.5)*output_scaling;
        s = d;  /* default is a no-op mapping */
        switch (method)
        {
          case AffineDistortion:
          {
            s.x=coefficients[0]*d.x+coefficients[1]*d.y+coefficients[2];
            s.y=coefficients[3]*d.x+coefficients[4]*d.y+coefficients[5];
            /* Affine partial derivitives are constant -- set above */
            break;
          }
          case PerspectiveDistortion:
          {
            double
              p,q,r,abs_r,abs_c6,abs_c7,scale;
            /* perspective is a ratio of affines */
            p=coefficients[0]*d.x+coefficients[1]*d.y+coefficients[2];
            q=coefficients[3]*d.x+coefficients[4]*d.y+coefficients[5];
            r=coefficients[6]*d.x+coefficients[7]*d.y+1.0;
            /* Pixel Validity -- is it a 'sky' or 'ground' pixel */
            validity = (r*coefficients[8] < 0.0) ? 0.0 : 1.0;
            /* Determine horizon anti-aliase blending */
            abs_r = fabs(r)*2;
            abs_c6 = fabs(coefficients[6]);
            abs_c7 = fabs(coefficients[7]);
            if ( abs_c6 > abs_c7 ) {
              if ( abs_r < abs_c6 )
                validity = 0.5 - coefficients[8]*r/coefficients[6];
            }
            else if ( abs_r < abs_c7 )
              validity = .5 - coefficients[8]*r/coefficients[7];
            /* Perspective Sampling Point (if valid) */
            if ( validity > 0.0 ) {
              /* divide by r affine, for perspective scaling */
              scale = 1.0/r;
              s.x = p*scale;
              s.y = q*scale;
              /* Perspective Partial Derivatives or Scaling Vectors */
              scale *= scale;
              ScaleFilter( resample_filter[id],
                (r*coefficients[0] - p*coefficients[6])*scale,
                (r*coefficients[1] - p*coefficients[7])*scale,
                (r*coefficients[3] - q*coefficients[6])*scale,
                (r*coefficients[4] - q*coefficients[7])*scale );
            }
            break;
          }
          case BilinearDistortion:
          {
            s.x=coefficients[0]*d.x+coefficients[1]*d.y
                    +coefficients[2]*d.x*d.y+coefficients[3];
            s.y=coefficients[4]*d.x+coefficients[5]*d.y
                    +coefficients[6]*d.x*d.y+coefficients[7];
            /* Bilinear partial derivitives of scaling vectors */
            ScaleFilter( resample_filter[id],
                coefficients[0] + coefficients[2]*d.y,
                coefficients[1] + coefficients[2]*d.x,
                coefficients[4] + coefficients[6]*d.y,
                coefficients[5] + coefficients[6]*d.x );
            break;
          }
          case PolynomialDistortion:
          {
            register long
              k;
            long
              nterms=(long)coefficients[1];

            PointInfo
              du,dv; /* the du,dv vectors from dx,dy -- derivatives */

            s.x=s.y=du.x=du.y=dv.x=dv.y=0.0;
            for(k=0; k < nterms; k++) {
              s.x  += poly_basis_fn(k,d.x,d.y)*coefficients[2+k];
              du.x += poly_basis_dx(k,d.x,d.y)*coefficients[2+k];
              du.y += poly_basis_dy(k,d.x,d.y)*coefficients[2+k];
              s.y  += poly_basis_fn(k,d.x,d.y)*coefficients[2+k+nterms];
              dv.x += poly_basis_dx(k,d.x,d.y)*coefficients[2+k+nterms];
              dv.y += poly_basis_dy(k,d.x,d.y)*coefficients[2+k+nterms];
            }
            ScaleFilter( resample_filter[id], du.x,du.y,dv.x,dv.y );
            break;
          }
          case ArcDistortion:
          {
            /* what is the angle and radius in the destination image */
            s.x  = (atan2(d.y,d.x) - coefficients[0])/Magick2PI;
            s.x -= MagickRound(s.x);    /* angle */
            s.y  = hypot(d.x,d.y);       /* radius */

            /* Arc Distortion Partial Scaling Vectors
              Are derived by mapping the perpendicular unit vectors
              dR  and  dA*R*2PI  rather than trying to map dx and dy
              The results is a very simple orthogonal aligned ellipse.
            */
            if ( s.y > MagickEpsilon )
              ScaleFilter( resample_filter[id],
                  coefficients[1]/(Magick2PI*s.y), 0, 0, coefficients[3] );
            else
              ScaleFilter( resample_filter[id],
                  distort_image->columns*2, 0, 0, coefficients[3] );

            /* now scale the angle and radius for source image lookup point */
            s.x = s.x*coefficients[1] + coefficients[4] + image->page.x +0.5;
            s.y = (coefficients[2] - s.y) * coefficients[3] + image->page.y;
            break;
          }
          case PolarDistortion:
          { /* Rect/Cartesain/Cylinder to Polar View */
            d.x -= coefficients[2];
            d.y -= coefficients[3];
            s.x  = atan2(d.x,d.y) - (coefficients[4]+coefficients[5])/2;
            s.x /= Magick2PI;
            s.x -= MagickRound(s.x);
            s.x *= Magick2PI;      /* angle - relative to centerline */
            s.y  = hypot(d.x,d.y);  /* radius */

            /* Polar Scaling vectors are basis on mapping dR and dA vectors
               This results in very simple orthogonal vectors
            */
            if ( s.y > MagickEpsilon )
              ScaleFilter( resample_filter[id],
                  coefficients[6]/(Magick2PI*s.y), 0, 0, coefficients[7] );
            else
              ScaleFilter( resample_filter[id],
                  distort_image->columns*2, 0, 0, coefficients[7] );

            /* now finish mapping radius/angle to source x,y coords */
            s.x = s.x*coefficients[6] + (double)image->columns/2.0
                            + image->page.x;
            s.y = (s.y - coefficients[1])*coefficients[7] + image->page.y+0.5;
            break;
          }
          case DePolarDistortion:
          { /* Polar to Cylindical  */
            /* ignore all destination virtual offsets */
            d.x = ((double)i+0.5)*output_scaling*coefficients[6]-coefficients[4];
            d.y = ((double)j+0.5)*output_scaling*coefficients[7]-coefficients[1];
            s.x = d.y*sin(d.x) + coefficients[2];
            s.y = d.y*cos(d.x) + coefficients[3];
            /* derivatives are usless - better to use SuperSampling */
            break;
          }
          case BarrelDistortion:
          {
            double r,f,g;
            /* recalculate the point in distort space (relative to center) */
            d.x -= coefficients[4];   d.y -= coefficients[5];
            r = sqrt(d.x*d.x+d.y*d.y)*coefficients[6];
            f = ((coefficients[0]*r + coefficients[1])*r + coefficients[2])*r
                    + coefficients[3];
            /* set scaling derivatives */
            g = coefficients[6]/r;
            g *= g;
            g *= (3*coefficients[0]*r + 2*coefficients[1])*r + coefficients[2];
            ScaleFilter( resample_filter[id],
                 g*d.x*d.x + f, g*d.x*d.y,
                 g*d.x*d.y,     g*d.y*d.y + f );
            /* convert destination point to source image location */
            s.x = f*d.x + coefficients[4];
            s.y = f*d.y + coefficients[5];
            break;
          }
          case ShepardsDistortion:
          { /* Shepards Method, or Inverse Weighted Distance for
              displacement around the destination image control points
              The input arguments are the coefficents to the function.
            */
            unsigned long
              i;
            double
              denominator;

            denominator = s.x = s.y = 0;
            for(i=0; i<number_arguments; i+=4) {
              double weight =
                  ((double)d.x-arguments[i+2])*((double)d.x-arguments[i+2])
                + ((double)d.y-arguments[i+3])*((double)d.y-arguments[i+3]);
              if ( weight != 0 )
                weight = 1/weight;
              else
                weight = 1;

              s.x += (arguments[ i ]-arguments[i+2])*weight;
              s.y += (arguments[i+1]-arguments[i+3])*weight;
              denominator += weight;
            }
            s.x /= denominator;
            s.y /= denominator;
            s.x += d.x;
            s.y += d.y;

            /* We can not determine derivatives using shepards method
               only color interpolatation, not area-resampling */
            break;
          }
          default:
            break; /* uset the default no-op given above */
        }
        /* map virtual canvas location back to real image coordinate */
        if ( bestfit && method != ArcDistortion ) {
          s.x -= image->page.x;
          s.y -= image->page.y;
        }
        s.x -= 0.5;
        s.y -= 0.5;

        if ( validity <= 0.0 ) {
          /* result of distortion is an invalid pixel - don't resample */
          SetPixelPacket(distort_image,&invalid,q,indexes);
        }
        else {
          /* resample the source image to find its correct color */
          pixel=ResamplePixelColor(resample_filter[id],s.x,s.y);
          /* if validity between 0.0 and 1.0 mix result with invalid pixel */
          if ( validity < 1.0 ) {
            /* Do a blend of sample color and invalid pixel */
            /* should this be a 'Blend', or an 'Over' compose */
            MagickPixelCompositeBlend(&pixel, validity,
                &invalid, (1.0-validity), &pixel);
          }
          SetPixelPacket(distort_image,&pixel,q,indexes);
        }
        q++;
        indexes++;
      }
      if (SyncCacheView(distort_view[id]) == MagickFalse)
        status=MagickFalse;
      if (image->progress_monitor != (MagickProgressMonitor) NULL)
        {
          MagickBooleanType
            proceed;

          #pragma omp critical
          proceed=SetImageProgress(image,DistortImageTag,progress++,
            image->rows);
          if (proceed == MagickFalse)
            status=MagickFalse;
        }
    }
    distort_view=DestroyCacheViewThreadSet(distort_view);
    resample_filter=DestroyResampleFilterThreadSet(resample_filter);

    if (status == MagickFalse)
      distort_image=DestroyImage(distort_image);
  }

  /* Arc does not return an offset unless 'bestfit' is in effect
     And the user has not provided an overriding 'viewport'.
   */
  if ( method == ArcDistortion && !bestfit && !viewport_given ) {
    distort_image->page.x = 0;
    distort_image->page.y = 0;
  }
  coefficients = (double *) RelinquishMagickMemory(coefficients);
  return(distort_image);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   S p a r s e C o l o r I n t e r p o l a t e                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  SparseColorInterpolate() Given a set of coordinates interpolate the colors
%  found at those coordinates, across the whole image, using various methods.
%
%  The format of the SparseColorInterpolate() method is:
%
%      Image *SparseColorInterpolate(const Image *image,
%        SparseColorInterpolateMethod method,
%        const unsigned long number_arguments,const double *arguments,
%        ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image: the image to be filled in.
%
%    o method: the method to fill in the gradient between the control points.
%
%    o number_arguments: the number of arguments given.
%
%    o arguments: the arguments for this distortion method.
%
%    o exception: Return any errors or warnings in this structure
%
%  The methods used for SparseColorInterpolate() are often simular to methods
%  used for DistortImage(), and even share the same code for determination
%  of the function coefficents, though with more dimensions (or resulting
%  values).
%
*/
MagickExport Image *SparseColorInterpolate(const Image *image,
  SparseColorInterpolateMethod method,const unsigned long number_arguments,
  const double *arguments,ExceptionInfo *exception)
{
#define SparseColorTag  "Distort/SparseColor"

  Image
    *sparse_image;

  double
    *coefficients;

  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"%s",image->filename);
  assert(exception != (ExceptionInfo *) NULL);
  assert(exception->signature == MagickSignature);

  /*
    Convert input arguments into mapping coefficients to apply the distortion.
    Note some Methods may fall back to other simpler methods.
  */
  coefficients = GenerateCoefficients(image, (DistortImageMethod *)&method,
       number_arguments, arguments, 3, exception);
  if ( coefficients == (double *) NULL )
    return((Image *) NULL);

  /* Verbose output */
  if ( GetImageArtifact(image,"verbose") != (const char *) NULL ) {
    register long
       i;

    switch (method) {
      case BarycentricColorInterpolate:
      {
        fprintf(stderr, "Barycentric Sparse Color:\n");
        fprintf(stderr, "  -channel R -fx '%+lf*i %+lf*j %+lf' \\\n",
            coefficients[0], coefficients[1], coefficients[2]);
        fprintf(stderr, "  -channel G -fx '%+lf*i %+lf*j %+lf' \\\n",
            coefficients[3], coefficients[4], coefficients[5]);
        fprintf(stderr, "  -channel B -fx '%+lf*i %+lf*j %+lf' \\\n",
            coefficients[6], coefficients[7], coefficients[8]);
        break;
      }
      case BilinearDistortion:
      {
        fprintf(stderr, "Bilinear Sparse Color\n");
        fprintf(stderr, "   -channel R -fx '%+lf*i %+lf*j %+lf*i*j %+lf;\n",
            coefficients[0], coefficients[1], coefficients[2], coefficients[3]);
        fprintf(stderr, "   -channel G -fx '%+lf*i %+lf*j %+lf*i*j %+lf;\n",
            coefficients[4], coefficients[5], coefficients[6], coefficients[7]);
        fprintf(stderr, "   -channel B -fx '%+lf*i %+lf*j %+lf*i*j %+lf;\n",
            coefficients[8], coefficients[9], coefficients[10], coefficients[11]);
        break;
      }
      case PolynomialDistortion:
      {
        unsigned long nterms = (unsigned long) coefficients[1];
        fprintf(stderr, "Polynomial (order %lg, terms %lu) Sparse Color\n",
             coefficients[0], nterms);
        fprintf(stderr, "   -channel R -fx 'ii=i; jj=j;\n");
        for (i=0; i < (long) nterms; i++) {
          if ( i%4 == 0 ) fprintf(stderr, "\n         ");
          fprintf(stderr, " %+lf%s", coefficients[2+i],
               poly_basis_str(i));
        }
        fprintf(stderr, "' \\\n");
        fprintf(stderr, "   -channel G -fx 'ii=i; jj=j;\n");
        for (i=0; i < (long) nterms; i++) {
          if ( i%4 == 0 ) fprintf(stderr, "\n         ");
          fprintf(stderr, " %+lf%s", coefficients[2+i+nterms],
               poly_basis_str(i));
        }
        fprintf(stderr, "' \\\n");
        fprintf(stderr, "   -channel B -fx 'ii=i; jj=j;\n");
        for (i=0; i < (long) nterms; i++) {
          if ( i%4 == 0 ) fprintf(stderr, "\n         ");
          fprintf(stderr, " %+lf%s", coefficients[2+i+2*nterms],
               poly_basis_str(i));
        }
        break;
      }
      default:
        break;
    }
  }

  /* Generate new image for generated interpolated gradient.
   * ASIDE: Actually we could have just replaced the colors of the original
   * image, but IM core policy, is if storage class could change then clone
   * the image.
   */

  sparse_image=CloneImage(image,0,0,MagickTrue,exception);
  if (sparse_image == (Image *) NULL)
    return((Image *) NULL);
  if (SetImageStorageClass(sparse_image,DirectClass) == MagickFalse)
    { /* if image is ColorMapped - change it to DirectClass */
      InheritException(exception,&image->exception);
      sparse_image=DestroyImage(sparse_image);
      return((Image *) NULL);
    }
  { /* ----- MAIN CODE ----- */
    long
      j,
      progress;

    MagickBooleanType
      status;

    ViewInfo
      **sparse_view;

    status=MagickTrue;
    progress=0;
    sparse_view=AcquireCacheViewThreadSet(sparse_image);
    #pragma omp parallel for
    for (j=0; j < (long) sparse_image->rows; j++)
    {
      MagickPixelPacket
        pixel;    /* pixel to assign to distorted image */

      register IndexPacket
        *indexes;

      register long
        i,
        id;

      register PixelPacket
        *q;

      id=GetCacheViewThreadId();
      q=SetCacheViewPixels(sparse_view[id],0,j,sparse_image->columns,1);
      if (q == (PixelPacket *) NULL)
        {
          status=MagickFalse;
          continue;
        }
      indexes=GetCacheViewIndexes(sparse_view[id]);
      GetMagickPixelPacket(sparse_image,&pixel);
      for (i=0; i < (long) sparse_image->columns; i++)
      {
        switch (method)
        {
          case BarycentricColorInterpolate:
          {
            pixel.red   = coefficients[0]*i+coefficients[1]*j+coefficients[2];
            pixel.green = coefficients[3]*i+coefficients[4]*j+coefficients[5];
            pixel.blue  = coefficients[6]*i+coefficients[7]*j+coefficients[8];
            break;
          }
          case BilinearColorInterpolate:
          {
            pixel.red   = coefficients[0]*i + coefficients[1]*j +
                             coefficients[2]*i*j + coefficients[3];
            pixel.green = coefficients[4]*i + coefficients[5]*j +
                             coefficients[6]*i*j + coefficients[7];
            pixel.blue  = coefficients[8]*i + coefficients[9]*j +
                             coefficients[10]*i*j + coefficients[11];
            break;
          }
          case PolynomialColorInterpolate:
          {
            register long
              k;
            long
              nterms=(long)coefficients[1];

            pixel.red = pixel.blue = pixel.green = 0.0;
            for(k=0; k < nterms; k++) {
              pixel.red  += poly_basis_fn(k,(double) i,(double) j)*coefficients[2+k];
              pixel.green+= poly_basis_fn(k,(double) i,(double) j)*coefficients[2+k+nterms];
              pixel.blue += poly_basis_fn(k,(double) i,(double) j)*coefficients[2+k+2*nterms];
            }
            break;
          }
          case ShepardsColorInterpolate:
          { /* Shepards Method, or IWD uses its own input arguments
               as coefficients.
            */
            unsigned long
              k;
            double
              denominator;

            denominator = pixel.red = pixel.blue = pixel.green = 0.0;
            for(k=0; k<number_arguments; k+=5) {
              double weight =
                  ((double)i-arguments[ k ])*((double)i-arguments[ k ])
                + ((double)j-arguments[k+1])*((double)j-arguments[k+1]);
              if ( weight != 0 )
                weight = 1/weight;
              else
                weight = 1;

              pixel.red   += arguments[k+2]*weight;
              pixel.green += arguments[k+3]*weight;
              pixel.blue  += arguments[k+4]*weight;
              denominator += weight;
            }
            pixel.red   /= denominator;
            pixel.green /= denominator;
            pixel.blue  /= denominator;
            break;
          }
          case VoronoiColorInterpolate:
          { /* Just use the closest control point you can find! */
            unsigned long
              k;
            double
              minimum = MagickHuge;

            for(k=0; k<number_arguments; k+=5) {
              double distance =
                  ((double)i-arguments[ k ])*((double)i-arguments[ k ])
                + ((double)j-arguments[k+1])*((double)j-arguments[k+1]);
              if ( distance < minimum ) {
                pixel.red   = arguments[k+2];
                pixel.green = arguments[k+3];
                pixel.blue  = arguments[k+4];
                minimum = distance;
              }
            }
            break;
          }
          default:
          {
            /* Just use the first color that was found! */
            pixel.red   = arguments[2];
            pixel.green = arguments[3];
            pixel.blue  = arguments[4];
            break;
          }
        }
        /* set the color directly back into the source image */
        pixel.red   *= QuantumRange;
        pixel.green *= QuantumRange;
        pixel.blue  *= QuantumRange;

        SetPixelPacket(sparse_image,&pixel,q,indexes);
        q++;
        indexes++;
      }
      if (SyncCacheView(sparse_view[id]) == MagickFalse)
        status=MagickFalse;
      if (image->progress_monitor != (MagickProgressMonitor) NULL)
        {
          MagickBooleanType
            proceed;

          #pragma omp critical
          proceed=SetImageProgress(image,SparseColorTag,progress++,image->rows);
          if (proceed == MagickFalse)
            status=MagickFalse;
        }
    }
    sparse_view=DestroyCacheViewThreadSet(sparse_view);
    if (status == MagickFalse)
      sparse_image=DestroyImage(sparse_image);
  }
  coefficients = (double *) RelinquishMagickMemory(coefficients);
  return(sparse_image);
}
