# -*- coding: utf-8 -*-
#@+leo-ver=5-thin
#@+node:ekr.20120401063816.10072: * @file leoIPython.py
#@@first

'''
Support for the --ipython command-line option and the IPython bridge:
http://leoeditor.com/IPythonBridge.html

This code will run on IPython 1.0 and higher, as well as the IPython
0.x versions that define the IPKernelApp class.

This module replaces leo.external.ipy_leo and
leo.plugins.internal_ipkernel.

The ``--ipython`` command-line argument creates g.app.ipk, a
*singleton* IPython shell, shared by all IPython consoles.

The startup code injects a single object, _leo, into the IPython namespace.
This object, a LeoNameSpace instance, simplifies dealing with multiple open
Leo commanders.
'''
#@+<< imports >>
#@+node:ekr.20130930062914.15990: ** << imports >> (leoIpython.py)
import sys
import leo.core.leoGlobals as g
import_trace = False and not g.unitTesting
# pylint: disable=no-name-in-module
try:
    from IPython.lib.kernel import connect_qtconsole
    if import_trace: print('ok: IPython.lib.kernel import connect_qtconsole')
except ImportError:
    connect_qtconsole = None
    print('leoIPython.py: can not import connect_qtconsole')
try:
    # First, try the IPython 0.x import.
    from IPython.zmq.ipkernel import IPKernelApp
    if import_trace: print('ok: from IPython.zmq.ipkernel import IPKernelApp')
except ImportError:
    # Next, try the IPython 1.x import.
    try:
        from IPython.kernel.zmq.kernelapp import IPKernelApp
        if import_trace: print('ok: from IPython.kernel.zmq.ipkernel import IPKernelApp')
    except ImportError:
        IPKernelApp = None
        print('leoIPython.py: can not import IPKernelApp')
g.app.ipython_inited = IPKernelApp is not None
#@-<< imports >>
#@+others
#@+node:ekr.20130930062914.15993: ** class InternalIPKernel
class InternalIPKernel(object):
    #@+others
    #@+node:ekr.20130930062914.15994: *3* ctor
    # Was init_ipkernel

    def __init__(self, backend='qt'):
        '''Ctor for InternalIPKernal class.'''
        # Start IPython kernel with GUI event loop and pylab support
        self.ipkernel = self.pylab_kernel(backend)
        # To create and track active qt consoles
        self.consoles = []
        # This application will also act on the shell user namespace
        self.namespace = self.ipkernel.shell.user_ns
        # Keys present at startup so we don't print the entire pylab/numpy
        # namespace when the user clicks the 'namespace' button
        self._init_keys = set(self.namespace.keys())
        # Example: a variable that will be seen by the user in the shell, and
        # that the GUI modifies (the 'Counter++' button increments it):
        self.namespace['app_counter'] = 0
        #self.namespace['ipkernel'] = self.ipkernel  # dbg
    #@+node:ekr.20130930062914.15992: *3* pylab_kernel
    def pylab_kernel(self,gui):
        '''Launch an IPython kernel with pylab support for the gui.'''
        log_debug = False
            # Produces a verbose IPython message log.
        tag = 'leoIPython.py:pylab_kernel'
        kernel = IPKernelApp.instance()
            # IPKernalApp is a singleton class.
            # Return the singleton instance, creating it if necessary.
        if kernel:
            # pylab is needed for Qt event loop integration.
            args = ['python','--pylab=%s' % (gui)]
            if log_debug: args.append('--debug')
                #'--log-level=10'
                # '--pdb', # User-level debugging
            try:
                # g.pdb()
                kernel.initialize(args)
                # kernel objects: (Leo --ipython arg)
                    # kernel.session: zmq.session.Session
                    # kernel.shell: ZMQInteractiveShell
                    # kernel.shell.comm_manager: comm.manager.CommManager.
                    # kernel.shell.event = events.EventManager
                    # kernel.shell.run_cell (method)
                    # kernel.shell.hooks
            except Exception:
                sys.stdout = sys.__stdout__
                print('%s: kernel.initialize failed!' % tag)
                raise
        else:
            print('%s IPKernelApp.instance failed' % (tag))
        return kernel
    #@+node:ekr.20130930062914.15995: *3* print_namespace
    def print_namespace(self,event=None):
        print("\n***Variables in User namespace***")
        for k, v in self.namespace.iteritems():
            if k not in self._init_keys and not k.startswith('_'):
                print('%s -> %r' % (k, v))
        sys.stdout.flush()
    #@+node:ekr.20130930062914.15996: *3* new_qt_console
    def new_qt_console(self,event=None):
        """start a new qtconsole connected to our kernel"""
        ipk = g.app.ipk
        console = None
        if ipk:
            if not ipk.namespace.get('_leo'):
                ipk.namespace['_leo'] = LeoNameSpace()
            # from IPython.lib.kernel import connect_qtconsole
            console = connect_qtconsole(
                self.ipkernel.connection_file,
                profile=self.ipkernel.profile)
            if console:
                self.consoles.append(console)
        return console
    #@+node:ekr.20130930062914.15997: *3* count
    def count(self,event=None):
        self.namespace['app_counter'] += 1

    #@+node:ekr.20130930062914.15998: *3* cleanup_consoles
    def cleanup_consoles(self,event=None):
        for c in self.consoles:
            c.kill()
    #@-others
#@+node:ekr.20130930062914.16002: ** class LeoNameSpace
class LeoNameSpace(object):
    
    '''An interface class passed to IPython that provides easy
    access to "g" and all commanders.
    
    A convenience property, "c" provides access to the first
    commander in g.app.windowList.
    '''

    def __init__ (self):
        '''LeoNameSpace ctor.'''
        self.commander = None
            # The commander returned by the c property.
        self.commanders_list = []
            # The list of commanders returned by the commanders property.
        self.g = g
        self.update()

    #@+others
    #@+node:ekr.20130930062914.16006: *3* LeoNS.c property
    def __get_c(self):
        '''Return the designated commander, or the only open commander.'''
        self.update()
        if self.commander and self.commander in self.commanders_list:
            return self.commander
        elif len(self.commanders_list) == 1:
            return self.commanders_list[0]
        else:
            return None

    def __set_c(self,c):
        '''Designate the commander to be returned by the getter.'''
        self.update()
        if c in self.commanders_list:
            self.commander = c
        else:
            g.trace(g.callers())
            raise ValueError(c)

    c = property(
        __get_c, __set_c,
        doc = "LeoNameSpace c property")
    #@+node:edward.20130930125732.11822: *3* LeoNS.commanders property
    def __get_commanders(self):
        '''Return the designated commander, or the only open commander.'''
        self.update()
        return self.commanders_list

    commanders = property(
        __get_commanders, None,
        doc = "LeoNameSpace commanders property (read-only)")
    #@+node:ekr.20130930062914.16009: *3* LeoNS.find_c
    def find_c(self,path):
        '''Return the commander associated with path, or None.'''
        g = self.g
        self.update()
        path = g.os_path_normcase(path)
        short_path = g.shortFileName(path)
        for c in self.commanders_list:
            fn = g.os_path_normcase(c.fileName())
            short_fn = g.shortFileName(fn)
            if fn == path or short_fn == short_path:
                return c
    #@+node:ekr.20130930062914.16003: *3* LeoNS.update
    def update (self):
        '''Update the list of available commanders.'''
        self.commanders_list = [frame.c for frame in g.app.windowList]
    #@-others
    
#@-others
#@@language python
#@@tabwidth -4
#@@pagewidth 70
#@-leo
