#!/bin/sh

# NVIDIA Graphics Driver bug reporting shell script.  This shell
# script will generate a log file named "nvidia-bug-report.log", which
# should be attached when emailing bug reports to NVIDIA.



LOG_FILENAME=nvidia-bug-report.log


PATH="/sbin:/usr/sbin:$PATH"


NVIDIA_BUG_REPORT_CHANGE="$Change: 1833133 $"
NVIDIA_BUG_REPORT_VERSION=`echo $NVIDIA_BUG_REPORT_CHANGE | tr -c -d [:digit:]`

#
# append() - append the contents of the specified file to the log
#

append() {
    echo "____________________________________________" >> $LOG_FILENAME
    echo ""                                             >> $LOG_FILENAME

    if [ ! -f "$1" ]; then
        echo "$1 does not exist"                        >> $LOG_FILENAME
    elif [ ! -r "$1" ]; then
        echo "$1 is not readable"                       >> $LOG_FILENAME
    else
        echo "$1"                                       >> $LOG_FILENAME
        cat  "$1"                                       >> $LOG_FILENAME
    fi
    echo ""                                             >> $LOG_FILENAME
}

#
# append_silent() - same as append(), but don't print anything
# if the file does not exist
#

append_silent() {
    if [ -f "$1" -a -r "$1" ]; then
        echo "____________________________________________" >> $LOG_FILENAME
        echo ""                                             >> $LOG_FILENAME
        echo "$1"                                           >> $LOG_FILENAME
        cat  "$1"                                           >> $LOG_FILENAME
        echo ""                                             >> $LOG_FILENAME
    fi
}

#
# append_glob() - use the shell to expand a list of files, and invoke
# append() for each of them
#

append_glob() {
    for i in `ls $1 2> /dev/null;`; do
        append "$i"
    done
}

#
# append_sysctl() - use sysctl to retrieve the value of interesting
# kernel or driver state variables.
#

append_sysctl() {
    echo "____________________________________________" >> $LOG_FILENAME
    echo ""                                             >> $LOG_FILENAME

    sysctl=`which sysctl 2> /dev/null | head -n 1`
    if [ $? -eq 0 -a "$sysctl" ]; then
        value=`$sysctl $1 2> /dev/null`
        if [ $? -eq 0 -a "$value" ]; then
            echo "$value"                               >> $LOG_FILENAME
        else
            echo "$1 does not exit"                     >> $LOG_FILENAME
        fi
    else
        echo "sysctl not found"                         >> $LOG_FILENAME
    fi
}


#
# Start of script
#


# check that we are root (needed for `lspci -vxxx` and potentially for
# accessing kernel log files)

if [ `id -u` -ne 0 ]; then
    echo "ERROR: Please run $(basename $0) as root."
    exit 1
fi


# move any old log file out of the way

if [ -f $LOG_FILENAME ]; then
    mv $LOG_FILENAME ${LOG_FILENAME}.old
fi


# make sure what we can write to the log file

touch $LOG_FILENAME 2> /dev/null

if [ $? -ne 0 ]; then
    echo
    echo "ERROR: Working directory is not writable; please cd to a directory"
    echo "       where you have write permission so that the $LOG_FILENAME"
    echo "       file can be written."
    echo
    exit 1
fi


# print a start message to stdout

echo ""
echo -n "Running $(basename $0)...";


# print prologue to the log file

echo "____________________________________________"                >> $LOG_FILENAME
echo ""                                                            >> $LOG_FILENAME
echo "Start of NVIDIA bug report log file.  Please send this"      >> $LOG_FILENAME
echo "report along with a description of your bug, to"             >> $LOG_FILENAME
echo "freebsd-gfx-bugs@nvidia.com."                                >> $LOG_FILENAME  
echo ""                                                            >> $LOG_FILENAME
echo "nvidia-bug-report.sh Version: $NVIDIA_BUG_REPORT_VERSION"    >> $LOG_FILENAME
echo ""                                                            >> $LOG_FILENAME
echo "Date: `date`"                                                >> $LOG_FILENAME
echo "uname: `uname -a`"                                           >> $LOG_FILENAME
echo ""                                                            >> $LOG_FILENAME


# append useful sysctl keys

append_sysctl "hw.nvidia.version"
append_sysctl "hw.nvidia.cards"
append_sysctl "hw.nvidia.agp"
append_sysctl "hw.nvidia.registry"

append_sysctl "vm.vmtotal"
append_sysctl "kern.version"
append_sysctl "kern.smp"

# append useful files

append "/etc/motd"


# append the X log; also, extract the config file named in the X log
# and append it

for i in /var/log/XFree86.0.log /var/log/Xorg.0.log ; do
    append "$i"
    if [ -f $i -a -r $i ]; then
        j=`grep "Using config file" $i | cut -f 2 -d \"`
        append "$j"
    fi
done

# append vmstat info

vmstat=`which vmstat 2> /dev/null | head -n 1`

echo "____________________________________________"                    >> $LOG_FILENAME
echo ""                                                                >> $LOG_FILENAME

if [ $? -eq 0 -a "$vmstat" ]; then
    echo "$vmstat"                                                     >> $LOG_FILENAME
    echo ""                                                            >> $LOG_FILENAME
    ( $vmstat >> $LOG_FILENAME ; exit 0 ) > /dev/null 2>&1
    echo ""                                                            >> $LOG_FILENAME
    echo "____________________________________________"                >> $LOG_FILENAME
    echo ""                                                            >> $LOG_FILENAME
    echo "$vmstat -a -i"                                               >> $LOG_FILENAME
    echo ""                                                            >> $LOG_FILENAME
    ( $vmstat -a -i >> $LOG_FILENAME ; exit 0 ) > /dev/null 2>&1
    echo ""                                                            >> $LOG_FILENAME
else
    echo "Skipping vmstat output (vmstat not found)"                   >> $LOG_FILENAME
    echo ""                                                            >> $LOG_FILENAME
fi

# append kldstat info

kldstat=`which kldstat 2> /dev/null | head -n 1`

echo "____________________________________________"                    >> $LOG_FILENAME
echo ""                                                                >> $LOG_FILENAME

if [ $? -eq 0 -a "$kldstat" ]; then
    echo "$kldstat -v"                                                 >> $LOG_FILENAME
    echo ""                                                            >> $LOG_FILENAME
    ( $kldstat -v >> $LOG_FILENAME ; exit 0 ) > /dev/null 2>&1
    echo ""                                                            >> $LOG_FILENAME
else
    echo "Skipping kldstat output (kldstat not found)"                 >> $LOG_FILENAME
    echo ""                                                            >> $LOG_FILENAME
fi


# append ldd info

glxinfo=`which glxinfo 2> /dev/null | head -n 1`

echo "____________________________________________"                    >> $LOG_FILENAME
echo ""                                                                >> $LOG_FILENAME

if [ $? -eq 0 -a "$glxinfo" ]; then
    echo "ldd $glxinfo"                                                >> $LOG_FILENAME
    echo ""                                                            >> $LOG_FILENAME
    ( ldd $glxinfo >> $LOG_FILENAME ; exit 0 ) > /dev/null 2>&1
    echo ""                                                            >> $LOG_FILENAME
else
    echo "Skipping ldd output (glxinfo not found)"                     >> $LOG_FILENAME
    echo ""                                                            >> $LOG_FILENAME
fi

# pciconf information

pciconf=`which pciconf 2> /dev/null | head -n 1`

echo "____________________________________________"                    >> $LOG_FILENAME
echo ""                                                                >> $LOG_FILENAME

if [ $? -eq 0 -a "$pciconf" ]; then
    echo "$pciconf -l -v"                                              >> $LOG_FILENAME
    echo ""                                                            >> $LOG_FILENAME
    ( $pciconf -l -v >> $LOG_FILENAME ; exit 0 ) > /dev/null 2>&1
    echo ""                                                            >> $LOG_FILENAME
else
    echo "Skipping pciconf output (pciconf not found)"                 >> $LOG_FILENAME
    echo ""                                                            >> $LOG_FILENAME
fi

# get any relevant kernel messages

echo "____________________________________________"                    >> $LOG_FILENAME
echo ""                                                                >> $LOG_FILENAME
echo "Scanning kernel log files for NVRM messages:"                    >> $LOG_FILENAME
echo ""                                                                >> $LOG_FILENAME

for i in /var/log/messages /var/log/kernel.log ; do
    if [ -f $i ]; then
        echo "  $i:"                                                   >> $LOG_FILENAME
        ( cat $i | grep NVRM >> $LOG_FILENAME ; exit 0 ) > /dev/null 2>&1
    fi
done

# append dmesg output

echo ""                                                                >> $LOG_FILENAME
echo "____________________________________________"                    >> $LOG_FILENAME
echo ""                                                                >> $LOG_FILENAME
echo "dmesg:"                                                          >> $LOG_FILENAME
echo ""                                                                >> $LOG_FILENAME
( dmesg >> $LOG_FILENAME ; exit 0 ) > /dev/null 2>&1

which gcc >/dev/null 2>&1
if [ $? -eq 0 ]; then
    echo "____________________________________________"                >> $LOG_FILENAME
    echo ""                                                            >> $LOG_FILENAME
    ( gcc -v 2>> $LOG_FILENAME ; exit 0 ) > /dev/null 2>&1
fi

which g++ >/dev/null 2>&1
if [ $? -eq 0 ]; then
    echo "____________________________________________"                >> $LOG_FILENAME
    echo ""                                                            >> $LOG_FILENAME
    ( g++ -v 2>> $LOG_FILENAME ; exit 0 ) > /dev/null 2>&1
fi

echo "____________________________________________"                    >> $LOG_FILENAME
echo ""                                                                >> $LOG_FILENAME
echo "xset -q:"                                                        >> $LOG_FILENAME
echo ""                                                                >> $LOG_FILENAME

( xset -q >> $LOG_FILENAME & sleep 1 ; kill -9 $! ) > /dev/null 2>&1

if [ $? -eq 0 ]; then
    # The xset process is still there.
    echo "xset could not connect to an X server"                       >> $LOG_FILENAME
fi

echo "____________________________________________"                    >> $LOG_FILENAME
echo ""                                                                >> $LOG_FILENAME
echo "nvidia-settings -q all:"                                         >> $LOG_FILENAME
echo ""                                                                >> $LOG_FILENAME

( nvidia-settings -q all >> $LOG_FILENAME & sleep 1 ; kill -9 $! ) > /dev/null 2>&1

if [ $? -eq 0 ]; then
    # The nvidia-settings process is still there.
    echo "nvidia-settings could not connect to an X server"            >> $LOG_FILENAME
fi
echo "____________________________________________"                    >> $LOG_FILENAME

# print epilogue to log file

echo ""                                                                >> $LOG_FILENAME
echo "End of NVIDIA bug report log file."                              >> $LOG_FILENAME


# Done

echo " complete."
echo ""
echo "The file $LOG_FILENAME has been created; please send this"
echo "report, along with a description of your bug, to"
echo "freebsd-gfx-bugs@nvidia.com."
echo ""
