/**********************************************************************
*
*  nvhw.c
*
*  Descripion  - hw access details and common functions needed for channel
*
*  Copyright (c) 2002-2003 NVIDIA Corporation
*
***********************************************************************
*/
#define __NO_VERSION__
#include <linux/module.h>
#include <linux/version.h>
#include <asm/uaccess.h>

#include "nvhw.h"
#include "nvrec.h"
#include "nvwavout.h"
#include "nvspdif.h"

/* allocate DMA buffer, playback and recording buffer should be allocated seperately */
int alloc_dmabuf(struct Nvaudio_state *state)
{
    void *rawbuf = NULL;
    int order, size;
    struct page *page, *pend;
    struct dmabuf *dmabuf = &state->dmabuffer;

    /* If we don't have any oss frag params, then use our default ones */
    if(dmabuf->ossmaxfrags == 0)
        dmabuf->ossmaxfrags = 4;

    if(dmabuf->ossfragsize == 0)
        dmabuf->ossfragsize = (PAGE_SIZE<<DMABUF_DEFAULTORDER)/dmabuf->ossmaxfrags;

    size = dmabuf->ossfragsize * dmabuf->ossmaxfrags;

    if(dmabuf->rawbuf && (PAGE_SIZE << dmabuf->buforder) == size){
        return 0;
    }

    /* alloc enough to satisfy the oss params */
    for (order = DMABUF_DEFAULTORDER; order >= DMABUF_MINORDER; order--) {

        if ( (PAGE_SIZE<<order) > size )
            continue;

        if ((rawbuf = pci_alloc_consistent(state->card->pci_dev,
                           PAGE_SIZE << order,
                           &dmabuf->dma_handle)))
            break;

    }

    if (!rawbuf)
        return -ENOMEM;


#ifdef NV_DEBUG_NORMAL
    printk("Nvaudio: allocated %ld (order = %d) bytes at %p\n",
           PAGE_SIZE << order, order, rawbuf);
#endif

    state->ready     = state->mapped = 0;
    dmabuf->rawbuf   = rawbuf;
    dmabuf->buforder = order;

    /* now mark the pages as reserved; otherwise remap_page_range doesn't do what we want */

    pend = virt_to_page(rawbuf + (PAGE_SIZE << order) - 1);
    for (page = virt_to_page(rawbuf); page <= pend; page++)
        mem_map_reserve(page);

    return 0;
}

/* free DMA buffer */

void dealloc_dmabuf(struct Nvaudio_state *state)
{
    struct page *page, *pend;
    struct dmabuf *dmabuf = &state->dmabuffer;

    if (dmabuf->rawbuf) {
        /* undo marking the pages as reserved */

        pend = virt_to_page(dmabuf->rawbuf + (PAGE_SIZE << dmabuf->buforder) - 1);
        for (page = virt_to_page(dmabuf->rawbuf); page <= pend; page++)
            mem_map_unreserve(page);

        pci_free_consistent(state->card->pci_dev, PAGE_SIZE << dmabuf->buforder,
                    dmabuf->rawbuf, dmabuf->dma_handle);

#ifdef NV_DEBUG_NORMAL
    printk("Nvaudio: dealloc_dmabuf bytes at %p\n",dmabuf->rawbuf);
#endif

    }
    dmabuf->rawbuf = NULL;
    state->mapped  = state->ready = 0;
}

/* function that program the prds for the channels*/
int prog_dmabuf(struct Nvaudio_state *state,unsigned rec)
{
    struct sg_item *sg;
    unsigned long flags;
    int ret = 0, i = 0, channelnum = 0;
    unsigned fragint = 0;
    unsigned long port = 0;
    struct dmabuf *dmabuf = &state->dmabuffer;

    spin_lock_irqsave(&state->lock, flags);

    if(state->enable & DAC_RUNNING){
        __stop_dac(state);
    }

    if(state->enable & ADC_RUNNING){
        __stop_adc(state);
    }

    if(state->enable & SPDIF_RUNNING) {
        __stop_spdif(state);
    }

    dmabuf->total_bytes = 0;
    dmabuf->count = dmabuf->error = 0;
    dmabuf->swptr = dmabuf->hwptr = 0;
    spin_unlock_irqrestore(&state->lock, flags);

    /* allocate DMA buffer, let alloc_dmabuf determine if we are already
     * allocated well enough or if we should replace the current buffer
     * (assuming one is already allocated, if it isn't, then allocate it).
     */
    if ((ret = alloc_dmabuf(state)))
        return ret;

    dmabuf->dmasize      = PAGE_SIZE << dmabuf->buforder;
    dmabuf->numfrag      = SG_LEN;
    dmabuf->fragsize     = dmabuf->dmasize/dmabuf->numfrag;
    dmabuf->fragsamples  = dmabuf->fragsize >> 1;
    dmabuf->userfragsize = dmabuf->ossfragsize;
    dmabuf->userfrags    = dmabuf->dmasize/dmabuf->ossfragsize;

    memset(dmabuf->rawbuf, 0, dmabuf->dmasize);

    if(dmabuf->ossmaxfrags == 4) {
        fragint = 8;
        dmabuf->fragshift = 2;
    } else if (dmabuf->ossmaxfrags == 8) {
        fragint = 4;
        dmabuf->fragshift = 3;
    } else if (dmabuf->ossmaxfrags == 16) {
        fragint = 2;
        dmabuf->fragshift = 4;
    } else {
        fragint = 1;
        dmabuf->fragshift = 5;
    }

#ifdef NV_DEBUG_NORMAL
        printk("Nvaudio: prog_dmabuf, fragint = %x \n",fragint);
#endif

    port = state->card->iobase;

    switch(rec)    {
     case 0:  /*audio out */
        port += PO_BDBAR;
        channelnum = 0;
        break;
     case 1:  /*audio in */
        port += PI_BDBAR;
        channelnum = 1;
        break;
     case 2:  /*spdif out */
        port += SP_BDBAR;
        channelnum = 2;
        break;
    }

    if(! state->card->channel) return ret;

    sg=&state->card->channel[channelnum].sg[0];

    for(i=0;i<dmabuf->numfrag;i++)
    {
        sg->busaddr=(u32)dmabuf->dma_handle+ (dmabuf->fragsize*i);

        // the card will always be doing 16bit stereo
        sg->control=dmabuf->fragsamples;

        sg->control|=CON_BUFPAD;

        // set us up to get IOC interrupts as often as needed to
        // satisfy numfrag requirements, no more
        if( ((i+1) % fragint) == 0) {
            sg->control|=CON_IOC;
        }

        sg++;
    }

    spin_lock_irqsave(&state->lock, flags);
    outb(2, port+OFF_CR);         /* reset DMA machine */
    while(inb(port+OFF_CR)&0x02); /* wait until reset done */
    outl((u32)(state->card->chandma + (channelnum * sizeof(struct Nvaudio_channel))),
                port+OFF_BDBAR);
    spin_unlock_irqrestore(&state->lock, flags);

    /* set the ready flag for the dma buffer */
    state->ready = 1;

#ifdef NV_DEBUG_NORMAL
    printk("Nvaudio: prog_dmabuf, sample rate = %d, format = %d,\n\tnumfrag = %d, "
           "fragsize = %d dmasize = %d\n",
           state->rate, state->fmt, dmabuf->numfrag,
           dmabuf->fragsize, dmabuf->dmasize);
#endif

    return 0;
}


/* Write AC97 codec registers */
/* TO DO :- check whether to use card spinlock on accessing codec registers */

u16 Nvaudio_ac97_get(struct ac97_codec *dev, u8 reg)
{
    struct Nvaudio_card *card = dev->private_data;
    int count  = 100;
    u8 reg_set = ((dev->id)?((reg&0x7f)|0x80):(reg&0x7f));

    while(count-- && (inb(card->iobase + CAS) & 1))
        udelay(1);

    return inw(card->ac97base + reg_set);
}

void Nvaudio_ac97_set(struct ac97_codec *dev, u8 reg, u16 data)
{
    struct Nvaudio_card *card = dev->private_data;
    int count  = 100;
    u8 reg_set = ((dev->id)?((reg&0x7f)|0x80):(reg&0x7f));

    while(count-- && (inb(card->iobase + CAS) & 1))
        udelay(1);
    outw(data, card->ac97base + reg_set);
}

