/****************************************************************** \
|*                                                                 *|
|*                                                                 *|
|*  (c) NVIDIA Corporation. All rights reserved                    *| 
|*                                                                 *|
|*  THE INFORMATION CONTAINED HEREIN IS PROPRIETARY AND            *|
|*  CONFIDENTIAL                                                   *|
|*  TO NVIDIA, CORPORATION. USE, REPORDUCTION OR DISCLOSURE TO ANY *|
|*  THIRD PARTY IS SUBJECT TO WRITTEN PRE-APPROVAL BY NVIDIA CORP. *|
|*                                                                 *|
|*  THE INFORMATION CONTAINED HEREIN IS PROVIDED "AS IS" WITHOUT   *|
|*  EXPRESS OR IMPLIED WARRANTY OF ANY KIND, INCLUDING ALL IMPLIED *|
|*  WARRANTIES OF MERCHANTABILITY, NONINFRINGEMENT, AND FITNESS    *|
|*  FOR A PARTICULAR PURPOSE.                                      *|
|*                                                                 *|
********************************************************************/

#include "basetype.h"
#include "os.h"
#include "adapter.h"
#include "nvnet.h"

char *linux_src_version_string = "NVNET Source $Revision: #13 $";
char *common_hdA_version_string = HDA_VERSION_STRING;
char *common_hdB_version_string = HDB_VERSION_STRING;
char *common_hdO_version_string = HDO_VERSION_STRING;
char *common_hdP_version_string = HDP_VERSION_STRING;

/*
 * Driver information
 */ 

#define DRIVER_NAME           "nvnet"
#define DRIVER_VERSION        "1.0.5"
#define DRIVER_RELEASE_DATE   "March 24, 2003"

MODULE_AUTHOR("NVIDIA (linux-nforce-bugs@nvidia.com");
MODULE_DESCRIPTION("NVIDIA Corporation NVNET Ethernet Driver Version"
                   DRIVER_VERSION " " DRIVER_RELEASE_DATE);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,10)
MODULE_LICENSE("NVIDIA");
#endif

/*
 * Duplex settings 
 *         0 - Auto duplex
 *         1 - Half duplex
 *         2 - Full duplex
 */
enum duplex_settings { duplex_auto, duplex_half, duplex_full };
#define DEFAULT_DUPLEX_SETTING duplex_auto
static int duplex = -1;

/*
 * Speed settings
 *          0 - auto negotiation
 *          1 - 10 Mbps
 *          2 - 100 Mbps
 */
enum speed_settings { speed_auto, speed_10, speed_100 };
#define DEFAULT_SPEED_SETTING speed_auto
static int speed = -1;

/*
 * Optimization settings
 *          0 - Optimize for throughput
 *          1 - Optimize for CPU
 */

enum optimization_settings {optimize_for_throughput, optimize_for_cpu};
#define DEFAULT_OPTIMIZATION_SETTING optimize_for_throughput
static int optimization = -1;

/*
 * Module parameters
 */

MODULE_PARM(speed, "i");
MODULE_PARM_DESC(speed, "Control speed by setting 0 for auto, 1 for 10 Mbps and 2 for 100 Mbps");

MODULE_PARM(duplex, "i");
MODULE_PARM_DESC(duplex, "Control duplex by setting 0 for auto, 1 for half and 2 for full");

MODULE_PARM(optimization, "i");
MODULE_PARM_DESC(optimization, "Control optimization by setting 0 for Throughput, 1 for CPU");
/* 
 * Table with PCI device identification
 */
static struct pci_device_id
nvnet_pci_table[]  __devinitdata = {
    {PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_NVNET, PCI_ANY_ID, PCI_ANY_ID},
    {PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_NVNET2, PCI_ANY_ID, PCI_ANY_ID},
    {PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_NVNET3, PCI_ANY_ID, PCI_ANY_ID},
    {0, 0, 0, 0}
};

MODULE_DEVICE_TABLE(pci, nvnet_pci_table);

static struct pci_driver nvnet_driver = 
{
    name:     "nvnet",
    id_table: nvnet_pci_table,
    probe:    nvnet_probe,
    remove:   nvnet_remove,
#ifdef CONFIG_PM
    suspend:  nvnet_suspend,
    resume:   nvnet_resume,
#endif

};

static int linuxlockalloc(void *ctx, int type, void **pLock)
{
    struct net_device *dev = (struct net_device *)ctx;
    struct nvnet_private *priv  = dev->priv;
    PRINTK(DEBUG_LOCK, "linuxlockalloc : IN\n");
    spin_lock_init(&priv->phylock);
    *pLock = (void **)&priv->phylock;
    PRINTK(DEBUG_LOCK, "linuxlockalloc : Out\n");
    return 1;

}

static int linuxlockacquire(void *ctx, int type, void *lock)
{
    PRINTK(DEBUG_LOCK, "linuxlockacquire: IN\n");
    spin_lock(lock);
    PRINTK(DEBUG_LOCK, "linuxlockacquire: OUT\n");
    return 1;

}

static int linuxlockrelease(void *ctx, int type, void *lock)
{
    PRINTK(DEBUG_LOCK, "linuxlockrelease: In\n");
    spin_unlock(lock);
    PRINTK(DEBUG_LOCK, "linuxlockrelease: Out\n");
    return 1;
}
/*
 * Maps to pfnAllocateMemory
 */
static int linuxalloc(void *ctx, MEMORY_BLOCK *mem)
{
    struct net_device *dev = (struct net_device *)ctx;
    struct nvnet_private *priv  = dev->priv;
    u32 mem_physical;
    dma_addr_t dma_addr;

    PRINTK(DEBUG_INIT, "linuxalloc: In\n");
    mem->pLogical = (PVOID)pci_alloc_consistent(priv->pdev,  
                              (size_t)mem->uiLength, &dma_addr);
    if (!mem->pLogical) 
    {
        PRINTK_ERROR("linuxalloc: memory allocation failed\n");
        PRINTK(DEBUG_INIT, "linuxalloc: Out\n");
        return 0;
    }

    memset(mem->pLogical, 0, mem->uiLength);
    mem_physical   = (u32)dma_addr;
    mem->pPhysical = (PVOID)mem_physical;

    PRINTK(DEBUG_INIT,"linuxalloc: Out\n");
    return 1;

}

/*
 * Maps to pfnFreeMemory
 */
static int linuxfree(void *ctx, MEMORY_BLOCK *mem)
{
    struct net_device *dev    = (struct net_device *)ctx;
    struct nvnet_private *priv  = dev->priv;

    u32 mem_physical    = (u32)mem->pPhysical;
    dma_addr_t dma_addr = (dma_addr_t)mem_physical;
 
    PRINTK(DEBUG_INIT, "linuxfree: In\n");

    pci_free_consistent(priv->pdev, (size_t)mem->uiLength, mem->pLogical, \
                        dma_addr);

    PRINTK(DEBUG_INIT, "linuxfree: Out\n");
    return 1;

}


/*
 * Maps to pfnAllocateMemoryEx
 */
static int linuxallocex(void *ctx, MEMORY_BLOCKEX *mem_block_ex)
{
    MEMORY_BLOCK mem_block;
    
    mem_block_ex->pLogical = NULL;
    mem_block_ex->uiLengthOrig = mem_block_ex->uiLength;

    if ((mem_block_ex->AllocFlags & ALLOC_MEMORY_ALIGNED) &&
        (mem_block_ex->AlignmentSize > 1))
    {
        mem_block_ex->uiLengthOrig += mem_block_ex->AlignmentSize;
    }

    mem_block.uiLength = mem_block_ex->uiLengthOrig;

    if (linuxalloc(ctx, &mem_block) == 0)
    {
        return 0;
    }

    mem_block_ex->pLogicalOrig = mem_block.pLogical;
    mem_block_ex->pPhysicalOrigLow = (ULONG)mem_block.pPhysical;
    mem_block_ex->pPhysicalOrigHigh = 0;

    mem_block_ex->pPhysical = mem_block.pPhysical;
    mem_block_ex->pLogical = mem_block.pLogical;

    if (mem_block_ex->uiLength != mem_block_ex->uiLengthOrig)
    {
        unsigned int offset;
        offset = mem_block_ex->pPhysicalOrigLow & (mem_block_ex->AlignmentSize - 1);

        if (offset)
        {
            mem_block_ex->pPhysical = (PVOID)((ULONG)mem_block_ex->pPhysical + 
                                              mem_block_ex->AlignmentSize - offset);
            mem_block_ex->pLogical = (PVOID)((ULONG)mem_block_ex->pLogical +
                                            mem_block_ex->AlignmentSize - offset);
        }   // if (offset)
    }       // if (mem_block_ex->uiLength != mem_block_ex->uiLengthOrig)
                
    return 1;
}

/*
 * Maps to pfnFreeMemoryEx
 */
static int linuxfreeex(void *ctx, MEMORY_BLOCKEX *mem_block_ex) 
{
    MEMORY_BLOCK mem_block;

    mem_block.pLogical = mem_block_ex->pLogicalOrig;
    mem_block.pPhysical = (PVOID)mem_block_ex->pPhysicalOrigLow;
    mem_block.uiLength = mem_block_ex->uiLengthOrig;

    return linuxfree(ctx, &mem_block);
}

/* 
 * Maps to pfnClearMemory
 */
static int linuxclear(void *ctx, void *mem, int length)
{
    memset(mem, 0, length);
    return 1;
}

/*
 * Maps to pfnStallExecution
 */
static int linuxdelay(void *ctx, unsigned long usec)
{
    udelay(usec);
    return 1;
}

/*
 * Maps to pfnAllocateReceiveBuffer
 */
static int linuxallocrxbuf(void *ctx, MEMORY_BLOCK *mem, void **id) 
{
    struct net_device *dev   = (struct net_device *)ctx;
    struct nvnet_private *priv = (struct nvnet_private *)dev->priv;
    struct sk_buff *skb;
    struct nvnet_rx_info *info;

    skb = dev_alloc_skb(MAX_PACKET_SIZE + 2 + sizeof(struct nvnet_rx_info));
    if(!skb) 
    {
        PRINTK_ERROR("linuxallocrxbuf - skb allocation failed\n");
        return 0;
    }

    info = (struct nvnet_rx_info *)((u8 *)skb->tail + 2 + MAX_PACKET_SIZE);

    info->skb      = skb;
    info->uiLength = MAX_PACKET_SIZE;
 
    mem->pLogical = (void *)skb;
    mem->uiLength = MAX_PACKET_SIZE;              
    *id           = (void *)info;        

    //skb_reserve(skb, 2);        /* Get 16 byte alignment of IP part */

    info->dma  = pci_map_single(priv->pdev, skb->tail, \
                       MAX_PACKET_SIZE, PCI_DMA_FROMDEVICE);
    mem->pPhysical = (void *)cpu_to_le32(info->dma);
    return 1;

}

/*
 * Maps to pfnFreeReceiveBuffer 
 */
static int linuxfreerxbuf(void *ctx, MEMORY_BLOCK *mem, void *id)
{
    struct net_device *dev = (struct net_device *)ctx;
    struct nvnet_private *priv = (struct nvnet_private *)dev->priv;
    struct sk_buff *skb;
    struct nvnet_rx_info *info;

    PRINTK(DEBUG_RX, "linuxfreerxbuf: In\n");

    info = (struct nvnet_rx_info *)id;
    skb  = info->skb;

    NVNET_ASSERT(skb != NULL);

    if (skb) 
    {
       pci_unmap_single(priv->pdev, info->dma, \
               (unsigned long)info->uiLength, PCI_DMA_FROMDEVICE);
        dev_kfree_skb(skb);
    }

    PRINTK(DEBUG_RX,  "linuxfreerxbuf: Out\n");
    return 1;
}

/*
 * Maps to pfnPacketWasSent
 */

static int linuxpackettx(void *ctx, void *id, unsigned long success)
{
    struct net_device *dev = (struct net_device *)ctx;
    struct nvnet_private *priv = (struct nvnet_private *)dev->priv;
    unsigned long addr;
    struct sk_buff *skb;
    nvnet_tx_info_os_t *tx_ring = (nvnet_tx_info_os_t *)id;


    PRINTK(DEBUG_TX, "linuxpackettx - In\n");
    skb             = (struct sk_buff *)tx_ring->pvID;

    NVNET_ASSERT(skb != NULL);

    addr     = tx_ring->dma_addr[0];
    pci_unmap_single(priv->pdev, le32_to_cpu(addr), skb->len, PCI_DMA_TODEVICE);

    tx_ring->pvID    = NULL;

    dev_kfree_skb_irq(skb);
    priv->tx_ring_count--;

    if ((priv->tx_ring_full) && (priv->tx_ring_count < priv->tx_ring_entries)) 
    {

        NVNET_ASSERT (priv->tx_ring_count < priv->tx_ring_entries);
        priv->tx_ring_full = 0;
        PRINTK_ERROR("linuxpackettx - waking up queue \n");
        netif_wake_queue(dev);
    }

    PRINTK(DEBUG_TX, "linuxpackettx - Out\n");
    return 1;
}

/*
 * Maps to pfnPacketWasReceived
 */
static int linuxpacketrx(void *ctx, void *data, unsigned long success, 
unsigned char *newbuf, unsigned char priority)
{
    struct sk_buff *skb;
    struct net_device *dev;
    struct nvnet_private *priv;
    ADAPTER_READ_DATA *readdata;
    struct nvnet_rx_info *info;
   
    PRINTK(DEBUG_RX, "linuxpacketrx- In\n");
    dev         = (struct net_device *)ctx;
    priv        = (struct nvnet_private *)dev->priv;
    readdata    = (ADAPTER_READ_DATA *)data;
    
    info         = (struct nvnet_rx_info *)readdata->pvID;
    skb          = info->skb;

    NVNET_ASSERT(skb != NULL);
    skb->dev     = dev;

    if(success)
    {
        pci_unmap_single(priv->pdev, info->dma, info->uiLength, \
                         PCI_DMA_FROMDEVICE);

        skb_put(skb, readdata->ulTotalLength);
        skb->protocol = eth_type_trans(skb, dev);

        /*
         * Send the received packet up
         */
        netif_rx(skb); //send the received packet up

        /* 
         * Update receive statistics
         */
        priv->stats.rx_bytes += readdata->ulTotalLength;
    }
    else 
    {
        pci_unmap_single(priv->pdev, info->dma, info->uiLength, \
                         PCI_DMA_FROMDEVICE);
        dev_kfree_skb(skb);
    }

    PRINTK(DEBUG_RX, "linuxpacketrx- Out\n");
    return 1;
}

/*
 * Maps to pfnLinkStateHasChanged
 */
static int linuxlinkchanged(void *ctx, int enabled)
{
    struct net_device *dev;

    dev = (struct net_device *)ctx;
    PRINTK(DEBUG_LINK, "linuxlinkchanged - In\n");

    if(enabled) {
        netif_carrier_on(dev);
        dev->flags |= IFF_RUNNING; 
    }
    else {
    netif_carrier_off(dev);
        dev->flags &= ~IFF_RUNNING; 
    }

    PRINTK(DEBUG_LINK, "linuxlinkchanged - Out\n");
    return 1;
}

/*
 * Maps to pfnPeriodicTimer
 */
#ifdef AMHPNA10
static int linuxtimer(void *ctx, void *timerapi)
{
    return 1;
}
#endif


static struct net_device_stats * nvnet_stats(struct net_device *dev)
{
    struct nvnet_private *priv = (struct nvnet_private *)dev->priv;
    ADAPTER_API *hw;
    ADAPTER_STATS stats;

    PRINTK(DEBUG_STATS, "nvnet_stats - In\n");
    hw = priv->hwapi;
    if (hw) 
    {    
        hw->pfnGetStatistics(hw->pADCX, &stats);
        PRINTK(DEBUG_STATS, "rx_packets = %x\n", stats.ulSuccessfulReceptions);

        priv->stats.rx_packets = stats.ulSuccessfulReceptions;
        priv->stats.tx_packets = stats.ulSuccessfulTransmissions;
        priv->stats.rx_errors  = stats.ulMissedFrames +
                                 stats.ulFailedReceptions +
                                 stats.ulCRCErrors +
                                 stats.ulFramingErrors +
                                 stats.ulOverFlowErrors ;

        priv->stats.tx_errors  = stats.ulFailedTransmissions +
                                 stats.ulRetryErrors +
                                 stats.ulUnderflowErrors +
                                 stats.ulLossOfCarrierErrors +
                                 stats.ulLateCollisionErrors;

        priv->stats.collisions = stats.ulLateCollisionErrors; 

        priv->stats.rx_over_errors = stats.ulOverFlowErrors;
        priv->stats.rx_crc_errors = stats.ulCRCErrors; 
        priv->stats.rx_frame_errors = stats.ulFramingErrors;
        priv->stats.rx_missed_errors = stats.ulMissedFrames;

        priv->stats.tx_carrier_errors = stats.ulLossOfCarrierErrors;
        

    }

    PRINTK(DEBUG_STATS, "nvnet_stats - Out\n");
    return &priv->stats;
}

/*
 * Get multicast addresses that OS is interested in
 */
static void nvnet_multicast(struct net_device *dev)
{
    struct nvnet_private *priv;
    struct dev_mc_list *mcptr;
    ADAPTER_API *hw;
    PACKET_FILTER hwfilter;
    u8 oraddr[6];
    u8 andaddr[6];
    int i;

    PRINTK(DEBUG_OPEN, "nvnet_multicast: In\n");

    priv = (struct nvnet_private *)dev->priv;
    hw = priv->hwapi;

    /*
     * Initialize filter
     */
    hwfilter.ulFilterFlags = 0;
    for(i = 0; i < 6; i++)
    {
        hwfilter.acMulticastAddress[i]  = 0;
        hwfilter.acMulticastMask[i]     = 0;
    }

    spin_lock_irqsave(&priv->lock, priv->lockflags);

    if(dev->flags & IFF_PROMISC)
    {
        hwfilter.ulFilterFlags |= ACCEPT_ALL_PACKETS;
    } /* if(dev->flags & IFF_PROMISC) */
    else if (dev->mc_count || (dev->flags & IFF_ALLMULTI))
    {
        hwfilter.ulFilterFlags |= ACCEPT_MULTICAST_PACKETS;
        hwfilter.ulFilterFlags |= ACCEPT_UNICAST_PACKETS;
        hwfilter.ulFilterFlags |= ACCEPT_BROADCAST_PACKETS;

    if (dev->flags & IFF_ALLMULTI)
    {
            hwfilter.acMulticastMask[0]    = 0xff;
            hwfilter.acMulticastAddress[0] = 0x01;
            hwfilter.acMulticastMask[1]    = 0xff;
            hwfilter.acMulticastAddress[1] = 0x00;
    } /* if (dev->flags & IFF_ALLMULTI) */
    else
    {

            /*
             * Process multicast address list
             */

            mcptr = dev->mc_list;
            for(i = 0; i < 6; i++)
                oraddr[i] = andaddr[i] = mcptr->dmi_addr[i];

            mcptr = mcptr->next;
            for(;mcptr; mcptr = mcptr->next)
            {
                PRINTK(DEBUG_OPEN, "multicast: addr len for addr: %d\n",\
                   mcptr->dmi_addrlen);
                for(i = 0; i < 6; i++)
                {
                    u8 mcaddr   = mcptr->dmi_addr[i];
                    andaddr[i]  &= mcaddr;
                    oraddr[i]   |= mcaddr;
                }
            }

            /* 
             * Write 1s to mask where all addresses have 0 bit or 1 bit. 
             * Compute mask and address value.
             */
            for(i = 0; i < 6; i++)
            {
                hwfilter.acMulticastAddress[i] = andaddr[i] & oraddr[i];
                hwfilter.acMulticastMask[i]    = andaddr[i] | (~oraddr[i]);
                PRINTK(DEBUG_OPEN, "multicast: mask[%d] = 0x%x addr[%d] = 0x%x\n",
                    i, hwfilter.acMulticastMask[i],
                    i, hwfilter.acMulticastAddress[i]);
            } /* for (i=0; i < 6; i++) */
        } /* else */

    } /* else if (dev->mc_count || (dev->flags & IFF_ALLMULTI)) */
    else
    {

        hwfilter.ulFilterFlags |= ACCEPT_UNICAST_PACKETS;
        hwfilter.ulFilterFlags |= ACCEPT_BROADCAST_PACKETS;
    } /* else */

    hw->pfnSetPacketFilter(hw->pADCX, &hwfilter);
    spin_unlock_irqrestore(&priv->lock, priv->lockflags);
    PRINTK(DEBUG_OPEN, "nvnet_multicast: Out\n");

}

/*
 * Send a packet
 */
static int nvnet_xmit(struct sk_buff *skb, struct net_device *dev)
{
    struct nvnet_private *priv;
    ADAPTER_WRITE_DATA txdata;
    ADAPTER_API *hw;
    int status;
    dma_addr_t addr;
    int retval = 0;
    nvnet_tx_info_os_t *tx_ring;
    unsigned long lockflags;

    PRINTK(DEBUG_TX, "nvnet_xmit : In\n");

    priv    = (struct nvnet_private *)dev->priv;
    hw      = priv->hwapi;

    if(!skb) 
    {

        PRINTK_ERROR("nvnet_xmit - invalid skb\n");
        PRINTK(DEBUG_TX, "nvnet_xmit : Out\n");
        return retval;
    }

    spin_lock_irqsave(&priv->lock, lockflags);

    NVNET_ASSERT (priv->tx_ring_count <= priv->tx_ring_entries);

    /*
     * Check if index is free
     */
    if (priv->tx_ring_count == priv->tx_ring_entries) 
    {
        /*
         * all tx ring entries are consumed 
         */
        priv->tx_ring_full = 1;
        PRINTK_ERROR("nvnet_xmit - skb - tx ring full\n");
        netif_stop_queue(dev);
        spin_unlock_irqrestore(&priv->lock, lockflags);
        return 1;

    }
    /*
     * Put the fragment information in tx ring
     */
    tx_ring = priv->tx_ring + priv->tx_ring_index;

    txdata.ulNumberOfElements = 1;
    txdata.pvID               =  (PVOID)tx_ring;

    tx_ring->pvID = (PVOID)skb;
    tx_ring->pCoalesceBuffer = NULL;

    addr  = cpu_to_le32(
               pci_map_single(priv->pdev, skb->data, skb->len, \
               PCI_DMA_TODEVICE));

    tx_ring->dma_addr[0] = addr;
    txdata.sElement[0].pPhysical = (void *)((u32)addr);
    txdata.ulTotalLength = txdata.sElement[0].ulLength = skb->len;

    status = hw->pfnWrite(hw->pADCX, &txdata);

    switch(status) 
    {
        case ADAPTERERR_NONE:
            priv->stats.tx_bytes += skb->len;
            priv->tx_ring_count++;
            priv->tx_ring_index = (priv->tx_ring_index + 1) %
                                   priv->tx_ring_entries;
            dev->trans_start = jiffies; 
            break;

        case ADAPTERERR_TRANSMIT_QUEUE_FULL:
            netif_stop_queue(dev);
            retval = 1;
            PRINTK_ERROR("nvnet_xmit -  queue full\n");
            break;

        default:
            PRINTK_ERROR("nvnet_xmit - transmit error\n");
            break;
    }

    spin_unlock_irqrestore(&priv->lock, priv->lockflags);
    PRINTK(DEBUG_TX, "nvnet_xmit - Out\n");
    return retval;

}


static int nvnet_config(struct net_device *dev, struct ifmap *map)
{
    return 0;
}

static int nvnet_open(struct net_device *dev)
{
    struct nvnet_private *priv;
    int status;


    PRINTK(DEBUG_OPEN, "nvnet_open: In\n");
    priv = (struct nvnet_private *)dev->priv;

    /*
     * Initialize hardware
     */
    PRINTK(DEBUG_OPEN, "nvnet_open: init hardware\n");
    status = priv->hwapi->pfnInit(priv->hwapi->pADCX, 
                                  speed, /* force speed */ 
                                  duplex, /* force full duplex */
                                  0, /* force mode */
                                  &priv->linkup);
    PRINTK(DEBUG_OPEN, "nvnet_open: init h/w complete\n");

    if(status != ADAPTERERR_NONE) 
    {
    free_irq(dev->irq, dev); 
        PRINTK_ERROR("nvnet_init - ADAPTER_Open failed\n");
        PRINTK(DEBUG_INIT, "nvnet_init -  Out\n");
        return -EAGAIN;
    }

    /* Set the node address */
    priv->hwapi->pfnSetNodeAddress(priv->hwapi->pADCX, dev->dev_addr);

    /*
     * Register the interrupt handler
     */
    PRINTK(DEBUG_OPEN, "nvnet_open: set interrupt\n");
    if(request_irq(dev->irq, &nvnet_interrupt, SA_SHIRQ, dev->name, dev)) 
    {

        PRINTK_ERROR("nvnet_open - request_irq failed\n");
        PRINTK(DEBUG_OPEN, "nvnet_open -  Out\n");
        return -EAGAIN;
    }

    /*
     * Turn on hardware
     */
    PRINTK(DEBUG_OPEN, "nvnet_open: start h/w\n");
    priv->hwapi->pfnStart(priv->hwapi->pADCX);

    PRINTK(DEBUG_OPEN, "nvnet_open: start queue\n");
    netif_start_queue(dev);
    PRINTK(DEBUG_OPEN, "nvnet_open - Out\n");
    return 0;
}


static void nvnet_interrupt(int irq, void *dev_instance, struct pt_regs *regs)
{
    struct net_device *dev;
    struct nvnet_private *priv;

    dev     = (struct net_device *)dev_instance;
    priv    = (struct nvnet_private *)dev->priv;

    PRINTK(DEBUG_INTR, "nvnet_interrupt - In (%d, %p, %p)\n", \
                 irq, dev_instance, regs);

    if(!dev)
        return;

    if(priv->hwapi->pfnQueryInterrupt(priv->hwapi->pADCX)) 
    {   
        priv->hwapi->pfnHandleInterrupt(priv->hwapi->pADCX);
        priv->hwapi->pfnEnableInterrupts(priv->hwapi->pADCX);

    }

    PRINTK(DEBUG_INTR, "nvnet_interrupt -  Out\n");

}

static int nvnet_close(struct net_device *dev)
{
    struct nvnet_private *priv;

    PRINTK(DEBUG_OPEN, "nvnet_close - In \n");
    netif_stop_queue(dev);

    priv = (struct nvnet_private *)dev->priv;


    /*
     * Release allocated resource for net_device
     */
    if(priv->hwapi->pfnDeinit)
        priv->hwapi->pfnDeinit(priv->hwapi->pADCX, false);


    free_irq(dev->irq, dev); 

    PRINTK(DEBUG_OPEN, "nvnet_close - OUT\n");
    return 0;
}

static int nvnet_ioctl(struct net_device *dev, struct ifreq *request, int cmd)
{
    u16 *data = (u16 *)&request->ifr_data;
    struct nvnet_private *priv;
    int retval  = 0;
    ADAPTER_API *hw;

    PRINTK(DEBUG_IOCTL, "nvnet_ioctl : In cmd = %x\n", cmd);
    priv = (struct nvnet_private *)dev->priv;
    hw = priv->hwapi;


    switch(cmd) 
    {
        case SIOCETHTOOL: 
        { 
               u32 ethcmd;
        
            if (copy_from_user(&ethcmd, (char *)request->ifr_data, sizeof(ethcmd))) {

                PRINTK_ERROR("copy_from_user failed\n");
                return -EFAULT;
            }

            switch(ethcmd) {
                case ETHTOOL_GDRVINFO: 
                {
                    struct ethtool_drvinfo info = {ETHTOOL_GDRVINFO};
                    
                    strcpy(info.driver, DRIVER_NAME);
                    strcpy(info.version, DRIVER_VERSION);

                    if (priv->pdev)
                        strcpy(info.bus_info, priv->pdev->slot_name);

                    if (copy_to_user((char *)request->ifr_data, &info, sizeof(info))) 
                    {
                        PRINTK_ERROR("nvnet_ioctl - copy_to_user failed\n");
                        return -EFAULT;
                    }      
                    break;
                 }
                default:
                    retval = -EOPNOTSUPP;
                    break;

                 }
            }
            break;

        case SIOCGMIIPHY:             /* Get address of MII PHY in use. */
        case SIOCDEVPRIVATE:          /* for binary compat, remove in 2.5 */
            data[0] = priv->phyaddr; 
            break;
        
        case SIOCGMIIREG:             /* Read MII PHY register. */
        case SIOCDEVPRIVATE+1:        /* for binary compat, remove in 2.5 */

            if(pass != ADAPTER_ReadPhy(hw->pADCX, priv->phyaddr, \
                                       data[1] & 0x1f, (ulong *)&data[3])) 
            {
                PRINTK_ERROR("nvnet_ioctl - ReadPhy failed\n");
                retval = -EBUSY;
            }
            break;
        case SIOCSMIIREG:             /* Write MII PHY register. */
        case SIOCDEVPRIVATE+2:        /* for binary compat, remove in 2.5 */

            if(pass != ADAPTER_WritePhy(hw->pADCX, priv->phyaddr, \
                                        data[1] & 0x1f, data[3])) 
            {
                 PRINTK_ERROR("nvnet_ioctl - WritePhy failed\n");
                 retval = -EBUSY; 
            }
            break;


        default:
            retval = -EOPNOTSUPP;
            break;
    }

    PRINTK(DEBUG_IOCTL, "nvnet_ioctl : out with ret =%x\n", retval);

    return retval;
}

static int nvnet_init(struct net_device *dev)
{
    u16 command, enable; //command register values in PCI config space
    OS_API *osapi;
    int status;
    struct nvnet_private *priv;
    u32 i;
    ADAPTER_OPEN_PARAMS        OpenParams;

    PRINTK(DEBUG_INIT, "nvnet_init - In\n");

    priv = (struct nvnet_private *)dev->priv;


    /* 
     * Check the status of the net_device and turn it on if necessary
     */
    pci_read_config_word(priv->pdev, PCI_COMMAND, &command);
    PRINTK(DEBUG_INIT, "nvnet_init - get config word 0x%x\n", command);

    /*
     * Make sure it can bus master and access memory mapped registers
     */
    enable = command | PCI_COMMAND_MASTER | PCI_COMMAND_MEMORY;

    if(command != enable) //net_device not enabled, turn it on
        pci_write_config_word(priv->pdev, PCI_COMMAND, enable);

    PRINTK(DEBUG_INIT, "nvnet_init - config word set\n");

    /*
     * Plug into the API
     */
    osapi = &priv->linuxapi;
    osapi->pOSCX = dev;
    osapi->pfnAllocMemory            = linuxalloc;
    osapi->pfnFreeMemory             = linuxfree;
    osapi->pfnClearMemory            = linuxclear;
    osapi->pfnAllocMemoryEx          = linuxallocex;
    osapi->pfnFreeMemoryEx           = linuxfreeex;
    osapi->pfnStallExecution         = linuxdelay;
    osapi->pfnAllocReceiveBuffer     = linuxallocrxbuf;
    osapi->pfnFreeReceiveBuffer      = linuxfreerxbuf;
    osapi->pfnPacketWasReceived      = linuxpacketrx;
    osapi->pfnPacketWasSent          = linuxpackettx;
    osapi->pfnLinkStateHasChanged    = linuxlinkchanged;
#ifdef AMHPNA10
    osapi->pfnPeriodicTimer          = linuxtimer;
#endif
    osapi->pfnLockAlloc              = linuxlockalloc;
    osapi->pfnLockAcquire            = linuxlockacquire;
    osapi->pfnLockRelease            = linuxlockrelease;

    priv->linkup    = true;

    PRINTK(DEBUG_INIT, "nvnet_init - call adapter open\n");
    OpenParams.MaxDpcLoop = 2;
    OpenParams.MaxRxPkt = 64;
    OpenParams.MaxTxPkt = 64;
    OpenParams.SentPacketStatusSuccess = 1;
    OpenParams.SentPacketStatusFailure = 0;
    OpenParams.MaxRxPktToAccumulate = 6;
    if (optimization == optimize_for_throughput)
    {
        OpenParams.ulPollInterval = 0;
    }
    else
    {
        OpenParams.ulPollInterval = 425;
    }
    OpenParams.SetForcedModeEveryNthRxPacket = 0;
    OpenParams.SetForcedModeEveryNthTxPacket = 0;
    OpenParams.RxForcedInterrupt = 0;
    OpenParams.TxForcedInterrupt = 0;
    OpenParams.pOSApi = osapi;
    OpenParams.pvHardwareBaseAddress = priv->regs;

    status = ADAPTER_Open (&OpenParams, (void **)&(priv->hwapi), 
                           &priv->phyaddr);

    if(status != ADAPTERERR_NONE) 
    {
        PRINTK_ERROR("nvnet_init - ADAPTER_Open failed\n");
        PRINTK(DEBUG_INIT, "nvnet_init -  Out\n");
        return -EAGAIN;
    }

    /* 
     * Mac address is loaded at boot time into h/w reg, but it's loaded backwards
     * we get the address, save it, and reverse it. The saved value is loaded
     * back into address at close time.
     */
    PRINTK(DEBUG_INIT, "nvnet_init - get mac address\n");
    priv->hwapi->pfnGetNodeAddress(priv->hwapi->pADCX, priv->original_mac_addr);

    for (i=0; i < 6; i++)
    {
        dev->dev_addr[i] = priv->original_mac_addr[6 - 1 - i];
    }

    /*
     * Tell hardware what the ethernet address is
     */
    PRINTK(DEBUG_INIT, "nvnet_init - set mac address\n");
    priv->hwapi->pfnSetNodeAddress(priv->hwapi->pADCX, dev->dev_addr);


    PRINTK(DEBUG_INIT, "nvnet_init - Out\n");
    return 0;
}



static int __devinit nvnet_probe(struct pci_dev *pdev, const struct pci_device_id *entry)
{
    char *memptr; /* pointer to devices hardware memory */
    struct net_device *dev; /* pointer to network driver struct */
    struct nvnet_private *priv; /* pointer to dev's private data */
    int initpower;
    int pm;
    int status;

    PRINTK(DEBUG_PROBE, "nvnet_probe - In\n");

    if (pci_enable_device (pdev)) 
    {

        PRINTK_ERROR("pci_enable_device failed\n");
        PRINTK(DEBUG_PROBE, "nvnet_probe - Out \n");
        return -EIO;
    }

    if (pci_set_dma_mask(pdev, 0xffffffff)) 
    {
        PRINTK_ERROR("nvnet, pci_set_dma_mask failed\n");
        PRINTK(DEBUG_PROBE, "nvnet_probe - Out \n");
        return -EIO;
    }


    pm = pci_find_capability(pdev, PCI_CAP_ID_PM);
    if(pm)
    {
        u16 power;
        pci_read_config_word(pdev, pm + PCI_PM_CTRL, &power);
        initpower = power & PCI_PM_CTRL_STATE_MASK; 
    }
    else
    {
        /*
         * No power management support
         */
         initpower = 0; 
    }


    status = pci_enable_device(pdev);

    if(status) 
        return status;


    if(!request_mem_region(pci_resource_start(pdev, 0),
                           pci_resource_len(pdev, 0),
                           "nvnet"))  
    {

        PRINTK_ERROR("probe - request_mem_region failed\n");
        pci_disable_device(pdev); 
        PRINTK(DEBUG_PROBE, "nvnet_probe - Out\n");
        return -ENODEV;
    }

    memptr = ioremap(pci_resource_start(pdev, 0),
                     pci_resource_len(pdev, 0));
    if(!memptr)
    {
        PRINTK_ERROR("probe - Couldn't remap memory\n");
        pci_disable_device(pdev); 
        release_mem_region(pci_resource_start(pdev, 0), 
                           pci_resource_len(pdev, 0));
        PRINTK(DEBUG_PROBE, "nvnet_probe - Out\n");
        return -ENODEV;
    }    

    dev = alloc_etherdev(sizeof(*priv));
    if (!dev) 
    {

        PRINTK_ERROR("probe - alloc_etherdev failed\n");
        pci_disable_device(pdev); 
        release_mem_region(pci_resource_start(pdev, 0), 
                           pci_resource_len(pdev, 0));
        PRINTK(DEBUG_PROBE, "nvnet_probe - Out\n");
        return -ENOMEM;

    }

    pci_set_drvdata(pdev, dev);
    priv = (struct nvnet_private *)dev->priv;

    /*
     * Allocate TX structures
     */
    priv->tx_ring_full       = 0;
    priv->tx_ring_entries    = DEFAULT_TX_RING_ENTRIES;
    priv->tx_ring_size       = priv->tx_ring_entries * 
                               sizeof(nvnet_tx_info_os_t);
    priv->tx_ring_index     = 0;
    priv->tx_ring_count     = 0;

    /*
     * Allocate space for the tx ring
     */
     priv->tx_ring = (nvnet_tx_info_os_t *)pci_alloc_consistent(
                      priv->pdev, priv->tx_ring_size,
                      &priv->tx_ring_dma);

    if (!priv->tx_ring) 
    {

        PRINTK_ERROR("nvnet_probe: Unable to allocate tx ring\n");
        pci_disable_device(pdev); 
        release_mem_region(pci_resource_start(pdev, 0), 
                           pci_resource_len(pdev, 0));
        kfree(dev);
        PRINTK(DEBUG_PROBE, "nvnet_probe - Out\n");
        return -ENOMEM;

    }
    spin_lock_init(&priv->lock); 
    /*
     * -1 means no power management support on the device
     */
    priv->initacpi    = initpower; 

    priv->pdev     = pdev; 

    SET_MODULE_OWNER(dev);
    priv->regs = (char *)memptr;
    pci_set_master(pdev);

    dev->base_addr    = pci_resource_start(pdev,0);
    dev->irq = pdev->irq;

    /*
     * Check if module parameters have been modified by the user
     */
    if (duplex != -1) 
    {

        char *str = NULL;
        if ((duplex < duplex_auto) || (duplex > duplex_full)) 
        {

            printk(KERN_INFO "nvnet: wrong value for duplex. Using default value.\n");
            duplex = DEFAULT_DUPLEX_SETTING;
        }
        switch(duplex) 
        {
            case 0: str = "auto"; break;
            case 1: str = "half"; break;
            case 2: str = "full"; break;
        }
        printk(KERN_INFO "nvnet: selecting duplex setting as %s duplex mode.\n", str);
    }
    else 
    {
          duplex = DEFAULT_DUPLEX_SETTING;
    }

    if (speed != -1) 
    {
        char *str = NULL;
        if ((speed < speed_auto) || (duplex > speed_100)) 
        {
            printk(KERN_INFO "nvnet: wrong value for speed. Using default value.\n");
            speed = DEFAULT_SPEED_SETTING;
        }
        switch(speed) 
        {
            case 0: str = "auto-negotiation"; break;
            case 1: str = "10 Mbps"; break;
            case 2: str = "100 Mbps"; break;
        }
        printk(KERN_NOTICE "nvnet: selecting speed settings as %s.\n", str);
    }
    else
    {
        speed = DEFAULT_SPEED_SETTING;
    }
     
    if (optimization != -1)
    {
        if ((optimization < optimize_for_throughput) || (optimization > optimize_for_cpu))
        {
            printk(KERN_NOTICE "nvnet: wrong value for optimization. Using default value 0x%x\n", 
                            DEFAULT_OPTIMIZATION_SETTING);

            optimization = DEFAULT_OPTIMIZATION_SETTING;
        }
        else
        {
            if (optimization == optimize_for_throughput)
            {
                printk(KERN_INFO "nvnet: Optimizing driver for throughput \n");
            }
            else
            {
                printk(KERN_INFO "nvnet: Optimizing driver for CPU \n");
            }
        }
    }
    else
    {
        optimization = DEFAULT_OPTIMIZATION_SETTING;
    }
    /*
     * Initialize the hardware
     */
    if(nvnet_init(dev))
    {
        PRINTK_ERROR("nvnet_probe: nvnet_init() failed\n");
        pci_disable_device(pdev); 
        release_mem_region(pci_resource_start(pdev, 0), 
                           pci_resource_len(pdev, 0));
        kfree(dev);
        PRINTK(DEBUG_PROBE, "nvnet_probe - Out\n");
    return -ENOMEM;
    }

    /* nvnet ethernet specific functions */
    dev->open               = nvnet_open;
    dev->stop               = nvnet_close;
    dev->set_config         = nvnet_config;
    dev->hard_start_xmit    = nvnet_xmit;
    dev->do_ioctl           = nvnet_ioctl;
    dev->get_stats          = nvnet_stats;
    dev->set_multicast_list = nvnet_multicast;
#if 0
    dev->tx_timeout         = nvnet_watchdog;
    dev->watchdog_timeo     = 1 * HZ; 
#endif

    register_netdev(dev); /* Let OS know about device */

    PRINTK(DEBUG_PROBE, "nvnet_probe: Out\n");
    return 0;

}


/*
 * Shutdown the hardware
 */
static void __devexit nvnet_remove(struct pci_dev *pdev)
{
    struct net_device *dev = pdev->driver_data;
    struct nvnet_private *priv = dev->priv;
    int i;

    PRINTK(DEBUG_OPEN, "nvnet_remove: IN\n");
    if (!dev) 
    {
        PRINTK_ERROR("nvnet_remove called with NULL device\n");
        return;
    }

    /*
     * Reload unreversed address back into MAC so we always start from a known state
     */
    for (i=0; i < 6; i++)
    {
        dev->dev_addr[i] = priv->original_mac_addr[i];
    }    
    priv->hwapi->pfnSetNodeAddress(priv->hwapi->pADCX, dev->dev_addr); /* save back into register */

    if(priv->hwapi->pfnClose)
    {
        priv->hwapi->pfnClose(priv->hwapi->pADCX);
        priv->hwapi = NULL;
    }

    unregister_netdev(dev);

    /* 
     * Free the transmit ring
     */
    pci_free_consistent(priv->pdev, priv->tx_ring_size,
                        priv->tx_ring, priv->tx_ring_dma);

    release_mem_region(pci_resource_start(pdev, 0), pci_resource_len(pdev,0));

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,5)
    pci_disable_device(pdev);
#endif
    kfree(dev);
    pci_set_drvdata(pdev, NULL);
    PRINTK(DEBUG_OPEN, "nvnet_remove -  Out\n");
}

#ifdef MODULE

static int __init nvnet_init_module(void)
{
    PRINTK(DEBUG_INIT, "nvnet_init_module - In\n");
    return  pci_module_init(&nvnet_driver);
}

static void __exit nvnet_cleanup_module(void)
{
    PRINTK(DEBUG_OPEN, "nvnet_cleanup_module - In\n");
    pci_unregister_driver(&nvnet_driver);
    PRINTK(DEBUG_OPEN, "nvnet_cleanup_module - Out\n");
}

module_init(nvnet_init_module);
module_exit(nvnet_cleanup_module);

#endif /* MODULE */


#ifdef CONFIG_PM

int nvnet_suspend(struct pci_dev *pcidev, u32 state)
{
    struct net_device *dev;
    struct nvnet_private *priv;

    PRINTK(DEBUG_OPEN, "nvnet_suspend - In\n");

    dev = (struct net_device *)pci_get_drvdata(pcidev);
    priv = (struct nvnet_private *)dev->priv;

    if(!netif_running(dev)) 
    {
        PRINTK(DEBUG_OPEN, "nvnet_suspend - Out, interface !running\n");
        return 0;
    }

    netif_device_detach(dev);
    nvnet_close(dev);

    PRINTK(DEBUG_OPEN, "nvnet_suspend - Out\n");

    return 0;
}

int nvnet_resume(struct pci_dev *pcidev)
{
    struct net_device *dev;
    struct nvnet_private *priv;

    PRINTK(DEBUG_OPEN, "nvnet_resume - In\n");

    dev  = (struct net_device *)pci_get_drvdata(pcidev);
    priv = (struct nvnet_private *)dev->priv;

    if (!dev) return 0;
    if (!dev->priv) return 0;

    if(!netif_running(dev))
        return 0;

    netif_device_attach(dev);
    
    nvnet_open(dev);
    PRINTK(DEBUG_OPEN, "nvnet_resume - Out\n");
    return 0;
}

#endif //CONFIG_PM

#if 0
static void nvnet_watchdog(struct net_device *dev)
{
    PRINTK(DEBUG_OPEN, "nvnet_watchdog - In\n");
    nvnet_close(dev);
    nvnet_open(dev);
    PRINTK(DEBUG_OPEN, "nvnet_watchdog - Out\n");
}
#endif

