/**********************************************************************
*
*  nvhw.c
*
*  Descripion  - hw access details and common functions needed for channel
*
*  Copyright (c) 2002-2003 NVIDIA Corporation
*
***********************************************************************
*/
#define __NO_VERSION__
#include <linux/module.h>
#include <linux/version.h>
#include <asm/uaccess.h>

#include "nvhw.h"
#include "nvrec.h"
#include "nvwavout.h"
#include "nvspdif.h"

extern int mastervolright;
extern int mastervolleft;
extern int pcmvolright;
extern int pcmvolleft;

extern short SPDIFLeftovers[6];
extern u32   SPDIFLeftoverCount;
extern short AnalogLeftovers[6];
extern u32   AnalogLeftoverCount;


/* allocate DMA buffer, playback and recording buffer should be allocated seperately */
int alloc_dmabuf(struct Nvaudio_state *state)
{
    void *rawbuf = NULL;
    int order, size;
    struct page *page, *pend;
    struct dmabuf *dmabuf = &state->dmabuffer;

    /* If we don't have any oss frag params, then use our default ones */
    if(dmabuf->ossmaxfrags == 0)
        dmabuf->ossmaxfrags = 4;

    if(dmabuf->ossfragsize == 0)
        dmabuf->ossfragsize = (PAGE_SIZE<<DMABUF_DEFAULTORDER)/dmabuf->ossmaxfrags;

    size = dmabuf->ossfragsize * dmabuf->ossmaxfrags;

    if(dmabuf->rawbuf && (PAGE_SIZE << dmabuf->buforder) == size){
        return 0;
    }

    /* alloc enough to satisfy the oss params */
    for (order = DMABUF_DEFAULTORDER; order >= DMABUF_MINORDER; order--) {

        if ( (PAGE_SIZE<<order) > size )
            continue;

        if ((rawbuf = pci_alloc_consistent(state->card->pci_dev,
                           PAGE_SIZE << order,
                           &dmabuf->dma_handle)))
            break;

    }

    if (!rawbuf)
        return -ENOMEM;


#ifdef NV_DEBUG_NORMAL
    printk("Nvaudio: allocated %ld (order = %d) bytes at %p\n",
           PAGE_SIZE << order, order, rawbuf);
#endif

    state->ready     = state->mapped = 0;
    dmabuf->rawbuf   = rawbuf;
    dmabuf->buforder = order;

    /* now mark the pages as reserved; otherwise remap_page_range doesn't do what we want */

    pend = virt_to_page(rawbuf + (PAGE_SIZE << order) - 1);
    for (page = virt_to_page(rawbuf); page <= pend; page++)
        mem_map_reserve(page);

    return 0;
}

/* free DMA buffer */

void dealloc_dmabuf(struct Nvaudio_state *state)
{
    struct page *page, *pend;
    struct dmabuf *dmabuf = &state->dmabuffer;

    if (dmabuf->rawbuf) {
        /* undo marking the pages as reserved */

        pend = virt_to_page(dmabuf->rawbuf + (PAGE_SIZE << dmabuf->buforder) - 1);
        for (page = virt_to_page(dmabuf->rawbuf); page <= pend; page++)
            mem_map_unreserve(page);

        pci_free_consistent(state->card->pci_dev, PAGE_SIZE << dmabuf->buforder,
                    dmabuf->rawbuf, dmabuf->dma_handle);

#ifdef NV_DEBUG_NORMAL
    printk("Nvaudio: dealloc_dmabuf bytes at %p\n",dmabuf->rawbuf);
#endif

    }
    dmabuf->rawbuf = NULL;
    state->mapped  = state->ready = 0;
}

/* function that program the prds for the channels*/
int prog_dmabuf(struct Nvaudio_state *state,unsigned rec)
{
    struct sg_item *sg;
    unsigned long flags;
    int ret = 0, i = 0, channelnum = 0;
    unsigned fragint = 0;
    unsigned long port = 0;
    struct dmabuf *dmabuf = &state->dmabuffer;

    spin_lock_irqsave(&state->lock, flags);

    if(state->enable & DAC_RUNNING){
        __stop_dac(state);
    }

    if(state->enable & ADC_RUNNING){
        __stop_adc(state);
    }

    if(state->enable & SPDIF_RUNNING) {
        __stop_spdif(state);
    }

    dmabuf->total_bytes = 0;
    dmabuf->count = dmabuf->error = 0;
    dmabuf->swptr = dmabuf->hwptr = 0;
    spin_unlock_irqrestore(&state->lock, flags);

    /* allocate DMA buffer, let alloc_dmabuf determine if we are already
     * allocated well enough or if we should replace the current buffer
     * (assuming one is already allocated, if it isn't, then allocate it).
     */
    if ((ret = alloc_dmabuf(state)))
        return ret;

    dmabuf->dmasize      = PAGE_SIZE << dmabuf->buforder;
    dmabuf->numfrag      = SG_LEN;
    dmabuf->fragsize     = dmabuf->dmasize/dmabuf->numfrag;
    dmabuf->fragsamples  = dmabuf->fragsize >> 1;
    dmabuf->userfragsize = dmabuf->ossfragsize;
    dmabuf->userfrags    = dmabuf->dmasize/dmabuf->ossfragsize;

    memset(dmabuf->rawbuf, 0, dmabuf->dmasize);

    if(dmabuf->ossmaxfrags == 4) {
        fragint = 8;
        dmabuf->fragshift = 2;
    } else if (dmabuf->ossmaxfrags == 8) {
        fragint = 4;
        dmabuf->fragshift = 3;
    } else if (dmabuf->ossmaxfrags == 16) {
        fragint = 2;
        dmabuf->fragshift = 4;
    } else {
        fragint = 1;
        dmabuf->fragshift = 5;
    }

#ifdef NV_DEBUG_NORMAL
        printk("Nvaudio: prog_dmabuf, fragint = %x \n",fragint);
#endif

    port = state->card->iobase;

    switch(rec)    {
     case 0:  /*audio out */
        port += PO_BDBAR;
        channelnum = 0;
        break;
     case 1:  /*audio in */
        port += PI_BDBAR;
        channelnum = 1;
        break;
     case 2:  /*spdif out */
        port += SP_BDBAR;
        channelnum = 2;
        break;
    }

    if(! state->card->channel) return ret;

    sg=&state->card->channel[channelnum].sg[0];

    for(i=0;i<dmabuf->numfrag;i++)
    {
        sg->busaddr=(u32)dmabuf->dma_handle+ (dmabuf->fragsize*i);

        // the card will always be doing 16bit stereo
        sg->control=dmabuf->fragsamples;

        sg->control|=CON_BUFPAD;

        // set us up to get IOC interrupts as often as needed to
        // satisfy numfrag requirements, no more
        if( ((i+1) % fragint) == 0) {
            sg->control|=CON_IOC;
        }

        sg++;
    }

    spin_lock_irqsave(&state->lock, flags);
    outb(2, port+OFF_CR);         /* reset DMA machine */
    while(inb(port+OFF_CR)&0x02); /* wait until reset done */
    outl((u32)(state->card->chandma + (channelnum * sizeof(struct Nvaudio_channel))),
                port+OFF_BDBAR);
    spin_unlock_irqrestore(&state->lock, flags);

    /* set the ready flag for the dma buffer */
    state->ready = 1;

#ifdef NV_DEBUG_NORMAL
    printk("Nvaudio: prog_dmabuf, sample rate = %d, format = %d,\n\tnumfrag = %d, "
           "fragsize = %d dmasize = %d\n",
           state->rate, state->fmt, dmabuf->numfrag,
           dmabuf->fragsize, dmabuf->dmasize);
#endif

    return 0;
}


/* Write AC97 codec registers */
/* TO DO :- check whether to use card spinlock on accessing codec registers */

u16 Nvaudio_ac97_get(struct ac97_codec *dev, u8 reg)
{
    struct Nvaudio_card *card = dev->private_data;
    int count  = 100;
    u8 reg_set = ((dev->id)?((reg&0x7f)|0x80):(reg&0x7f));

    while(count-- && (inb(card->iobase + CAS) & 1))
        udelay(1);

    return inw(card->ac97base + reg_set);
}

void Nvaudio_ac97_set(struct ac97_codec *dev, u8 reg, u16 data)
{
    struct Nvaudio_card *card = dev->private_data;
    int count  = 100;
    u8 reg_set = ((dev->id)?((reg&0x7f)|0x80):(reg&0x7f));

    while(count-- && (inb(card->iobase + CAS) & 1))
        udelay(1);
    outw(data, card->ac97base + reg_set);
}

/* Function to recalculate the free space based on the downmix info */
int Nvaudio_Recalfreespace(struct Nvaudio_card* card,int cnt)
{
    int result = 0;
    unsigned int i_glob_cnt = 0;
    switch( card->numchannels) {
        case CHANNEL_51:    
            switch(card->spkrselect) {
                case NV_SPKR_STEREO:
                    result = cnt + (((cnt >> 1) / CHANNEL_51) * ( 4 * sizeof(short)));           
                    break;
                case NV_SPKR_QUAD:
                    result = cnt + (((cnt >> 1) / CHANNEL_51) * ( 2 * sizeof(short)));           
                    break;
                case NV_SPKR_51:
                    result = cnt;
                    break;
            }
            break;
        case CHANNEL_QUAD:
            switch(card->spkrselect) {
                case NV_SPKR_STEREO:
                    result = cnt + (((cnt >> 1) / CHANNEL_QUAD) *( 2 * sizeof(short)));           
                    break;
                case NV_SPKR_QUAD:
                    result = cnt;
                    break;
                case NV_SPKR_51:
                    result = cnt;
                    break;
            }
            break;
        case CHANNEL_STEREO:
            switch(card->spkrselect) {
                case NV_SPKR_STEREO:
                    result = cnt;
                    break;
                case NV_SPKR_QUAD:
                    
                    /* Temp hack added as mplayer isnot sending the ioctl for
                    channel stereo - so we always open the wave as stereo and 
                    set the spkr config here - later find a better solution */
                    i_glob_cnt = inl(card->iobase + GLOB_CNT);
                    outl((i_glob_cnt & 0xcfffff) | 0x100000,card->iobase + GLOB_CNT);
                    result = cnt >> 1;
                    break;
                case NV_SPKR_51:
                    /* Temp hack added as mplayer isnot sending the ioctl for
                    channel stereo - so we always open the wave as stereo and 
                    set the spkr config here - later find a better solution */
                    i_glob_cnt = inl(card->iobase + GLOB_CNT);
                    outl((i_glob_cnt & 0xcfffff) | 0x100000,card->iobase + GLOB_CNT);
                    result = cnt >> 1;
                    break;
            }
            break;
    }

    return result;

}

/* Set the Multichannel Volume registers - temp */
int Nvaudio_MultiChannel_volume(struct Nvaudio_card* card, int channel)
{
    struct ac97_codec *codec    = card->ac97_codec[0];
    struct ac97_codec *seccodec = NULL;

     switch (card->codecType){
        case ADI_CODEC:
            break;
        case SIGMATEL_CODEC:
            {
                seccodec = card->ac97_codec[1];
                if(seccodec) {
                    if(channel >= CHANNEL_QUAD) { 
                         Nvaudio_ac97_set(seccodec,AC97_MASTER_VOL_STEREO,
                            Nvaudio_ac97_get(codec,AC97_MASTER_VOL_STEREO));
                    }
                    
                    if(channel == CHANNEL_51) {
                         Nvaudio_ac97_set(seccodec,AC97_SURROUND_MASTER,
                            Nvaudio_ac97_get(codec,AC97_MASTER_VOL_STEREO));
                    }
                }
            }
            break;
        default:
            if(channel >= CHANNEL_QUAD) {
                Nvaudio_ac97_set(codec,AC97_SURROUND_MASTER,
                    Nvaudio_ac97_get(codec,AC97_MASTER_VOL_STEREO));
                Nvaudio_ac97_set(codec,0x64,
                    Nvaudio_ac97_get(codec,AC97_PCMOUT_VOL));
            }
            
            if(channel == CHANNEL_51) {
                Nvaudio_ac97_set(codec,AC97_CENTER_LFE_MASTER,
                    Nvaudio_ac97_get(codec,AC97_MASTER_VOL_STEREO));
                Nvaudio_ac97_set(codec,0x66,
                    Nvaudio_ac97_get(codec,AC97_PCMOUT_VOL));
            }
            
            break;
    }
    return 0;
}

/* Function to set the GLOBAL CTRL register based on Speaker Selection */
void Nvaudio_SetSpkrRegister(struct Nvaudio_card* card)
{
    unsigned int   i_glob_cnt;
    i_glob_cnt = inl(card->iobase + GLOB_CNT);

    switch (card->spkrselect) {
        case NV_SPKR_QUAD:
            outl((i_glob_cnt & 0xcfffff) | 0x100000,
              card->iobase + GLOB_CNT);
            break;
        case  NV_SPKR_51:
            outl((i_glob_cnt & 0xcfffff) | 0x200000,
                  card->iobase + GLOB_CNT);
            break;
        case NV_SPKR_STEREO:
            outl(i_glob_cnt & 0xcfffff,
                    card->iobase + GLOB_CNT);
            break;
        default:
            break;
    }
    /* Do we need to set the Mixer regs here */

    /* Temp hack added for few selection now - will remove later */
    if(((card->numchannels == CHANNEL_QUAD) || (card->numchannels == CHANNEL_STEREO)) && (card->spkrselect == NV_SPKR_51)) {
        /* set to four spkr now - until center channel is created */
        outl((i_glob_cnt & 0xcfffff) | 0x100000,
                  card->iobase + GLOB_CNT);
    }

    
}

/* Function to Store the leftover data for copying */
void Nvaudio_StoreLeftOver(int loop, int loopend, short** pSrc, int bSpdif)
{
    int newloop = 0, diff =0;
    short *pSource = NULL; 

    pSource = (short *) *pSrc;
    
    if(loop != loopend) {

        diff     = loopend - loop;
      
        if(bSpdif) {  // spdif
            newloop  = SPDIFLeftoverCount;
            diff    += SPDIFLeftoverCount;
            for(; newloop < diff; newloop++) {
                SPDIFLeftovers[newloop] = *pSource++;
                SPDIFLeftoverCount += 1;
            }
        }

        else { //Analog
            newloop  = AnalogLeftoverCount;
            diff    += AnalogLeftoverCount;
            for(; newloop < diff; newloop++) {
                AnalogLeftovers[newloop] = *pSource++;
                AnalogLeftoverCount += 1;
            }

        }
    } else  {

        if(bSpdif) {
            SPDIFLeftoverCount = 0;
        }
        else {
            AnalogLeftoverCount = 0;
        }
    }

    //make sure we  update pointer
    *pSrc = pSource;

}

/*  Function to use the Leftover with Spkrselect as Quad and 2 channel wave*/
u32 Nvaudio_UseLeftOverToStereo(short** pIn, short** pOut, int loopend,int carrylimit, u32* StartIndex)
{
    u32 SampleCnt  = 0;
    int shiftdone  = FALSE;
    long left=0,   right=0;
    short *pDest = NULL, *pSource = NULL; 
    
    pSource = (short *) *pIn;
    pDest   = (short *) *pOut;

    if(AnalogLeftoverCount > 0) {
        left = AnalogLeftovers[0];
        
        switch(AnalogLeftoverCount)
        {
            case 1:
                if(loopend >= (carrylimit - 1)){
                    right = *pSource++;
                    shiftdone = TRUE;
                }
                break;
        }

        if(shiftdone == FALSE) {
            return SampleCnt;
        }

        *pDest++ = (short)left;    
        *pDest++ = (short)right;   
        *pDest++ = (short)left;    
        *pDest++ = (short)right;   

        *StartIndex = carrylimit - AnalogLeftoverCount;
        SampleCnt   = 2;
        AnalogLeftoverCount = 0;

    }

    // Make sure we update the pointers
    *pIn  = pSource;
    *pOut = pDest;

    return SampleCnt;

}

/*  Function to use the Leftover with Spkrselect as Quad and 5.1 channel*/
u32 Nvaudio_UseLeftOverToQuad(short** pIn, short** pOut, int loopend,int carrylimit, u32* StartIndex, int bSpdif)
{
    u32 SampleCnt  = 0;
    int shiftdone  = FALSE;
    long left=0,right=0,center=0,lfe=0,sleft=0,sright=0;
    short *pDest = NULL, *pSource = NULL; 
    float sMix1  = 0.0, sMix2 = 0.0;

    pSource = (short *) *pIn;
    pDest   = (short *) *pOut;

    if((AnalogLeftoverCount > 0) && (!bSpdif)) {

        left = AnalogLeftovers[0];
        
        switch(AnalogLeftoverCount)
        {
            case 1:
                if(loopend >= (carrylimit - 1)) {
                    right  = *pSource++;
                    center = *pSource++;
                    lfe    = *pSource++;
                    sleft  = *pSource++;
                    sright = *pSource++;
                    shiftdone = TRUE;
                }
                break;

            case 2:
                if(loopend >= (carrylimit - 2)) {
                    right  = AnalogLeftovers[1];
                    center = *pSource++;
                    lfe    = *pSource++;
                    sleft  = *pSource++;
                    sright = *pSource++;
                    shiftdone = TRUE;
                }
                break;
            case 3:
                if(loopend >= (carrylimit - 3)) {
                    right  = AnalogLeftovers[1];
                    center = AnalogLeftovers[2];
                    lfe    = *pSource++;
                    sleft  = *pSource++;
                    sright = *pSource++;
                    shiftdone = TRUE;
                }
                break;
            case 4:
                if(loopend >= (carrylimit - 4)) {
                    right  = AnalogLeftovers[1];
                    center = AnalogLeftovers[2];
                    lfe    = AnalogLeftovers[3];
                    sleft  = *pSource++;
                    sright = *pSource++;
                    shiftdone = TRUE;
                }
                break;
            case 5:
                if(loopend >= (carrylimit - 5)) {
                    right  = AnalogLeftovers[1];
                    center = AnalogLeftovers[2];
                    lfe    = AnalogLeftovers[3];
                    sleft  = AnalogLeftovers[4];
                    sright = *pSource++;
                    shiftdone = TRUE;
                }
                break;
            default:
                break;

        } // switch

        if(shiftdone == FALSE) {
            return SampleCnt;
        }

        sMix1 = 0.707f * center;
        sMix2 = 0.707f * lfe;
        left   += (long)(sMix1 + sMix2);
        right  += (long)(sMix1 + sMix2);
        sleft  += (long)sMix2;
        sright += (long)sMix2;

        RANGECHECK(left,MINVOLRANGE,MAXVOLRANGE);
        RANGECHECK(right,MINVOLRANGE,MAXVOLRANGE);
        RANGECHECK(sleft,MINVOLRANGE,MAXVOLRANGE);
        RANGECHECK(sright,MINVOLRANGE,MAXVOLRANGE);
        
        *pDest++ = (short)left;
        *pDest++ = (short)right;
        *pDest++ = (short)sleft;
        *pDest++ = (short)sright;
        
        *StartIndex = carrylimit - AnalogLeftoverCount;
        SampleCnt   = 2;
        AnalogLeftoverCount = 0;
    } // if

    // Make sure we update the pointers
    *pIn  = pSource;
    *pOut = pDest;

    return SampleCnt;

}

/* Function to use the left over data */
u32 Nvaudio_UseLeftOver(short** pIn, short** pOut, int loopend,int carrylimit, u32* StartIndex, int bSpdif, int ToQuad)
{
    u32 SampleCnt  = 0;
    int shiftdone  = FALSE;
    long left=0,right=0,center=0,lfe=0,sleft=0,sright=0;
    short *pDest = NULL, *pSource = NULL; 
    
    pSource = (short *) *pIn;
    pDest   = (short *) *pOut;

    if((AnalogLeftoverCount > 0) && (!bSpdif)) {

        left = AnalogLeftovers[0];
        
        switch(AnalogLeftoverCount)
        {
            case 1:
                if(loopend >= (carrylimit - 1)) {
                    right = *pSource++;                
                    if(carrylimit == CHANNEL_51) {
                        center = *pSource++;
                        lfe    = *pSource++;
                    }
                    sleft  = *pSource++;
                    sright = *pSource++;
                    shiftdone = TRUE;
                }
                break;

            case 2:
                if(loopend >= (carrylimit - 2)) {
                    right  = AnalogLeftovers[1];                
                    if(carrylimit == CHANNEL_51) {
                        center = *pSource++;
                        lfe    = *pSource++;
                    }
                    sleft  = *pSource++;
                    sright = *pSource++;
                    shiftdone = TRUE;
                }
                break;
            case 3:
                if(loopend >= (carrylimit - 3)) {
                    right  = AnalogLeftovers[1];
                    if(carrylimit == CHANNEL_51) {
                        center = AnalogLeftovers[2];
                        lfe    = *pSource++;
                        sleft  = *pSource++;
                    }
                    else if(carrylimit == CHANNEL_QUAD) {
                        sleft = AnalogLeftovers[2];
                    }
                    sright = *pSource++;
                    shiftdone = TRUE;
                }
                break;
            case 4:
                if(loopend >= (carrylimit - 4)) {
                    right  = AnalogLeftovers[1];
                    if(carrylimit == CHANNEL_51) {
                        center = AnalogLeftovers[2];
                        lfe    = AnalogLeftovers[3];
                        sleft  = *pSource++;
                        sright = *pSource++;
                    }
                    shiftdone = TRUE;
                }
                break;
            case 5:
                if(loopend >= (carrylimit - 5)) {
                    right  = AnalogLeftovers[1];
                    if(carrylimit == CHANNEL_51) {
                        center = AnalogLeftovers[2];
                        lfe    = AnalogLeftovers[3];
                        sleft  = AnalogLeftovers[4];
                        sright = *pSource++;
                    }
                    shiftdone = TRUE;
                }
                break;
            default:
                break;

        } // switch

        if(shiftdone == FALSE) {
            return SampleCnt;
        }

        if(ToQuad == NV_SPKR_STEREO) {
        
            left  = Nvaudio_MixDown(left,center,lfe,sleft);
            right = Nvaudio_MixDown(right,center,lfe,sright);
        
            RANGECHECK(left,MINVOLRANGE,MAXVOLRANGE);
            RANGECHECK(right,MINVOLRANGE,MAXVOLRANGE);
            
        }
        
        *pDest++ = (short)left;
        *pDest++ = (short)right;
        
        *StartIndex = carrylimit - AnalogLeftoverCount;
        SampleCnt = 1;
        AnalogLeftoverCount = 0;

        if(ToQuad == NV_SPKR_51) {
            *pDest++ = (short)center;
            *pDest++ = (short)lfe;
            SampleCnt += 1;
        }
        
        if((ToQuad == NV_SPKR_QUAD) || (ToQuad == NV_SPKR_51)) {
            *pDest++ = (short)sleft;
            *pDest++ = (short)sright;
            SampleCnt += 1;
        }

    } // if

    else if( (SPDIFLeftoverCount > 0) && (bSpdif)) {

        left = SPDIFLeftovers[0];
        
        switch(SPDIFLeftoverCount)
        {
            case 1:
                if(loopend >= (carrylimit - 1)) {
                    right = *pSource++;
                    if(carrylimit == CHANNEL_51) {
                        center = *pSource++;
                        lfe    = *pSource++;
                    }
                    sleft  = *pSource++;
                    sright = *pSource++;
                    shiftdone = TRUE;
                }
                break;

            case 2:
                if(loopend >= (carrylimit - 2)) {
                    right  = SPDIFLeftovers[1];
                    if(carrylimit == CHANNEL_51) {
                        center = *pSource++;
                        lfe    = *pSource++;
                    }
                    sleft  = *pSource++;
                    sright = *pSource++;
                    shiftdone = TRUE;
                }
                break;
            case 3:
                if(loopend >= (carrylimit - 3)) {
                    right  = SPDIFLeftovers[1];
                    if(carrylimit == CHANNEL_51) {
                        center = SPDIFLeftovers[2];
                        lfe    = *pSource++;
                        sleft  = *pSource++;
                    }
                    else if(carrylimit == CHANNEL_QUAD) {
                        sleft = SPDIFLeftovers[2];
                    }
                    sright = *pSource++;
                    shiftdone = TRUE;
                }
                break;
            case 4:
                if(loopend >= (carrylimit - 4)) {
                    right  = SPDIFLeftovers[1];
                    if(carrylimit == CHANNEL_51) {
                        center = SPDIFLeftovers[2];
                        lfe    = SPDIFLeftovers[3];
                        sleft  = *pSource++;
                        sright = *pSource++;
                    }
                    shiftdone = TRUE;
                }
                break;
            case 5:
                if(loopend >= (carrylimit - 5)) {
                    right  = SPDIFLeftovers[1];
                    if(carrylimit == CHANNEL_51) {
                        center = SPDIFLeftovers[2];
                        lfe    = SPDIFLeftovers[3];
                        sleft  = SPDIFLeftovers[4];
                        sright = *pSource++;
                    }
                    shiftdone = TRUE;
                }
                break;
            default:
                break;

        } // switch

        if(shiftdone == FALSE) {
            return SampleCnt;
        }

        left  = Nvaudio_MixDown(left,center,lfe,sleft);
        right = Nvaudio_MixDown(right,center,lfe,sright);
        
        RANGECHECK(left,MINVOLRANGE,MAXVOLRANGE);
        RANGECHECK(right,MINVOLRANGE,MAXVOLRANGE);
        
        /* apply the volume */
        left   = ( left * mastervolleft * pcmvolleft) / 10000; 
        right  = ( right * mastervolright * pcmvolright) / 10000; 
        
        *pDest++ = (short)left;
        *pDest++ = (short)right;
        
        *StartIndex = carrylimit - SPDIFLeftoverCount;
        SampleCnt = 1;
        SPDIFLeftoverCount = 0;

    } // if
    // Make sure we update the pointers
    *pIn  = pSource;
    *pOut = pDest;

    return SampleCnt;

}

/*
   Function that helps to mix down the data -
   used for spdif and analog with speaker selection
*/
long Nvaudio_MixDown( long channel, long center, long lfe, long schannel)
{
     float result;
     result  = channel + 0.7f * ( schannel + center + lfe);
     return result;
}

/*
    Function to mix down - QUAD TO STEREO
*/
u32  Nvaudio_MixDownQuadToStereo(const char* pIn,void *pOut,int cnt, int carrylimit, int bSpdif)
{
    u32 Samplecnt = 0 ,StartIndex = 0;
    long left=0,right=0,sleft=0,sright=0;
    short *pDest = NULL, *pSource = NULL; 
    int loop = 0;

    pSource = (short *) pIn;
    pDest   = (short *) pOut;

    Samplecnt = Nvaudio_UseLeftOver(&pSource,&pDest,cnt,carrylimit,&StartIndex,bSpdif, NV_SPKR_STEREO);

    for(loop= StartIndex;loop <= (cnt - carrylimit);loop += carrylimit) {
        left   = *pSource++;
        right  = *pSource++;
        sleft  = *pSource++;
        sright = *pSource++;

        left  = Nvaudio_MixDown(left,0,0,sleft);
        right = Nvaudio_MixDown(right,0,0,sright);
        
        RANGECHECK(left,MINVOLRANGE,MAXVOLRANGE);
        RANGECHECK(right,MINVOLRANGE,MAXVOLRANGE);
        
        if(bSpdif) {
            /* apply the volume */
            left   = ( left * mastervolleft * pcmvolleft) / 10000; 
            right  = ( right * mastervolright * pcmvolright) / 10000; 
        }

        *pDest++ = (short)left;
        *pDest++ = (short)right;
        
        Samplecnt += 1;
    }

    //Check for leftovers
    Nvaudio_StoreLeftOver(loop,cnt,&pSource,bSpdif);

    return Samplecnt;
}

/*
    Function to mix down - 5.1 TO STEREO
*/
u32  Nvaudio_MixDown5_1ToStereo(const char* pIn,void *pOut,int cnt, int carrylimit, int bSpdif)
{
    u32 Samplecnt = 0 ,StartIndex = 0;
    long left=0,right=0,center=0, lfe=0, sleft=0,sright=0;
    short *pDest = NULL, *pSource = NULL; 
    int loop = 0;

    pSource = (short *) pIn;
    pDest   = (short *) pOut;

    Samplecnt = Nvaudio_UseLeftOver(&pSource,&pDest,cnt,carrylimit,&StartIndex,bSpdif,NV_SPKR_STEREO);

    for(loop= StartIndex;loop <= (cnt - carrylimit);loop += carrylimit) {
        left   = *pSource++;
        right  = *pSource++;
        center = *pSource++;
        lfe    = *pSource++;
        sleft  = *pSource++;
        sright = *pSource++;
        
        left  = Nvaudio_MixDown(left,center,lfe,sleft);
        right = Nvaudio_MixDown(right,center,lfe,sright);
        
        RANGECHECK(left,MINVOLRANGE,MAXVOLRANGE);
        RANGECHECK(right,MINVOLRANGE,MAXVOLRANGE);

        if(bSpdif) {
            /* apply the volume */
            left   = ( left * mastervolleft * pcmvolleft) / 10000; 
            right  = ( right * mastervolright * pcmvolright) / 10000; 
        }

        *pDest++ = (short)left;
        *pDest++ = (short)right;
        
        Samplecnt += 1;
    }

    //Check for leftovers
    Nvaudio_StoreLeftOver(loop,cnt,&pSource,bSpdif);

    return Samplecnt;
}

/*
    Function to do normal mix down 
*/
u32  Nvaudio_MixNormal(const char* pIn,void *pOut,int cnt, int carrylimit, int bSpdif)
{
    u32 Samplecnt = 0 ,StartIndex = 0;
    long left=0,right=0;
    short *pDest = NULL, *pSource = NULL; 
    int loop = 0;

    pSource = (short *) pIn;
    pDest   = (short *) pOut;

    switch(carrylimit) {
        case CHANNEL_STEREO:
            Samplecnt = Nvaudio_UseLeftOver(&pSource,&pDest,cnt,carrylimit,&StartIndex,bSpdif,NV_SPKR_STEREO);
            break;
        case CHANNEL_QUAD:
            Samplecnt = Nvaudio_UseLeftOver(&pSource,&pDest,cnt,carrylimit,&StartIndex,bSpdif,NV_SPKR_QUAD);
            break;
        case CHANNEL_51:
            Samplecnt = Nvaudio_UseLeftOver(&pSource,&pDest,cnt,carrylimit,&StartIndex,bSpdif,NV_SPKR_51);
            break;
    }
    

    for(loop= StartIndex;loop <= (cnt - carrylimit);loop += carrylimit) {
        left  = *pSource++;
        right = *pSource++;
        
        if(bSpdif) {
            /* apply the volume */
            left   = ( left * mastervolleft * pcmvolleft) / 10000; 
            right  = ( right * mastervolright * pcmvolright) / 10000; 
        }

        *pDest++ = (short)left;
        *pDest++ = (short)right;
        Samplecnt += 1;
        
        if(carrylimit == CHANNEL_51) {
            *pDest++ = *pSource++;
            *pDest++ = *pSource++;
            Samplecnt += 1;
        }

        if((carrylimit == CHANNEL_51) || (carrylimit == CHANNEL_QUAD)) {
            *pDest++ = *pSource++;
            *pDest++ = *pSource++;
            Samplecnt += 1;
        }

    }

    //Check for leftovers
    Nvaudio_StoreLeftOver(loop,cnt,&pSource,bSpdif);

    return Samplecnt;
}

/* Function to mix down - 5.1 TO Quad */
u32  Nvaudio_MixDown5_1ToQuad(const char* pIn,void *pOut,int cnt, int carrylimit, int bSpdif)
{
    u32 Samplecnt = 0 ,StartIndex = 0;
    long left=0,right=0,center=0, lfe=0, sleft=0,sright=0;
    short *pDest = NULL, *pSource = NULL; 
    int loop = 0;
    float sMix1 = 0.0, sMix2 = 0.0;

    pSource = (short *) pIn;
    pDest   = (short *) pOut;

    Samplecnt = Nvaudio_UseLeftOverToQuad(&pSource,&pDest,cnt,carrylimit,&StartIndex,bSpdif);

    for(loop= StartIndex;loop <= (cnt - carrylimit);loop += carrylimit) {
        left   = *pSource++;
        right  = *pSource++;
        center = *pSource++;
        lfe    = *pSource++;
        sleft  = *pSource++;
        sright = *pSource++;
        
        sMix1   = 0.707f * center;    
        sMix2   = 0.707f * lfe;    
        left   += (long)(sMix1 + sMix2);
        right  += (long)(sMix1 + sMix2);
        sleft  += (long)sMix2;
        sright += (long)sMix2;
        
        RANGECHECK(left,MINVOLRANGE,MAXVOLRANGE);
        RANGECHECK(right,MINVOLRANGE,MAXVOLRANGE);

        RANGECHECK(sleft,MINVOLRANGE,MAXVOLRANGE);
        RANGECHECK(sright,MINVOLRANGE,MAXVOLRANGE);

        *pDest++ = (short)left;
        *pDest++ = (short)right;
        *pDest++ = (short)sleft;
        *pDest++ = (short)sright;

        Samplecnt += 2;  // Two samples generated
    }

    //Check for leftovers
    Nvaudio_StoreLeftOver(loop,cnt,&pSource,bSpdif);

    return Samplecnt;
}

/* Function to Replicate Stereo To Rears */
u32  Nvaudio_ReplicateStereotoRear(const char* pIn,void* pOut,int cnt,int carrylimit)
{
    u32 Samplecnt = 0, StartIndex = 0;
    long left=0,  right=0;
    short *pDest = NULL, *pSource = NULL; 
    int loop = 0;
    
    pSource = (short *) pIn;
    pDest   = (short *) pOut;

    Samplecnt = Nvaudio_UseLeftOverToStereo(&pSource,&pDest,cnt,carrylimit,&StartIndex);

    for(loop= StartIndex;loop <= (cnt - carrylimit);loop += carrylimit) {
        left  = *pSource++;
        right = *pSource++;
        
        *pDest++ = (short)left;
        *pDest++ = (short)right;
        *pDest++ = (short)left;
        *pDest++ = (short)right;

        Samplecnt += 2;  // Two samples generated
    }

    //Check for leftovers
    Nvaudio_StoreLeftOver(loop,cnt,&pSource,FALSE);

    return Samplecnt;
}
