/*
 *                 Sun Public License Notice
 * 
 * The contents of this file are subject to the Sun Public License
 * Version 1.0 (the "License"). You may not use this file except in
 * compliance with the License. A copy of the License is available at
 * http://www.sun.com/
 * 
 * The Original Code is NetBeans. The Initial Developer of the Original
 * Code is Sun Microsystems, Inc. Portions Copyright 1997-2002 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.vcs.profiles.cvsprofiles.commands;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Hashtable;
import java.util.Map;

import org.netbeans.modules.vcscore.commands.CommandDataOutputListener;
import org.netbeans.modules.vcscore.commands.CommandOutputListener;
import org.netbeans.modules.vcscore.cmdline.VcsAdditionalCommand;

/**
 * The automatic recognizer of CVS filesystems from CVS/ folders.
 *
 * @author  Martin Entlicher
 */
public class CvsFSAutorecognizer extends Object implements VcsAdditionalCommand {
    
    private static final String CVS_DIR = "CVS";
    private static final String CVS_ROOT = "Root";
    private static final String CVS_REPOSITORY = "Repository";
    
    private static final String CVS_METHOD_LOCAL = "local";
    
    /**
     * The map of root directories, that were assigned different from the provided - to be explicitly mounted,
     * and name of the originally provided directory. This is used to not recognize
     * other directories again with the same different root.
     */
    private static Map changedFSRoots = new Hashtable();

    /** Creates a new instance of CvsFSAutorecognizer */
    public CvsFSAutorecognizer() {
    }
    
    public boolean exec(Hashtable vars, String[] args,
                        CommandOutputListener stdoutNRListener, CommandOutputListener stderrNRListener,
                        CommandDataOutputListener stdoutListener, String dataRegex,
                        CommandDataOutputListener stderrListener, String errorRegex) {
        
        String dirName = (String) vars.get("ROOTDIR");
        File root = new File(dirName);
        if (!root.isDirectory()) return false;
        File parent = root.getParentFile();
        if (parent != null) {
            File parentRoot = new File(parent, CVS_DIR + File.separator + CVS_ROOT);
            if (parentRoot.exists()) {
                return false;
            }
            String changedDirName = (String) changedFSRoots.get(parent.getAbsolutePath());
            if (changedDirName != null && !changedDirName.equals(root.getName())) {
                //System.out.println("  Root '"+parent.getAbsolutePath()+"' already mounted from "+changedDirName);
                // The parent was already mounted as a CVS FS.
                return false;
            }
        }
        File rootCVS = new File(root, CVS_DIR);
        boolean status;
        //System.out.println("CvsFSAutorecognizer("+root+"), CVS.isDirectory() = "+rootCVS.isDirectory());
        if (rootCVS.isDirectory()) {
            status = fillData(rootCVS, vars, root.getName(), parent);
            //System.out.println("fillData("+rootCVS+", "+true+") = "+status);
        } else {
            status = false;
        }
        return status;
    }
    
    /**
     * Fill the data from the CVS/Root file to the variables map.
     * @param root The CVS/ folder
     * @param vars The map of variables.
     * @return <code>true</code> if the data was successfully filled,
     *         <code>false</code> otherwise.
     */
    private static boolean fillData(File root, Hashtable vars, String rootName, File parent) {
        File CVSRoot = new File(root, CVS_ROOT);
        File CVSRepository = new File(root, CVS_REPOSITORY);
        String serverType = null;
        String repository = null;
        String userName = null;
        String serverName = null;
        String serverPort = null;
        if (CVSRepository.exists()) {
            BufferedReader buff = null;
            try {
                buff = new BufferedReader(new InputStreamReader(new FileInputStream(CVSRepository)));
                String line = buff.readLine();
                if (line != null) {
                    if (line.length() > 0 && !line.equals(".")) {
                        if (rootName.equals(line)) {
                            if (parent != null) {
                                vars.put("ROOTDIR", parent.getAbsolutePath());
                                changedFSRoots.put(parent.getAbsolutePath(), rootName);
                            }
                        }
                    }
                }
                //System.out.println("  fillData("+root+") Repository = "+line);
            } catch (IOException exc) { //doesn't matter - nothing will be filled in
            } finally {
                if (buff != null) {
                    try {
                        buff.close();
                    } catch (IOException eIO) {}
                }
            }
            //System.out.println("        "+((!isCorrectRepository) ? "NOT " : "IS") + "CORRECT REPOSITORY.");
            //if (!isCorrectRepository) return false;
        } else {
            return false;
        }
        if (CVSRoot.exists()) {
            BufferedReader buff = null;
            try {
                buff = new BufferedReader(new InputStreamReader(new FileInputStream(CVSRoot)));
                String line = buff.readLine();
                if (line != null && line.startsWith(":")) {
                    line = line.substring(1);
                    //StringTokenizer token = new StringTokenizer(line, ":",false);
                    int begin = 0;
                    int end = line.indexOf(":");
                    if (end > 0) {
                        serverType = line.substring(begin, end);
                        if (serverType.equals(CVS_METHOD_LOCAL))  {
                            repository =  line.substring(end + 1);
                            userName = "";
                            serverName = "";
                        } else {   //some kind of server..
                            begin = end + 1;
                            end = line.indexOf(":", begin);
                            if (begin < line.length()) {
                                String userServer =  line.substring(begin, (end > 0) ? end : line.length());
                                int atIndex = userServer.indexOf('@');
                                if (atIndex >= 0) {
                                    userName = userServer.substring(0, atIndex);
                                    serverName = userServer.substring(atIndex + 1);
                                } else {
                                    userName = "";
                                    serverName = userServer;
                                }
                            }
                            if (end > 0) repository = line.substring(end + 1);
                            StringBuffer port = new StringBuffer();
                            char c;
                            for (int i = 0; repository.length() > i && Character.isDigit(c = repository.charAt(i)); i++) {
                                port.append(c);
                            }
                            if (port.length() > 0) {
                                serverPort = port.toString();
                                repository = repository.substring(port.length());
                            }
                        }
                    }
                }
            } catch (IOException exc) { //doesn't matter - nothing will be filled in
            }
            finally {
                if (buff != null) {
                    try {
                        buff.close();
                    } catch (IOException eIO) {}
                }
            }
        }
        //System.out.println("CvsFSAutorecognizer("+CVSRoot+") = SERVERTYPE = "+serverType);
        if (serverType == null) return false;
        if (serverType != null) vars.put("SERVERTYPE", serverType);
        if (repository != null) vars.put("CVS_REPOSITORY", repository);
        if (userName != null) vars.put("CVS_USERNAME", userName);
        if (serverName != null) vars.put("CVS_SERVER", serverName);
        if (serverPort != null) vars.put("ENVIRONMENT_VAR_CVS_CLIENT_PORT", serverPort);
        else vars.remove("ENVIRONMENT_VAR_CVS_CLIENT_PORT");
        //System.out.println("Filled data '"+root+"': :"+serverType+":"+userName+"@"+serverName+":"+repository+", port = "+serverPort);
        return true;
        
    }
    
    /**
     * Find the CVS/ subfolder, that represents the common CVS settings.
     * If there is no CVS/ folder in the root directory, we use this approach
     * to get the information from subfolders.
     */
    private static File findCommonSubCVS(File root) {
        File[] subfiles = root.listFiles();
        if (subfiles != null) {
            String cvsRootName = CVS_DIR + File.separator + CVS_ROOT;
            for (int i = 0; i < subfiles.length; i++) {
                if (subfiles[i].isDirectory()) {
                    File test = new File(subfiles[i], cvsRootName);
                    if (test.exists()) return new File(subfiles[i], CVS_DIR);
                }
            }
        }
        return null;
    }
}
