/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: packer2.cxx,v $
 *
 *  $Revision: 1.9 $
 *
 *  last change: $Author: vg $ $Date: 2006/03/16 13:00:05 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include <ucbhelper/content.hxx>
//#include <ucbhelper/contentbroker.hxx>
//
#ifndef _COM_SUN_STAR_UCB_INTERACTIVEAUGMENTEDIOEXCEPTION_HPP_
#include <com/sun/star/ucb/InteractiveAugmentedIOException.hpp>
#endif
#ifndef _COM_SUN_STAR_UCB_INTERACTIVEWRONGMEDIUMEXCEPTION_HPP_
#include <com/sun/star/ucb/InteractiveWrongMediumException.hpp>
#endif
#ifndef _COM_SUN_STAR_UCB_XCOMMANDENVIRONMENT_HPP_
#include <com/sun/star/ucb/XCommandEnvironment.hpp>
#endif

#ifndef _OSL_FILE_HXX_
#include <osl/file.hxx>
#endif
#ifndef _OSL_FILE_HXX_
#include <osl/file.hxx>
#endif
#ifndef _STREAM_HXX
#include <tools/stream.hxx>
#endif
#ifndef _ZCODEC_HXX
#include <tools/zcodec.hxx>
#endif
#ifndef _TOOLS_TEMPFILE_HXX
#include <tools/tempfile.hxx>
#endif
#ifndef _URLOBJ_HXX
#include <tools/urlobj.hxx>
#endif

#include "packer2.hxx"
#include "errorhdl.hxx"

using namespace ::rtl;
using namespace ::osl;
using namespace ::ucb;
using namespace ::com::sun::star;
using ::com::sun::star::beans::PropertyValue;

const ULONG BUFFER_SIZE = 8*1024; // a 8k Buffer

// #define CL_DEBUG 1

/* ***************************************************************** */

LocalFileHeader::LocalFileHeader()
{
	init();
}

LocalFileHeader::LocalFileHeader( const LocalFileHeader& rFile )
{
	m_nCompression = rFile.m_nCompression;
	m_nCompressedSize = rFile.m_nCompressedSize;
	m_nUnCompressedSize = rFile.m_nUnCompressedSize;
	m_aFileName = rFile.m_aFileName;
	m_nDiskNumberStart = rFile.m_nDiskNumberStart;
	m_nRelativeOffset = rFile.m_nRelativeOffset;
	m_aSourcePath = rFile.m_aSourcePath;
}

LocalFileHeader::LocalFileHeader( const OUString& rFileName, ULONG nSize, USHORT nCompression /* = 0 */ )
{
	init();
	m_aSourcePath = rFileName;

	INetURLObject aURL; aURL.SetSmartProtocol( INET_PROT_FILE ); aURL.SetSmartURL( rFileName );
	m_aFileName = aURL.getName();

/*
	sal_Int32 nIndex = rFileName.lastIndexOf( sal_Unicode('/'), 0 );
	if( nIndex != - 1 )
	{
		m_aFileName = rFileName.copy( nIndex + 1 );
	}
	else
	{
		m_aFileName = rFileName;
	}
*/
	m_nUnCompressedSize = nSize;
	m_nCompressedSize = nSize;
	m_nCompression = nCompression;
}

void LocalFileHeader::init()
{
	m_nUnCompressedSize = 0;
	m_nCompressedSize = 0;
	m_nCompression = 0;

	m_nDiskNumberStart = 0;
	m_nRelativeOffset = NULL;
}

USHORT LocalFileHeader::getHeaderSize() const
{
	return sizeof( m_nDiskNumberStart ) +
		   sizeof( m_nRelativeOffset ) +
		   sizeof( m_nCompression ) +
		   sizeof( m_nUnCompressedSize ) +
		   sizeof( m_nCompressedSize ) +
		   sizeof( USHORT ) +
		   m_aFileName.getLength() * sizeof( USHORT );
}

ULONG LocalFileHeader::writeHeader( SvFileStream& rStrm ) const
{
	rStrm << m_nDiskNumberStart;
	rStrm << m_nRelativeOffset;
	rStrm << m_nCompression;
	rStrm << m_nUnCompressedSize;
	rStrm << m_nCompressedSize;
	USHORT nLen = m_aFileName.getLength();
	rStrm << nLen;
	const sal_Unicode* pStr = m_aFileName;
	for( int i = 0; i < nLen; i++ )
		rStrm << (USHORT)pStr[i];

	return rStrm.GetErrorCode();
}

ULONG LocalFileHeader::readHeader( SvFileStream& rStrm )
{
	rStrm >> m_nDiskNumberStart;
	rStrm >> m_nRelativeOffset;
	rStrm >> m_nCompression;
	rStrm >> m_nUnCompressedSize;
	rStrm >> m_nCompressedSize;
	USHORT nLen;
	rStrm >> nLen;
	sal_Unicode* pStr = new sal_Unicode[nLen+1];
	for( int i = 0; i < nLen; i++ )
	{
		USHORT nTemp;
		rStrm >> nTemp;
		pStr[i] = (sal_Unicode)nTemp;
	}

	pStr[nLen] = 0;

	m_aFileName = pStr;
	delete pStr;

	return rStrm.GetErrorCode();
}

/* ***************************************************************** */

/*
** initialize the packer implementation
*/
UnoPacker_Impl::UnoPacker_Impl( const ::com::sun::star::uno::Sequence< PropertyValue >& aArgs )
{
	m_nDiskSize = -1;
	m_nDisk = 0;
	m_nMaxDisk = 0;
	m_nUID = 0;
	mnProgress = 0;
	mnProgressBytes = 0;
	mbDynamic = TRUE;
	mbRemovable = TRUE;
	getParameter( aArgs );
}

/*
** frees all memory resources of the packer implementation
*/
UnoPacker_Impl::~UnoPacker_Impl()
{
	for( LocalFileHeader* pHdr = m_aFiles.First(); pHdr; pHdr = m_aFiles.Next() )
		delete pHdr;

	if( m_aTempFile.Len() != 0 )
	{
		OUString aUNCPathBefore;
		FileBase::searchFileURL( m_aTempFile, aUNCPathBefore, aUNCPathBefore );
		File::remove( aUNCPathBefore );
	}
}

// =======================================================================
// here starts the code for packing archives
// =======================================================================

/*
** packs the given files in the given archive
*/
BOOL UnoPacker_Impl::pack( const uno::Sequence< OUString >& input, const OUString& destination )
{
	ULONG nLastErrorCode = ERRCODE_NONE;

	// init directory header
	do
	{
		nLastErrorCode = initDirectoryHeader( input );
		if( nLastErrorCode == ERRCODE_ABORT )
			return FALSE;

		if( nLastErrorCode != ERRCODE_NONE )
		{
			if( !HandleError( nLastErrorCode, EC_RETRY|EC_ABORT ) )
				return FALSE;
		}
	}
	while( nLastErrorCode != ERRCODE_NONE );

	// if compression is on, pack all files to a temp file
	if( m_nCompression != 0 )
	{
		mnProgressBytes >>= 1;

		do
		{
			mnProgress = 0;
			nLastErrorCode = packFiles();
			if( nLastErrorCode == ERRCODE_ABORT )
				return FALSE;

			if( nLastErrorCode != ERRCODE_NONE )
			{
				if( !HandleError( nLastErrorCode, EC_RETRY|EC_ABORT ) )
					return FALSE;
			}
		}
		while( nLastErrorCode != ERRCODE_NONE );
	}

	if( !mbDynamic )
		CalculateFileOffsets();

	// go...
	BOOL bRecover = FALSE;
	do
	{
		mnProgress = mnProgressBytes >> 1;

		nLastErrorCode = writeArchive( destination, bRecover );
		if( nLastErrorCode == ERRCODE_ABORT )
			return FALSE;

		if( nLastErrorCode != ERRCODE_NONE )
		{
			if( !HandleError( nLastErrorCode, EC_RETRY|EC_ABORT ) )
				return FALSE;

			bRecover = TRUE;
		}
	}
	while( nLastErrorCode != ERRCODE_NONE );

	if( mbDynamic ) do
	{
		nLastErrorCode = WriteDynamicHeader( destination );
		if( nLastErrorCode == ERRCODE_ABORT )
			return FALSE;

		if( nLastErrorCode != ERRCODE_NONE )
		{
			if( !HandleError( nLastErrorCode, EC_RETRY|EC_ABORT ) )
				return FALSE;
		}
	}
	while( nLastErrorCode != ERRCODE_NONE );

	return TRUE;
}

/*
** initialise class with parameters from the Property Sequence
*/
void UnoPacker_Impl::getParameter( const uno::Sequence< PropertyValue >& aArgs )
{
	const INT32 nArgs = aArgs.getLength();
	const PropertyValue* pValues = aArgs.getConstArray();

	for( INT32 nArg = 0; nArg < nArgs; nArg++ )
	{
		const OUString& rName = pValues[nArg].Name;
		const uno::Any& rAny = pValues[nArg].Value;

		if( rName.compareToAscii("Compression") == 0 )
		{
			rAny >>= m_nCompression;
		}
		else if( rName.compareToAscii("VolumeSize") == 0 )
		{
			rAny >>= m_nDiskSize;
			m_nDiskSize *= 1024;
			if( m_nDiskSize == 0 )
			{
				m_nDiskSize = (ULONG)-1;
				mbDynamic = sal_True;
			}
			else
			{
				mbDynamic = sal_False;
			}
		}
		else if( rName.compareToAscii("RootDirectory") == 0 )
		{
			rAny >>= m_aRootDir;
		}
		else if( rName.compareToAscii("InteractionHandler") == 0 )
		{
			rAny >>= mxErrorHandler;
		}
		else if( rName.compareToAscii("ProgressHandler") == 0 )
		{
			rAny >>= mxProgressHandler;
		}
		else if( rName.compareToAscii("ExtraData") == 0 )
		{
			rAny >>= m_aExtraData;
		}
	}
}

/*
** inits the directory header with the given file list.
** The files can have subdirs relative to the base dir
*/
ULONG UnoPacker_Impl::initDirectoryHeader( const uno::Sequence< OUString >& input )
{
	ULONG nLastErrorCode=ERRCODE_NONE;

	const INT32 nFiles = input.getLength();
	const OUString* pStr = input.getConstArray();

	mnProgressBytes = 0;

	// init name, size and compression for each file
	for( int i = 0; i < nFiles && nLastErrorCode == ERRCODE_NONE; i++ )
	{
		LocalFileHeader* pHeader = new LocalFileHeader( pStr[i], 0, m_nCompression );

		// open input-stream
		maActualFile = pHeader->getSourcePath();

		SvFileStream* pInStream = createInputStream( pHeader );
		nLastErrorCode = pInStream->GetErrorCode();

		if( nLastErrorCode == ERRCODE_NONE )
		{
			pInStream->Seek( STREAM_SEEK_TO_END );
			nLastErrorCode = pInStream->GetErrorCode();

			if( nLastErrorCode == ERRCODE_NONE )
			{
				ULONG nSize = pInStream->Tell();
				nLastErrorCode = pInStream->GetErrorCode();
				if( nLastErrorCode == ERRCODE_NONE )
				{
					pHeader->setCompressedSize( nSize );
					pHeader->setUnCompressedSize( nSize );

					mnProgressBytes += nSize;

					m_aFiles.Insert( pHeader, CONTAINER_APPEND );
				}
			}
		}
		delete pInStream;
	}

	return nLastErrorCode;
}

/*
** pack all files in the directory header into a single tempfile
*/
ULONG UnoPacker_Impl::packFiles()
{
	ULONG nLastErrorCode=ERRCODE_NONE;

	{
		m_aTempFile = TempFile::CreateTempName();
	}

	BYTE *pBuffer = new BYTE[BUFFER_SIZE];
	ZCodec aZCoder;

	// open output-stream
	maActualFile = m_aTempFile;
	SvFileStream aOutStream(m_aTempFile, STREAM_STD_WRITE | STREAM_TRUNC);
	nLastErrorCode = aOutStream.GetErrorCode();

	if( nLastErrorCode == ERRCODE_NONE )
	{
		ULONG nLastPos = 0;
		for( LocalFileHeader* pHdr = m_aFiles.First(); pHdr && nLastErrorCode==ERRCODE_NONE; pHdr = m_aFiles.Next() )
		{
			// start compression-mode
			aZCoder.BeginCompression();

			// open input-stream
			maActualFile = pHdr->getSourcePath();
			SvFileStream* pInStream = createInputStream( pHdr );
			nLastErrorCode = pInStream->GetErrorCode();

			if( nLastErrorCode == ERRCODE_NONE )
			{
				while( !pInStream->IsEof() )
				{
					long nReadBytes = pInStream->Read(pBuffer, BUFFER_SIZE);
					UpdateProgress( nReadBytes );
					nLastErrorCode = pInStream->GetErrorCode();
					if( nLastErrorCode == ERRCODE_NONE )
					{
						if( aZCoder.Write( aOutStream, pBuffer, nReadBytes ) == -1 )
						{
							nLastErrorCode = ERRCODE_IO_BADCRC;
						}
						else
						{
							nLastErrorCode = aOutStream.GetErrorCode();
						}

						if( nLastErrorCode != ERRCODE_NONE )
						{
							maActualFile = m_aTempFile;
							break;
						}
					}
					else
					{
						break;
					}
				}
			}

			aZCoder.EndCompression();

			if( nLastErrorCode == ERRCODE_NONE )
			{
				ULONG nPos = aOutStream.Tell();
				pHdr->setCompressedSize( nPos - nLastPos );
				nLastPos = nPos;
			}

			delete pInStream;
		}
	}

	if( nLastErrorCode != 0 )
	{
		maActualFile = m_aTempFile;

		if( m_aTempFile.Len() )
		{
			OUString aUNCPathBefore;
			FileBase::searchFileURL( m_aTempFile, aUNCPathBefore, aUNCPathBefore );
			File::remove( aUNCPathBefore );
		}
	}

	delete [] pBuffer;
	return nLastErrorCode;
}

/*
** calculates the start offsets of each file in this archive and the number of disks needed
*/
void UnoPacker_Impl::CalculateFileOffsets()
{
	// calc length of directory header
	ULONG nDirHeaderSize = sizeof( USHORT );
	LocalFileHeader* pHdr;

	for( pHdr = m_aFiles.First(); pHdr; pHdr = m_aFiles.Next() )
		nDirHeaderSize += pHdr->getHeaderSize();

	// calculate offsets for each file
	USHORT nDisk = 0;
	ULONG nDiskSize = m_nDiskSize;

	ULONG nOffset = HEADER_SIZE + nDirHeaderSize +
				    sizeof( USHORT ) + m_aExtraData.getLength() * sizeof( USHORT );

	nDiskSize -= nOffset;

	for( pHdr = m_aFiles.First(); pHdr; pHdr = m_aFiles.Next() )
	{
		pHdr->setDisk( nDisk );
		pHdr->setOffset( nOffset );

		ULONG nFileSizeToGo = pHdr->getCompressedSize();
		while( nFileSizeToGo > 0 )
		{
			if( nFileSizeToGo < nDiskSize )
			{
				nDiskSize -= nFileSizeToGo;
				nOffset += nFileSizeToGo;
				nFileSizeToGo = 0;
			}
			else
			{
				nFileSizeToGo -= nDiskSize;

				// new disk
				nDisk++;
				nOffset = HEADER_SIZE;
				nDiskSize = m_nDiskSize - nOffset;
			}
		}
	}
	m_nMaxDisk = nDisk + 1;
}

/*
** writes the archive to the destination file, splitting files if necessary
*/
ULONG UnoPacker_Impl::writeArchive( const OUString& destination, BOOL bRecover )
{
	ULONG nLastErrorCode=ERRCODE_NONE;

	INetURLObject	aDestName; aDestName.SetSmartProtocol( INET_PROT_FILE ); aDestName.SetSmartURL( destination );
	BYTE*			pBuffer = new BYTE[BUFFER_SIZE];

	LocalFileHeader* pHdr = m_aFiles.First();
	ULONG nFileSizeToGo = pHdr->getCompressedSize();

	SvFileStream* pInStream = NULL;
	if( m_nCompression != 0 )
		pInStream = new SvFileStream( m_aTempFile, STREAM_STD_READ );

	// write position on the current disk
	ULONG nDiskOffset = 0;
	USHORT nDisk = 0;

	// check if we need to restart with the last unsuccesfull
	// written disk
	if( bRecover )
	{
		nDisk = mRecover.mnDisk;
		nFileSizeToGo = mRecover.mnFileSizeToGo;

		while( pHdr != mRecover.mpHeader )
			pHdr = m_aFiles.Next();

		maActualFile = pHdr->getFileName();
		if( pInStream == 0 )
		{
			pInStream = createInputStream( pHdr );
			nLastErrorCode = pInStream->GetErrorCode();
		}

		if( nLastErrorCode == ERRCODE_NONE )
		{
			pInStream->Seek( mRecover.mnStartOffset );
			nLastErrorCode = pInStream->GetErrorCode();
		}

		mnProgress = mRecover.mnProgress;
		UpdateProgress(0);

		if( nDisk > 0 )
		{
			String aExt( aDestName.getExtension() );
			aExt.Erase(1);
			aExt += (char)((nDisk / 10) + '0' - 1);
			aExt += (char)((nDisk % 10) + '0' - 1);
			aDestName.setExtension( aExt );
		}
	}

	for( ; pHdr && nLastErrorCode == ERRCODE_NONE; nDisk++ )
	{
		// store recover information
		mRecover.mnDisk = nDisk;
		mRecover.mnFileSizeToGo = nFileSizeToGo;
		mRecover.mpHeader = pHdr;
		mRecover.mnProgress = mnProgress;
		if( pInStream )
			mRecover.mnStartOffset = pInStream->Tell();
		else
			mRecover.mnStartOffset = 0;

		// check if we need a new disk
		if( mbRemovable && (nDisk > 0) )
		{
			nLastErrorCode = RequestDisk( aDestName, nDisk );
			if( nLastErrorCode == ERRCODE_IO_NOTSAMEDEVICE )
			{
				// this is not a removable media
				mbRemovable = FALSE;
				nLastErrorCode = ERRCODE_NONE;
			}
			else if( nLastErrorCode != ERRCODE_NONE )
				break;
		}

		ULONG nDiskSize = m_nDiskSize;

		// open output-stream
		maActualFile = aDestName.PathToFileName();
		SvFileStream aOutStream(maActualFile, STREAM_STD_WRITE | STREAM_TRUNC);
		aOutStream.SetBufferSize( 0 );

		// write header
		aOutStream << PACK_ID;
		nLastErrorCode = aOutStream.GetErrorCode();
		if( nLastErrorCode != ERRCODE_NONE )
			break;

		aOutStream << (USHORT)1;
		aOutStream << nDisk;
		aOutStream << m_nMaxDisk;
		aOutStream << PACK_ID;		// TODO: calculate unique id
		if( nLastErrorCode != ERRCODE_NONE )
			break;

		nDiskOffset = HEADER_SIZE;

		// write table of contents if we are on first disc
		if( nDisk == 0 )
		{
			// write extra data on first disc
			USHORT nTemp16 = m_aExtraData.getLength();
			aOutStream << nTemp16;
			const sal_Unicode* pStr = m_aExtraData;
			for( USHORT i = 0; i < nTemp16; i++ )
				aOutStream << (USHORT)pStr[i];

			// write directory header on first disc
			nTemp16 = m_aFiles.Count();
			aOutStream << nTemp16;

			nDiskOffset += sizeof( USHORT ) + m_aExtraData.getLength() * sizeof( USHORT ) + sizeof( USHORT );

			for( LocalFileHeader* pHdr = m_aFiles.First(); pHdr && nLastErrorCode == ERRCODE_NONE; pHdr = m_aFiles.Next() )
			{
				nLastErrorCode = pHdr->writeHeader( aOutStream );
				nDiskOffset += pHdr->getHeaderSize();
			}

			pHdr = m_aFiles.First();
		}
		if( nLastErrorCode != ERRCODE_NONE )
			break;

		// subtract header size from virtual free disk space
		nDiskSize -= aOutStream.Tell();
		nLastErrorCode = aOutStream.GetErrorCode();

		ULONG nBytesToWrite, nBytesWritten;

		// write until all files finished or disk is full
		while( nDiskSize && pHdr && nLastErrorCode == ERRCODE_NONE )
		{
			// if the header of this file is not initialiesed
			// yet, do it now!
			if( pHdr->getOffset() == 0 )
			{
				pHdr->setDisk( nDisk );
				pHdr->setOffset( nDiskOffset );
			}

			// can we write the whole file on this chunk?
			if( nFileSizeToGo < nDiskSize )
			{
				nBytesToWrite = nFileSizeToGo;
				nDiskSize -= nFileSizeToGo;
			}
			else
			{
				nBytesToWrite = nDiskSize;
				nDiskSize = 0;
			}

			nBytesWritten = 0;

			if( nBytesToWrite != 0 )
			{
				if( pInStream == 0 )
				{
					pInStream = createInputStream( pHdr );
					nLastErrorCode = pInStream->GetErrorCode();
				}

				while( nBytesToWrite && nLastErrorCode == ERRCODE_NONE )
				{
					ULONG nWrite = (nBytesToWrite < BUFFER_SIZE)?nBytesToWrite:BUFFER_SIZE;

				    pInStream->Read( pBuffer, nWrite );
					nLastErrorCode = pInStream->GetErrorCode();

					if( nLastErrorCode == ERRCODE_NONE )
					{
						ULONG nWritten = aOutStream.Write( pBuffer, nWrite );
						UpdateProgress( nWritten );
						nLastErrorCode = aOutStream.GetErrorCode();

						nDiskOffset += nWritten;
						nBytesWritten += nWritten;

						if( nLastErrorCode != ERRCODE_NONE && (nLastErrorCode != SVSTREAM_DISK_FULL || !mbDynamic ))
						{
							// break only if this is not a disk full error or
							// if this is not a dynamic disk size pack
							break;
						}

						// seek back if we have not written all we read
						if( nWritten < nWrite )
						{
							pInStream->SeekRel( nWritten - nWrite );
							nLastErrorCode = SVSTREAM_DISK_FULL;
						}

						if( nLastErrorCode == SVSTREAM_DISK_FULL )
						{
							nLastErrorCode = ERRCODE_NONE;
							nDiskSize = 0;
							break;
						}
					}
					else
					{
						maActualFile = pHdr->getSourcePath();
						break;
					}

					nBytesToWrite -= nWrite;
				}
			}

			nFileSizeToGo -= nBytesWritten;

			if( nFileSizeToGo == 0 )
			{
				if( m_nCompression == 0 )
				{
					delete pInStream;
					pInStream = NULL;
				}

				pHdr = m_aFiles.Next();
				if( pHdr )
					nFileSizeToGo = pHdr->getCompressedSize();
			}
		}

		// compute next disk file name
		String aExt( aDestName.getExtension() );
		aExt.Erase(1);
		aExt += (char)((nDisk / 10) + '0');
		aExt += (char)((nDisk % 10) + '0');
	    aDestName.setExtension( aExt );
	}

	delete pInStream;
	delete pBuffer;

	return nLastErrorCode;
}

ULONG UnoPacker_Impl::WriteDynamicHeader( const OUString& destination )
{
	ULONG nLastErrorCode = ERRCODE_NONE;

	INetURLObject aSourceEntry; aSourceEntry.SetSmartProtocol( INET_PROT_FILE ); aSourceEntry.SetSmartURL( destination );
	maActualFile = aSourceEntry.PathToFileName();

	SvFileStream* pInOutStream = NULL;

	// force disk 0
	USHORT nCurrentDisk = (USHORT)-1;

	while( nCurrentDisk != 0 && nLastErrorCode == ERRCODE_NONE )
	{
		if( pInOutStream )
		{
			delete pInOutStream;
			pInOutStream = NULL;
		}

		// we need to open a new one so the old one is not locking the disc
		pInOutStream = new SvFileStream( maActualFile, STREAM_STD_READWRITE|STREAM_NOCREATE );

		if( nLastErrorCode == ERRCODE_NONE )
		{
			sal_uInt32 nPID;
			USHORT nVersion;
			USHORT nMaxDisk;
			sal_uInt32 nID;
			nLastErrorCode = readHeader( *pInOutStream, nPID, nVersion, nCurrentDisk, nMaxDisk, nID );
		}
		else
		{
			nCurrentDisk = (USHORT)-1;
		}

		if( mbRemovable && (nLastErrorCode != ERRCODE_NONE || nCurrentDisk != 0) )
		{
			nCurrentDisk = (USHORT)-1;

			if( pInOutStream )
			{
				delete pInOutStream;
				pInOutStream = NULL;
			}

			nLastErrorCode = RequestDisk( aSourceEntry, 0 );
			if( nLastErrorCode == ERRCODE_IO_NOTSAMEDEVICE )
			{
				// this is not a removable media
				mbRemovable = FALSE;
				nLastErrorCode = ERRCODE_NONE;
			}
		}
	}

	if( nLastErrorCode == ERRCODE_NONE )
	{
		// open output-stream

		// skip header & extradata
		pInOutStream->Seek(HEADER_SIZE + sizeof( USHORT ) + m_aExtraData.getLength() * sizeof( USHORT ) + sizeof( USHORT ) );
		nLastErrorCode = pInOutStream->GetErrorCode();
		if( nLastErrorCode == ERRCODE_NONE )
		{
			// update header
			for( LocalFileHeader* pHdr = m_aFiles.First(); pHdr && nLastErrorCode == ERRCODE_NONE; pHdr = m_aFiles.Next() )
				nLastErrorCode = pHdr->writeHeader( *pInOutStream );
		}
	}

	delete pInOutStream;

	return nLastErrorCode;
}

/*
** creates an input stream for a local file header
*/
SvFileStream* UnoPacker_Impl::createInputStream( LocalFileHeader* pHeader )
{
	return new SvFileStream( pHeader->getSourcePath(), STREAM_STD_READ );
}

// =======================================================================
// here starts the code for unpacking archives
// =======================================================================

/*
** unpacks all files in the given archive to the given destination path
*/
BOOL UnoPacker_Impl::unpack( const OUString& source, const OUString& destinationpath, const uno::Sequence< OUString >& files )
{
	ULONG nLastErrorCode = ERRCODE_NONE;

	if( !readHeaders(source) )
		return FALSE;
	m_aRootDir = destinationpath;

	// unpack it
	do
	{
		mnProgress = 0;

		nLastErrorCode = unpackArchive( source );
		if( nLastErrorCode == ERRCODE_ABORT )
			return FALSE;

		if( nLastErrorCode != ERRCODE_NONE )
		{
			if( !HandleError( nLastErrorCode, EC_RETRY|EC_ABORT ) )
				return FALSE;
		}
	}
	while( nLastErrorCode != ERRCODE_NONE );

	return TRUE;
}

/*
** opens the given archive and returns the stored extra data
*/
OUString UnoPacker_Impl::getExtraData( const OUString& source )
{
	OUString aExtraData;

	INetURLObject aURL; aURL.SetSmartProtocol( INET_PROT_FILE ); aURL.SetSmartURL( source );
	OUString aSource( aURL.PathToFileName() );

	if(readHeaders(aSource))
		aExtraData = m_aExtraData;

	return aExtraData;
}

/*
** reads the file header and extra data from an archive
*/
BOOL UnoPacker_Impl::readHeaders( const OUString& source )
{
	ULONG nLastErrorCode = ERRCODE_NONE;

	sal_uInt32 nPID;
	USHORT nDisk;
	sal_uInt32 nID;
	USHORT nVersion;

	maActualFile = source ;
	// check the header to see what we have here
	do
	{
		SvFileStream aInStream( source, STREAM_STD_READ );
		nLastErrorCode = readHeader( aInStream, nPID, nVersion, nDisk, m_nMaxDisk, nID );

		if( nLastErrorCode == ERRCODE_NONE )
		{
			if( nPID != PACK_ID || nDisk != 0 )
			{
				nLastErrorCode = SVSTREAM_FILEFORMAT_ERROR;
			}
			else
			{
				// read in the directory header
				nLastErrorCode = readFileHeader( aInStream );
			}
		}

		if( nLastErrorCode != ERRCODE_NONE )
		{
			if( !HandleError( nLastErrorCode, EC_ABORT|EC_RETRY ) )
				return FALSE;
		}
	}
	while( nLastErrorCode != ERRCODE_NONE );

	return TRUE;
}

/*
** read the archive header
*/
ULONG UnoPacker_Impl::readHeader( SvFileStream& rStream, sal_uInt32& nPID, USHORT &nVersion, USHORT &nDisk, USHORT& nMaxDisk, sal_uInt32& nID )
{
	nPID = NULL; nVersion = 0; nDisk = 0; nMaxDisk = 0; nID = 0;

	rStream >> nPID;
	rStream >> nVersion;
	rStream >> nDisk;
	rStream >> nMaxDisk;
	rStream >> nID;

	return rStream.GetErrorCode();
}

/*
** reads the directory header
*/
ULONG UnoPacker_Impl::readFileHeader( SvFileStream& rStream )
{
	ULONG nLastErrorCode = ERRCODE_NONE;

	// read extra data
	USHORT nLen = 0;
	rStream >> nLen;
	sal_Unicode* pStr = new sal_Unicode[nLen+1];
	for( int i = 0; i < nLen; i++ )
	{
		USHORT nTemp;
		rStream >> nTemp;
		pStr[i] = (sal_Unicode)nTemp;
	}

	pStr[nLen] = 0;

	m_aExtraData = pStr;
	delete pStr;

	// read directory header
	USHORT nFiles = 0;
	rStream >> nFiles;
	for( USHORT nFile = 0; nFile < nFiles && nLastErrorCode == ERRCODE_NONE; nFile++ )
	{
		LocalFileHeader* pHeader = new LocalFileHeader();
		nLastErrorCode = pHeader->readHeader(rStream);
		m_aFiles.Insert( pHeader, CONTAINER_APPEND );
		mnProgressBytes += pHeader->getUnCompressedSize();
		if( pHeader->getCompression() )
			mnProgressBytes += pHeader->getCompressedSize();
	}

	if( nLastErrorCode != ERRCODE_NONE )
	{
		// in case of an error delete all directory entrys so far
		for( LocalFileHeader* pHdr = m_aFiles.First(); pHdr; pHdr = m_aFiles.Next() )
			delete pHdr;
		m_aFiles.Clear();

	}
	return nLastErrorCode;
}

/*
** unpacks the files in the m_aFiles list to the destination directory
*/
ULONG UnoPacker_Impl::unpackArchive( const OUString& source )
{
	INetURLObject aSourceEntry; aSourceEntry.SetSmartProtocol( INET_PROT_FILE ); aSourceEntry.SetSmartURL( source );

	ULONG nLastErrorCode = ERRCODE_NONE;

	BYTE* pBuffer = new BYTE[BUFFER_SIZE];
	SvFileStream* pInStream = NULL;
	USHORT nCurrentDisk = (USHORT)-1;
	for( LocalFileHeader* pHdr = m_aFiles.First(); pHdr && nLastErrorCode == ERRCODE_NONE; pHdr = m_aFiles.Next() )
	{
		USHORT nDisk = pHdr->getDisk();
		BOOL bCompressed = pHdr->getCompression() != 0;

		ULONG nBytesToGo = pHdr->getCompressedSize();

		SvFileStream* pOutStream = NULL;

		String aInputFile;

		// open destination stream
		if( bCompressed )
		{
			// open tempfile for later decompression
			m_aTempFile = TempFile::CreateTempName();
			aInputFile = m_aTempFile;
			pOutStream = new SvFileStream( m_aTempFile, STREAM_STD_WRITE | STREAM_TRUNC );
		}
		else
		{
			aInputFile = pHdr->getSourcePath();
			pOutStream = createOutputStream( pHdr );
		}
		maActualFile = aInputFile;
		nLastErrorCode = pOutStream->GetErrorCode();

		while( nBytesToGo && nLastErrorCode == ERRCODE_NONE )
		{
			// get disk file
			while( nDisk != nCurrentDisk && nLastErrorCode == ERRCODE_NONE )
			{
				delete pInStream;
				pInStream = NULL;

				if( nDisk != 0 )
				{
					String aExt( aSourceEntry.getExtension() );
					aExt.Erase(1);
					aExt += (char)(((nDisk-1) / 10) + '0');
					aExt += (char)(((nDisk-1) % 10) + '0');
					aSourceEntry.setExtension( aExt );
				}

				maActualFile = aSourceEntry.PathToFileName();
				pInStream = new SvFileStream( maActualFile, STREAM_STD_READ );

				if( nLastErrorCode == ERRCODE_NONE )
				{
					sal_uInt32 nPID;
					USHORT nVersion;
					USHORT nMaxDisk;
					sal_uInt32 nID;
					nLastErrorCode = readHeader( *pInStream, nPID, nVersion, nCurrentDisk, nMaxDisk, nID );
				}
				else
				{
					nCurrentDisk = (USHORT)-1;
				}

				if( mbRemovable && ( nLastErrorCode != ERRCODE_NONE || nCurrentDisk != nDisk ) )
				{
					delete pInStream;
					pInStream = NULL;

					nLastErrorCode = RequestDisk( aSourceEntry, nDisk );
					if( nLastErrorCode == ERRCODE_IO_NOTSAMEDEVICE )
					{
						// this is not a removable media
						mbRemovable = FALSE;
						nLastErrorCode = ERRCODE_NONE;
					}
					else if( nLastErrorCode != ERRCODE_NONE )
						break;
				}
			}

			// if this is the first disk of this file seek to file
			if( nDisk == pHdr->getDisk() )
			{
				pInStream->Seek( pHdr->getOffset() );
				nLastErrorCode = pInStream->GetErrorCode();
			}

			if( nLastErrorCode != ERRCODE_NONE )
				break;

			// stream file
			while( !pInStream->IsEof() && nBytesToGo && nLastErrorCode == ERRCODE_NONE )
			{
				long nReadBytes = pInStream->Read(pBuffer, BUFFER_SIZE<nBytesToGo?BUFFER_SIZE:nBytesToGo);
				nLastErrorCode = pOutStream->GetErrorCode();

				if( nLastErrorCode == ERRCODE_NONE )
				{
					nBytesToGo -= nReadBytes;
					pOutStream->Write (pBuffer, nReadBytes);
					UpdateProgress( nReadBytes );
					nLastErrorCode = pOutStream->GetErrorCode();

					if( nLastErrorCode != ERRCODE_NONE )
						maActualFile = aInputFile;
				}
			}

			// do we need another disk?
			if( nBytesToGo != 0 )
				nDisk++;
		}

		delete pOutStream;

		// unpack file
		if( bCompressed )
		{
			if( nLastErrorCode == ERRCODE_NONE )
			{
				SvFileStream* pOutStream = createOutputStream( pHdr );
				SvFileStream* pInStream = new SvFileStream( m_aTempFile, STREAM_STD_READ );

				ZCodec aZCoder;
				aZCoder.BeginCompression();
				aZCoder.Decompress( *pInStream, *pOutStream );
				aZCoder.EndCompression();

				UpdateProgress( pHdr->getUnCompressedSize() );

				nLastErrorCode = pInStream->GetErrorCode();
				if( nLastErrorCode != ERRCODE_NONE )
				{
					maActualFile = m_aTempFile;
				}
				else
				{
					nLastErrorCode = pOutStream->GetErrorCode();
					if( nLastErrorCode != ERRCODE_NONE )
					{
						maActualFile = pHdr->getSourcePath();
					}
				}

				delete pOutStream;
				delete pInStream;
			}

			if( m_aTempFile.Len() != 0 )
			{
				OUString aUNCPathBefore;
				FileBase::searchFileURL( m_aTempFile, aUNCPathBefore, aUNCPathBefore );
				File::remove( aUNCPathBefore );
			}
		}
	}

	delete pInStream;
	delete pBuffer;

	return nLastErrorCode;
}

/*
** creates an output stream for a local file header
*/
SvFileStream* UnoPacker_Impl::createOutputStream( LocalFileHeader* pHeader )
{
	INetURLObject aURL; aURL.SetSmartProtocol( INET_PROT_FILE ); aURL.SetSmartURL( m_aRootDir );

	aURL.Append( pHeader->getFileName() );

	String aFull( aURL.PathToFileName() );
	return new SvFileStream( aFull, STREAM_STD_WRITE | STREAM_TRUNC );
}

ULONG UnoPacker_Impl::RequestDisk( INetURLObject& rEntry, sal_Int32 nDisk )
{
/*
	String aOldActualFile( maActualFile );
	rEntry.ToAbs();
#ifdef CL_DEBUG
	fprintf(stderr, "[CL] RequestDisk(%ld); inpath=%s\n", nDisk, rEntry.GetFull().GetStr() );
#endif

	ULONG nLastErrorCode = ERRCODE_NONE;

#ifdef UNX
	// current directory may a lock removable medium
	// so set it to root here
	{
		const String aSlash( "/" );
		DirEntry aEntry( aSlash );
		aEntry.SetCWD();
	}
#endif // UNX

	VolumeInfo aInfo( osl_VolumeInfo_Mask_DeviceHandle|osl_VolumeInfo_Mask_Attributes );
	OUString aUNCPathBefore;

	FileBase::searchNormalizedPath( rEntry.GetPath().GetFull(), aUNCPathBefore, aUNCPathBefore );

#ifdef UNX
	osl_getRealPath( aUNCPathBefore.pData, &aUNCPathBefore.pData );
#endif

	maActualFile = aUNCPathBefore;

#ifdef CL_DEBUG
	fprintf(stderr, "[CL] UNC: %s\n", maActualFile.GetStr() );
#endif

	OUString aOldMountPath;
	VolumeDevice aDevice;

	// get volume info for current disk
	do
	{
		nLastErrorCode = Directory::getVolumeInfo( aUNCPathBefore, aInfo );
		if( nLastErrorCode == ERRCODE_NONE )
		{
			aDevice = aInfo.getDeviceHandle();
#ifdef UNX
			aOldMountPath = aDevice.getMountPath();
#endif // UNX
		}

		if( nLastErrorCode != ERRCODE_NONE )
		{
			if( !HandleError( ERRCODE_IO_INVALIDDEVICE, EC_RETRY|EC_ABORT ) )
				return ERRCODE_ABORT;
		}
	}
	while( nLastErrorCode != ERRCODE_NONE );

	// if this is not a removable media, return an error
	if( !aInfo.getRemoveableFlag() )
	{
#ifdef CL_DEBUG
		fprintf(stderr,"[CL] Device not a removable!\n");
#endif
		maActualFile = aOldActualFile;
		return ERRCODE_IO_NOTSAMEDEVICE;
	}

#ifdef UNX
	// unmount current disk
	do
	{
		nLastErrorCode = aDevice.unmount();

		if( nLastErrorCode != ERRCODE_NONE )
		{
			if( !HandleError( ERRCODE_IO_ACCESSDENIED, EC_RETRY|EC_ABORT ) )
				return ERRCODE_ABORT;
		}
	}
	while( nLastErrorCode != ERRCODE_NONE );
#endif

#ifdef CL_DEBUG
//	fprintf(stderr, "[CL] Old Mount Path: %s\n", aOldMountPath.GetStr() );
#endif

	// open dialog to request next disk from user
	{
		UsrAny aAny;
		aAny.setINT32( nDisk );
		InteractiveWrongMediumException e;
		e.Medium = aAny;
		aAny.set( &e, InteractiveWrongMediumException_getReflection() );

		if( !HandleError( aAny, EC_YES|EC_ABORT ) )
			return ERRCODE_ABORT;
	}

#ifdef UNX
	// try to mount current disk
	// we don't check here if its the right
	// disk, we will be called again if its
	// not
	do
	{
		nLastErrorCode = aDevice.automount();

		if( nLastErrorCode != ERRCODE_NONE )
		{
			if( !HandleError( ERRCODE_IO_ACCESSDENIED, EC_RETRY|EC_ABORT ) )
				return ERRCODE_ABORT;
		}
	}
	while( nLastErrorCode != ERRCODE_NONE );

	OUString aNewMountPath( aDevice.getMountPath() );

#ifdef CL_DEBUG
//	fprintf(stderr, "[CL] New Mount Path: %s\n", OUStringToString(aNewMountPath, CHARSET_SYSTEM).GetStr() );
#endif

	if( aNewMountPath != aOldMountPath )
	{
		aNewMountPath += aUNCPathBefore.copy( aOldMountPath.getLength() );

		OUString aSystemPath;
		FileBase::getSystemPathFromNormalizedPath( aNewMountPath, aSystemPath );

		DirEntry aEntry( aNewMountPath );

		aEntry += rEntry.GetName();
		rEntry = aEntry;

#ifdef CL_DEBUG
//		fprintf(stderr, "[CL] New System Path: %s\n", rEntry.GetFull().GetStr() );
#endif
	}
#endif

	maActualFile = aOldActualFile;
*/
	return ERRCODE_NONE;
}

/*
** calls the simple error handler if present
*/
sal_Bool UnoPacker_Impl::HandleError( uno::Any& rAny, USHORT eContinuations )
{
	if( !mxErrorHandler.is() )
		return sal_False;

	InteractionRequest_impl* pReq = new InteractionRequest_impl( rAny, eContinuations );
	uno::Reference< task::XInteractionRequest > xRef( pReq );
	mxErrorHandler->handle( xRef );

	const USHORT nSelection = pReq->getSelection();
	return  nSelection == EC_YES || nSelection == EC_RETRY;
}

sal_Bool UnoPacker_Impl::HandleError( ULONG nError, USHORT eContinuations )
{
	uno::Any aAny;
	::com::sun::star::ucb::InteractiveAugmentedIOException e;
	e.Code = SvStreamErrorToUCBIoErrorCode( nError );
	e.Arguments.realloc( 1 );
	PropertyValue aProperty;
	aProperty.Name = OUString ( RTL_CONSTASCII_USTRINGPARAM ( "Uri" ) );
	aProperty.Handle = static_cast < sal_Int32 > ( -1 );
	aProperty.Value <<= OUString ( maActualFile );
	e.Arguments[0] <<= aProperty;

	aAny <<= e;

	return HandleError( aAny, eContinuations );
}

void UnoPacker_Impl::UpdateProgress( ULONG nDeltaBytes )
{
	mnProgress += nDeltaBytes;

	if( mxProgressHandler.is() && mnProgressBytes != 0 )
	{
		sal_Int32 nPercent = mnProgress * 100;
		nPercent /= mnProgressBytes;

		uno::Any aAny;
		aAny <<= (sal_Int16)nPercent;
		mxProgressHandler->update( aAny );
	}
}

// =======================================================================
// here starts the code for archive content
// =======================================================================

uno::Reference< container::XIndexAccess > UnoPacker_Impl::getContent( const OUString& source )
{
	uno::Reference< container::XIndexAccess > rIndex;

	if(readHeaders(source))
		rIndex = new UnoArchiveContent( m_aFiles );

	return rIndex;
}

UnoArchiveContent::UnoArchiveContent( const LocalFileHeaderList& rFiles ) throw()
{
	// const as const can
	const ULONG nCount = rFiles.Count();
	for( ULONG nPos = 0; nPos < nCount; nPos++ )
		m_aFiles.Insert( new LocalFileHeader( *rFiles.GetObject(nPos)), CONTAINER_APPEND );
}

UnoArchiveContent::~UnoArchiveContent() throw()
{
	for( LocalFileHeader* pHdr = m_aFiles.First(); pHdr; pHdr = m_aFiles.Next() )
		delete pHdr;
}

// XServiceInfo
OUString SAL_CALL UnoArchiveContent::getImplementationName() throw()
{
    return getImplementationName_Static();
}

sal_Bool SAL_CALL UnoArchiveContent::supportsService(const OUString& ServiceName) throw()
{
	uno::Sequence< OUString > aSNL( getSupportedServiceNames() );
    const OUString * pArray = aSNL.getConstArray();

    for( INT32 i = 0; i < aSNL.getLength(); i++ )
        if( pArray[i] == ServiceName )
            return sal_True;

    return sal_False;
}

uno::Sequence< OUString > SAL_CALL UnoArchiveContent::getSupportedServiceNames(void) throw()
{
    return getSupportedServiceNames_Static();
}

uno::Sequence< OUString > SAL_CALL UnoArchiveContent::getSupportedServiceNames_Static(void) throw()
{
	uno::Sequence< OUString > aSNS( 1 );
    aSNS.getArray()[0] = OUString( RTL_CONSTASCII_USTRINGPARAM("com.sun.star.util.ArchiverContent"));
    return aSNS;
}

// XIndexAccess
sal_Int32 SAL_CALL UnoArchiveContent::getCount(void) throw(  ::com::sun::star::uno::RuntimeException)
{
	return m_aFiles.Count();
}

uno::Any UnoArchiveContent::getByIndex( sal_Int32 Index) throw(  uno::RuntimeException, lang::IndexOutOfBoundsException, lang::WrappedTargetException )
{
	if( Index < 0 || Index >= m_aFiles.Count() )
		throw lang::IndexOutOfBoundsException();

	uno::Any aAny;
	OUString aFileName( m_aFiles.GetObject( Index )->getFileName() );
	aAny <<= aFileName;
	return aAny;
}

// XElementAccess
uno::Type SAL_CALL UnoArchiveContent::getElementType(void) throw(  ::com::sun::star::uno::RuntimeException)
{
	return getCppuType((const ::rtl::OUString*)0);
}

sal_Bool SAL_CALL UnoArchiveContent::hasElements(void) throw(  ::com::sun::star::uno::RuntimeException)
{
	return m_aFiles.Count() != 0;
}

