/*************************************************************************
 *
 *  $RCSfile: pkgchk_misc.cxx,v $
 *
 *  $Revision: 1.14 $
 *
 *  last change: $Author: vg $ $Date: 2003/05/28 13:26:50 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2002 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include "pkgchk_env.h"
#include "pkgchk_unorc.h"
#include "rtl/uri.hxx"
#include "rtl/digest.h"
#include "osl/process.h"
#include "osl/pipe.hxx"

#include "com/sun/star/ucb/GlobalTransferCommandArgument.hpp"
#include "com/sun/star/ucb/NameClash.hpp"


using namespace ::rtl;
using namespace ::osl;
using namespace ::com::sun::star::uno;
using namespace ::com::sun::star;

namespace pkgchk
{

//==============================================================================
void file_write(
    oslFileHandle file, OString const & text, OUString const & fileName )
{
    sal_uInt64 pos = 0;
    oslFileError rc;
    do
    {
        sal_uInt64 written;
        rc = osl_writeFile(
            file, text.getStr() + pos, text.getLength() - pos, &written );
        if (osl_File_E_None == rc)
            pos += written;
        if (osl_File_E_AGAIN == rc)
            rc = osl_File_E_None;
    }
    while (osl_File_E_None == rc && pos < text.getLength());
    
    if (osl_File_E_None != rc)
    {
        throw RuntimeException(
            fileName + OUSTR(" cannot be written!"),
            Reference< XInterface >() );
    }
}

//==============================================================================
sal_uInt64 file_get_size( oslFileHandle file )
{
    sal_uInt64 size;
	sal_uInt64 nOldPos;
	OSL_VERIFY(
        osl_File_E_None == osl_getFilePos( file, &nOldPos ) &&
        osl_File_E_None == osl_setFilePos( file, osl_Pos_End, 0 ) &&
        osl_File_E_None == osl_getFilePos( file, &size ) &&
        osl_File_E_None == osl_setFilePos( file, osl_Pos_Absolut, nOldPos ) );
    return size;
}

//==============================================================================
OUString path_make_absolute( OUString const & path )
{
    static OUString s_workingDir;
    if (! s_workingDir.getLength())
        osl_getProcessWorkingDir( &s_workingDir.pData );
    
    OUString abs;
    if (FileBase::E_None !=
          FileBase::getAbsoluteFileURL( s_workingDir, path, abs ))
    {
        throw RuntimeException(
            path + OUSTR(" cannot be made absolute!"),
            Reference< XInterface >() );
    }
    return abs;
}

//==============================================================================
OUString const & path_get_executable()
{
    static OUString s_path;
    if (! s_path.getLength())
    {
        OSL_VERIFY( osl_File_E_None == osl_getExecutableFile( &s_path.pData ) );
        sal_Int32 nDirEnd = s_path.lastIndexOf( '/' );
        if (nDirEnd < 0)
        {
            throw RuntimeException(
                OUSTR("cannot locate executable directory!"),
                Reference< XInterface >() );
        }
        s_path = s_path.copy( 0, nDirEnd );
    }
    return s_path;
}

//==============================================================================
void diritem_get_status(
    FileStatus * status, DirectoryItem & dirItem, sal_uInt32 mask )
{
    if (DirectoryItem::E_None != dirItem.getFileStatus( *status ))
    {
        throw RuntimeException(
            OUSTR("no file status!"),
            Reference< XInterface >() );
    }
    if (! status->isValid( mask ))
    {
        throw RuntimeException(
            OUSTR("invalid file status!"),
            Reference< XInterface >() );
    }
}

//==============================================================================
void path_get_status(
    FileStatus * status, OUString const & path, sal_uInt32 mask )
{
    DirectoryItem dirItem;
    bool ret = (DirectoryItem::E_None == DirectoryItem::get( path, dirItem ));
    if (ret)
    {
        diritem_get_status( status, dirItem, mask );
    }
    else
    {
        throw RuntimeException(
            path + OUSTR(" does not exist!"),
            Reference< XInterface >() );
    }
}

//==============================================================================
OUString file_status_get_encoded_name( FileStatus const & status )
{
    OUString ret;
    if (FileBase::E_None !=
          FileBase::getFileURLFromSystemPath( status.getFileName(), ret ))
    {
        throw RuntimeException(
            status.getFileName() + OUSTR(" cannot be converted to file url!"),
            Reference< XInterface >() );
    }
    return ret;
}

//==============================================================================
void dir_create( OUString const & path )
{
    if (Directory::E_None != Directory::create( path ))
    {
        throw RuntimeException(
            path + OUSTR(" cannot be created!"),
            Reference< XInterface >() );
    }
}

//==============================================================================
void dir_open( Directory & dir, OUString const & path, bool create )
{
    Directory::RC rc = dir.open();
    if (Directory::E_NOENT == rc)
    {
        if (create)
        {
            dir_create( path );
        }
        else
        {
            throw RuntimeException(
                path + OUSTR(" does not exist!"),
                Reference< XInterface >() );
        }
        // re-open
        dir_open( dir, path, create );
    }
    else if (Directory::E_None != rc)
    {
        throw RuntimeException(
            path + OUSTR(" cannot be opened!"),
            Reference< XInterface >() );
    }
}

//##############################################################################
//##############################################################################
//##############################################################################

//==============================================================================
void path_erase( OUString const & path )
{
    if (! path_exists( path ))
        return;
    
    FileStatus status( c_default_file_status_mask );
    path_get_status( &status, path, c_default_file_status_mask );
    FileStatus::Type file_type = status.getFileType();
    
    if (FileStatus::Directory == file_type)
    {
        {
        Directory dir( path );
        dir_open( dir, path );
        while (true)
        {
            {
            DirectoryItem dirItem;
            Directory::RC rc = dir.getNextItem( dirItem );
            if (Directory::E_NOENT == rc)
                break;
            if (Directory::E_None != rc || !dirItem.is())
            {
                OUStringBuffer buf( 64 );
                buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(
                                     "cannot get next dir item from ") );
                buf.append( path );
                buf.append( (sal_Unicode) '!' );
                throw RuntimeException(
                    buf.makeStringAndClear(),
                    Reference< XInterface >() );
            }
            diritem_get_status( &status, dirItem, c_default_file_status_mask );
            } // release dir item
            // recurse
            path_erase( status.getFileURL() );
        }
        } // release dir
        // remove (empty) dir
        if (Directory::E_None != Directory::remove( path ))
        {
            OUStringBuffer buf( 64 );
            buf.appendAscii(
                RTL_CONSTASCII_STRINGPARAM("removing directory ") );
            buf.append( path );
            buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(" failed!") );
            throw RuntimeException(
                buf.makeStringAndClear(),
                Reference< XInterface >() );
        }
    }
    else
    {
        // remove file
        if (File::E_None != File::remove( path ))
        {
            throw RuntimeException(
                path + OUSTR(" cannot be removed!"),
                Reference< XInterface >() );
        }
    }
}

//==============================================================================
void path_copy(
    OUString const & dest_path_, OUString const & src_path,
    pkgchk_env const & env, bool overwrite )
{
    FileStatus status( c_default_file_status_mask );
    path_get_status( &status, src_path, c_default_file_status_mask );
    FileStatus::Type file_type = status.getFileType();
    
    OUString dest_path( dest_path_ );
    
    if (FileStatus::Directory == file_type)
    {
        if (path_exists( dest_path )) // dest_path exists
        {
            // copy into dest path
            dest_path = path_concat(
                dest_path, file_status_get_encoded_name( status ) );
        } // else create dest path and copy contents into it
        dir_ensure( dest_path );
        
        Directory dir( src_path );
        dir_open( dir, src_path );
        while (true)
        {
            {
            DirectoryItem dirItem;
            Directory::RC rc = dir.getNextItem( dirItem );
            if (Directory::E_NOENT == rc)
                break;
            if (Directory::E_None != rc || !dirItem.is())
            {
                throw RuntimeException(
                    OUSTR("cannot get next dir item from ") + src_path,
                    Reference< XInterface >() );
            }
            diritem_get_status( &status, dirItem, c_default_file_status_mask );
            } // release dir item
            // recurse
            path_copy( dest_path, status.getFileURL(), env, overwrite );
        }
    }
    else // copy file
    {
        OSL_ASSERT(
            FileStatus::Regular == file_type || FileStatus::Link == file_type );
        if (FileStatus::Regular != file_type && FileStatus::Link != file_type)
        {
            throw RuntimeException(
                src_path + OUSTR(" has unexpected file type!"),
                Reference< XInterface >() );
        }
        if (path_exists( dest_path ))
        {
            FileStatus dest_status( c_default_file_status_mask );
            path_get_status(
                &dest_status, dest_path, c_default_file_status_mask );
            if (FileStatus::Directory == dest_status.getFileType())
            {
                // copy into dest path
                dest_path = path_concat(
                    dest_path, file_status_get_encoded_name( status ) );
            }
            if (!overwrite && path_exists( dest_path ))
            {
                OUStringBuffer buf( 64 );
                buf.appendAscii( RTL_CONSTASCII_STRINGPARAM("copying ") );
                buf.append( src_path );
                buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(" to ") );
                buf.append( dest_path );
                buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(
                                     ": failed!  destination file exists!") );
                throw RuntimeException(
                    buf.makeStringAndClear(),
                    Reference< XInterface >() );
            }
        }
        
        if (File::E_None != File::copy( src_path, dest_path ))
        {
            OUStringBuffer buf( 64 );
            buf.appendAscii( RTL_CONSTASCII_STRINGPARAM("copying ") );
            buf.append( src_path );
            buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(" to ") );
            buf.append( dest_path );
            buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(": failed!") );
            throw RuntimeException(
                buf.makeStringAndClear(),
                Reference< XInterface >() );
        }
        
        // try preserve time stamp, at least modify stamp
        FileStatus src_status(
            c_default_file_status_mask |
            FileStatusMask_CreationTime | FileStatusMask_AccessTime );
        path_get_status( &src_status, src_path, c_default_file_status_mask );
        TimeValue tModify( src_status.getModifyTime() );
        TimeValue tCreation( src_status.getCreationTime() );
        TimeValue tAccess( src_status.getAccessTime() );
        // if creation time not supported
        if (! src_status.isValid( FileStatusMask_CreationTime ))
            tCreation = tModify;
        // if access time not supported
        if (! src_status.isValid( FileStatusMask_AccessTime ))
            tAccess = tModify;
        File::RC rc = File::setTime( dest_path, tCreation, tAccess, tModify );
        if (File::E_None != rc)
        {
            OUStringBuffer buf( 64 );
            buf.appendAscii(
                RTL_CONSTASCII_STRINGPARAM("setting time stamp of ") );
            buf.append( dest_path );
            buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(": failed!") );
            env.err( buf.makeStringAndClear() );
        }
    }
}

//##############################################################################

//==============================================================================
void zip_inflate(
    OUString const & dest_path, OUString const & src_path,
    pkgchk_env const & env )
{
    // create dest dir
    dir_ensure( dest_path );
    
    OUString err_msg;
    try
    {
        // inflate into dest dir
        OUStringBuffer buf( 64 );
        buf.appendAscii( RTL_CONSTASCII_STRINGPARAM("vnd.sun.star.pkg://") );
        OUString src_uri(
            Uri::encode(
                src_path, rtl_getUriCharClass( rtl_UriCharClassRegName ),
                rtl_UriEncodeIgnoreEscapes, RTL_TEXTENCODING_UTF8 ) );
        buf.append( src_uri );
        buf.append( (sal_Unicode) '/' );
        src_uri = buf.makeStringAndClear();
        
        // this may bootstrap uno
        // (necessary here, because of types from types.rdb)
        Reference< ucb::XCommandProcessor > xCmdProc( env.get_ucb_cmdproc() );
        // global transfer
        ucb::GlobalTransferCommandArgument transfer_arg(
            ucb::TransferCommandOperation_COPY, src_uri, dest_path,
            OUString(), ucb::NameClash::OVERWRITE );
        ucb::Command cmd(
            OUSTR("globalTransfer"), -1, makeAny( transfer_arg ) );
        xCmdProc->execute( cmd, 0, Reference< ucb::XCommandEnvironment >() );
    }
    catch (ucb::CommandAbortedException & exc)
    {
        err_msg = OUSTR("ucb.CommandAbortedException occured: ") + exc.Message;
    }
    catch (RuntimeException &)
    {
        throw;
    }
    catch (Exception & exc)
    {
        err_msg = OUSTR("exception occured: ") + exc.Message;
    }
    // log msg
    OUStringBuffer buf( 128 );
    buf.appendAscii( RTL_CONSTASCII_STRINGPARAM("inflating ") );
    buf.append( src_path );
    buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(" into directory ") );
    buf.append( dest_path );
    if (err_msg.getLength())
    {
        buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(": failed!  ") );
        buf.append( err_msg );
        env.err( buf.makeStringAndClear() );
        File::remove( dest_path ); // try to remove destination
    }
    else
    {
        buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(": ok.") );
        env.log( buf.makeStringAndClear() );
    }
}

//##############################################################################

//==============================================================================
OUString get_from_bootstrap( OUString const & key, Bootstrap const & bstrap )
{
    OUString value;
    if (! bstrap.getFrom( key, value ))
    {
        OUStringBuffer buf( 64 );
        buf.appendAscii( RTL_CONSTASCII_STRINGPARAM("missing key \"") );
        buf.append( key );
        buf.appendAscii( RTL_CONSTASCII_STRINGPARAM("\" in ") );
        OUString iniPath;
        bstrap.getIniName( iniPath );
        buf.append( iniPath );
        throw RuntimeException(
            buf.makeStringAndClear(), Reference< XInterface >() );
    }
    return value;
}

//##############################################################################

#define IPC_PREFIX "SingleOfficeIPC_"

//==============================================================================
bool office_is_running( Bootstrap const & unorc )
{
    OUString install_path(
        RTL_CONSTASCII_USTRINGPARAM(RC_USER_INSTPATH_MACRO) );
    unorc.expandMacrosFrom( install_path );
    if (! path_exists( install_path ))
        return false;
    // extra normalize path for hash
    FileStatus status( FileStatusMask_FileURL );
    path_get_status( &status, install_path, FileStatusMask_FileURL );
    install_path = status.getFileURL();
    
	rtlDigest digest = rtl_digest_create( rtl_Digest_AlgorithmMD5 );
	if (digest <= 0)
    {
        throw RuntimeException(
            OUSTR("cannot get digest rtl_Digest_AlgorithmMD5!"),
            Reference< XInterface >() );
    }
    
    sal_uInt8 const * data = (sal_uInt8 const *) install_path.getStr();
    sal_uInt32 size = (install_path.getLength() * sizeof (sal_Unicode));
    sal_uInt32 md5_key_len = rtl_digest_queryLength( digest );
    sal_uInt8 * md5_buf = new sal_uInt8 [ md5_key_len ];
    
    rtl_digest_init( digest, data, size );
    rtl_digest_update( digest, data, size );
    rtl_digest_get( digest, md5_buf, md5_key_len );
    rtl_digest_destroy( digest );
    
    // create hex-value string from the MD5 value to keep
    // the string size minimal
    OUStringBuffer buf( sizeof (IPC_PREFIX) -1 + (md5_key_len * 2) +1 );
    buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(IPC_PREFIX) );
    for ( sal_uInt32 i = 0; i < md5_key_len; ++i )
    {
        buf.append( (sal_Int32) md5_buf[ i ], 0x10 );
    }
    
    delete [] md5_buf;
    
    OUString pipe_id( buf.makeStringAndClear() );    
    Security sec;
	Pipe pipe( pipe_id, osl_Pipe_OPEN, sec );
    
	return (sal_False != pipe.is());
}

}
