/*************************************************************************
 *
 *  $RCSfile: pkgchk_packages.cxx,v $
 *
 *  $Revision: 1.11 $
 *
 *  last change: $Author: vg $ $Date: 2003/05/28 13:38:10 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2002 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <vector>

#include "pkgchk_env.h"
#include "rtl/string.hxx"
#include "rtl/strbuf.hxx"


using namespace ::std;
using namespace ::rtl;
using namespace ::osl;
using namespace ::com::sun::star::uno;
using namespace ::com::sun::star;

namespace pkgchk
{

enum content_type { ZIP, SO, JAR, PY, XLB, RDB, XCU, XCS, OTHER };

//==============================================================================
static inline content_type path_get_content_type( OUString const & path )
{
    sal_Int32 dot = path.lastIndexOf( '.' );
    if (dot >= 0)
    {
        if (0 == rtl_ustr_ascii_compareIgnoreAsciiCase(
                path.pData->buffer + dot, ".zip" ))
        {
            return ZIP;
        }
        if (0 == rtl_ustr_ascii_compareIgnoreAsciiCase(
                path.pData->buffer + dot, SAL_DLLEXTENSION ))
        {
            return SO;
        }
        if (0 == rtl_ustr_ascii_compareIgnoreAsciiCase(
                path.pData->buffer + dot, ".jar" ))
        {
            return JAR;
        }
        if (0 == rtl_ustr_ascii_compareIgnoreAsciiCase(
                path.pData->buffer + dot, ".rdb" ))
        {
            return RDB;
        }
        if (0 == rtl_ustr_ascii_compareIgnoreAsciiCase(
                path.pData->buffer + dot, ".xlb" ))
        {
            return XLB;
        }
        if (0 == rtl_ustr_ascii_compareIgnoreAsciiCase(
                path.pData->buffer + dot, ".xcu" ))
        {
            return XCU;
        }
        if (0 == rtl_ustr_ascii_compareIgnoreAsciiCase(
                path.pData->buffer + dot, ".xcs" ))
        {
            return XCS;
        }
        if (0 == rtl_ustr_ascii_compareIgnoreAsciiCase(
                path.pData->buffer + dot, ".py" ))
        {
            return PY;
        }
    }
    return OTHER;
}

//==============================================================================
static inline OUString content_type_to_loadername( content_type ct )
{
    OUString ret;
    switch (ct)
    {
    case SO:
        ret = OUSTR("com.sun.star.loader.SharedLibrary");
        break;
    case JAR:
        ret = OUSTR("com.sun.star.loader.Java2");
        break;
    case PY:
        ret = OUSTR("com.sun.star.loader.Python");
        break;
    default:
        throw RuntimeException(
            OUSTR("cannot determine loader type!"),
            Reference< XInterface >() );
    }
    return ret;
}

//==============================================================================
static void package_unlink(
    OUString const & package,
    OUString const & base_path, OUString const & reg_base_path,
    pkgchk_env & env, bool skip_registration = false )
{
    OUString path( path_concat( base_path, package ) );
    FileStatus status( c_default_file_status_mask );
    path_get_status( &status, path, c_default_file_status_mask );
    FileStatus::Type file_type = status.getFileType();
    
    if (FileStatus::Directory == file_type)
    {
        if (platform_ignore_path( path ))
        {
#if defined DIAG
            OString cstr_path(
                OUStringToOString( path, osl_getThreadTextEncoding() ) );
            printf(
                "[diag] package_unlink(): ignoring path %s\n",
                cstr_path.getStr() );
#endif
            return;
        }
        skip_registration |=
            ends_with_ignore_ascii_case(
                path, RTL_CONSTASCII_STRINGPARAM("/skip_registration") );
        
        Directory dir( path );
        dir_open( dir, path );
        while (true)
        {
            {
            DirectoryItem dirItem;
            Directory::RC rc = dir.getNextItem( dirItem );
            if (Directory::E_NOENT == rc)
                break;
            if (Directory::E_None != rc || !dirItem.is())
            {
                throw RuntimeException(
                    OUSTR("cannot get next packages dir item from ") + path,
                    Reference< XInterface >() );
            }
            diritem_get_status( &status, dirItem, c_default_file_status_mask );
            } // release dir item
            // recurse
            package_unlink(
                path_concat( package, file_status_get_encoded_name( status ) ),
                base_path, reg_base_path, env, skip_registration );
        }
    }
    // do something
    else if (FileStatus::Regular == file_type || FileStatus::Link == file_type)
    {
        OUString url( env.make_reg_url( reg_base_path, package ) );
        content_type ct = path_get_content_type( path );
        switch (ct)
        {
        case JAR:
            // remove from classpath file
            env.classpath_remove( package );
        case SO:
        case PY:
        {
            if (skip_registration)
            {
#if defined DIAG
                OString cstr_url(
                    OUStringToOString( url, osl_getThreadTextEncoding() ) );
                printf(
                    "[diag] package_unlink(): skipping registration of %s\n",
                    cstr_url.getStr() );
#endif
            }
            else // revoke from services rdb
            {
                Reference< registry::XSimpleRegistry > xServices_rdb(
                    env.get_services_rdb() );
                
                OUStringBuffer buf( 128 );
                buf.appendAscii( RTL_CONSTASCII_STRINGPARAM("revoking ") );
                buf.append( url );
                buf.appendAscii(
                    RTL_CONSTASCII_STRINGPARAM(" from registry ") );
                buf.append( xServices_rdb->getURL() );
                
                if (env.get_impreg()->revokeImplementation(
                        url, xServices_rdb ))
                {
                    buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(": ok.") );
                }
                else
                {
                    buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(": failed!") );
                    // just log errors when revoking
                }
                env.log( buf.makeStringAndClear() );
            }
            break;
        }
        case XLB:
        {
            env.basic_remove( url );
            break;
        }
        case XCU:
        {
            env.xcu_remove( url );
            break;
        }
        case XCS:
        {
            env.xcs_remove( url );
            break;
        }
        
        default:
            break;
        }
    }
    else
    {
        env.warn( path + OUSTR(" has unexpected file type!") );
    }
}


//------------------------------------------------------------------------------
static void demacrofy_xcu_origin(
    OUString const & path, OUString const & url )
{
    // encode conforming xml
    OUString dir( url.copy( 0, url.lastIndexOf( '/' ) ) );
    sal_Int32 len = dir.getLength();
    OUStringBuffer buf( len + 12 );
    for ( sal_Int32 pos = 0; pos < len; ++pos )
    {
        sal_Unicode c = dir[ pos ];
        switch (c)
        {
        case '<':
            buf.appendAscii( RTL_CONSTASCII_STRINGPARAM("&lt;") );
            break;
        case '>':
            buf.appendAscii( RTL_CONSTASCII_STRINGPARAM("&gt;") );
            break;
        case '&':
            buf.appendAscii( RTL_CONSTASCII_STRINGPARAM("&amp;") );
            break;
        case '\'':
            buf.appendAscii( RTL_CONSTASCII_STRINGPARAM("&apos;") );
            break;
        case '\"':
            buf.appendAscii( RTL_CONSTASCII_STRINGPARAM("&quot;") );
            break;
        default:
            buf.append( c );
            break;
        }
    }
    OString origin(
        OUStringToOString( buf.makeStringAndClear(), RTL_TEXTENCODING_UTF8 ) );
    OString str_origin( RTL_CONSTASCII_STRINGPARAM("%origin%") );
    
    File file( path );
    if (File::E_None != file.open( OpenFlag_Read ))
	{
        throw RuntimeException(
            OUSTR("cannot read ") + path, Reference< XInterface >() );
	}
    sal_uInt64 file_size = file_get_size( file );
    
    bool write_file_back = false;
    ByteSequence seq;
    vector< OString > lines;
    lines.reserve( 64 );
    while (true)
    {
        sal_uInt64 nPos;
        if (File::E_None != file.getPos( nPos ) || nPos >= file_size)
            break;
        if (File::E_None != file.readLine( seq ))
            break;
        
        OString line( (sal_Char const *) seq.getConstArray(), seq.getLength() );
        sal_Int32 pos = line.indexOf( '%' );
        if (pos >= 0)
        {
            sal_Int32 len = line.getLength();
            OStringBuffer buf( len + 64 );
            buf.append( line.getStr(), pos );
            while (pos < len)
            {
                sal_Char c = line[ pos ];
                if (c == '%')
                {
                    if ((pos +1) < len && line[ pos +1 ] == '%') // %%
                    {
                        ++pos; // consume extra %
                        write_file_back = true;
                    }
                    else if (line.match( str_origin, pos )) // %origin%
                    {
                        buf.append( origin );
                        pos += str_origin.getLength();
                        write_file_back = true;
                        continue;
                    }
                }
                buf.append( c );
                ++pos;
            }
            line = buf.makeStringAndClear();
        }
        lines.push_back( line );
    }
    file.close();
    
    if (write_file_back)
    {
        // write to unorc
        if (File::E_None != file.open( OpenFlag_Write ))
        {
            throw RuntimeException(
                OUSTR("cannot open ") + path + OUSTR(" for writing"),
                Reference< XInterface >() );
        }
        
        for ( size_t n = 0; n < lines.size(); ++n )
        {
            OString const & line = lines[ n ];
            file_write( file, line, path );
            file_write( file, LF, path );
        }
        file.close();
    }
}

//==============================================================================
static void package_link(
    OUString const & package,
    OUString const & base_path, OUString const & reg_base_path,
    pkgchk_env & env, bool skip_registration = false )
{
    OUString path( path_concat( base_path, package ) );
    FileStatus status( c_default_file_status_mask );
    path_get_status( &status, path, c_default_file_status_mask );
    FileStatus::Type file_type = status.getFileType();
    
    if (FileStatus::Directory == file_type)
    {
        if (platform_ignore_path( path ))
        {
#if defined DIAG
            OString cstr_path(
                OUStringToOString( path, osl_getThreadTextEncoding() ) );
            printf(
                "[diag] package_link(): ignoring path %s\n",
                cstr_path.getStr() );
#endif
            return;
        }
        skip_registration |=
            ends_with_ignore_ascii_case(
                path, RTL_CONSTASCII_STRINGPARAM("/skip_registration") );
        
        Directory dir( path );
        dir_open( dir, path );
        while (true)
        {
            {
            DirectoryItem dirItem;
            Directory::RC rc = dir.getNextItem( dirItem );
            if (Directory::E_NOENT == rc)
                break;
            if (Directory::E_None != rc || !dirItem.is())
            {
                throw RuntimeException(
                    OUSTR("cannot get next packages dir item from ") + path,
                    Reference< XInterface >() );
            }
            diritem_get_status( &status, dirItem, c_default_file_status_mask );
            } // release dir item
            // recurse
            package_link(
                path_concat( package, file_status_get_encoded_name( status ) ),
                base_path, reg_base_path, env, skip_registration );
        }
    }
    // do something
    else if (FileStatus::Regular == file_type || FileStatus::Link == file_type)
    {
        OUString url( env.make_reg_url( reg_base_path, package ) );
        content_type ct = path_get_content_type( path );
        switch (ct)
        {
        case SO:
        case JAR:
        case PY:
        {
            if (skip_registration)
            {
#if defined DIAG
                OString cstr_url(
                    OUStringToOString( url, osl_getThreadTextEncoding() ) );
                printf(
                    "[diag] package_link(): skipping registration for %s\n",
                    cstr_url.getStr() );
#endif
            }
            else // register into services rdb
            {
                Reference< registry::XSimpleRegistry > xServices_rdb(
                    env.get_services_rdb() );
                
                OUStringBuffer buf( 128 );
                buf.appendAscii( RTL_CONSTASCII_STRINGPARAM("registering ") );
                buf.append( url );
                buf.appendAscii(
                    RTL_CONSTASCII_STRINGPARAM(" into registry ") );
                buf.append( xServices_rdb->getURL() );
                
                try
                {
                    env.get_impreg()->registerImplementation(
                        content_type_to_loadername( ct ), url, xServices_rdb );
                    
                    buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(": ok.") );
                    env.log( buf.makeStringAndClear() );
                }
                catch (registry::CannotRegisterImplementationException & exc)
                {
                    buf.appendAscii(
                        RTL_CONSTASCII_STRINGPARAM(
                            ": failed! (cannot register implementation") );
                    if (exc.Message.getLength())
                    {
                        buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(": ") );
                        buf.append( exc.Message );
                    }
                    buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(")!") );
                    env.err( buf.makeStringAndClear() );
                }
            }
            if (JAR == ct)
            {
                // add jar to classpath file
                env.classpath_insert( package );
            }
            break;
        }
        case RDB:
        {
            try
            {
                // merge types in
                env.get_types_rdb()->mergeKey( OUSTR("/"), path );
            }
            catch (registry::InvalidRegistryException & exc)
            {
                throw RuntimeException(
                    0 == exc.Message.getLength() // work around empty Message
                    ? OUSTR("registry::InvalidRegistryException "
                            "exception occured!")
                    : exc.Message,
                    Reference< XInterface >() );
            }
            catch (registry::MergeConflictException & exc)
            {
                throw RuntimeException(
                    0 == exc.Message.getLength() // work around empty Message
                    ? OUSTR("registry::MergeConflictException "
                            "exception occured!")
                    : exc.Message,
                    Reference< XInterface >() );
            }
            break;
        }
        case XLB:
        {
            env.basic_insert( url );
            break;
        }
        case XCU:
        {
            demacrofy_xcu_origin( path, url );
            env.xcu_insert( url );
            break;
        }
        case XCS:
        {
            env.xcs_insert( url );
            break;
        }
        
        default:
            break;
        }
    }
    else
    {
        env.warn( path + OUSTR(" has unexpected file type!") );
    }
}

//##############################################################################

//______________________________________________________________________________
bool pkgchk_env::packages_check()
{
    // ensure packages path
    dir_ensure( m_packages_path );
    
    // new check
    m_packages_to_be_installed.clear();
    m_packages_to_be_removed.clear();
    
    log( OUSTR("scanning uno packages from ") + m_packages_path );
    
    FileStatus status( c_default_file_status_mask );
    Directory packages_dir( m_packages_path );
    dir_open( packages_dir, m_packages_path );
    while (true)
    {
        DirectoryItem dirItem;
        Directory::RC rc = packages_dir.getNextItem( dirItem );
        if (Directory::E_NOENT == rc)
            break;
        if (Directory::E_None != rc || !dirItem.is())
        {
            throw RuntimeException(
                OUSTR("cannot get next packages dir item from ") +
                  m_packages_path,
                Reference< XInterface >() );
        }
        diritem_get_status( &status, dirItem, c_default_file_status_mask );
        
        OUString package_path( status.getFileURL() );
        
        // ignore cache dir
        if (package_path.equals( m_cache_path ))
        {
            continue;
        }
        FileStatus::Type file_type = status.getFileType();
        if (FileStatus::Directory == file_type)
        {
            err( OUSTR("directories not allowed: ") + package_path );
            continue;
        }
        if (FileStatus::Regular != file_type && FileStatus::Link != file_type)
        {
            err( package_path + OUSTR(" is of unsupported file type!") );
            continue;
        }
        
        m_packages_to_be_installed.insert(
            file_status_get_encoded_name( status ) );
    }
    
    OUString cache_packages_path(
        path_concat(
            m_cache_path,
            RTL_CONSTASCII_STRINGPARAM("uno_packages") ) );
    // output packages cache directory
    Directory cache_packages_dir( cache_packages_path );
    dir_open(
        cache_packages_dir, cache_packages_path,
        true /* create if not existing */ );
    
    // read installed entries from cache directory
    while (true)
    {
        {
        DirectoryItem dirItem;
        Directory::RC rc = cache_packages_dir.getNextItem( dirItem );
        if (Directory::E_NOENT == rc)
            break;
        if (Directory::E_None != rc || !dirItem.is())
        {
            throw RuntimeException(
                OUSTR("cannot get next cache dir item from ") +
                cache_packages_path,
                Reference< XInterface >() );
        }
        diritem_get_status( &status, dirItem, c_default_file_status_mask );
        FileStatus::Type file_type = status.getFileType();
        if (FileStatus::Directory != file_type &&
            FileStatus::Regular != file_type &&
            FileStatus::Link != file_type)
        {
            err( status.getFileURL() + OUSTR(" is of unexpected file type!") );
            continue;
        }
        } // release dir item
        
        OUString package( file_status_get_encoded_name( status ) );
        
        if (FileStatus::Directory == status.getFileType()) // zip inflation dir
        {
            // decode time stamp of inflation dir
            sal_Int32 dot = package.lastIndexOf( '.' );
            OSL_ENSURE( 0 <= dot, "no zip inflation dir?!" );
            if (0 <= dot)
            {
                sal_uInt32 tCacheEntry =
                    (sal_uInt32)package.copy( dot +1 ).toInt64();
                OSL_ENSURE(
                    tCacheEntry,
                    "zip inflation dir without encoded time stamp?!" );
                if (tCacheEntry)
                {
                    OUString package_name = package.copy( 0, dot );
                    OSL_ASSERT( ZIP == path_get_content_type( package_name ) );
                    // is in packages dir?
                    if (m_packages_to_be_installed.find( package_name ) !=
                        m_packages_to_be_installed.end())
                    {
                        path_get_status(
                            &status,
                            path_concat( m_packages_path, package_name ),
                            c_default_file_status_mask );
                        TimeValue time( status.getModifyTime() );
                        if (tCacheEntry == time.Seconds)
                        {
                            // same version
                            m_packages_to_be_installed.erase( package_name );
                            continue;
                        }
                    }
                }
            }
        }
        else if (m_packages_to_be_installed.find( package ) !=
                 m_packages_to_be_installed.end()) // in packages dir?
        {
            TimeValue time( status.getModifyTime() );
            sal_uInt32 tCacheEntry = time.Seconds;
            path_get_status(
                &status, path_concat( m_packages_path, package ),
                c_default_file_status_mask );
            time = status.getModifyTime();
            if (tCacheEntry == time.Seconds)
            {
                // same version
                m_packages_to_be_installed.erase( package );
                continue;
            }
        }
        
        // to be removed
        m_packages_to_be_removed.insert( package );
    }    
    
    log( OUSTR("finished scanning uno packages.") );
    
    return packages_to_be_balanced();
}

//______________________________________________________________________________
void pkgchk_env::packages_balance()
{
    // hack: ensure UCB is initialized, because some
    // Office components rely on it.
    get_ucb_cmdproc();
    
    OUString uno_packages( RTL_CONSTASCII_USTRINGPARAM("uno_packages") );
    OUString cache_packages_path( path_concat( m_cache_path, uno_packages ) );
    
    log( OUSTR("balancing cache dir ") + m_cache_path );
    
    // first uninstall removed packages
    t_string_set::const_iterator iPos( m_packages_to_be_removed.begin() );
    t_string_set::const_iterator iEnd( m_packages_to_be_removed.end() );
    for ( ; iPos != iEnd; ++iPos )
    {
        OUString const & package = *iPos;
        // prefix with uno_packages/
        OUString rel_package_path( path_concat( uno_packages, package ) );
        // remove from services registry
        package_unlink(
            rel_package_path, m_cache_path, m_reg_cache_path, *this );
        // erase path
        path_erase( path_concat( m_cache_path, rel_package_path ) );
    }
    
    // then install added packages
    FileStatus status( c_default_file_status_mask );
    iPos = m_packages_to_be_installed.begin();
    iEnd = m_packages_to_be_installed.end();
    for ( ; iPos != iEnd; ++iPos )
    {
        OUString const & package = *iPos;
        if (ZIP == path_get_content_type( package )) // inflate
        {
            // decorate name with time stamp
            OUStringBuffer buf( 128 );
            buf.append( package );
            buf.append( (sal_Unicode)'.' );
            path_get_status(
                &status, path_concat( m_packages_path, package ),
                c_default_file_status_mask );
            TimeValue time( status.getModifyTime() );
            buf.append( OUString::valueOf( (sal_Int64) time.Seconds ) );
            // prefix with uno_packages/
            OUString reg_package_path(
                path_concat( uno_packages, buf.makeStringAndClear() ) );
            // inflate
            zip_inflate(
                path_concat( m_cache_path, reg_package_path ),
                path_concat( m_packages_path, package ), *this );
            // link into installation
            package_link(
                reg_package_path,
                m_cache_path, m_reg_cache_path, *this );
        }
        else
        {
            // copy into cache dir
            path_copy(
                cache_packages_path, path_concat( m_packages_path, package ),
                *this );
            // link into installation
            package_link(
                // prefix with uno_packages/
                path_concat( uno_packages, package ),
                m_cache_path, m_reg_cache_path, *this );
        }
    }
    
    log( OUSTR("finished balancing cache dir.") );
}

//______________________________________________________________________________
void pkgchk_env::packages_remove_all()
{
    // erase cache entries from script.xlc, dialog.xlc
    basic_remove_cache_entries();
    
    // log files resides in cache dir?
    if (0 == m_log_file_path.compareTo(
            m_cache_path, m_cache_path.getLength() ))
    {
        if (m_log_file) // if Initialized
        {
            // close old log file
            OSL_VERIFY( osl_File_E_None == osl_closeFile( m_log_file ) );
            m_log_file = 0;
        }
        // de-install log file
        OUString log_file_path( m_log_file_path );
        m_log_file_path = OUString();
        
        // erase cache dir
        path_erase( m_cache_path );
        // ensure cache path
        dir_create( m_cache_path );
        
        // new log
        m_log_file_path = log_file_path;
    }
    else
    {
        // erase cache dir
        path_erase( m_cache_path );
        // ensure cache path
        dir_create( m_cache_path );
    }
}

}
