/*************************************************************************
 *
 *  $RCSfile: speedctl.cxx,v $
 *
 *  $Revision: 1.5 $
 *
 *  last change: $Author: ka $ $Date: 2001/08/22 14:23:16 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifdef WNT
#include <tools/svwin.h>
#endif // WNT

#include <math.h>
#include <tools/time.hxx>
#include <vcl/poly.hxx>
#include <vcl/salbtype.hxx>
#include <vcl/svapp.hxx>
#include <vcl/window.hxx>
#include "speedctl.hxx"

// -----------
// - Defines -
// -----------

#define SPEED_MAGIC                 0x56789ABC
#define SPEED_SCALINGDIST			10000
#define SPEED_TIMEDIST_MS			40
#define SPEED_MIN_INC			    0.001
#define SPEED_RESCHEDULE			5
#define TIME_TO_MSEC( _aTime )		((_aTime).Seconds*1000UL+(_aTime).Nanosec/1000000)

// --------------
// - DistStruct -
// --------------

struct DistInfo
{
	Point	maPt;
	double	mfCumulatedDist;
};

// ----------------
// - SpeedControl -
// ----------------

SpeedControl::SpeedControl() :
	mpWin				( NULL ),
	mfUnitsPerSec		( 0.0 ),
	mfCurStep			( 0.0 ),
	mfLastStep			( 0.0 ),
	mfCumulatedSteps	( 0.0 ),
	mfLastCumulatedSteps( 0.0 ),
	mnLastTime			( 0UL ),
	mnCurStep			( 0UL ),
	mnLastStep			( 0UL ),
	mnMagic     		( SPEED_MAGIC )
{
}

// ------------------------------------------------------------------------

SpeedControl::~SpeedControl()
{
}

// ------------------------------------------------------------------------
void SpeedControl::ImplInit( Window* pWin, const double& rUnitsPerSec )
{
	mpWin = pWin;
	mfUnitsPerSec = rUnitsPerSec;
	mfCurStep =	mfLastStep = mfCumulatedSteps = mfLastCumulatedSteps = 0.0;
	mnLastTime = 0UL;
	mnCurStep = mnLastStep = 0UL;
	mnMagic = SPEED_MAGIC;
}

// ------------------------------------------------------------------------

void SpeedControl::Reset( Window* pWin, const double& rUnitsPerSec )
{
	ImplInit( pWin, rUnitsPerSec );
}

// ------------------------------------------------------------------------

ULONG SpeedControl::GetNextStep()
{
	if( SPEED_MAGIC != mnMagic )
		return 0;

	static ULONG nTimeCount = 0UL;

	if( !mnLastTime )
		mnLastTime = Time::GetSystemTicks();

	const ULONG	nDiff = Time::GetSystemTicks() - mnLastTime;

	if( nDiff >= SPEED_TIMEDIST_MS )
	{
		const double mfCurSpeed = ( mfCumulatedSteps - mfLastCumulatedSteps ) * 1000. / nDiff;

		mfLastStep = mfCurStep;

		if( mfCurSpeed > 0. )
			mfCurStep *= ( mfUnitsPerSec / mfCurSpeed );
		else
			mfCurStep = SPEED_MIN_INC;

		mfCurStep = ( mfCurStep + mfLastStep ) * 0.5;
		mfLastCumulatedSteps = mfCumulatedSteps;

		// reschedule only a few times
		if( !( nTimeCount++ % SPEED_RESCHEDULE ) )
			Application::Reschedule();

		if( SPEED_MAGIC != mnMagic )
			return 0;

		mnLastTime = Time::GetSystemTicks();
	}

	mfCumulatedSteps += mfCurStep;
	mnLastStep = mnCurStep;
	mnCurStep = (ULONG) ( mfCumulatedSteps + .5 );

	if( ( SPEED_MAGIC == mnMagic ) && mpWin )
		mpWin->Sync();

#ifdef WNT
		Sleep( 1 );
#endif // WNT

	return( mnCurStep - mnLastStep );
}

// ---------------------
// - MoverSpeedControl -
// ---------------------

MoverSpeedControl::MoverSpeedControl() :
	mpDist			( NULL ),
	mnDistCount		( 0UL ),
	mnActDistPos	( 0UL ),
	mfTotalDist		( 0.0 ),
	mfStartScaleX	( 1.0 ),
	mfStartScaleY	( 1.0 ),
	mfEndScaleX		( 1.0 ),
	mfEndScaleY		( 1.0 ),
	mbScaleOnly		( FALSE )
{
}

// -----------------------------------------------------------------------------

MoverSpeedControl::~MoverSpeedControl()
{
	delete[] mpDist;
}

// ------------------------------------------------------------------------

void MoverSpeedControl::Reset( const Polygon& rPath, Window* pWin, const double& rUnitsPerSec, 
							   const double& rStartScaleX, const double& rEndScaleX,
							   const double& rStartScaleY, const double& rEndScaleY )
{
	ImplInit( pWin, rUnitsPerSec );

	delete[] mpDist;
	
	mnDistCount = rPath.GetSize();
	mnActDistPos = 0UL;
	mfStartScaleX = rStartScaleX;
	mfStartScaleY = rStartScaleY;
	mfEndScaleX = rEndScaleX;
	mfEndScaleY = rEndScaleY;

	if( mnDistCount )
	{
		Point		aLast( rPath[ 0 ] );
		DistInfo*	pTmp = ( mpDist = new DistInfo[ mnDistCount ] );

		pTmp->maPt = aLast;
		pTmp->mfCumulatedDist = 0.0;
		pTmp++;

		for( USHORT i = 1; i < mnDistCount; i++, pTmp++ )
		{
			const Point&	rAct = rPath[ i ];
			const double	fDistX = rAct.X() - aLast.X();
			const double	fDistY = rAct.Y() - aLast.Y();

			pTmp->maPt = rAct;
			pTmp->mfCumulatedDist = ( pTmp - 1 )->mfCumulatedDist + sqrt( fDistX * fDistX + fDistY * fDistY );
			aLast = rAct;
		}

		if( mpDist[ mnDistCount - 1 ].mfCumulatedDist != 0.0 )
		{
			mfTotalDist = mpDist[ mnDistCount - 1 ].mfCumulatedDist;
			mbScaleOnly = FALSE;
		}
		else if( ( rStartScaleX != 1.0 ) || ( rEndScaleX != 1.0 ) ||
				 ( rStartScaleY != 1.0 ) || ( rEndScaleY != 1.0 ) )
		{
			mfTotalDist = SPEED_SCALINGDIST;
			mbScaleOnly = TRUE;
		}
		else
		{
			mfTotalDist = 0.0;
			mbScaleOnly = FALSE;
		}
					
	}
	else
	{
		mfTotalDist = 0.0;
		mbScaleOnly = FALSE;
		mpDist = NULL;
	}
}

// ------------------------------------------------------------------------

BOOL MoverSpeedControl::GetNextPathPoint( Point& rPt, double& rScaleX, double& rScaleY )
{
	BOOL bRet = FALSE;

	if( SPEED_MAGIC != mnMagic )
		return FALSE;

	if( mpDist )
	{
		BOOL bFound = FALSE;

		if( !mnLastTime )
		{
			rPt = mpDist[ 0 ].maPt;
			rScaleX = mfStartScaleX;
			rScaleY = mfStartScaleY;
			mnLastTime = Time::GetSystemTicks();
			bRet = TRUE;
		}
		else
		{
			GetNextStep();

        	if( SPEED_MAGIC != mnMagic )
				return FALSE;

			if( mfCumulatedSteps <= mfTotalDist )
			{
				const double fTau = ( ( mfTotalDist != 0.0 ) ? ( mfCumulatedSteps / mfTotalDist ) : 1.0 );

				if( mbScaleOnly )
				{
					rPt = mpDist[ 0 ].maPt;
					rScaleX = mfStartScaleX + ( mfEndScaleX - mfStartScaleX ) * fTau;
					rScaleY = mfStartScaleY + ( mfEndScaleY - mfStartScaleY ) * fTau;
					bRet = TRUE;
				}
				else
				{
					for( ULONG i = mnActDistPos + 1; i < mnDistCount; i++, mnActDistPos++ )
					{
						const DistInfo& rAct = mpDist[ i ];

						if( rAct.mfCumulatedDist >= mfCumulatedSteps )
						{
							const DistInfo&	rLast = mpDist[ i - 1UL ];
							const Point&	rStart = rLast.maPt;
							const Point&	rEnd = rAct.maPt;
							const double	fCurSegmentLength = rAct.mfCumulatedDist - rLast.mfCumulatedDist;

							if( fCurSegmentLength != 0.0 )
							{
								const double fFact = ( mfCumulatedSteps - rLast.mfCumulatedDist ) / fCurSegmentLength;

								rPt.X() = rStart.X() + FRound( fFact * ( rEnd.X() - rStart.X() ) );
								rPt.Y() = rStart.Y() + FRound( fFact * ( rEnd.Y() - rStart.Y() ) );
							}
							else
								rPt = rStart;

							rScaleX = mfStartScaleX + ( mfEndScaleX - mfStartScaleX ) * fTau;
							rScaleY = mfStartScaleY + ( mfEndScaleY - mfStartScaleY ) * fTau;
							bRet = TRUE;
							break;
						}
					}
				}
			}
		}

		if( !bRet )
		{
			rPt = mpDist[ mnDistCount - 1 ].maPt;
			rScaleX = mfEndScaleX;
			rScaleY = mfEndScaleY;
		}
	}
	else
	{
		rPt = Point();
		rScaleX = rScaleY = 1.0;
	}

	return bRet;
}
