/*************************************************************************
 *
 *  $RCSfile: instdb.cxx,v $
 *
 *  $Revision: 1.12 $
 *
 *  last change: $Author: hr $ $Date: 2003/04/04 15:09:57 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/
#include <stdio.h>

#include "decltor.hxx"
#include "sistream.hxx"
#include "instdb.hxx"
#include "script.hxx"

SiDatabase::SiDatabase(SiFileStream& aStream)
: m_aDoneList(2017)
, m_aStream(aStream)
{
	m_eInstallMode	= IM_INVALID;
	m_nListPos		= 0;
	m_nLaguage		= USHRT_MAX;
	m_bScpMode 		= FALSE;
}

ByteString SiDatabase::Quote(ByteString const& s)
{
	ByteString Quoted;

	for (USHORT i=0; i<s.Len(); i++)
	{
		if (s.GetChar(i) == '"')
			Quoted += '\\';

		Quoted += s.GetChar(i);
	}

	// Fuer folgenden Fall: " ... = "a:\" -> "a:\\"
	if (Quoted.GetBuffer()[Quoted.Len()-1] == '\\')
		Quoted += '\\';

	return Quoted;
}

void SiDatabase::BeginDeclaration (ByteString const& aName, SiDeclarator const* pDec)
{
	m_aStream << aName << " " << pDec->GetID() << endl;
}

void SiDatabase::EndDeclaration ()
{
	m_aStream << "End" << endl << endl;
}

void SiDatabase::BeginProperty (ByteString const& aName)
{
	ByteString aTab;
	ByteString aLangName( aName );

	if( m_nLaguage != USHRT_MAX )
	{
		ByteString aLangStr = ByteString::CreateFromInt32( m_nLaguage );
		if( aLangStr.Len() == 1 )
			aLangStr.Insert( "0", 0 );
		aLangName += " (";
		aLangName += aLangStr;
		aLangName += ")";
	}

	if( (aLangName.Len()+4) < 16 )
		aTab.Fill( 3-(aLangName.Len()/4), '\t' );
	m_aStream << '\t' << aLangName << aTab << " = ";
}

void SiDatabase::EndProperty()
{
	m_aStream << ';' << endl;
}

void SiDatabase::BeginList()
{
	m_aStream << '(';
	m_nListPos = 0;
}

void SiDatabase::EndList()
{
	m_aStream << ')';
	m_nListPos = 0;
}

void SiDatabase::WriteFreeStyle(const ByteString& rFreeCode)
{
	ByteString aCode( rFreeCode );
	LineEnd eLineEnd = m_aStream.GetLineDelimiter();
	aCode.ConvertLineEnd( eLineEnd );
	aCode.SearchAndReplaceAll( "}", "\\}" );

	m_aStream << '{';
	m_aStream << aCode;
	m_aStream << '}';
}

void SiDatabase::AddSeparator()
{
	if (m_nListPos > 0)
	{
		m_aStream << ", ";

		if ((m_nListPos % 10) == 0)
		{
			ByteString aTab; aTab.Fill(5,'\t');
			m_aStream << endl; // Achtung: endl liefert SvStream& zurueck
			m_aStream << aTab;
		}
	}

	m_nListPos++;
}

void SiDatabase::AddListValue(long aValue)
{
	AddSeparator();
	m_aStream << aValue;
}

void SiDatabase::AddListValue(ByteString const& aValue, BOOL bNewLine)
{
	AddSeparator();
	m_aStream << '"' << Quote(aValue) << '"';

	if( bNewLine && m_nListPos > 0 )
		m_aStream << endl;
}

void SiDatabase::AddListValue(SiIdentifier const& aValue)
{
	AddSeparator();
	m_aStream << aValue.AsByteString();
}

void SiDatabase::AddListValue(SiDeclarator const* pValue)
{
	AddSeparator();
	m_aStream << pValue->GetID();
}

void SiDatabase::WriteProperty( ByteString const& aName, BOOL bValue, USHORT nLanguage)
{
	m_nLaguage = nLanguage;
	BeginProperty(aName);
	m_aStream << (bValue ? "YES" : "NO");
	EndProperty();
}

void SiDatabase::WriteProperty (ByteString const& aName, int nValue, USHORT nLanguage)
{
	m_nLaguage = nLanguage;
	WriteProperty (aName, (long)nValue);
}

void SiDatabase::WriteProperty (ByteString const& aName, long lValue, USHORT nLanguage)
{
	m_nLaguage = nLanguage;
	if (lValue != 0)
	{
		BeginProperty(aName);
		m_aStream << ByteString::CreateFromInt32( lValue );
		EndProperty();
	}
}

void SiDatabase::WriteProperty (ByteString const& aName, sal_uInt32 lValue, USHORT nLanguage)
{
	m_nLaguage = nLanguage;
	if (lValue != 0)
	{
		sal_Char aBuf[50];
		sprintf( aBuf, "%u", lValue );      // #100211# - checked

		BeginProperty(aName);
		m_aStream << ByteString( aBuf );
		EndProperty();
	}
}

void SiDatabase::WriteProperty (ByteString const& aName, ByteString const& aValue, USHORT nLanguage)
{
	m_nLaguage = nLanguage;
//	if (aValue.Len() != 0)
//	{
		BeginProperty(aName);
		m_aStream << '"' << Quote(aValue) << '"';
		EndProperty();
//	}
}

void SiDatabase::WriteProperty (ByteString const& aName, SiIdentifier const& aValue, USHORT nLanguage)
{
	m_nLaguage = nLanguage;
	if (aValue.AsByteString().Len() != 0)
	{
		BeginProperty(aName);
		m_aStream << aValue.AsByteString();
		EndProperty();
	}
}

void SiDatabase::WriteProperty (ByteString const& aName, SiDeclarator const* pValue, USHORT nLanguage)
{
	m_nLaguage = nLanguage;
	if (pValue != NULL)
	{
		BeginProperty(aName);
		m_aStream << pValue->GetID();
		EndProperty();
	}
}

void SiDatabase::WriteProperty (ByteString const& aName, Date const& aDate, USHORT nLanguage)
{
	m_nLaguage = nLanguage;
	if (aDate.GetYear() != 0)
	{
		BeginProperty(aName);
		m_aStream << '"';

		if (aDate.GetDay() < 10)
			m_aStream << '0';

		m_aStream <<  ByteString::CreateFromInt32( aDate.GetDay() );

		if (aDate.GetMonth() < 10)
			m_aStream << '0';

		m_aStream << ByteString::CreateFromInt32( aDate.GetMonth() );
		m_aStream << ByteString::CreateFromInt32( aDate.GetYear() );
		m_aStream << '"';
		EndProperty();
	}
}

void SiDatabase::WriteProperty( ByteString const& aName, Time const& aTime, USHORT nLanguage)
{
	m_nLaguage = nLanguage;

	BeginProperty(aName);
	m_aStream << '"';

	if (aTime.GetHour() < 10)
		m_aStream << '0';

	m_aStream << ByteString::CreateFromInt32( aTime.GetHour() );

	if (aTime.GetMin() < 10)
		m_aStream << '0';

	m_aStream << ByteString::CreateFromInt32( aTime.GetMin() );
	m_aStream << '"';
	EndProperty();
}

BOOL SiDatabase::WriteObject(SiDeclarator const* pDec)
// pre:  Objekte _koennte_ schon in der Datenbank sein
//
// post: Mit Ausnahme von Systemobjekten befindet sich
//		 das Objekt in der Datenbank und eine Referenz
//		 in der DoneList.
{
	if (!pDec->IsSystemObject()
	&&   m_aDoneList.Find(pDec->GetID()) == 0)
	{
		m_aDoneList.Insert(pDec->GetID(), TRUE);
		return pDec->WriteTo(*this);
	}

	return TRUE;
}

BOOL SiDatabase::Create
(
	SiDirectoryList const& aDirList
)
{
	for (USHORT i=0; i<aDirList.Count(); i++)
	{
		SiDirectory *pDir = aDirList.GetObject(i);
		Create(pDir);
		Create(pDir->GetSubDirList());
	}

	return TRUE;
}

BOOL SiDatabase::Create(SiDirectory const* pDir)
{
	if (pDir == NULL)
		return TRUE;

	if( m_eInstallMode != IM_WORKSTATION ||
		pDir->InstallOnWorkstation() )
		Create( pDir->GetParent() );

	WriteObject(pDir);
	return TRUE;
}

BOOL SiDatabase::Create(SiOs2Creator const* pOs2Creator)
{
	if (pOs2Creator->GetIconDir() != NULL)
		Create(pOs2Creator->GetIconDir());

	WriteObject(pOs2Creator);
	return TRUE;
}

BOOL SiDatabase::Create(SiOs2Class const* pOs2Class)
{
	if (pOs2Class->GetDLL() != NULL)
		Create(pOs2Class->GetDLL());

	WriteObject(pOs2Class);
	return TRUE;
}

BOOL SiDatabase::Create(SiOs2Template const* pOs2Template)
{
	if (pOs2Template->GetIcon() != NULL)
		Create(pOs2Template->GetIcon());

	if (pOs2Template->GetClass() != NULL)
		Create(pOs2Template->GetClass());

	WriteObject(pOs2Template);
	return TRUE;
}

BOOL SiDatabase::Create(SiFile const* pFile)
{
	Create(pFile->GetDirectory());
	Create(pFile->GetNetDirectory());

	WriteObject(pFile->GetDataCarrier()); // z.Zt. noch kein Pflichtfeld

	// Referenzierte Objekte zuerst erzeugen
	//

	// s.o.
	// if (pFile->GetPartOf() != NULL)
	//  Create(pFile->GetPartOf());

	if( pFile->IsUnoComponent() )
	{
		const SiStarRegistryList* pLst = pFile->GetRegistryList();
		for( USHORT i = 0; i < pLst->Count(); ++i )
		{
			SiStarRegistry* pObj = pLst->GetObject(i);
			WriteObject( pObj );
		}
	}

	if (pFile->GetOs2EAFile() != NULL)
		Create(pFile->GetOs2EAFile());

	if (pFile->GetOs2Creator() != NULL)
		Create(pFile->GetOs2Creator());

	// aktuelle Datei erzeugen
	//
	WriteObject(pFile);

	// Fuer die Nachinstallation muessen auch die Dateiteile
	// mit in die DB geschrieben werden. Bei einer WORKSTATION-
	// Installation ist dies nicht erforderlich, weil die Datei
	// der Netzinstallation schon zusammengesetzt ist.
	if (m_eInstallMode != IM_WORKSTATION)
	{
		SiFileList const& aPartList = pFile->GetPartList();
		for (USHORT i=0; i<aPartList.Count(); i++)
			Create(aPartList.GetObject(i));
	}

	const SiShortcutList* pLst = pFile->GetShortcutList();
	for( USHORT i = 0; i < pLst->Count(); ++i )
		WriteObject( pLst->GetObject(i) );

	return TRUE;
}

BOOL SiDatabase::Create(SiProcedure const* pProc)
{
	WriteObject(pProc);
	return TRUE;
}

BOOL SiDatabase::Create(SiCustom const* pCustom)
{
	WriteObject(pCustom);
	return TRUE;
}

BOOL SiDatabase::Create(SiFolder const* pFolder)
{
	if (pFolder->GetOs2OpenIcon() != NULL)
		Create(pFolder->GetOs2OpenIcon());

	if (pFolder->GetOs2CloseIcon() != NULL)
		Create(pFolder->GetOs2OpenIcon());

	WriteObject(pFolder);

	return TRUE;
}

BOOL SiDatabase::Create(SiFolderItem const* pFolderItem)
{
	Create(pFolderItem->GetFolder());

	if (pFolderItem->GetFile() != NULL)
		Create(pFolderItem->GetFile());

	if (pFolderItem->GetOs2Reference() != NULL)
		Create(pFolderItem->GetOs2Reference());

	if (pFolderItem->GetOs2Class() != NULL)
		Create(pFolderItem->GetOs2Class());

	if (pFolderItem->GetOs2Creator() != NULL)
		Create(pFolderItem->GetOs2Creator());

	WriteObject(pFolderItem);
	return TRUE;
}

BOOL SiDatabase::Create(SiRegistryItem const* pItem)
{
	if (pItem == NULL)
		return TRUE;

	Create(pItem->GetParent());
	WriteObject(pItem);
	return TRUE;
}

BOOL SiDatabase::Create(SiModule const* pModule)
{
	if (pModule != NULL)
	{
		// Dateien und abhaengige Objekte
		USHORT i;
		for (i=0; i<pModule->GetFileList().Count();i++)
			Create(pModule->GetFileList().GetObject(i));

		for (i=0; i<pModule->GetDirList().Count();i++)
			Create(pModule->GetDirList().GetObject(i));

		for (i=0; i<pModule->GetProcedureList().Count();i++)
			Create(pModule->GetProcedureList().GetObject(i));

		for (i=0; i<pModule->GetCustomList().Count();i++)
			Create(pModule->GetCustomList().GetObject(i));

		// Module: pre order Traversierung
		WriteObject(pModule);

		for (i=0; i<pModule->GetModuleList()->Count(); i++)
			Create(pModule->GetModuleList()->GetObject(i));

		// modulabhaengige Objekte
		SiFolderItemList const& rFIList = pModule->GetFolderItemList();
		for (i=0; i<rFIList.Count(); i++)
			Create(rFIList.GetObject(i));

		SiProfileList const& rPList = pModule->GetProfileList();
		for (i=0; i<rPList.Count(); i++)
		{
			SiProfile const* pProfile = rPList.GetObject(i);

			if( m_bScpMode ||
				(m_eInstallMode == IM_PATCH			||
				 m_eInstallMode == IM_STANDALONE	||
				 m_eInstallMode == IM_NETWORK		||
				 m_eInstallMode == IM_APPSERVER		||
				(m_eInstallMode == IM_WORKSTATION && !pProfile->InstallOnNetwork())) )
			{
				//! Create(pProfile->GetDirectory());
				WriteObject(pProfile);
			}
		}

		SiProfileItemList const& rPIList = pModule->GetProfileItemList();
		for (i=0; i<rPIList.Count(); i++)
		{
			SiProfileItem const* pProfileItem = rPIList.GetObject(i);
			SiProfile const* pProfile = pProfileItem->GetProfile();
			if( !pProfile )
			{
				fprintf( stdout, "FATAL: ProfileItem without a Profile: %s\n", pProfileItem->GetID().GetBuffer() );
				exit(0);
			}

			if( m_bScpMode ||
				(m_eInstallMode == IM_PATCH			||
				 m_eInstallMode == IM_STANDALONE 	||
				 m_eInstallMode == IM_NETWORK		||
				 m_eInstallMode == IM_APPSERVER		||
				(m_eInstallMode == IM_WORKSTATION && !pProfile->InstallOnNetwork())) )
			{
				WriteObject(pProfile);
				// Create(pProfile->GetDirectory());
				WriteObject(pProfile);
				WriteObject(pProfileItem);
			}
		}

		// StarRegistry-Items
		const SiStarRegistryItemList& rStarRegList = pModule->GetStarRegistryItemList();
		for( i = 0; i < rStarRegList.Count(); ++i )
		{
			SiStarRegistryItem* pItem = rStarRegList.GetObject(i);
			WriteObject( pItem->GetRegistry() );
			WriteObject( pItem );
		}

		SiConfigurationItemList const& rCfgList = pModule->GetConfigurationItemList();
		for( i = 0; i < rCfgList.Count(); i++ )
		{
			SiConfigurationItem* pItem = rCfgList.GetObject(i);
			WriteObject( pItem );
		}

		SiRegistryItemList const& rRIList = pModule->GetRegistryItemList();
		for (i=0; i<rRIList.Count(); i++)
			Create(rRIList.GetObject(i));

		SiRegistryAreaList const& rRAList = pModule->GetRegistryAreaList();
		for (i=0; i<rRAList.Count(); i++)
			WriteObject(rRAList.GetObject(i));

		SiOs2ClassList const& rCList = pModule->GetOs2ClassList();
		for (i=0; i<rCList.Count(); i++)
			Create(rCList.GetObject(i));

		SiOs2TemplateList const& rTList = pModule->GetOs2TemplateList();
		for (i=0; i<rTList.Count(); i++)
			Create(rTList.GetObject(i));
	}

	return TRUE;
}

BOOL SiDatabase::Create
(
	SiCompiledScript const* pCS,
	SiInstallMode			eInstallMode,
	ByteString			 const& aSourcePath,
	BOOL					bInstallFromNet
)
{
	m_eInstallMode = eInstallMode;

	SiInstallation Inst(SiIdentifier("INSTALLATION"), NULL);
	SiInstallation *pInst = &Inst;

	if (pCS->GetInstallation() != NULL)
		pInst = pCS->GetInstallation();

	pInst->SetSourcePath(aSourcePath);
	if( eInstallMode != IM_PATCH )
		pInst->SetInstallMode(eInstallMode);
	pInst->SetInstallFromNet(bInstallFromNet);
	WriteObject(pInst);

	// ScpActions
	if( pCS->DoWriteScpAction() )
	for( USHORT y=0; y < pCS->GetScpActionList().Count(); y++ )
		WriteObject( pCS->GetScpActionList().GetObject(y) );

	// Hilfetext
	for( USHORT x=0; x < pCS->GetHelpTextList().Count(); x++ )
		WriteObject( pCS->GetHelpTextList().GetObject(x) );

	// Installationskino
	for( USHORT i=0; i < pCS->GetSlideList().Count(); i++ )
		WriteObject( pCS->GetSlideList().GetObject(i) );

	// Auch nicht referenzierte Verzeichnisse schreiben
	// (fuer Deinstallation)
	BOOL bReturn = Create(pCS->GetDirectoryList()) &&
				   Create(pCS->GetRootModule());

	// event. von dem Setup comp. HilfeFile eintragen.
	if( pCS->GetCompiledHelp() )
		Create(pCS->GetCompiledHelp());

	// write the ModuleIds list and the ModuleSet list, too
    for ( USHORT j=0; j < pCS->GetModuleIdsList().Count(); j++ )
        WriteObject( pCS->GetModuleIdsList().GetObject(j) );

    for ( USHORT k=0; k < pCS->GetModuleSetList().Count(); k++ )
        WriteObject( pCS->GetModuleSetList().GetObject(k) );

	return bReturn;
}

BOOL SiDatabase::Create( SiCompiledScript const* pCS )
{
	if ( !pCS || !pCS->GetInstallation() )
        return FALSE;

    m_bScpMode 		= pCS->IsSecondLevel();
	m_eInstallMode	= pCS->GetInstallation()->GetInstallMode();
	WriteObject(pCS->GetInstallation());

	// ScpActions
	if( pCS->DoWriteScpAction() )
	for( USHORT y=0; y < pCS->GetScpActionList().Count(); y++ )
		WriteObject( pCS->GetScpActionList().GetObject(y) );

	// Hilfetext
	for (USHORT x=0; x<pCS->GetHelpTextList().Count(); x++)
		WriteObject(pCS->GetHelpTextList().GetObject(x));

	// Installationskino
	for (USHORT i=0; i<pCS->GetSlideList().Count(); i++)
		WriteObject(pCS->GetSlideList().GetObject(i));

	// Auch nicht referenzierte Verzeichnisse schreiben
	// (fuer Deinstallation)
	BOOL bReturn = Create(pCS->GetDirectoryList()) &&
				   Create(pCS->GetRootModule());

	// event. von dem Setup comp. HilfeFile eintragen.
	if( pCS->GetCompiledHelp() )
		Create(pCS->GetCompiledHelp());

	// write the ModuleIds list and the ModuleSet list, too
    for ( USHORT j=0; j < pCS->GetModuleIdsList().Count(); j++ )
        WriteObject( pCS->GetModuleIdsList().GetObject(j) );

    for ( USHORT k=0; k < pCS->GetModuleSetList().Count(); k++ )
        WriteObject( pCS->GetModuleSetList().GetObject(k) );

	return bReturn;
}

