/*************************************************************************
 *
 *  $RCSfile: cfgmgr.cxx,v $
 *
 *  $Revision: 1.22 $
 *
 *  last change: $Author: hr $ $Date: 2003/03/27 11:27:51 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

// include ---------------------------------------------------------------

#include <stdio.h>

#ifndef _COM_SUN_STAR_UCB_XCOMMANDENVIRONMENT_HPP_
#include <com/sun/star/ucb/XCommandEnvironment.hpp>
#endif
#ifndef _COM_SUN_STAR_LANG_WRAPPEDTARGETEXCEPTION_HPP_
#include <com/sun/star/lang/WrappedTargetException.hpp>
#endif
#ifndef _COM_SUN_STAR_UCB_INTERACTIVEAUGMENTEDIOEXCEPTION_HPP_
#include <com/sun/star/ucb/InteractiveAugmentedIOException.hpp>
#endif
#ifndef _COM_SUN_STAR_UCB_IOERRORCODE_HPP_
#include <com/sun/star/ucb/IOErrorCode.hpp>
#endif
#include <com/sun/star/uno/Reference.hxx>
#include <com/sun/star/uno/Any.hxx>

#include <ucbhelper/content.hxx>
#include <tools/urlobj.hxx>
#include <svtools/pathoptions.hxx>
#include <so3/svstor.hxx>
#include <sot/storinfo.hxx>

#ifndef _SVSTDARR_ULONGS_DECL
#define _SVSTDARR_ULONGS
#include <svtools/svstdarr.hxx>
#endif

#pragma hdrstop

#include "cfgmgr.hxx"
#include "cfgitem.hxx"
#include "objsh.hxx"
#include "cfgimpl.hxx"
#include "docfile.hxx"

using namespace ::com::sun::star;

static const char pStorageName[] = "Configurations";

DBG_NAME(SfxConfigManager);

// ----------------------------------------------------------------------------

SotStorage* GetStorage( const String& rName, StreamMode nMode )
{
    try
	{
		::ucb::Content aContent( rName, uno::Reference < ::com::sun::star::ucb::XCommandEnvironment >() );
		uno::Any aAny;
		try
		{
			aAny = aContent.getPropertyValue( ::rtl::OUString(RTL_CONSTASCII_USTRINGPARAM("IsDocument")) );
		}
		catch(com::sun::star::ucb::InteractiveAugmentedIOException&)
		{
		}
		sal_Bool bIsDocument;
		if ( (aAny >>= bIsDocument) && bIsDocument )
			return new SotStorage( TRUE, rName, nMode, STORAGE_TRANSACTED );
		else
			return new SotStorage( aContent, rName, nMode, STORAGE_TRANSACTED );
	}
    catch( com::sun::star::uno::Exception& e)
    {
		// Fatal error, possible corrupted configuration
		com::sun::star::lang::WrappedTargetException wte;
		rtl::OUString ouName = rName;
		wte.Message = rtl::OUString::createFromAscii("GetStorage, name: '") + ouName +
	        rtl::OUString::createFromAscii("'");
		wte.TargetException <<= e;
		throw wte;
	}
}

SfxConfigManager::SfxConfigManager( SotStorage* pStorage )
	: pObjShell( NULL )
	, nErrno( ERR_NO )
	, bModified( FALSE )
{
	DBG_CTOR(SfxConfigManager, 0);

    // configuration stored in any storage
    pItemArr = new SfxConfigItemArr_Impl;

	if ( !pStorage )
	{
		// default config manager
        {
            // first check the upper layers of configuration
            String aPathes = SvtPathOptions().GetUIConfigPath();
            sal_uInt16 i, nIdx=0, nCount=aPathes.GetTokenCount( ';' );
            for ( i=0; i<nCount; ++i )
			{
                String aPathToken = aPathes.GetToken( 0, ';', nIdx );
                INetURLObject aPath( aPathToken );
                String aCfgFileName;
                aPath.insertName( String::CreateFromAscii("soffice.cfg") );
                aCfgFileName = aPath.GetMainURL( INetURLObject::NO_DECODE );
                SotStorageRef xStorage( GetStorage( aCfgFileName, STREAM_STD_READ ) );

                // load all the items, user layer will overwrite them if necessary
                // every item from the upper layer will reference xStorage
                if ( !xStorage->GetError() )
                    LoadConfiguration( *xStorage );
            }
        }

        String aCfgFileName;
        INetURLObject aPath( SvtPathOptions().GetUserConfigPath() );
        aPath.insertName( String::CreateFromAscii("soffice.cfg") );
        aCfgFileName = aPath.GetMainURL( INetURLObject::NO_DECODE );
        pStorage = GetStorage( aCfgFileName, STREAM_STD_READWRITE );
	}

    if ( !pStorage->IsOLEStorage() )
	{
		// take the storage ( otherwise no configitems could be loaded )
		m_xStorage = pStorage;

        // any other configuration storage or document in 6.0 format
        // read directory information
        if ( !LoadConfiguration( *pStorage ) )
            nErrno = ERR_READ;
	}
	else
	{
        // any other configuration storage or document in 5.0 format
        // create temporary storage for the converted data in 6.0 format
        m_xStorage = new SotStorage( TRUE, String(), STREAM_STD_READWRITE, STORAGE_TRANSACTED );

        // convert 5.0 configuration into own 6.0 storage
		SfxConfigManagerImExport_Impl aImporter( pObjShell, pItemArr );
        nErrno = aImporter.Import( pStorage, m_xStorage );
	}
}

SfxConfigManager::SfxConfigManager( SfxObjectShell& rShell )
    : pObjShell( &rShell )
	, nErrno( ERR_NO )
	, bModified( FALSE )
{
	DBG_CTOR(SfxConfigManager, 0);

    // configuration stored in a document or template
    pItemArr = new SfxConfigItemArr_Impl;

    // create temporary storage
    m_xStorage = new SotStorage( TRUE, String(), STREAM_STD_READWRITE, STORAGE_TRANSACTED );

    SotStorageRef xStorage = pObjShell->GetStorage();
    if ( !xStorage->IsOLEStorage() )
	{
		// use the configuration substorage of the document
		SotStorageRef xCfgStorage = xStorage->OpenSotStorage( String::CreateFromAscii(pStorageName), STREAM_STD_READWRITE );

		// copy data, using the original storage is only possible for SvPersist objects
		// the retransfer of the data is organized by the document
		xCfgStorage->CopyTo( m_xStorage );

        // read directory information
        if ( !LoadConfiguration( *m_xStorage ) )
            nErrno = ERR_READ;
	}
	else
	{
        // convert 5.0 configuration into own 6.0 storage
		SfxConfigManagerImExport_Impl aImporter( pObjShell, pItemArr );
        nErrno = aImporter.Import( xStorage, m_xStorage );
	}

    rShell.SetConfigManager( this );
}

SfxConfigManager::~SfxConfigManager()
{
    for( USHORT i = 0; i < pItemArr->Count(); ++i )
    {
        SfxConfigItem_Impl* pItem = (*pItemArr)[i];
        if ( pItem->pCItem )
        {
            DBG_ERROR("SfxConfigItem not removed!");
            pItem->pCItem->ReleaseConfigManager();
        }
		delete pItem;
    }

    delete pItemArr;
}

// ------------------------------------------------------------------------

SotStorage* SfxConfigManager::GetConfigurationStorage( SotStorage* pDocStor )
{
	return pDocStor->OpenSotStorage( String::CreateFromAscii(pStorageName), STREAM_STD_READWRITE );
}

String SfxConfigManager::GetURL()
{
    if ( pObjShell )
        return pObjShell->GetMedium()->GetName();
    else if ( m_xStorage.Is() )
        return m_xStorage->GetName();

    DBG_ERROR("No storage!")
    return String();
}

void SfxConfigManager::SetModified(BOOL bMod)
{
	bModified = bMod;
	if ( bMod && pObjShell)
		pObjShell->SetModified( TRUE );
}

BOOL SfxConfigManager::HasConfiguration( SotStorage& rStorage )
{
    if ( rStorage.IsOLEStorage() )
	{
        return SfxConfigManagerImExport_Impl::HasConfiguration( rStorage );
    }
    else
    {
        return rStorage.IsStorage( String::CreateFromAscii(pStorageName) );
    }
}

// ----------------------------------------------------------------------------

BOOL SfxConfigManager::LoadConfiguration( SotStorage& rStorage )
{
    // for the case that there are several layers of configuration
    USHORT nOldCount = pItemArr->Count();

    SvStorageInfoList aList;
    rStorage.FillInfoList( &aList );
    for( USHORT i = 0; i < aList.Count(); i++ )
    {
        SvStorageInfo& rInfo = aList.GetObject( i );
        if ( rInfo.IsStream() )
        {
            // get StreamName and ItemType
            SfxConfigItem_Impl* pItem = NULL;
            String aStreamName = rInfo.GetName();
            USHORT nType = SfxConfigManagerImExport_Impl::GetType( aStreamName );

            // check items from upper layers if one of them is of the same type
            for ( USHORT nPos=0; nPos<nOldCount; nPos++ )
            {
                if ( (*pItemArr)[nPos]->nType == nType )
                {
                    pItem = (*pItemArr)[nPos];
                    break;
                }
            }

            if ( !pItem )
            {
                // new item
                pItem = new SfxConfigItem_Impl;
                pItemArr->Insert( pItem, pItemArr->Count() );
                pItem->aStreamName = aStreamName;
                pItem->nType = nType;

                // compatibility for 5.0 format
                pItem->bDefault = FALSE;
            }

            // every item will hold a reference to its storage
            pItem->xStorage = &rStorage;
        }
    }

    return TRUE;
}

BOOL SfxConfigManager::StoreConfiguration( SotStorage* pStorage )
{
    // What about the Modified flag ?! ( see also appmisc, config dialog, objcont! )
    BOOL bOwnSaveDone = FALSE;
    BOOL bRet = TRUE;
    if ( m_xStorage.Is() )
    {
        // first update own storage
        bRet = !bModified || StoreConfiguration_Impl( m_xStorage ) && m_xStorage->Commit();
        bOwnSaveDone = TRUE;
        if ( !pStorage && pObjShell )
        {
            // pStorage == NULL means : storage of document should be updated also
            SotStorage* pDocumentStorage = pObjShell->GetStorage();
            if ( !pDocumentStorage->IsOLEStorage() )
            {
                // use the configuration substorage of the document
                SotStorageRef xCfgStorage = pDocumentStorage->OpenSotStorage(
                        String::CreateFromAscii(pStorageName), STREAM_STD_READWRITE, STORAGE_TRANSACTED );
                bRet = m_xStorage->CopyTo( xCfgStorage ) && xCfgStorage->Commit();
            }
            else
            {
                // 5.0 format : store compact configuration stream
                SfxConfigManagerImExport_Impl aExporter( pObjShell, pItemArr );
                nErrno = aExporter.Export( m_xStorage, pDocumentStorage );
                bRet = ( nErrno == ERR_NO );
            }

            if ( bRet )
            {
                // can't commit changes if the documents' storage is under control of the document
                if( pObjShell->GetCreateMode() == SFX_CREATE_MODE_ORGANIZER )
                    bRet = pDocumentStorage->Commit();
                else
                {
					// The configuration will be stored together the document, so the modified flag
					// must remain set!
                    return TRUE;
                }
            }
        }

        if ( bRet && !pStorage || pStorage == (SotStorage*) m_xStorage )
        {
            // only storing into own storage was requested
            bModified = FALSE;
            return TRUE;
        }
    }
	else
		DBG_ASSERT( pStorage, "Can't save configuration!" );

    if ( !bRet || !pStorage )
		return FALSE;

    // store also into storage passed as parameter, but don't commit the changes,  because this will be done by the caller
    if ( !pStorage->IsOLEStorage() )
	{
        // 6.0 format
        if ( bOwnSaveDone )
        {
            // if own storage is updated, just copy it to the destination storage
            bRet = m_xStorage->CopyTo( pStorage );
        }
        else
        {
            bRet = StoreConfiguration_Impl( pStorage );
        }
	}
	else
	{
        // 5.0 format : store compact configuration stream
		SfxConfigManagerImExport_Impl aExporter( pObjShell, pItemArr );
        nErrno = aExporter.Export( m_xStorage, pStorage );
        bRet = ( nErrno == ERR_NO );
	}

	bModified = !bRet;
    return bRet;
}

BOOL SfxConfigManager::StoreConfiguration_Impl( SotStorage* pStorage )
{
    BOOL bRet = TRUE;

    // store directly into the destination storage
    USHORT nCount = pItemArr->Count();
    for ( USHORT nPos=0; bRet && nPos<nCount; nPos++ )
    {
        SfxConfigItem_Impl* pItem = (*pItemArr)[nPos];
        SfxConfigItem* pCItem = pItem->pCItem;
        if ( pCItem && pCItem->IsModified() )
			pItem->bDefault = pCItem->IsDefault();

        if ( pItem->bDefault )
        {
            // elements with default configuration will not be stored
            if ( pStorage->IsContained( pItem->aStreamName ) )
                pStorage->Remove( pItem->aStreamName );
        }
        else if ( pCItem )
        {
            if( pCItem->IsModified() || !pStorage->IsContained( pItem->aStreamName ) )
			{
                // store all items that have not been stored before
                bRet = pCItem->Store( *pStorage );
				if ( pStorage == m_xStorage )
                	pItem->xStorage = m_xStorage;
			}
        }

		if ( pCItem )
			pCItem->SetModified( FALSE );
    }

    return bRet;
}

// ----------------------------------------------------------------------------

void SfxConfigManager::AddConfigItem( SfxConfigItem& rCItem )
{
    for( USHORT i = 0; i < pItemArr->Count(); ++i )
    {
        SfxConfigItem_Impl* pItem = (*pItemArr)[i];
        if ( pItem->nType == rCItem.GetType() )
        {
            if ( pItem->pCItem )
            {
                SfxConfigItems_Impl& rItems = pItem->aItems;
                USHORT nCount = rItems.Count();
#ifdef DBG_UTIL
                for ( USHORT nItem=0; nItem<nCount; nItem++ )
                    if ( rItems[nItem] == &rCItem )
                        DBG_ERROR("Item already inserted!");
#endif
                rItems.Insert( &rCItem, nCount );
            }
            else
                pItem->pCItem = &rCItem;
            return;
        }
    }

    SfxConfigItem_Impl* pItem = new SfxConfigItem_Impl( &rCItem );
    pItemArr->Insert( pItem, pItemArr->Count() );
    pItem->bDefault = rCItem.IsDefault();
    pItem->aStreamName = SfxConfigManagerImExport_Impl::GetStreamName( pItem->nType );
}

void SfxConfigManager::RemoveConfigItem( SfxConfigItem& rCItem )
{
    for( USHORT i = 0; i < pItemArr->Count(); ++i )
    {
        SfxConfigItem_Impl* pItem = (*pItemArr)[i];
        if ( pItem->nType == rCItem.GetType() )
        {
			if ( pItem->pCItem == &rCItem )
			{
            	if ( pItem->aItems.Count() )
            	{
                	pItem->pCItem = pItem->aItems[0];
                	pItem->aItems.Remove( 0 );
            	}
            	else
                	pItem->pCItem = NULL;
			}
			else
			{
                USHORT nCount = pItem->aItems.Count();
                for ( USHORT nItem=0; nItem<nCount; nItem++ )
                	if ( pItem->aItems[nItem] == &rCItem )
					{
						pItem->aItems.Remove(nItem);
						break;
					}
			}

            return;
        }
    }

    DBG_ERROR( "Item not registered!" );
}

BOOL SfxConfigManager::HasConfigItem( USHORT nType )
{
    for( USHORT i = 0; i < pItemArr->Count(); ++i )
    {
        SfxConfigItem_Impl* pItem = (*pItemArr)[i];
        if ( pItem->nType == nType )
            return TRUE;
    }

    return FALSE;
}

BOOL SfxConfigManager::LoadConfigItem( SfxConfigItem& rCItem )
{
    DBG_ASSERT( m_xStorage.Is(), "No storage for configurations!" );

    // search for item
    for( USHORT i = 0; i < pItemArr->Count(); ++i )
    {
        SfxConfigItem_Impl* pItem = (*pItemArr)[i];
        if ( pItem->nType == rCItem.GetType() )
        {
            // update configuration befor using it
            // find last modified item
            SfxConfigItem* pCfgItem = NULL;
            if ( pItem->pCItem && pItem->pCItem->IsModified() )
                pCfgItem = pItem->pCItem;

            USHORT nCount = pItem->aItems.Count();
            for ( USHORT nItem=0; nItem<nCount; nItem++ )
                if ( pItem->aItems[nItem]->IsModified() )
                    pCfgItem = pItem->aItems[nItem];

            if ( pCfgItem )
                pCfgItem->StoreConfig();

            if ( pItem->bDefault )
            {
                // no configuration in storage
                rCItem.UseDefault();
            }
            else if ( pItem->xStorage.Is() )
            {
                // item configuration is in upper layer of configuration
                if ( rCItem.Load( *pItem->xStorage ) != SfxConfigItem::ERR_OK )
					rCItem.UseDefault();
            }
            else
            {
                DBG_ERROR("Item without Storage!");
                rCItem.UseDefault();
                return FALSE;
            }

            return TRUE;
        }
    }

    DBG_ERROR("ItemType not registered!");
    return FALSE;
}

BOOL SfxConfigManager::StoreConfigItem( SfxConfigItem& rCItem )
{
    DBG_ASSERT( m_xStorage.Is(), "No storage for configurations!" );
    if ( !m_xStorage.Is() )
        return FALSE;

	BOOL bRet = TRUE;

    // search for item
    for( USHORT i = 0; i < pItemArr->Count(); ++i )
    {
        SfxConfigItem_Impl* pItem = (*pItemArr)[i];
        if ( pItem->nType == rCItem.GetType() )
        {
            pItem->bDefault = rCItem.IsDefault();
            if ( rCItem.IsDefault() )
            {
                // elements with default configuration will not be stored
                if ( m_xStorage->IsContained( pItem->aStreamName ) )
                    bRet = m_xStorage->Remove( pItem->aStreamName );
            }
            else
            {
                pItem->xStorage = m_xStorage;
                if( rCItem.IsModified() || !m_xStorage->IsContained( pItem->aStreamName ) )
				{
                    // store all items that have not been stored before
                    bRet = rCItem.Store( *m_xStorage );
				}
            }

            if ( rCItem.GetConfigManager() == this  )
				rCItem.SetModified( FALSE );

			break;
        }
    }

    DBG_ASSERT( i<pItemArr->Count(), "Item not registered!" );
    return bRet;
}

BOOL SfxConfigManager::StoreAlwaysConfigItem( SfxConfigItem& rCItem )
{
    DBG_ASSERT( m_xStorage.Is(), "No storage for configurations!" );
    if ( !m_xStorage.Is() )
        return FALSE;

	BOOL bRet = TRUE;

    // search for item
    for( USHORT i = 0; i < pItemArr->Count(); ++i )
    {
        SfxConfigItem_Impl* pItem = (*pItemArr)[i];
        if ( pItem->nType == rCItem.GetType() )
        {
            pItem->bDefault = FALSE;
            pItem->xStorage = m_xStorage;

            // store all items that have not been stored before
            bRet = rCItem.Store( *m_xStorage );

            if ( rCItem.GetConfigManager() == this  )
				rCItem.SetModified( FALSE );

			break;
		}
	}

	return bRet;
}

void SfxConfigManager::CopyConfigItem( SfxConfigManager& rMgr, USHORT nType )
{
	BOOL bIsDefault = TRUE;
	for( USHORT i = 0; i < rMgr.pItemArr->Count(); ++i )
    {
        SfxConfigItem_Impl* pItem = (*rMgr.pItemArr)[i];
        if ( pItem->nType == nType )
        {
            if ( rMgr.m_xStorage->IsContained( pItem->aStreamName ) )
			{
				rMgr.m_xStorage->CopyTo( pItem->aStreamName, m_xStorage, pItem->aStreamName );
				bIsDefault = FALSE;

				// Important: Set modified flag, because SfxConfigManager is modified due to CopyTo!
				// Otherwise StoreConfiguration doesn't call Commit() which is needed to write back to the storage.
				SetModified( TRUE );
			}

			break;
		}
	}

    for( USHORT n = 0; n < pItemArr->Count(); ++n )
    {
        SfxConfigItem_Impl* pItem = (*pItemArr)[n];
		if ( pItem->nType == nType )
		{
			// needs update of internal data!
			pItem->bDefault = bIsDefault;
			pItem->xStorage = m_xStorage;
			if ( bIsDefault && m_xStorage->IsContained( pItem->aStreamName ))
				 m_xStorage->Remove( pItem->aStreamName );
			return;
		}
	}

	if ( !bIsDefault )
	{
		SfxConfigItem_Impl* pItem = new SfxConfigItem_Impl;
    	pItemArr->Insert( pItem, pItemArr->Count() );
    	pItem->aStreamName = SfxConfigManagerImExport_Impl::GetStreamName( nType );
    	pItem->nType = nType;
		pItem->xStorage = m_xStorage;

    	// compatibility for 5.0 format
    	pItem->bDefault = FALSE;
	}
}

void SfxConfigManager::RemovePersistentConfigItem( USHORT nType )
{
    for( USHORT i = 0; i < pItemArr->Count(); ++i )
    {
        SfxConfigItem_Impl* pItem = (*pItemArr)[i];
        if ( pItem->nType == nType )
		{
            if ( m_xStorage->IsContained( pItem->aStreamName ) )
                m_xStorage->Remove( pItem->aStreamName );
			break;
		}
	}

    for( USHORT n = 0; n < pItemArr->Count(); ++n )
    {
        SfxConfigItem_Impl* pItem = (*pItemArr)[n];
		if ( pItem->nType == nType )
		{
            if ( pItem->pCItem )
            {
                pItem->pCItem->ReConnect( NULL );
                SfxConfigItems_Impl& rItems = pItem->aItems;
                USHORT nCount = rItems.Count();
                for ( USHORT nItem=0; nItem<nCount; nItem++ )
            	    rItems[nItem]->ReConnect( NULL );
            }

            delete pItemArr->GetObject( n );
            pItemArr->Remove( n );
			break;
		}
	}
}

void SfxConfigManager::ReInitialize( USHORT nType )
{
    for( USHORT i = 0; i < pItemArr->Count(); ++i )
    {
        SfxConfigItem_Impl* pItem = (*pItemArr)[i];
        if ( pItem->nType == nType )
        {
            if ( pItem->pCItem )
            {
                pItem->pCItem->ReInitialize();
                SfxConfigItems_Impl& rItems = pItem->aItems;
                USHORT nCount = rItems.Count();
                for ( USHORT nItem=0; nItem<nCount; nItem++ )
                        rItems[nItem]->ReInitialize();
            }

            break;
        }
    }
}

void SfxConfigManager::ReInitialize( const String& rStreamName )
{
	if ( !pObjShell && m_xStorage.Is() )
	{
		//StoreConfiguration(m_xStorage);
/*
		String aName = m_xStorage->GetName();
		SotStorageRef xStorage = new SotStorage( aName, STREAM_STD_READWRITE );
	    for( USHORT i = 0; i < pItemArr->Count(); ++i )
	    {
        	SfxConfigItem_Impl* pItem = (*pItemArr)[i];
			if ( pItem->xStorage == m_xStorage )
				pItem->xStorage = xStorage;
		}

		m_xStorage = xStorage;
*/
		m_xStorage->Revert();

		USHORT nType = SfxConfigManagerImExport_Impl::GetType( rStreamName );
		if ( nType )
			ReInitialize( nType );
	}
}

void SfxConfigManager::ReInitialize( SfxConfigItem* pCItem )
{
    for( USHORT i = 0; i < pItemArr->Count(); ++i )
    {
        SfxConfigItem_Impl* pItem = (*pItemArr)[i];
        if ( pItem->nType == pCItem->GetType() )
        {
            if ( pItem->pCItem )
            {
                if ( pItem->pCItem != pCItem )
                    pItem->pCItem->ReInitialize();
                SfxConfigItems_Impl& rItems = pItem->aItems;
                USHORT nCount = rItems.Count();
                for ( USHORT nItem=0; nItem<nCount; nItem++ )
                    if ( rItems[nItem] != pCItem )
                        rItems[nItem]->ReInitialize();
            }

            break;
        }
    }
}

void SfxConfigManager::ResetConfigItem( USHORT nType )
{
    for( USHORT i = 0; i < pItemArr->Count(); ++i )
    {
        SfxConfigItem_Impl* pItem = (*pItemArr)[i];
        if ( pItem->nType == nType )
        {
            // elements with default configuration will not be stored
            if ( m_xStorage->IsContained( pItem->aStreamName ) )
            {
                m_xStorage->Remove( pItem->aStreamName );

				// Important: Set modified flag, because SfxConfigManager is modified due to m_xStorage->Remove!
				// Otherwise StoreConfiguration doesn't call Commit() which is needed to write back to the storage.
				SetModified( TRUE );
			}
            pItem->bDefault = TRUE;
            break;
        }
    }
}

void SfxConfigManager::ReConnect( USHORT nType, SfxConfigManager* pOther )
{
    for( USHORT i = 0; i < pItemArr->Count(); ++i )
    {
        SfxConfigItem_Impl* pItem = (*pItemArr)[i];
        if ( pItem->nType == nType )
        {
            if ( pItem->pCItem )
            {
                pItem->pCItem->ReConnect( pOther );
                SfxConfigItems_Impl& rItems = pItem->aItems;
                USHORT nCount = rItems.Count();
                for ( USHORT nItem=0; nItem<nCount; nItem++ )
                        rItems[nItem]->ReConnect( pOther );
            }

            break;
        }
    }
}

