/*************************************************************************
 *
 *  $RCSfile: gnome-vfs-filetype-registration.c,v $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <dlfcn.h>
#if !(defined NETBSD || defined FREEBSD || defined MACOSX)
#include <alloca.h>
#endif
#include <string.h>

/* 
 * This file produces an executable that can manage file type 
 * registrations for GNOME. This becomes necessary due to a 
 * bug in Nautilus (GNOME bug id ... ) that ignores additional
 * files in the users .gnome/share/mime-info directory
 *
 * To reduce build dependencies, most of the needed gnome
 * functionality is retrieved via dlsym, so a lot of definitions
 * have to be copied from the original gnome files.
 */

/* stolen from gnome-vfs-mime-info.h */
typedef enum {
        GNOME_VFS_OK,
        GNOME_VFS_ERROR_NOT_FOUND,
        GNOME_VFS_ERROR_GENERIC,
        GNOME_VFS_ERROR_INTERNAL,
        GNOME_VFS_ERROR_BAD_PARAMETERS,
        GNOME_VFS_ERROR_NOT_SUPPORTED,
        GNOME_VFS_ERROR_IO,
        GNOME_VFS_ERROR_CORRUPTED_DATA,
        GNOME_VFS_ERROR_WRONG_FORMAT,
        GNOME_VFS_ERROR_BAD_FILE,
        GNOME_VFS_ERROR_TOO_BIG,
        GNOME_VFS_ERROR_NO_SPACE,
        GNOME_VFS_ERROR_READ_ONLY,
        GNOME_VFS_ERROR_INVALID_URI,
        GNOME_VFS_ERROR_NOT_OPEN,
        GNOME_VFS_ERROR_INVALID_OPEN_MODE,
        GNOME_VFS_ERROR_ACCESS_DENIED,
        GNOME_VFS_ERROR_TOO_MANY_OPEN_FILES,
        GNOME_VFS_ERROR_EOF,
        GNOME_VFS_ERROR_NOT_A_DIRECTORY,
        GNOME_VFS_ERROR_IN_PROGRESS,
        GNOME_VFS_ERROR_INTERRUPTED,
        GNOME_VFS_ERROR_FILE_EXISTS,
        GNOME_VFS_ERROR_LOOP,
        GNOME_VFS_ERROR_NOT_PERMITTED,
        GNOME_VFS_ERROR_IS_DIRECTORY,
        GNOME_VFS_ERROR_NO_MEMORY,
        GNOME_VFS_ERROR_HOST_NOT_FOUND,
        GNOME_VFS_ERROR_INVALID_HOST_NAME,
        GNOME_VFS_ERROR_HOST_HAS_NO_ADDRESS,
        GNOME_VFS_ERROR_LOGIN_FAILED,
        GNOME_VFS_ERROR_CANCELLED,
        GNOME_VFS_ERROR_DIRECTORY_BUSY,
        GNOME_VFS_ERROR_DIRECTORY_NOT_EMPTY,
        GNOME_VFS_ERROR_TOO_MANY_LINKS,
        GNOME_VFS_ERROR_READ_ONLY_FILE_SYSTEM,
        GNOME_VFS_ERROR_NOT_SAME_FILE_SYSTEM,
        GNOME_VFS_ERROR_NAME_TOO_LONG,
        GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE,
        GNOME_VFS_ERROR_SERVICE_OBSOLETE,
        GNOME_VFS_ERROR_PROTOCOL_ERROR,
        GNOME_VFS_NUM_ERRORS
} GnomeVFSResult;

/*
 * the bunch of gnome vfs functions needed
 */
 
typedef struct  
{
    void  (* application_registry_remove_application) ( const char *app_id );
    void  (* application_registry_add_mime_type)      ( const char *app_id, const char *mime_type );
    void  (* application_registry_set_value)          ( const char *app_id, const char *key, const char *value);
    void  (* application_registry_set_bool_value)     ( const char *app_id, const char *key, int value);
    void  (* application_registry_remove_mime_type)   ( const char *app_id, const char *mime_type );

    void  (* mime_freeze)                      ( void );
    void  (* mime_thaw)                        ( void );
    void  (* mime_registered_mime_type_delete) ( const char * mime_type );
    
    GnomeVFSResult (* mime_set_default_application)           ( const char *mime_type, const char *application_id );
    GnomeVFSResult (* mime_set_registered_type_key)           ( const char *mime_type, const char *key, const char *data );
    GnomeVFSResult (* mime_set_value)                         ( const char *mime_type, const char *key, const char *value );
} gnome_vfs_functions;

/*
 * the mime info as data structur
 */

typedef struct 
{
    const char * mime_type;
    const char * description;
    const char * extensions;
    const char * icon_filename;
} mime_info;
    
/*
 * make the code read as normal gnome app
 */
 
#define gnome_vfs_application_registry_remove_application gnome_vfs->application_registry_remove_application
#define gnome_vfs_application_registry_add_mime_type gnome_vfs->application_registry_add_mime_type
#define gnome_vfs_application_registry_remove_mime_type gnome_vfs->application_registry_remove_mime_type
#define gnome_vfs_application_registry_set_value gnome_vfs->application_registry_set_value
#define gnome_vfs_application_registry_set_bool_value gnome_vfs->application_registry_set_bool_value
 
#define gnome_vfs_mime_thaw gnome_vfs->mime_thaw
#define gnome_vfs_mime_freeze gnome_vfs->mime_freeze
#define gnome_vfs_mime_registered_mime_type_delete gnome_vfs->mime_registered_mime_type_delete
#define gnome_vfs_mime_set_registered_type_key gnome_vfs->mime_set_registered_type_key
#define gnome_vfs_mime_set_value gnome_vfs->mime_set_value
#define gnome_vfs_mime_set_default_application gnome_vfs->mime_set_default_application

/*
 * global variables
 */
 
static const char * application_name = NULL;

static void * gnome_vfs_handle = NULL;

static gnome_vfs_functions * gnome_vfs = NULL;

static const mime_info writer_types[] = 
{   { "application/vnd.sun.xml.writer", "%s Text Document", "sxw", "002_text_document.png" },
    { "application/vnd.sun.xml.writer.template", "%s Text Document Template", "stw", "003_text_template.png" },
    { "application/vnd.sun.xml.writer.global", "%s Master Document", "sxg", "011_master_document.png" },
    { "application/vnd.sun.xml.writer.math", "%s Formula", "sxm", "015_math_document.png" },
    { "application/vnd.stardivision.writer", "StarOffice 5 Text Document", "sdw", "002_text_document.png" },
    { "application/vnd.stardivision.writer-global", "StarOffice 5 Master Document", "sgl", "011_master_document.png" },
    { "application/vnd.stardivision.math", "StarOffice 5 Formula", "sdf", "015_math_document.png" }
};
    
static const mime_info calc_types[] = 
{   { "application/vnd.sun.xml.calc", "%s Spreadsheet", "sxc", "004_spreadsheet_document.png" },
    { "application/vnd.sun.xml.calc.template", "%s Spreadsheet Template", "stc", "005_spreadsheet_template.png" },
    { "application/vnd.stardivsion.calc", "StarOffice 5 Spreadsheet", "sdc", "004_spreadsheet_document.png" },
    { "application/vnd.stardivsion.chart", "StarOffice 5 Chart", "sdc", "004_spreadsheet_document.png" }
};

static const mime_info impress_types[] = 
{   { "application/vnd.sun.xml.impress", "%s Presentation", "sxi", "008_presentation_document.png" },
    { "application/vnd.sun.xml.impress.template", "%s Presentation Template", "sti", "009_presentation_template.png" },
    { "application/vnd.stardivsion.impress", "StarOffice 5 Presentation", "sdd", "008_presentation_document.png" }
};

static const mime_info draw_types[] = 
{   { "application/vnd.sun.xml.draw", "%s Drawing", "sxd", "006_drawing_document.png" },
    { "application/vnd.sun.xml.draw.template", "%s Drawing Template", "std", "007_drawing_template.png" },
    { "application/vnd.stardivsion.draw", "StarOffice 5 Drawing", "sdd", "006_drawing_document.png" }
};

/*
 * Prototypes
 */
 
int gnome_vfs_init();
void gnome_vfs_shutdown();

GnomeVFSResult register_mime_type( 
    const mime_info * info,  
    const char * application_id, 
    const char * product_name, 
    const char * install_path 
);

GnomeVFSResult deregister_mime_type(
    const char * mime_type, 
    const char * application_id 
);

GnomeVFSResult register_application(
    const char * application_id, 
    const char * product_name, 
    const char * product_version,
    const char * command, 
    const char * install_path 
);

GnomeVFSResult deregister_application( const char * application_id );


/*
 * load the gnome-vfs library and initialize the function pointers
 */
 
int gnome_vfs_init()
{
    void (* gnome_vfs_mime_get_value) ( const char *, const char * ) = NULL;
    
    /* load the gnomevfs-2 library */
    gnome_vfs_handle = dlopen("libgnomevfs-2.so.0", RTLD_LAZY);
    
    /* retry with gnomevfs if not found */
    if( NULL == gnome_vfs_handle )
    {
        gnome_vfs_handle = dlopen("libgnomevfs.so.0", RTLD_LAZY);
    }

    if( NULL == gnome_vfs_handle )
    {
/*        fprintf( stderr, "%s.\n", dlerror() );
*/
        return 0;
    }
    
    gnome_vfs = ( gnome_vfs_functions * ) calloc( 1, sizeof( gnome_vfs_functions ) );     

    /* this call causes gnome-vfs to become initialized */
    gnome_vfs_mime_get_value = (void(*)(const char*,const char*)) dlsym( gnome_vfs_handle, "gnome_vfs_mime_get_value" );
    
    if( NULL == gnome_vfs_mime_get_value )
    {
/*        fprintf( stderr, "%s.\n", dlerror() );
*/
        return 0;
    }
    
    gnome_vfs_mime_get_value( "text/plain", "description" );

    /* extract needed symbols from the library */
    gnome_vfs->mime_set_registered_type_key = (GnomeVFSResult(*)(const char*,const char*,const char*)) dlsym( gnome_vfs_handle, "gnome_vfs_mime_set_registered_type_key" );    

    if( NULL == gnome_vfs->mime_set_registered_type_key )
    {
/*        fprintf( stderr, "%s.\n", dlerror() );
*/        return 0;
    }

    gnome_vfs->mime_freeze = (void(*)()) dlsym( gnome_vfs_handle, "gnome_vfs_mime_freeze" );    

    if( NULL == gnome_vfs->mime_freeze )
    {
/*        fprintf( stderr, "%s.\n", dlerror() );
*/        return 0;
    }

    gnome_vfs->mime_thaw = (void(*)()) dlsym( gnome_vfs_handle, "gnome_vfs_mime_thaw" );    

    if( NULL == gnome_vfs->mime_thaw )
    {
/*        fprintf( stderr, "%s.\n", dlerror() );
*/        return 0;
    }
    
    gnome_vfs->mime_set_value = (GnomeVFSResult(*)(const char*,const char*,const char*)) dlsym( gnome_vfs_handle, "gnome_vfs_mime_set_value" );    

    if( NULL == gnome_vfs->mime_set_value )
    {
/*        fprintf( stderr, "%s.\n", dlerror() );
*/        return 0;
    }

    gnome_vfs->mime_registered_mime_type_delete = (void(*)(const char*)) dlsym( gnome_vfs_handle, "gnome_vfs_mime_registered_mime_type_delete" );    

    if( NULL == gnome_vfs->mime_registered_mime_type_delete )
    {
/*        fprintf( stderr, "%s.\n", dlerror() );
*/        return 0;
    }

    gnome_vfs->mime_set_default_application = (GnomeVFSResult(*)(const char*,const char*)) dlsym( gnome_vfs_handle, "gnome_vfs_mime_set_default_application" );
     
    if( NULL == gnome_vfs->mime_set_default_application )
    {
/*        fprintf( stderr, "%s.\n", dlerror() );
*/        return 0;
    }
    
    gnome_vfs->application_registry_remove_application  = (void(*)(const char*)) dlsym( gnome_vfs_handle, "gnome_vfs_application_registry_remove_application" );
    
    if( NULL == gnome_vfs->application_registry_remove_application )
    {
/*        fprintf( stderr, "%s.\n", dlerror() );
*/        return 0;
    }
    
    gnome_vfs->application_registry_add_mime_type = (void(*)(const char*,const char*)) dlsym( gnome_vfs_handle, "gnome_vfs_application_registry_add_mime_type" );
    
    if( NULL == gnome_vfs->application_registry_add_mime_type )
    {
/*        fprintf( stderr, "%s.\n", dlerror() );
*/        return 0;
    }
    

    gnome_vfs->application_registry_set_value = (void(*)(const char*,const char*,const char*)) dlsym( gnome_vfs_handle, "gnome_vfs_application_registry_set_value" );
    
    if( NULL == gnome_vfs->application_registry_set_value )
    {
/*        fprintf( stderr, "%s.\n", dlerror() );
*/        return 0;
    }
    

    gnome_vfs->application_registry_set_bool_value = (void(*)(const char*,const char*,int)) dlsym( gnome_vfs_handle, "gnome_vfs_application_registry_set_bool_value" );
    
    if( NULL == gnome_vfs->application_registry_set_bool_value )
    {
/*        fprintf( stderr, "%s.\n", dlerror() );
*/        return 0;
    }
    

    gnome_vfs->application_registry_remove_mime_type = (void(*)(const char*,const char*)) dlsym( gnome_vfs_handle, "gnome_vfs_application_registry_remove_mime_type" );
    
    if( NULL == gnome_vfs->application_registry_remove_mime_type )
    {
/*        fprintf( stderr, "%s.\n", dlerror() );
*/        return 0;
    }
}

/*
 * release the gnome-vfs library and free the function pointer struct
 */
 
void gnome_vfs_shutdown()
{
    /* release the gnomevfs library */
    if( gnome_vfs_handle )
    {
        void (* gnome_vfs_mime_info_shutdown) ( void ) = NULL;
        void (* gnome_vfs_application_registry_sync) ( void ) = NULL;

        /* extract and call shutdown functions */
        gnome_vfs_mime_info_shutdown = (void(*)()) dlsym( gnome_vfs_handle, "gnome_vfs_mime_info_shutdown" );

        if( NULL == gnome_vfs_mime_info_shutdown )
        {
/*            fprintf( stderr, "%s.\n", dlerror() );
*/            return;
        }

        gnome_vfs_application_registry_sync = (void(*)()) dlsym( gnome_vfs_handle, "gnome_vfs_application_registry_sync" );
    
        if( NULL == gnome_vfs_application_registry_sync )
        {
/*            fprintf( stderr, "%s.\n", dlerror() );
*/            return;
        }

        gnome_vfs_mime_info_shutdown();
        gnome_vfs_application_registry_sync();
        dlclose( gnome_vfs_handle );
    }
    
    if( gnome_vfs )
        free( gnome_vfs );
}


/*
 *  add a specific mime type with all its properties to the gnome mime info registry
 */
 
GnomeVFSResult register_mime_type( const mime_info * info,  const char * application_id, const char * product_name,  
                                   const char * install_path )
{
    char * icon_filename = (char *) alloca( strlen( install_path ) + strlen( info->icon_filename ) + 29 );
    char * description   = (char *) alloca( strlen( product_name ) + strlen( info->description ) + 2 );

    sprintf( icon_filename, "%s/share/icons/document-icons/%s", install_path, info->icon_filename );
    sprintf( description, info->description, product_name );
        
    /* stores data in user.mime file */
    gnome_vfs_mime_set_registered_type_key( info->mime_type, "description", description );
    gnome_vfs_mime_set_registered_type_key( info->mime_type, "ext", info->extensions );
    gnome_vfs_mime_set_registered_type_key( info->mime_type, "deleted", "" );
    
    /* stores data in the user.keys file */
    gnome_vfs_mime_set_value( info->mime_type, "description", description );
    gnome_vfs_mime_set_value( info->mime_type, "default_action_type", "application" );
    gnome_vfs_mime_set_value( info->mime_type, "icon_filename", icon_filename );
    gnome_vfs_mime_set_value( info->mime_type, "short_list_application_user_removals", "" );
    gnome_vfs_mime_set_value( info->mime_type, "short_list_application_user_additions", application_id );

    /* register mime type for application */
    gnome_vfs_application_registry_add_mime_type( application_id, info->mime_type );
    gnome_vfs_mime_set_default_application( info->mime_type, application_id ); 
    
    return GNOME_VFS_OK;
}

/*
 *  remove a specific mime type from the gnome mime info registry
 */

GnomeVFSResult deregister_mime_type( const char * mime_type, const char * application_id )
{
    gnome_vfs_application_registry_remove_mime_type( application_id, mime_type );
    gnome_vfs_mime_registered_mime_type_delete( mime_type );

    return GNOME_VFS_OK;
}

/*
 *  add the installed application to the gnome application info registry
 */

GnomeVFSResult register_application( const char * application_id, const char * product_name, const char * product_version,
                                     const char * command, const char * install_path )
{
    char * command_full = (char *) alloca( strlen( install_path ) + strlen( command ) + 10 );
    char * product = (char *) alloca( strlen( product_name ) + strlen( product_version ) + 2 );

    sprintf( command_full, "%s/program/%s", install_path, command );
    sprintf( product, "%s %s", product_name, product_version );

    /* entry will be created automatically */
    gnome_vfs_application_registry_set_value( application_id, "name", product );
    gnome_vfs_application_registry_set_value( application_id, "command", command_full );
    gnome_vfs_application_registry_set_bool_value( application_id, "can_open_multiple_files", 1 );
    gnome_vfs_application_registry_set_bool_value( application_id, "requires_terminal", 0 );

    return GNOME_VFS_OK;
}

/*
 *  remove the installed application from the gnome application info registry
 */

GnomeVFSResult deregister_application( const char * application_id )
{
    /* entry will be created automatically */
    gnome_vfs_application_registry_remove_application( application_id );

    return GNOME_VFS_OK;
}

int main(int argc, const char *argv[] )
{
    char * application_id = NULL;
    int application_major, application_minor; 
    int current = 4;
    
    if( argc < 5 )
    {
        fprintf( stderr, "Usage: gnomeint <product_name> <product_version> <install_path> module [module2] ...\n" );
        fprintf( stderr, "Usage: gnomeint -remove <product_name> <product_version> module [module2] ...\n" );
        return -1;
    }

    /*
     * initialize function pointers
     */
     
    if( 0 == gnome_vfs_init() )
    {
/*        fprintf( stderr, "%s: initialization failed.\n", argv[0] );
*/        return -1;
    }
    
    /*
     * register file mime type
     */
     
    /* freeze the lists to enable multiple updates */
    gnome_vfs_mime_freeze();
    
    if( 0 == strcmp( "-remove", argv[1] ) )
    {
        /*
         * construct application id without '.' and blanks
         */

        application_id = (char *) alloca( strlen( argv[2] ) + 4 );
        sscanf( argv[3], "%d.%d", &application_major, &application_minor );
        sprintf( application_id, "%s%d%d", argv[2], application_major, application_minor ); 

        /*
         * deregister the mime types first and then the application
         */

        while( current < argc )
        {
            if( 0 == strcmp( "writer", argv[current] ) )
            {
                int n = 0, nmax = sizeof(writer_types) / sizeof(mime_info);

                while( n < nmax )
                {
                    deregister_mime_type( writer_types[n++].mime_type, application_id );
                }
            }

            else if( 0 == strcmp( "calc", argv[current] ) )
            {
                int n = 0, nmax = sizeof(calc_types) / sizeof(mime_info);

                while( n < nmax )
                {
                    deregister_mime_type( calc_types[n++].mime_type, application_id );
                }
            }

            else if( 0 == strcmp( "draw", argv[current] ) )
            {
                int n = 0, nmax = sizeof(draw_types) / sizeof(mime_info);

                while( n < nmax )
                {
                    deregister_mime_type( draw_types[n++].mime_type, application_id );
                }
            }

            else if( 0 == strcmp( "impress", argv[current] ) )
            {
                int n = 0, nmax = sizeof(impress_types) / sizeof(mime_info);

                while( n < nmax )
                {
                    deregister_mime_type( impress_types[n++].mime_type, application_id );
                }
            }

            current++;
        }

        deregister_application( application_id );
    }
    else
    {
        /*
         * construct application id without '.' and blanks
         */

        application_id = (char *) alloca( strlen( argv[1] ) + 4 );
        sscanf( argv[2], "%d.%d", &application_major, &application_minor );
        sprintf( application_id, "%s%d%d", argv[1], application_major, application_minor ); 

        /*
         * register the application first and then the mime types
         */
         
        register_application( application_id, argv[1], argv[2], "soffice", argv[3] );

        while( current < argc )
        {
            if( 0 == strcmp( "writer", argv[current] ) )
            {
                int n = 0, nmax = sizeof(writer_types) / sizeof(mime_info);

                while( n < nmax )
                {
                    register_mime_type( writer_types + n++, application_id, argv[1], argv[3] );
                }

                /* register for MS Word and Richtext format */
                gnome_vfs_application_registry_add_mime_type( application_id, "application/msword" );
                gnome_vfs_application_registry_add_mime_type( application_id, "text/richtext" );
            }

            else if( 0 == strcmp( "calc", argv[current] ) )
            {
                int n = 0, nmax = sizeof(calc_types) / sizeof(mime_info);

                while( n < nmax )
                {
                    register_mime_type( calc_types + n++, application_id, argv[1], argv[3] );
                }

                /* register for MS Excel format */
                gnome_vfs_application_registry_add_mime_type( application_id, "application/ms-excel" );
            }

            else if( 0 == strcmp( "draw", argv[current] ) )
            {
                int n = 0, nmax = sizeof(draw_types) / sizeof(mime_info);

                while( n < nmax )
                {
                    register_mime_type( draw_types + n++, application_id, argv[1], argv[3] );
                }
            }

            else if( 0 == strcmp( "impress", argv[current] ) )
            {
                int n = 0, nmax = sizeof(impress_types) / sizeof(mime_info);

                while( n < nmax )
                {
                    register_mime_type( impress_types + n++, application_id, argv[1], argv[3] );
                }

                /* register for MS Powerpoint format */
                gnome_vfs_application_registry_add_mime_type( application_id, "application/mspowerpoint" );
            }

            current++;
        }
    }
        
    /* force the user.mime and user.keys to be written  */
    gnome_vfs_mime_thaw();
    
    /*
     * cleanup
     */
     
    gnome_vfs_shutdown();
}





