/*************************************************************************
 *
 *  $RCSfile: pkgchk_app.cxx,v $
 *
 *  $Revision: 1.10 $
 *
 *  last change: $Author: vg $ $Date: 2003/05/26 08:37:42 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2002 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <stdio.h>
#include <vector>

#include "pkgchk_misc.h"
#include "pkgchk.hxx"
#include "osl/process.h"
#include "osl/thread.h"

#include "com/sun/star/uno/RuntimeException.hpp"


using namespace ::rtl;
using namespace ::osl;
using namespace ::com::sun::star::uno;

namespace pkgchk
{

//==============================================================================
static bool read_option(
    OUString const & opt, sal_Unicode copt, sal_uInt32 * pIndex )
{
    if (osl_getCommandArgCount() <= *pIndex)
        return false;
    
    OUString arg;
    osl_getCommandArg( *pIndex, &arg.pData );
    sal_Int32 len = arg.getLength();
    
    if (len < 2 || '-' != arg[ 0 ])
        return false;
    
    if (2 == len && copt == arg[ 1 ])
    {
        ++(*pIndex);
#if defined DIAG
        printf( "[diag] identified option \'%c\'\n", copt );
#endif
        return true;
    }
    if ('-' == arg[ 1 ] &&
        0 == rtl_ustr_compare( arg.pData->buffer + 2, opt.pData->buffer ))
    {
        ++(*pIndex);
#if defined DIAG
        OString cstr_opt(
            OUStringToOString( opt, osl_getThreadTextEncoding() ) );
        printf( "[diag] identified option \'%s\'\n", cstr_opt.getStr() );
#endif
        return true;
    }
    return false;
}

//==============================================================================
static bool read_argument(
    OUString * pValue,
    OUString const & opt, sal_Unicode copt,
    sal_uInt32 * pIndex )
{
    if (read_option( opt, copt, pIndex ))
    {
        if (*pIndex < osl_getCommandArgCount())
        {
            osl_getCommandArg( *pIndex, &pValue->pData );
            ++(*pIndex);
#if defined DIAG
            OString cstr_val(
                OUStringToOString( *pValue, osl_getThreadTextEncoding() ) );
            printf( "[diag] argument value: %s\n", cstr_val.getStr() );
#endif
            return true;
        }
        --(*pIndex);
    }
    return false;
}

//------------------------------------------------------------------------------
static char const s_usingText [] =
"\n"
"using: pkgchk <switches> [package1 package2 ...]\n\n"
"switches:\n"
" -s, --shared            check/install shared components; default: user\n"
" -r, --renewal           re-install all packages\n"
" -f, --force_overwrite   force overwrite existing packages when copying from\n"
"                         command line\n"
" -v, --verbose           verbose output to stdout\n"
" -l, --log <file>        custom log file; default: <cache-dir>/log.txt\n"
" --strict_error_handling break process immediately if an error occurs\n"
" --supersede_basic_libs  overwrite basic library entries of same name\n"
" --check_unorc_only      check/patch only the " SAL_CONFIGFILE("uno") "\n"
" --no_unorc_patch        don't touch " SAL_CONFIGFILE("uno") "\n"
" -h, --help              this message\n"
"\n"
"pkgchk looks for UNO packages in the UNO_USER_PACKAGES or"
" UNO_SHARED_PACKAGES\n"
"directory (see option -s) which are specified in the "
SAL_CONFIGFILE("uno") ".\n"
"It registers and revokes packages from a cache directory specified via the\n"
"corresponding UNO_USER_PACKAGES_CACHE or UNO_SHARED_PACKAGES_CACHE "
"(option -s)\n"
"variables.\n"
"If any packages are given via command line, then these are copied into the\n"
"packages directory in advance.\n"
"For details concerning package structure, please read the developer's guide:\n"
"http://api.openoffice.org/DevelopersGuide/\n"
"  Components/Components.htm#1+7+1+UNO+Package+Installation\n"
"\n";

//==============================================================================
class AppOptionsContainer : public OptionsContainer
{
    bool m_verbose;
    
public:
    inline AppOptionsContainer( bool verbose )
        : m_verbose( verbose )
        {}
    
    virtual bool get_option(
        OUString const & name, sal_Unicode shortcut ) const;
    virtual bool get_argument(
        Any * value, OUString const & name, sal_Unicode shortcut ) const;
};

//______________________________________________________________________________
bool AppOptionsContainer::get_option(
    OUString const & name, sal_Unicode shortcut ) const
{
#if defined DIAG /* diag: strict error handling */
    if (name.equalsAsciiL(
            RTL_CONSTASCII_STRINGPARAM("strict_error_handling") ))
    {
        return true;
    }
#endif
    if (name.equalsAsciiL(
            RTL_CONSTASCII_STRINGPARAM("verbose") ))
    {
        return m_verbose;
    }
    // application: be verbose when ignoring errors
    if (name.equalsAsciiL(
            RTL_CONSTASCII_STRINGPARAM("verbose_on_ignored_errors") ))
    {
        return true;
    }
    
    sal_uInt32 nCount = osl_getCommandArgCount();
    for ( sal_uInt32 nPos = 0; nPos < nCount; )
    {
        if (read_option( name, shortcut, &nPos ))
            return true;
        ++nPos;
    }
    return false;
}

//______________________________________________________________________________
bool AppOptionsContainer::get_argument(
    Any * value, OUString const & name, sal_Unicode shortcut ) const
{
    OUString val;
    sal_uInt32 nCount = osl_getCommandArgCount();
    for ( sal_uInt32 nPos = 0; nPos < nCount; )
    {
        if (read_argument( &val, name, shortcut, &nPos ))
        {
            value->setValue( &val, ::getCppuType( &val ) );
            return true;
        }
        ++nPos;
    }
    return false;
}

}

using namespace ::pkgchk;

//##############################################################################
extern "C" int SAL_CALL main( int argc, char const * argv [] )
{
	int ret = 0;
    
	try
	{
        bool verbose = false;
		bool renewal = false;
        bool check_unorc_only = false;
        bool patch_unorc = true;
        ::std::vector< OUString > additional_packages;
        
        OUString str_help = OUSTR("help");
        OUString str_verbose = OUSTR("verbose");
        OUString str_renewal = OUSTR("renewal");
        OUString str_check_unorc_only = OUSTR("check_unorc_only");
        OUString str_no_unorc_patch = OUSTR("no_unorc_patch");
        
        sal_uInt32 nCount = osl_getCommandArgCount();
		for ( sal_uInt32 nPos = 0; nPos < nCount; )
		{
            // options
			if (read_option( str_help, 'h', &nPos ))
			{
                printf( s_usingText );
                return 0;
			}
			else if (read_option( str_verbose, 'v', &nPos ))
			{
                verbose = true;
			}
			else if (read_option( str_renewal, 'r', &nPos ))
			{
                renewal = true;
			}
			else if (read_option( str_check_unorc_only, '\0', &nPos ))
			{
                check_unorc_only = true;
			}
			else if (read_option( str_no_unorc_patch, '\0', &nPos ))
			{
                patch_unorc = false;
			}
			else
            {
                OUString system_path;
                OSL_VERIFY(
                    osl_Process_E_None ==
                      osl_getCommandArg( nPos, &system_path.pData ) );
                ++nPos;
                if (system_path.getLength() && '-' != system_path[ 0 ])
                {
                    // no option
                    OUString file_url;
                    oslFileError rc = osl_getFileURLFromSystemPath(
                        system_path.pData, &file_url.pData );
                    if (osl_File_E_None == rc)
                    {
                        // package file
                        additional_packages.push_back(
                            path_make_absolute( file_url ) );
                    }
                    else
                    {
                        throw RuntimeException(
                            OUSTR("cannot get file url from system path: ") +
                              system_path,
                            Reference< XInterface >() );
                    }
                }
            }
		}
        
        OUString unorc_path(
            path_concat(
                path_get_executable(),
                RTL_CONSTASCII_STRINGPARAM(SAL_CONFIGFILE("uno")) ) );
        Bootstrap unorc( unorc_path );
        if (0 == unorc.getHandle())
        {
            throw RuntimeException(
                OUSTR("cannot open ") + unorc_path,
                Reference< XInterface >() );
        }
        if (check_unorc_only)
        {
            // ensure correct entries in unorc only
            check_unorc(
                reinterpret_cast< rtlBootstrapHandle * >( &unorc ),
                patch_unorc, verbose );
            return 0;
        }
        
        // build up env
        UNO_PackagesManager pkg_mgr( unorc, AppOptionsContainer( verbose ) );
        
        // check if office process is running
        bool office_runs = office_is_running( unorc );
        if (office_runs && (renewal || !additional_packages.empty()))
        {
            throw RuntimeException(
                OUSTR("concurrent office process running!  close it!"),
                Reference< XInterface >() );
        }
        
        // copy additional packages
        for ( ::std::size_t n = additional_packages.size(); n--; )
        {
            pkg_mgr.copy( additional_packages[ n ] );
        }
        
        if (renewal)
        {
            // remove all installed packages
            pkg_mgr.remove_all();
        }
        
        if (pkg_mgr.check()) // check for packages to be balanced
        {
            // balancing needed
            if (office_runs)
            {
                throw RuntimeException(
                    OUSTR("concurrent office process running!  close it!"),
                    Reference< XInterface >() );
            }
            // do balancing
            pkg_mgr.balance();
        }
	}
	catch (Exception & exc)
	{
        OString cstr_msg(
            OUStringToOString( exc.Message, osl_getThreadTextEncoding() ) );
        fprintf(
            stderr, "> error: %s\n> dying abnormally...\n",
            cstr_msg.getStr() );
		ret = 1;
	}
    
	return ret;
}
