/*************************************************************************
 *
 *  $RCSfile: ConstantPool.java,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: vg $ $Date: 2003/06/12 11:14:22 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

package com.sun.star.lib.sandbox.generic;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;

import java.util.Vector;

/**
 * Eine Instanz werwaltet einen Pool von Konstanten fuer das
 * Java-ClassFile-Format.
 * <BR>
 * @version     $Version: 1.1 $
 * @author      Markus Meyer
 * @see			stardiv.js.ip.Ip
 */
class PoolConstant
{
    //7  CONSTANT_Class -> Class = s
    //9  CONSTANT_Fieldref -> Class = s, NameAndType = s1
    //10 CONSTANT_Methodref -> Class = s, NameAndType = s1
    //11 CONSTANT_Interfaceref -> Class = s, NameAndType = s1
    //8  CONSTANT_String -> String = s
    //3  CONSTANT_Integer -> i
    //4  CONSTANT_Float -> f
    //5  CONSTANT_Long -> l
    //6  CONSTANT_Double -> d
    //12 CONSTANT_NameAndType -> Name = s, Type = s1
    //1  CONSTANT_Utf8 ->aUTF

    static final byte  CONSTANT_Class = 7 ;
    static final byte  CONSTANT_Fieldref = 9;
    static final byte  CONSTANT_Methodref = 10;
    static final byte  CONSTANT_InterfaceMethodref = 11;
    static final byte  CONSTANT_String = 8;
    static final byte  CONSTANT_Integer = 3;
    static final byte  CONSTANT_Float = 4;
    static final byte  CONSTANT_Long = 5;
    static final byte  CONSTANT_Double = 6;
    static final byte  CONSTANT_NameAndType = 12;
    static final byte  CONSTANT_Utf8 = 1;

    byte    iTag;
    short   s, s1;
    int     i;
    float   f;
    long    l;
    double  d;
    String  aUTF;
    
    PoolConstant(){}
    PoolConstant( int t, int i ){ iTag = (byte)t; s = (byte)i;}
    PoolConstant( int t, int i, int i1 ){ iTag = (byte)t; s = (byte)i; s1 = (byte)i1; }
    PoolConstant( int t, String a ){ iTag = (byte)t; aUTF = a; }

    void read( DataInput aStm )
        throws java.io.IOException
    {
		iTag = aStm.readByte();
		switch( iTag )
		{
			case 7 :   // CONSTANT_Class
			case 8  :  // CONSTANT_String
				s = aStm.readShort();
				break;
			case 9 :   // CONSTANT_Fieldref
			case 10 :  // CONSTANT_Methodref
			case 11 :  // InterfaceMethodref
			case 12 :  // CONSTANT_NameAndType
				s = aStm.readShort();
				s1 = aStm.readShort();
				break;

			case 3  :  // CONSTANT_Integer
				i = aStm.readInt();
   				break;

			case 4  :  // CONSTANT_Float
				f = aStm.readFloat();
				break;

			case 5  :  // CONSTANT_Long
				l = aStm.readLong();
				break;

			case 6  :  // CONSTANT_Double
				d = aStm.readDouble();
				break;

			case 1  :  // CONSTANT_Utf8
			    aUTF = aStm.readUTF();
			    break;
			default:
   			    System.out.println( "error: Tag = " + iTag );
		}
    }

    void write( DataOutput aStm )
        throws java.io.IOException
    {
		aStm.writeByte( iTag );
		switch( iTag )
		{
			case 7 :   // CONSTANT_Class
			case 8  :  // CONSTANT_String
				aStm.writeShort( s );
				break;
			case 9 :   // CONSTANT_Fieldref
			case 10 :  // CONSTANT_Methodref
			case 11 :  // InterfaceMethodref
			case 12 :  // CONSTANT_NameAndType
				aStm.writeShort( s );
				aStm.writeShort( s1 );
				break;
			case 3  :  // CONSTANT_Integer
				aStm.writeInt( i );
   				break;

			case 4  :  // CONSTANT_Float
				aStm.writeFloat( f );
				break;

			case 5  :  // CONSTANT_Long
				aStm.writeLong( l );
				break;

			case 6  :  // CONSTANT_Double
				aStm.writeDouble( d );
				break;

			case 1  :  // CONSTANT_Utf8
			    aStm.writeUTF( aUTF );
			    break;
			default:
   			    System.out.println( "error: Tag = " + iTag );
		}
    }
}

/**
 * Diese Klasse ist die Laufzeitdarstellung des ConstantPool in einem
 * Java-Classfile. Die genau Spezifikation befindet sich 
 * im Buch "The Java Virtual Machine Specification".
 */
public class ConstantPool {
    Vector aAtoms = new Vector();

    /**
     * Es wird ein Pool angelegt. Der erste Eintrag wird mit Null belegt.
     */
    public ConstantPool() {
        aAtoms.addElement( null );
    }

    /**
     * Die Anzahl der Elemente im Pool. Initial hat der Pool die Groesse eins.
     * Die Double und Long Konstanten belegen zwei Eintraege im Pool.
     */
    public short size() {
        return (short)aAtoms.size();
    }

    private short addElement( PoolConstant aE  ) {
//          int iRet = 
		aAtoms.addElement( aE );
		if( aE.iTag == 5    // CONSTANT_Long
		  || aE.iTag == 6 ) // CONSTANT_Double
            aAtoms.addElement( null );
//  		return (short)iRet;

  		return (short)aAtoms.indexOf(aE);
    }

    /**
     * Es wird der Index auf die Utf8-Konstante mit dem Name aName zurueckgegeben.
     * Ist sie nicht vorhanden, wird sie angelegt.
     */
    public short getUtf8( String aName ) {
		for( short i = 1; i < aAtoms.size(); i++ ) {
    		PoolConstant pE = (PoolConstant)aAtoms.elementAt( i );
    		if( pE != null
    		  && pE.iTag == PoolConstant.CONSTANT_Utf8
		      && aName.equals( pE.aUTF ) )
    		    return i;
    	}

        PoolConstant aE = new PoolConstant();
   		aE.iTag = PoolConstant.CONSTANT_Utf8;
   		aE.aUTF = aName;
        return addElement( aE );
    }

    private short get( byte iTag, short s, short s1 ) {
		for( short i = 1; i < aAtoms.size(); i++ ) {
    		PoolConstant pE = (PoolConstant)aAtoms.elementAt( i );
    		if( pE != null && pE.iTag == iTag
    		  && pE.s == s && pE.s1 == s1 )
                return i;
    	}
        PoolConstant aE = new PoolConstant();
   		aE.iTag = iTag;
   		aE.s = s;
   		aE.s1 = s1;
        return addElement( aE );
    }

    /**
     * Es wird der Index auf die String-Konstante mit dem Name aName zurueckgegeben.
     * Ist sie nicht vorhanden, wird sie angelegt.
     */
    public short getString( String aName ) {
    	short iUtf = getUtf8( aName );
    	return get( PoolConstant.CONSTANT_String, iUtf, (short)0 );
    }

    /**
     * Es wird der Index auf die Class-Konstante mit dem Name aName zurueckgegeben.
     * Ist sie nicht vorhanden, wird sie angelegt.
     */
    public short getClass( String aName ) {
    	short iUtf = getUtf8( aName );
    	return get( PoolConstant.CONSTANT_Class, iUtf, (short)0 );
    }

    /**
     * Es wird der Index auf die Methodref-Konstante zurueckgegeben.
     * Ist sie nicht vorhanden, wird sie angelegt.
     * 
     * @param sClass ist der Index auf die Klasse in der sich die Methode befindet.
     * @param sNameAndType ist der Index auf den Namene und den Typ der Methode.
     *
     * @see #getClass(java.lang.String)
     * @see #getNameAndType(java.lang.String, java.lang.String)
     * @see #getNameAndType(short, short)
     */
    public short getMethodRef( short sClass, short sNameAndType ) {
    	return get( PoolConstant.CONSTANT_Methodref, sClass, sNameAndType );
    }

    /**
     * Es wird der Index auf die InterfaceMethodref-Konstante zurueckgegeben.
     * Ist sie nicht vorhanden, wird sie angelegt.
     * 
     * @param sClass ist der Index auf die Klasse in der sich die Methode befindet.
     * @param sNameAndType ist der Index auf den Namene und den Typ der Methode.
     *
     * @see #getClass(java.lang.String)
     * @see #getNameAndType(java.lang.String, java.lang.String)
     * @see #getNameAndType(short, short)
     */
    public short getInterfaceMethodRef( short sClass, short sNameAndType ) {
    	return get( PoolConstant.CONSTANT_InterfaceMethodref, sClass, sNameAndType );
    }

    /**
     * Es wird der Index auf die Fieldref-Konstante zurueckgegeben.
     * Ist sie nicht vorhanden, wird sie angelegt.
     * 
     * @param sClass ist der Index auf die Klasse in der sich die Variable befindet.
     * @param sNameAndType ist der Index auf den Namene und den Typ der Variablen.
     *
     * @see #getClass(java.lang.String)
     * @see #getNameAndType(java.lang.String, java.lang.String)
     * @see #getNameAndType(short, short)
     */
    public short getFieldRef( short sClass, short sNameAndType ) {
    	return get( PoolConstant.CONSTANT_Fieldref, sClass, sNameAndType );
    }

    /**
     * Es wird der Index auf die NameAndType-Konstante zurueckgegeben.
     * Ist sie nicht vorhanden, wird sie angelegt.
     * 
     * @param sName ist der Utf8-Index auf den Namen.
     * @param sType ist der Utf8-Index auf den Typ.
     *
     * @see #getUtf8(java.lang.String)
     * @see #getNameAndType(short, short)
     */
    public short getNameAndType( short sName, short sType ) {
    	return get( PoolConstant.CONSTANT_NameAndType, sName, sType );
    }

    /**
     * Es wird der Index auf die NameAndType-Konstante zurueckgegeben.
     * Ist sie nicht vorhanden, wird sie angelegt.
     * 
     * @param aName ist der Namen.
     * @param aType ist der Typ.
     *
     * @see #getUtf8(java.lang.String)
     * @see #getNameAndType(short, short)
     */
    public short getNameAndType( String aName, String aType ) {
    	short sName = getUtf8( aName );
    	short sType = getUtf8( aType );
    	return get( PoolConstant.CONSTANT_NameAndType, sName, sType );
    }

    /**
     * Aus dem Stream wird die Laenge des Pools und der Pool gelesen.
     */
    public void read( DataInput aStm ) throws java.io.IOException {
        short iPoolCount = aStm.readShort(); // Groesse des constant_pool

		for( short iIndex = 1; iIndex < iPoolCount; iIndex++ ) {
    		PoolConstant aE = new PoolConstant();
    		aE.read( aStm );
            aAtoms.addElement( aE );
    		if( aE.iTag == 5    // CONSTANT_Long
    		  || aE.iTag == 6 ) // CONSTANT_Double
   				iIndex++;  // Long/ Double belegt zwei Tabellenplaetze
    	}
    }

    /**
     * In den Stream wird die Laenge des Pools und der Pool geschrieben.
     * Der Aufbau im Stream entspricht genau der JVM 1.02 Spezifikation.
     */
    public void write( DataOutput aStm ) throws java.io.IOException {
		aStm.writeShort( aAtoms.size() ); // Groesse des constant_pool

		for( short i = 1; i < aAtoms.size(); i++ ) {
    		PoolConstant pE = (PoolConstant)aAtoms.elementAt( i );
    		if( pE != null )
    		    pE.write( aStm );
    		else
   				i++;  // Long/ Double belegt zwei Tabellenplaetze
    	}
    }

    /**
     * Der Inhalt eines Pooleintrags wird zurueckgegeben. Besteht
     * der Eintrag nicht nur aus einem String, dann wird der
     * uebergebene Parameter in eine String umgewandelt.
     */
	public String toString( short sIndex ) {
        PoolConstant pE = (PoolConstant)aAtoms.elementAt( sIndex );
        
        if( null != pE )
    		switch( pE.iTag ) {
    			case 7 :   // CONSTANT_Class
    			case 8  :  // CONSTANT_String
    			    return toString( pE.s );

    			case 1  :  // CONSTANT_Utf8
    			    return pE.aUTF;
    		}
        return String.valueOf( sIndex );
	}
	
    /**
     * Der gesammte Pool wird ausgegeben.
     */
	public String toString() {
		short iPoolCount = size();
        String aStr = "Constant Pool: " + iPoolCount + "\n";
        aStr += "\t0: VM entry\n";

		for( short i = 1; i < iPoolCount; i++ ) {
            PoolConstant pE = (PoolConstant)aAtoms.elementAt( i );
            if( pE == null )
                continue;
            aStr += "\t" + i + ": ";
            
    		switch( pE.iTag ) {
    			case 7 :   // CONSTANT_Class
                    aStr += "Class: " + toString( pE.s );
    				break;
    			case 9 :   // CONSTANT_Fieldref
                    aStr += "Fieldref: Class=" + toString( pE.s )
                                         + " NameType=" + toString( pE.s1 );
    				break;
    			case 10 :  // CONSTANT_Methodref
                    aStr += "Methodref: Class=" + toString( pE.s )
                                        + " NameType=" + toString( pE.s1 );
    				break;
    			case 11 :  // InterfaceMethodref
                    aStr += "InterfaceMethodref: Class=" + toString( pE.s )
                                         + " NameType=" + toString( pE.s1 );
    				break;
    			case 8  :  // CONSTANT_String
                    aStr += "String: " + toString( pE.s );
       			    break;
    			case 3  :  // CONSTANT_Integer
                    aStr += "Integer: " + pE.i;
       				break;
    			case 4  :  // CONSTANT_Float
                    aStr +=  "Float: " + pE.f;
    				break;
    			case 5  :  // CONSTANT_Long
                    aStr += "Long: " + pE.l;
    				break;
    			case 6  :  // CONSTANT_Double
                    aStr += "Double: " + pE.d;
    				break;
       			case 12 :  // CONSTANT_NameAndType
                    aStr +=  "NameAndType: "  + toString( pE.s ) + ", " + toString( pE.s1 );
    				break;
    			case 1  :  // CONSTANT_Utf8
                    aStr += "Utf8: " + pE.aUTF;
    			    break;
    			default:
       			    System.out.println( "error: Tag = " + pE.iTag );
    		}
    		aStr += "\n";
    	}
    	return aStr;
	}
}

