/*************************************************************************
 *
 *  $RCSfile: calculat.cxx,v $
 *
 *  $Revision: 1.5.58.1 $
 *
 *  last change: $Author: hr $ $Date: 2004/01/09 17:57:16 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

// header for Point, Rectangle
#ifndef _SV_GEN_HXX
#include <tools/gen.hxx>
#endif
// header for Polygon
#ifndef _SV_POLY_HXX
#include <vcl/poly.hxx>
#endif
// header for DBG_ASSERT
#ifndef _TOOLS_DEBUG_HXX
#include <tools/debug.hxx>
#endif
// header for XPolygon, XPolyPolygon
#ifndef _XPOLY_HXX
#include <svx/xpoly.hxx>
#endif
// header for Line
#ifndef _SV_LINE_HXX
#include <vcl/line.hxx>
#endif
#ifndef INCLUDED_RTL_MATH_HXX
#include <rtl/math.hxx>
#endif
// for performance measurement
#ifndef	_RTL_LOGFILE_HXX_
#include <rtl/logfile.hxx>
#endif

// Note: Enable the following to skip points in the resulting spline
// poly-polygon, if they have equal x-values rather than identical points.
// Unitl now, I think there are situations where the output might differ, if you
// do so, so it's not enabled by default.

// #define SPLINE_OPTIMIZE_POINTS

#include "calculat.hxx"

#include <algorithm>
#include <functional>

using namespace ::std;

void SchCalculationHelper::IntersectPolyPolygonWithRectangle(
    const XPolyPolygon & rPolyPolygon,
    const Rectangle& rRectangle,
    XPolyPolygon& rOutResult )
{
	RTL_LOGFILE_CONTEXT_AUTHOR( context, "sch", "bm93744", "SchCalculationHelper::IntersectPolyPolygonWithRectangle");

    const USHORT nCount = rPolyPolygon.Count();
    rOutResult.Clear();
    XPolyPolygon aSubResult;

    for( USHORT i = 0; i < nCount; ++i )
    {
        aSubResult.Clear();
        IntersectPolygonWithRectangle( rPolyPolygon.GetObject( i ),
                                       rRectangle,
                                       aSubResult );

        rOutResult.Insert( aSubResult, XPOLYPOLY_APPEND );
    }
    OSL_TRACE( "IntersectPolyPolygonWithRectangle: result has %d polygons", rOutResult.Count() );
}
    
void SchCalculationHelper::IntersectPolygonWithRectangle( const XPolygon& rPolygon, const Rectangle& rRectangle, XPolyPolygon& aResult )
{
	RTL_LOGFILE_CONTEXT_AUTHOR( context, "sch", "bm93744", "SchCalculationHelper::IntersectPolygonWithRectangle");

	aResult.Clear();

 	if( rRectangle.IsInside( rPolygon.GetBoundRect() ) )
	{
		aResult.Insert( rPolygon );
        OSL_TRACE( "IntersectPolygonWithRectangle: result has %d polygons", aResult.Count() );
		return;
	}

    Point aFrom;
    Point aTo;
	USHORT nCount = rPolygon.GetPointCount();

    // set last point to a position outside the rectangle, such that the first
    // time clip2d returns true, the comparison to last will always yield false
    Point aLast (rRectangle.TopLeft());
    aLast.Move (-1, -1);
    XPolygon aCurrentPoly;
    USHORT nIdx = 0;

	for (USHORT i=1; i<nCount; i++)
	{
		aFrom = rPolygon[i-1];
		aTo = rPolygon[i];
		if (clip2d (aFrom, aTo, rRectangle))
		{
            // compose an XPolygon of as many consecutive points as possible
            if (aFrom == aLast)
            {
                if (aTo != aFrom)
                    aCurrentPoly.Insert (nIdx++, aTo, XPOLY_NORMAL);
            }
            else
            {
                // create an XPolygon and put it into the XPolyPolygon
                if (aCurrentPoly.GetPointCount() > 0)
                    aResult.Insert (aCurrentPoly, XPOLYPOLY_APPEND);

                // start new sequence
				aCurrentPoly.SetPointCount (0);
                aCurrentPoly.Insert (0, aFrom, XPOLY_NORMAL);
                nIdx = 1;
                if (aTo != aFrom)
                    aCurrentPoly.Insert (nIdx++, aTo, XPOLY_NORMAL);
            }

            aLast = aTo;
        }
    }
    if (aCurrentPoly.GetPointCount() > 0)
        aResult.Insert (aCurrentPoly, XPOLYPOLY_APPEND);

    OSL_TRACE( "IntersectPolygonWithRectangle: result has %d polygons", aResult.Count() );
}

BOOL SchCalculationHelper::ClipLineAtRectangle( Line& aLine, const Rectangle& rRectangle )
{
	Point	aPoint0 = aLine.GetStart ();
	Point	aPoint1 = aLine.GetEnd ();
	BOOL	bVisible = clip2d (aPoint0, aPoint1, rRectangle);
	if (bVisible)
	{
		aLine.SetStart (aPoint0);
		aLine.SetEnd (aPoint1);
	}
	return bVisible;
}




BOOL	SchCalculationHelper::clip2d	(Point & rPoint0, 
										Point & rPoint1,
										const Rectangle & rRectangle)
{
	//	Direction vector of the line.						
	Point	aD = rPoint1 - rPoint0;

	if (aD.X()==0 && aD.Y()==0 && rRectangle.IsInside (rPoint0))
	{
		//	Degenerate case of a zero length line.
		return TRUE;
	}
	else
	{
		//	Values of the line parameter where the line enters resp. leaves the rectangle.
		double	fTE = 0,
				fTL = 1;
				
		//	Test wether at least a part lies in the four half-planes with respect to 
		//	the rectangles four edges.
		if (CLIPt (aD.X(), rRectangle.Left() - rPoint0.X(), fTE, fTL))
			if (CLIPt (-aD.X(), rPoint0.X() - rRectangle.Right(), fTE, fTL))
				if (CLIPt (aD.Y(), rRectangle.Top() - rPoint0.Y(), fTE, fTL))
					if (CLIPt (-aD.Y(), rPoint0.Y() - rRectangle.Bottom(), fTE, fTL))
					{
						//	At least a part is visible.
						if (fTL < 1)
						{
							//	Compute the new end point.
							rPoint1.X() = (long)(rPoint0.X() + fTL * aD.X() + 0.5);
							rPoint1.Y() = (long)(rPoint0.Y() + fTL * aD.Y() + 0.5);
						}
						if (fTE > 0)
						{
							//	Compute the new starting point.
							rPoint0.X() = (long)(rPoint0.X() + fTE * aD.X() + 0.5);
							rPoint0.Y() = (long)(rPoint0.Y() + fTE * aD.Y() + 0.5);
						}
						return TRUE;
					}
					
		//	Line is not visible.
		return FALSE;
	}
}




BOOL	SchCalculationHelper::CLIPt	(double fDenom, 
									double fNum, 
									double & fTE, 
									double & fTL)
{
	double	fT;
	
	if (fDenom > 0)				//	Intersection enters: PE
	{
		fT = fNum / fDenom;		//	Parametric value at the intersection.
		if (fT > fTL)			//	fTE and fTL crossover
			return FALSE;		//	  therefore reject the line.
		else if (fT > fTE)		//	A new fTE has been found.
			fTE = fT;
	}
	else if (fDenom < 0)		//	Intersection leaves: PL
	{
		fT = fNum / fDenom;		//	Parametric Value at the intersection.
		if (fT < fTE)			//	fTE and fTL crossover
			return FALSE;		//	  therefore reject the line.
		else if (fT < fTL)		//	A new fTL has been found.
			fTL = fT;
	}
	else if (fNum > 0)
		return FALSE;			//	Line lies on the outside of the edge.
	
	return TRUE;
}


// --------------------------------------------------------------------------------

// Calculation of Splines

namespace
{

class lcl_SplineCalculation
{
public:
    typedef pair< double, double >   tPointType;
    typedef vector< tPointType >     tPointVecType;

    /** @descr creates an object that calculates cublic splines on construction

        @param rPoints  the points for which splines shall be calculated
        @param fY1FirstDerivation the resulting spline should have the first
               derivation equal to this value at the x-value of the first point
               of rPoints.  If fY1FirstDerivation is set to infinity, a natural
               spline is calculated.
        @param fYnFirstDerivation the resulting spline should have the first
               derivation equal to this value at the x-value of the last point
               of rPoints
     */
    lcl_SplineCalculation( const tPointVecType & rPoints,
                           double fY1FirstDerivation,
                           double fYnFirstDerivation );

    /** @descr this function corresponds to the function splint in [1].

        [1] Numerical Recipies in C, 2nd edition
            William H. Press, et al.,
            Section 3.3, page 116
    */
    double GetInterpolatedValue( double x );

private:
    /// a copy of the points given in the CTOR
    tPointVecType            m_aPoints;

    /// the result of the Calculate() method
    vector< double >         m_aSecDerivY;

    double m_fYp1;
    double m_fYpN;

    // these values are cached for performance reasons
    tPointVecType::size_type m_nKLow;
    tPointVecType::size_type m_nKHigh;
    double m_fLastInterpolatedValue;

    /** @descr this function corresponds to the function spline in [1].

        [1] Numerical Recipies in C, 2nd edition
            William H. Press, et al.,
            Section 3.3, page 115
    */
    void Calculate();
};

template< typename T, typename U >
    struct lcl_LessFirstOfPair : binary_function< pair< T, U >, pair< T, U >, bool >
{
    inline bool operator() ( const pair< T, U > & rOne, const pair< T, U > & rOther )
    {
        return ( rOne.first < rOther.first );
    }
};

template< typename T >
    struct lcl_EqualsFirstDoubleOfPair : binary_function< pair< double, T >, pair< double, T >, bool >
{
    inline bool operator() ( const pair< double, T > & rOne, const pair< double, T > & rOther )
    {
        return ( ::rtl::math::approxEqual( rOne.first, rOther.first ) );
    }
};

// assume Point given in 100th of a mm
struct lcl_EqualsPointWithDPI : binary_function< Point, Point, bool >
{
    lcl_EqualsPointWithDPI( long nDPIX, long nDPIY ) :
            m_fFactX( static_cast< double >( nDPIX ) / 2540.0 ),
            m_fFactY( static_cast< double >( nDPIY ) / 2540.0 )
    {}
    
    inline bool operator() ( const Point & p1, const Point & p2 )
    {
        bool bXEqual = false;
        bool bYEqual = false;
        if( m_fFactX > 0 )
        {
            bXEqual =
                ( static_cast< long >( static_cast< double >( p1.getX()) * m_fFactX ) ==
                  static_cast< long >( static_cast< double >( p2.getX()) * m_fFactX ) );
        }
        else
        {
            bXEqual = ( p1.getX() == p2.getX() );
        }

        if( bXEqual )
        {
            if( m_fFactY > 0 )
            {
                bYEqual =
                    ( static_cast< long >( static_cast< double >( p1.getY()) * m_fFactY ) ==
                      static_cast< long >( static_cast< double >( p2.getY()) * m_fFactY ) );
            }
            else
            {
                bYEqual = ( p1.getY() == p2.getY() );
            }
        }

        return bXEqual && bYEqual;
    }

private:
    double m_fFactX;
    double m_fFactY;
};

lcl_SplineCalculation::lcl_SplineCalculation(
    const tPointVecType & rPoints,
    double fY1FirstDerivation,
    double fYnFirstDerivation )
        : m_aPoints( rPoints ),
          m_fYp1( fY1FirstDerivation ),
          m_fYpN( fYnFirstDerivation ),
          m_nKLow( 0 ),
          m_nKHigh( rPoints.size() - 1 )
{
    ::rtl::math::setInf( &m_fLastInterpolatedValue, sal_False );

    sort( m_aPoints.begin(), m_aPoints.end(),
          lcl_LessFirstOfPair< double, double >() );

    // #108301# remove points that have equal x-values
    m_aPoints.erase( unique( m_aPoints.begin(), m_aPoints.end(),
                             lcl_EqualsFirstDoubleOfPair< double >() ),
                     m_aPoints.end() );

    Calculate();
}

void lcl_SplineCalculation::Calculate()
{
    // n is the last valid index to m_aPoints
    const tPointVecType::size_type n = m_aPoints.size() - 1;
    if( n < 1 )
        return;

    vector< double > u( n );
    m_aSecDerivY.resize( n + 1, 0.0 );

    if( ::rtl::math::isInf( m_fYp1 ) )
    {
        // natural spline
        m_aSecDerivY[ 0 ] = 0.0;
        u[ 0 ] = 0.0;
    }
    else
    {
        m_aSecDerivY[ 0 ] = -0.5;
        double xDiff = ( m_aPoints[ 1 ].first - m_aPoints[ 0 ].first );
        u[ 0 ] = ( 3.0 / xDiff ) *
            ((( m_aPoints[ 1 ].second - m_aPoints[ 0 ].second ) / xDiff ) - m_fYp1 );
    }

    for( tPointVecType::size_type i = 1; i < n; ++i )
    {
        pair< double, double >
            p_i = m_aPoints[ i ],
            p_im1 = m_aPoints[ i - 1 ],
            p_ip1 = m_aPoints[ i + 1 ];
        
        double sig = ( p_i.first - p_im1.first ) /
            ( p_ip1.first - p_im1.first );
        double p = sig * m_aSecDerivY[ i - 1 ] + 2.0;

        m_aSecDerivY[ i ] = ( sig - 1.0 ) / p;
        u[ i ] =
            ( ( p_ip1.second - p_i.second ) /
              ( p_ip1.first - p_i.first ) ) -
            ( ( p_i.second - p_im1.second ) /
              ( p_i.first - p_im1.first ) );
        u[ i ] =
            ( 6.0 * u[ i ] / ( p_ip1.first - p_im1.first )
              - sig * u[ i - 1 ] ) / p;
    }

    // initialize to values for natural splines (used for m_fYpN equal to
    // infinity)
    double qn = 0.0;
    double un = 0.0;

    if( ! ::rtl::math::isInf( m_fYpN ) )
    {
        qn = 0.5;
        double xDiff = ( m_aPoints[ n ].first - m_aPoints[ n - 1 ].first );
        un = ( 3.0 / xDiff ) *
            ( m_fYpN - ( m_aPoints[ n ].second - m_aPoints[ n - 1 ].second ) / xDiff );
    }

    m_aSecDerivY[ n ] = ( un - qn * u[ n - 1 ] ) * ( qn * m_aSecDerivY[ n - 1 ] + 1.0 );

    // note: the algorithm in [1] iterates from n-1 to 0, but as size_type
    // may be (usuall is) an unsigned type, we can not write k >= 0, as this
    // is always true.
    for( tPointVecType::size_type k = n; k > 0; --k )
    {
        ( m_aSecDerivY[ k - 1 ] *= m_aSecDerivY[ k ] ) += u[ k - 1 ];
    }
}

double lcl_SplineCalculation::GetInterpolatedValue( double x )
{
    DBG_ASSERT( m_aPoints.size() < 1 ||
                ( ( m_aPoints[ 0 ].first <= x ) &&
                  ( x <= m_aPoints[ m_aPoints.size() - 1 ].first ) ),
                "Trying to extrapolate" );

    const tPointVecType::size_type n = m_aPoints.size() - 1;
    if( n < 1 )
    {
        double fNan;
        ::rtl::math::setNan( & fNan );
        return fNan;
    }

    if( x < m_fLastInterpolatedValue )
    {
        m_nKLow = 0;
        m_nKHigh = n;

        // calculate m_nKLow and m_nKHigh
        // first initialization is done in CTOR
        while( m_nKHigh - m_nKLow > 1 )
        {
            tPointVecType::size_type k = ( m_nKHigh + m_nKLow ) / 2;
            if( m_aPoints[ k ].first > x )
                m_nKHigh = k;
            else
                m_nKLow = k;
        }
    }
    else
    {
        while( ( m_aPoints[ m_nKHigh ].first < x ) &&
               ( m_nKHigh <= n ) )
        {
            ++m_nKHigh;
            ++m_nKLow;
        }
        DBG_ASSERT( m_nKHigh <= n, "Out of Bounds" );
    }
    m_fLastInterpolatedValue = x;

    double h = m_aPoints[ m_nKHigh ].first - m_aPoints[ m_nKLow ].first;
    DBG_ASSERT( h != 0, "Bad input to GetInterpolatedValue()" );

    double a = ( m_aPoints[ m_nKHigh ].first - x ) / h;
    double b = ( x - m_aPoints[ m_nKLow ].first  ) / h;

    return ( a * m_aPoints[ m_nKLow ].second +
             b * m_aPoints[ m_nKHigh ].second +
             (( a*a*a - a ) * m_aSecDerivY[ m_nKLow ] +
              ( b*b*b - b ) * m_aSecDerivY[ m_nKHigh ] ) *
             ( h*h ) / 6.0 );
}

typedef lcl_SplineCalculation::tPointVecType::size_type lcl_tSizeType;

// this is the maximum number of points that will be inserted into an XPolygon
const lcl_tSizeType nPolygonSizeThreshold = 0xff00;

} //  anonymous namespace

// ----------------------------------------

void SchCalculationHelper::CalculateCubicSplines(
    const vector< pair< double, double > > & rPoints,
    sal_Int32 nGranularity,
    XPolyPolygon & rOutResult,
    long nDPIX /* = 0 */, long nDPIY /* = 0 */ )
{
	RTL_LOGFILE_CONTEXT_AUTHOR( context, "sch", "bm93744", "SchCalculationHelper::CalculateCubicSplines");

    DBG_ASSERT( nGranularity != 0, "Zero-Granularity is invalid" );
    rOutResult.Clear();

    // calculate second derivates
    double fInfty;
    ::rtl::math::setInf( &fInfty, sal_False );
    lcl_SplineCalculation aSpline( rPoints, fInfty, fInfty );

    // fill result polygon with calculated values
    lcl_tSizeType nLastIndex = rPoints.size() - 1;

    // calculate all necessary points on spline curve
    vector< Point > aCalculatedPoints;
    aCalculatedPoints.reserve( nLastIndex * nGranularity + 1 );
#ifdef SPLINE_OPTIMIZE_POINTS
    long nLastX = -1;
#else
    Point aLast( -1, -1 );
#endif

    for( lcl_tSizeType i = 0; i < nLastIndex; ++i )
    {
        double fBaseX = rPoints[ i ].first;
        double fInc = ( rPoints[ i + 1 ].first - fBaseX ) /
            static_cast< double >( nGranularity );

        for( sal_Int32 j = 0; j < nGranularity; ++j )
        {
            double x = fBaseX + ( fInc * static_cast< double >( j ) );
            Point aPoint( static_cast< long >( x ),
                          static_cast< long >( aSpline.GetInterpolatedValue( x )));
#ifdef SPLINE_OPTIMIZE_POINTS
            if( aPoint.getX() != nLastX )
            {
                aCalculatedPoints.push_back( aPoint );
                nLastX = aPoint.getX();
            }
#else
            if( aPoint != aLast )
            {
                aCalculatedPoints.push_back( aPoint );
                aLast = aPoint;
            }
#endif
        }
    }
    // calculate last point
    Point aPoint( static_cast< long >( rPoints[ nLastIndex ].first ),
                  static_cast< long >( aSpline.GetInterpolatedValue( rPoints[ nLastIndex ].first )));
#ifdef SPLINE_OPTIMIZE_POINTS
    if( aPoint.getX() != nLastX )
        aCalculatedPoints.push_back( aPoint );
#else
    if( aPoint != aLast )
        aCalculatedPoints.push_back( aPoint );
#endif

    OSL_TRACE( "Used points: %u of %u (skipped %u)",
               aCalculatedPoints.size(),
               nLastIndex * nGranularity + 1,
               nLastIndex * nGranularity + 1 - aCalculatedPoints.size() );

    if( nDPIX > 0 || nDPIY > 0 )
    {
        size_t nOldSize = aCalculatedPoints.size();
        OSL_TRACE( "Reducing with %u x %u DPI", nDPIX, nDPIY );
        aCalculatedPoints.erase(
            unique( aCalculatedPoints.begin(), aCalculatedPoints.end(),
                    lcl_EqualsPointWithDPI( nDPIX, nDPIY ) ),
            aCalculatedPoints.end() );
        OSL_TRACE( "Used points: %u (removed: %u)",
                   aCalculatedPoints.size(),
                   nOldSize - aCalculatedPoints.size());
    }

    // put calculated points into XPolygons that are put into an XPolyPolygon

    // note: the threshold should be strictly smaller than the maximum capacity
    // of an XPolygon
    XPolygon aCurrentPolygon( static_cast< USHORT >(
                                  min< lcl_tSizeType >( aCalculatedPoints.size(),
                                                        nPolygonSizeThreshold )));
    aCurrentPolygon[ 0 ] = aCalculatedPoints[ 0 ];
    USHORT nIndexInPolygon = 1;
    for( vector< Point >::const_iterator aIt = aCalculatedPoints.begin();
         aIt != aCalculatedPoints.end(); ++aIt )
    {
        aCurrentPolygon[ nIndexInPolygon ] = (*aIt);
        if( ( ++nIndexInPolygon % nPolygonSizeThreshold ) == 0 )
        {
            rOutResult.Insert( aCurrentPolygon, XPOLYPOLY_APPEND );
            aCurrentPolygon[ 0 ] = (*aIt);
            nIndexInPolygon = 1;
        }
    }
    if( nIndexInPolygon > 1 )
    {
        aCurrentPolygon.SetSize( nIndexInPolygon );
        rOutResult.Insert( aCurrentPolygon, XPOLYPOLY_APPEND );
    }

    OSL_TRACE( "CalculateCubicSplines: result has %d polygons", rOutResult.Count() );
}
