/*************************************************************************
 *
 *  $RCSfile: machine.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: sb $ $Date: 2000/12/15 17:42:22 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef _UCB_MAIN_MACHINE_HXX_
#include "machine.hxx"
#endif

#ifndef _VOS_DIAGNOSE_H_
#include <vos/diagnose.hxx>
#endif
#ifndef _VOS_MUTEX_HXX_
#include <vos/mutex.hxx>
#endif

using ucb_main::Machine;
using ucb_main::MachineControl;

//============================================================================
//
//  MachineControl
//
//============================================================================

inline MachineControl::MachineData::MachineData(Mode eTheMode):
	m_eState(STATE_RUNNING),
	m_eMode(eTheMode)
{}

//============================================================================
void MachineControl::checkTryStop()
{
	if (m_eState == STATE_RUNNING
		&& m_nTotal > 0
		&& m_nIdle + (m_bEventOccured ? m_nIdleAwaitingEvent : 0) == m_nTotal)
	{
		m_eState = STATE_CALLING_TRY_STOP;
		for (Map::iterator aIt(m_aMap.begin()); aIt != m_aMap.end(); ++aIt)
			if (aIt->second.m_eState != MachineData::STATE_STOPPED)
			{
				aIt->second.m_eState = MachineData::STATE_STOPPING;
				aIt->first->tryStop();
			}
		m_eState = STATE_STOPPING;
	}
}

//============================================================================
void MachineControl::checkDoStop()
{
	if (m_eState == STATE_STOPPING
		&& m_nStopNo + m_nStopYes == m_nTotal)
	{
		m_eState = STATE_CALLING_DO_STOP;
		if (m_nStopNo == 0)
		{
			for (Map::iterator aIt(m_aMap.begin()); aIt != m_aMap.end();
				 ++aIt)
				if (aIt->second.m_eState != MachineData::STATE_STOPPED)
				{
					aIt->second.m_eState = MachineData::STATE_STOPPED;
					aIt->first->doStop(true);
				}
			m_eState = STATE_STOPPED;
		}
		else
		{
			for (Map::iterator aIt(m_aMap.begin()); aIt != m_aMap.end();
				 ++aIt)
				if (aIt->second.m_eState != MachineData::STATE_STOPPED)
				{
					aIt->second.m_eState = MachineData::STATE_RUNNING;
					aIt->first->doStop(false);
				}
			m_eState = STATE_RUNNING;
			m_nStopNo = 0;
			m_nStopYes = 0;
		}
	}
}

//============================================================================
void MachineControl::activate()
{
	vos::OGuard aGuard(m_aMutex);
	if (m_eState == STATE_INACTIVE)
	{
		m_eState = STATE_RUNNING;
		checkTryStop();
	}
}

//============================================================================
void MachineControl::busy(Machine * pMachine)
{
	VOS_ENSURE(pMachine, "ucb_main::MachineControl::busy(): Bad machine");
	if (pMachine)
	{
		vos::OGuard aGuard(m_aMutex);
		Map::iterator aIt(m_aMap.find(pMachine));
		if (aIt == m_aMap.end())
			if (m_eState == STATE_INACTIVE)
			{
				++m_nTotal;
				m_aMap[pMachine] = MachineData::MODE_BUSY;
			}
			else
				VOS_ENSURE(false,
						   "ucb_main::MachineControl::busy(): Bad state");
		else
			switch (aIt->second.m_eState)
			{
				case MachineData::STATE_RUNNING:
				case MachineData::STATE_STOPPING:
					switch (aIt->second.m_eMode)
					{
						case MachineData::MODE_IDLE:
							aIt->second.m_eMode = MachineData::MODE_BUSY;
							--m_nIdle;
							break;

						case MachineData::MODE_IDLE_AWAITING_EVENT:
							aIt->second.m_eMode = MachineData::MODE_BUSY;
							--m_nIdleAwaitingEvent;
							break;
					}
					break;

				default:
					VOS_ENSURE(false,
							   "ucb_main::MachineControl::busy(): Bad state");
					break;
			}
	}
}

//============================================================================
void MachineControl::idle(Machine * pMachine, bool bAwaitingEvent)
{
	VOS_ENSURE(pMachine, "ucb_main::MachineControl::idle(): Bad machine");
	if (pMachine)
	{
		vos::OGuard aGuard(m_aMutex);
		Map::iterator aIt(m_aMap.find(pMachine));
		if (aIt == m_aMap.end())
			if (m_eState == STATE_INACTIVE)
			{
				++m_nTotal;
				if (bAwaitingEvent)
				{
					m_aMap[pMachine] = MachineData::MODE_IDLE_AWAITING_EVENT;
					++m_nIdleAwaitingEvent;
				}
				else
				{
					m_aMap[pMachine] = MachineData::MODE_IDLE;
					++m_nIdle;
				}
			}
			else
				VOS_ENSURE(false,
						   "ucb_main::MachineControl::idle(): Bad state");
		else
			switch (aIt->second.m_eState)
			{
				case MachineData::STATE_RUNNING:
					switch (aIt->second.m_eMode)
					{
						case MachineData::MODE_BUSY:
							if (bAwaitingEvent)
							{
								aIt->second.m_eMode
									= MachineData::MODE_IDLE_AWAITING_EVENT;
								++m_nIdleAwaitingEvent;
							}
							else
							{
								aIt->second.m_eMode = MachineData::MODE_IDLE;
								++m_nIdle;
							}
							checkTryStop();
							break;

						case MachineData::MODE_IDLE:
							if (bAwaitingEvent)
							{
								aIt->second.m_eMode
									= MachineData::MODE_IDLE_AWAITING_EVENT;
								--m_nIdle;
								++m_nIdleAwaitingEvent;
							}
							break;

						case MachineData::MODE_IDLE_AWAITING_EVENT:
							if (!bAwaitingEvent)
							{
								aIt->second.m_eMode = MachineData::MODE_IDLE;
								--m_nIdleAwaitingEvent;
								++m_nIdle;
								checkTryStop();
							}
							break;
					}
					break;

				case MachineData::STATE_STOPPING:
					switch (aIt->second.m_eMode)
					{
						case MachineData::MODE_BUSY:
							if (bAwaitingEvent)
							{
								aIt->second.m_eMode
									= MachineData::MODE_IDLE_AWAITING_EVENT;
								++m_nIdleAwaitingEvent;
							}
							else
							{
								aIt->second.m_eMode = MachineData::MODE_IDLE;
								++m_nIdle;
							}
							break;

						case MachineData::MODE_IDLE:
							if (bAwaitingEvent)
							{
								aIt->second.m_eMode
									= MachineData::MODE_IDLE_AWAITING_EVENT;
								--m_nIdle;
								++m_nIdleAwaitingEvent;
							}
							break;

						case MachineData::MODE_IDLE_AWAITING_EVENT:
							if (!bAwaitingEvent)
							{
								aIt->second.m_eMode = MachineData::MODE_IDLE;
								--m_nIdleAwaitingEvent;
								++m_nIdle;
							}
							break;
					}
					break;

				default:
					VOS_ENSURE(aIt->second.m_eState
							       == MachineData::STATE_STOPPED,
							   "ucb_main::MachineControl::idle(): Bad state");
					break;
			}
	}
}

//============================================================================
void MachineControl::canStop(Machine * pMachine, bool bStop)
{
	vos::OGuard aGuard(m_aMutex);
	Map::iterator aIt(m_aMap.find(pMachine));
	VOS_ENSURE(aIt != m_aMap.end(),
			   "ucb_main::MachineControl::canStop(): Bad machine");
	if (aIt != m_aMap.end())
		if (aIt->second.m_eState == MachineData::STATE_STOPPING)
		{
			aIt->second.m_eState = MachineData::STATE_WAITING;
			++(bStop ? m_nStopYes : m_nStopNo);
			checkDoStop();
		}
		else
			VOS_ENSURE(false,
					   "ucb_main::MachineControl::canStop(): Bad state");
}

//============================================================================
void MachineControl::stopped(Machine * pMachine)
{
	vos::OGuard aGuard(m_aMutex);
	Map::iterator aIt(m_aMap.find(pMachine));
	VOS_ENSURE(aIt != m_aMap.end(),
			   "ucb_main::MachineControl::stopped(): Bad machine");
	if (aIt != m_aMap.end())
	{
		switch (aIt->second.m_eState)
		{
			case MachineData::STATE_RUNNING:
				--m_nTotal;
				aIt->second.m_eState = MachineData::STATE_STOPPED;
				switch (aIt->second.m_eMode)
				{
					case MachineData::MODE_BUSY:
						checkTryStop();
						break;

					case MachineData::MODE_IDLE:
						--m_nIdle;
						break;

					case MachineData::MODE_IDLE_AWAITING_EVENT:
						--m_nIdleAwaitingEvent;
						break;
				}
				break;

			case MachineData::STATE_STOPPING:
				--m_nTotal;
				aIt->second.m_eState = MachineData::STATE_STOPPED;
				switch (aIt->second.m_eMode)
				{
					case MachineData::MODE_IDLE:
						--m_nIdle;
						break;

					case MachineData::MODE_IDLE_AWAITING_EVENT:
						--m_nIdleAwaitingEvent;
						break;
				}
				checkDoStop();
				break;

			case MachineData::STATE_WAITING:
				--m_nTotal;
				aIt->second.m_eState = MachineData::STATE_STOPPED;
				switch (aIt->second.m_eMode)
				{
					case MachineData::MODE_IDLE:
						--m_nIdle;
						break;

					case MachineData::MODE_IDLE_AWAITING_EVENT:
						--m_nIdleAwaitingEvent;
						break;
				}
				break;
		}
	}
}

//============================================================================
void MachineControl::eventOccured()
{
	vos::OGuard aGuard(m_aMutex);
	if (!m_bEventOccured)
	{
		m_bEventOccured = true;
		checkTryStop();
	}
}

//============================================================================
//
//  Machine
//
//============================================================================

Machine::Machine(MachineControl * pTheControl):
	m_pControl(pTheControl)
{
	VOS_ENSURE(m_pControl, "ucb_main::Machine::Machine(): Null control");
}
