/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: b2dmbase.cxx,v $
 *
 *  $Revision: 1.6 $
 *
 *  last change: $Author: obo $ $Date: 2006/09/17 15:34:17 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

// MARKER(update_precomp.py): autogen include statement, do not remove
#include "precompiled_goodies.hxx"

#ifndef _B2D_MBASE_HXX
#include "b2dmbase.hxx"
#endif

#ifndef _TOOLS_DEBUG_HXX
#include <tools/debug.hxx>
#endif

#ifndef _SV_WINDOW_HXX
#include <vcl/window.hxx>
#endif

#ifndef _SV_SALBTYPE_HXX
#include <vcl/salbtype.hxx>
#endif

#ifndef _BXD_INTERPOLATOR_HXX
#include "bxdintpo.hxx"
#endif

//************************************************************
//   Basic geometric point object
//************************************************************

BOOL B2dIAOElement::IsInside(const Region& rRegion)
{
	Point aPoint(GetPositionX(), GetPositionY());
	if(IsPixel())
	{
		return rRegion.IsInside(aPoint);
	}
	else if(IsBitmap())
	{
		Rectangle aRectangle(aPoint, ((B2dIAOBitmap*)this)->GetBitmap().GetSizePixel());
		return rRegion.IsInside(aRectangle);
	}
//	else if(IsBmpRef())
//	{
//		Rectangle aRectangle(aPoint, ((B2dIAOBmpRef*)this)->GetBitmap()->GetSizePixel());
//		return rRegion.IsInside(aRectangle);
//	}
	else
	{
		Size aSize;

		if(((B2dIAOBmpVDev*)this)->GetEntry())
			aSize = ((B2dIAOBmpVDev*)this)->GetEntry()->GetSize();

		Rectangle aRectangle(aPoint, aSize);
		return rRegion.IsInside(aRectangle);
	}
}

BOOL B2dIAOElement::IsOutside(const Region& rRegion)
{
	Point aPoint(GetPositionX(), GetPositionY());
	if(IsPixel())
	{
		return !rRegion.IsInside(aPoint);
	}
	else if(IsBitmap())
	{
		Region aRegion(Rectangle(aPoint, ((B2dIAOBitmap*)this)->GetBitmap().GetSizePixel()));
		aRegion.Intersect(rRegion);
		return aRegion.IsEmpty();
	}
//	else if(IsBmpRef())
//	{
//		Region aRegion(Rectangle(aPoint, ((B2dIAOBmpRef*)this)->GetBitmap()->GetSizePixel()));
//		aRegion.Intersect(rRegion);
//		return aRegion.IsEmpty();
//	}
	else
	{
		Size aSize;

		if(((B2dIAOBmpVDev*)this)->GetEntry())
			aSize = ((B2dIAOBmpVDev*)this)->GetEntry()->GetSize();

		Region aRegion(Rectangle(aPoint, aSize));
		aRegion.Intersect(rRegion);
		return aRegion.IsEmpty();
	}
}

//************************************************************
//   Geometric Point Object provider
//************************************************************

void B2dIAOPixelProvider::CreateNewEntries()
{
	B2dIAOPixel* pNew = new B2dIAOPixel[B2D_NUM_IAOPIXEL_TO_ALLOCATE];
	DBG_ASSERT(pNew, "Got NO new B2dIAOElements in CreateNewEntries()");
	aMemList.Insert(pNew);
	for(UINT32 a=0;a<B2D_NUM_IAOPIXEL_TO_ALLOCATE;a++)
	{
		pNew->nType = B2D_IAO_ELEMENT_PIXEL;
		PutB2dIAOPixel(pNew++);
	}
}

void B2dIAOPixelProvider::TryToReleaseSomeMemory()
{
	if(pFirst)
	{
		UINT32 nListCount = aMemList.Count();
		for(UINT32 a = 0; pFirst && nListCount && a < nListCount; )
		{
			B2dIAOPixel* pObj = (B2dIAOPixel*)aMemList.GetObject(a);
			B2dIAOPixel* pSegStart = pObj;
			B2dIAOPixel* pSegEnd = pObj + B2D_NUM_IAOPIXEL_TO_ALLOCATE;
			UINT32 nCountInside = 0;
			B2dIAOPixel* pAct = pFirst;
			BOOL bDidRemoveOne(FALSE);

			while(pAct && nCountInside < B2D_NUM_IAOPIXEL_TO_ALLOCATE)
			{
				if(pAct >= pSegStart && pAct < pSegEnd)
					nCountInside++;
				pAct = (B2dIAOPixel*)pAct->pNext;
			}

			if(nCountInside == B2D_NUM_IAOPIXEL_TO_ALLOCATE)
			{
				// ALL objects of this memory part are free at the moment,
				// it can be deleted, do it.
				nCountInside = 0;
				pAct = pFirst;
				B2dIAOPixel* pPrev = NULL;

				// take objects out of the list
				while(pAct && nCountInside < B2D_NUM_IAOPIXEL_TO_ALLOCATE)
				{
					if(pAct >= pSegStart && pAct < pSegEnd)
					{
						if(pPrev)
						{
							pPrev->pNext = pAct->pNext;
						}
						else
						{
							pFirst = (B2dIAOPixel*)pAct->pNext;
						}

						nCountInside++;
					}
					else
					{
						pPrev = pAct;
					}

					pAct = (B2dIAOPixel*)pAct->pNext;
				}

				// remove memory part from the list
				aMemList.Remove(pObj);
				delete[] pObj;
				bDidRemoveOne = TRUE;
			}
			
			if(bDidRemoveOne)
			{
				// correct loop parameters
				nListCount = aMemList.Count();
			}
			else
			{
				// next list entry
				a++;
			}
		}
	}
}

//************************************************************
//   Geometric Bitmap Object provider
//************************************************************

void B2dIAOBitmapProvider::CreateNewEntries()
{
	B2dIAOBitmap* pNew = new B2dIAOBitmap[B2D_NUM_IAOBITMAP_TO_ALLOCATE];
	DBG_ASSERT(pNew, "Got NO new B2dIAOElements in CreateNewEntries()");
	aMemList.Insert(pNew);
	for(UINT32 a=0;a<B2D_NUM_IAOBITMAP_TO_ALLOCATE;a++)
	{
		pNew->nType = B2D_IAO_ELEMENT_BITMAP;
		PutB2dIAOBitmap(pNew++);
	}
}

void B2dIAOBitmapProvider::TryToReleaseSomeMemory()
{
	if(pFirst)
	{
		UINT32 nListCount = aMemList.Count();
		for(UINT32 a = 0; pFirst && nListCount && a < nListCount; )
		{
			B2dIAOBitmap* pObj = (B2dIAOBitmap*)aMemList.GetObject(a);
			B2dIAOBitmap* pSegStart = pObj;
			B2dIAOBitmap* pSegEnd = pObj + B2D_NUM_IAOBITMAP_TO_ALLOCATE;
			UINT32 nCountInside = 0;
			B2dIAOBitmap* pAct = pFirst;
			BOOL bDidRemoveOne(FALSE);

			while(pAct && nCountInside < B2D_NUM_IAOBITMAP_TO_ALLOCATE)
			{
				if(pAct >= pSegStart && pAct < pSegEnd)
					nCountInside++;
				pAct = (B2dIAOBitmap*)pAct->pNext;
			}

			if(nCountInside == B2D_NUM_IAOBITMAP_TO_ALLOCATE)
			{
				// ALL objects of this memory part are free at the moment,
				// it can be deleted, do it.
				nCountInside = 0;
				pAct = pFirst;
				B2dIAOBitmap* pPrev = NULL;

				// take objects out of the list
				while(pAct && nCountInside < B2D_NUM_IAOBITMAP_TO_ALLOCATE)
				{
					if(pAct >= pSegStart && pAct < pSegEnd)
					{
						if(pPrev)
						{
							pPrev->pNext = pAct->pNext;
						}
						else
						{
							pFirst = (B2dIAOBitmap*)pAct->pNext;
						}

						nCountInside++;
					}
					else
					{
						pPrev = pAct;
					}

					pAct = (B2dIAOBitmap*)pAct->pNext;
				}

				// remove memory part from the list
				aMemList.Remove(pObj);
				delete[] pObj;
				bDidRemoveOne = TRUE;
			}
			
			if(bDidRemoveOne)
			{
				// correct loop parameters
				nListCount = aMemList.Count();
			}
			else
			{
				// next list entry
				a++;
			}
		}
	}
}

//************************************************************
//   Geometric BitmapReference Object provider
//************************************************************

//void B2dIAOBmpRefProvider::CreateNewEntries()
//{
//	B2dIAOBmpRef* pNew = new B2dIAOBmpRef[B2D_NUM_IAOBMPREF_TO_ALLOCATE];
//	DBG_ASSERT(pNew, "Got NO new B2dIAOElements in CreateNewEntries()");
//	aMemList.Insert(pNew);
//	for(UINT32 a=0;a<B2D_NUM_IAOBMPREF_TO_ALLOCATE;a++)
//	{
//		pNew->nType = B2D_IAO_ELEMENT_BITMAPREFERENCE;
//		PutB2dIAOBmpRef(pNew++);
//	}
//}

//void B2dIAOBmpRefProvider::TryToReleaseSomeMemory()
//{
//	if(pFirst)
//	{
//		UINT32 nListCount = aMemList.Count();
//		for(UINT32 a = 0; pFirst && nListCount && a < nListCount; )
//		{
//			B2dIAOBmpRef* pObj = (B2dIAOBmpRef*)aMemList.GetObject(a);
//			B2dIAOBmpRef* pSegStart = pObj;
//			B2dIAOBmpRef* pSegEnd = pObj + B2D_NUM_IAOBMPREF_TO_ALLOCATE;
//			UINT32 nCountInside = 0;
//			B2dIAOBmpRef* pAct = pFirst;
//			BOOL bDidRemoveOne(FALSE);
//
//			while(pAct && nCountInside < B2D_NUM_IAOBMPREF_TO_ALLOCATE)
//			{
//				if(pAct >= pSegStart && pAct < pSegEnd)
//					nCountInside++;
//				pAct = (B2dIAOBmpRef*)pAct->pNext;
//			}
//
//			if(nCountInside == B2D_NUM_IAOBMPREF_TO_ALLOCATE)
//			{
//				// ALL objects of this memory part are free at the moment,
//				// it can be deleted, do it.
//				nCountInside = 0;
//				pAct = pFirst;
//				B2dIAOBmpRef* pPrev = NULL;
//
//				// take objects out of the list
//				while(pAct && nCountInside < B2D_NUM_IAOBMPREF_TO_ALLOCATE)
//				{
//					if(pAct >= pSegStart && pAct < pSegEnd)
//					{
//						if(pPrev)
//						{
//							pPrev->pNext = pAct->pNext;
//						}
//						else
//						{
//							pFirst = (B2dIAOBmpRef*)pAct->pNext;
//						}
//
//						nCountInside++;
//					}
//					else
//					{
//						pPrev = pAct;
//					}
//
//					pAct = (B2dIAOBmpRef*)pAct->pNext;
//				}
//
//				// remove memory part from the list
//				aMemList.Remove(pObj);
//				delete[] pObj;
//				bDidRemoveOne = TRUE;
//			}
//			
//			if(bDidRemoveOne)
//			{
//				// correct loop parameters
//				nListCount = aMemList.Count();
//			}
//			else
//			{
//				// next list entry
//				a++;
//			}
//		}
//	}
//}

//************************************************************
//   Geometric BmpVDev Object provider
//************************************************************

void B2dIAOBmpVDevProvider::CreateNewEntries()
{
	B2dIAOBmpVDev* pNew = new B2dIAOBmpVDev[B2D_NUM_IAOBMPVDEV_TO_ALLOCATE];
	DBG_ASSERT(pNew, "Got NO new B2dIAOElements in CreateNewEntries()");
	aMemList.Insert(pNew);
	for(UINT32 a=0;a<B2D_NUM_IAOBMPVDEV_TO_ALLOCATE;a++)
	{
		pNew->nType = B2D_IAO_ELEMENT_BMPVDEV;
		PutB2dIAOBmpVDev(pNew++);
	}
}

void B2dIAOBmpVDevProvider::TryToReleaseSomeMemory()
{
	if(pFirst)
	{
		UINT32 nListCount = aMemList.Count();
		for(UINT32 a = 0; pFirst && nListCount && a < nListCount; )
		{
			B2dIAOBmpVDev* pObj = (B2dIAOBmpVDev*)aMemList.GetObject(a);
			B2dIAOBmpVDev* pSegStart = pObj;
			B2dIAOBmpVDev* pSegEnd = pObj + B2D_NUM_IAOBMPVDEV_TO_ALLOCATE;
			UINT32 nCountInside = 0;
			B2dIAOBmpVDev* pAct = pFirst;
			BOOL bDidRemoveOne(FALSE);

			while(pAct && nCountInside < B2D_NUM_IAOBMPVDEV_TO_ALLOCATE)
			{
				if(pAct >= pSegStart && pAct < pSegEnd)
					nCountInside++;
				pAct = (B2dIAOBmpVDev*)pAct->pNext;
			}

			if(nCountInside == B2D_NUM_IAOBMPVDEV_TO_ALLOCATE)
			{
				// ALL objects of this memory part are free at the moment,
				// it can be deleted, do it.
				nCountInside = 0;
				pAct = pFirst;
				B2dIAOBmpVDev* pPrev = NULL;

				// take objects out of the list
				while(pAct && nCountInside < B2D_NUM_IAOBMPVDEV_TO_ALLOCATE)
				{
					if(pAct >= pSegStart && pAct < pSegEnd)
					{
						if(pPrev)
						{
							pPrev->pNext = pAct->pNext;
						}
						else
						{
							pFirst = (B2dIAOBmpVDev*)pAct->pNext;
						}

						nCountInside++;
					}
					else
					{
						pPrev = pAct;
					}

					pAct = (B2dIAOBmpVDev*)pAct->pNext;
				}

				// remove memory part from the list
				aMemList.Remove(pObj);
				delete[] pObj;
				bDidRemoveOne = TRUE;
			}
			
			if(bDidRemoveOne)
			{
				// correct loop parameters
				nListCount = aMemList.Count();
			}
			else
			{
				// next list entry
				a++;
			}
		}
	}
}

//************************************************************
//   Basis-InterActionObject (IAO)
//************************************************************

TYPEINIT0(B2dIAObject);

B2dIAObject::B2dIAObject(B2dIAOManager* pMan, Point aBasPos, Color aBasCol)
:	pGeomPositions(NULL),
	aBasePosition(GetGeomPositions(), aBasPos)
{
	// Set Manager and add to chain
	pManager = pMan;
	DBG_ASSERT(pManager, "NO Manager for new constructed IAO");

	// Insert at manager, do NOT set pNext, pPrev afterwards (!)
	pManager->InsertIAO(this);

	// Initialize Geometry
	pGeometry = NULL;

	// Initialize BaseColor
	aBaseColor = aBasCol;

	// Initialize BOOLs
	bVisible = TRUE;
	bGeometryValid = FALSE;
	bBaseRectValid = FALSE;
	bAnimationOn = FALSE;
	bAnimRegisteredAtManager = FALSE;
	bIsHittable = TRUE;
}

B2dIAObject::~B2dIAObject()
{
	// old positions should be invalidated at this point, do this before
	// calling the destructor
	// InvalidateRectangle(); 

	// eventually get out of animation
	if(bAnimRegisteredAtManager)
	{
		bVisible = FALSE;
		CheckAnimationState();
	}

	// Get rid of Geometry
	while(pGeometry)
	{
		// remove from chain
		B2dIAOElement* pEntry = pGeometry;
		pGeometry = pEntry->RemFirstFromList(pGeometry);
		if(pManager)
			pManager->PutB2dIAOElement(pEntry);
	}

	// mark geometry as not valid
	bGeometryValid = FALSE;

	// Remove from Manager (if still exists)
	if(pManager)
		pManager->RemoveIAO(this);
}

void B2dIAObject::InvalidateRectangle()
{ 
	if(pManager)
		pManager->InvalidateRectangle(GetBaseRect()); 
}

BOOL B2dIAObject::IsHit(const Point& rPixelPos, UINT16 nTol) const
{
	BOOL bRetval(FALSE);

	if(IsHittable())
	{
		if(nTol)
		{
			Rectangle aRect = GetBaseRect();
			aRect.Left() -= nTol;
			aRect.Right() += nTol;
			aRect.Top() -= nTol;
			aRect.Bottom() += nTol;
			bRetval = aRect.IsInside(rPixelPos);
		}
		else
		{
			bRetval = GetBaseRect().IsInside(rPixelPos);
		}
	}
	return bRetval;
}

void B2dIAObject::AnimationStep(UINT32 /*nAnimCounter*/)
{
}

void B2dIAObject::ApplyDevice(OutputDevice *pOut)
{
	DBG_ASSERT(pOut, "NO OutDev in ApplyDevice() call");

	B2dPositionEntity* pAct = pGeomPositions;
	BOOL bChange(FALSE);

	while(pAct)
	{
		Point aNewPixPos(pOut->LogicToPixel(pAct->GetPosition()));
		if(aNewPixPos != pAct->GetPositionPixel())
		{
			// remember change
			bChange = TRUE;

			// remember new value
			pAct->SetPositionPixel(aNewPixPos);
		}

		// get next point
		pAct = pAct->GetNext();
	}

	if(bChange)
	{
		// throw away geometry and BaseRect
		GeometryChange(); 
		BaseRectChange(); 
	}
}

const Rectangle& B2dIAObject::GetBaseRect() const
{
	if(!bBaseRectValid)
	{
		// needs to be calced, cast to non-const
		((B2dIAObject*)this)->CreateBaseRect();

		// set to valid
		((B2dIAObject*)this)->bBaseRectValid = TRUE;
	}

	return aBaseRect;
}

B2dIAOElement* B2dIAObject::GetGeometry()
{
	if(!bGeometryValid)
	{
		// needs to be build
		CreateGeometry();

		// set to vaild
		bGeometryValid = TRUE;

		// invalidate new regions
		InvalidateRectangle(); 
	}

	return pGeometry;
}

BOOL B2dIAObject::AddPixel(const Point& rPos, const Color& rCol)
{
	if(pManager && pManager->GetClipRegion().IsInside(rPos))
	{
		// get new element from Manager
		B2dIAOPixel* pNew = pManager->GetB2dIAOPixel();
		DBG_ASSERT(pNew, "NO Manager for new constructed IAO");

		// fill values
		pNew->SetPosition(rPos);
		pNew->SetColor(rCol);

		// add to chain
		pGeometry = pNew->AddToList(pGeometry);

		return TRUE;
	}
	return FALSE;
}

BOOL B2dIAObject::AddBitmap(const Point& rPos, const BitmapEx& rBmp)
{
	if(pManager)
	{
		// get new element from Manager
		B2dIAOBitmap* pNew = pManager->GetB2dIAOBitmap();
		DBG_ASSERT(pNew, "NO Manager for new constructed IAO");

		// fill values
		pNew->SetPosition(rPos);
		pNew->SetBitmap(rBmp);

		// add to chain
		pGeometry = pNew->AddToList(pGeometry);

		return TRUE;
	}
	return FALSE;
}

//BOOL B2dIAObject::AddBmpRef(const Point& rPos, BitmapEx* pBmp)
//{
//	if(pManager)
//	{
//		// get new element from Manager
//		B2dIAOBmpRef* pNew = pManager->GetB2dIAOBmpRef();
//		DBG_ASSERT(pNew, "NO Manager for new constructed IAO");
//
//		// fill values
//		pNew->SetPosition(rPos);
//		pNew->SetBitmap(pBmp);
//
//		// add to chain
//		pGeometry = pNew->AddToList(pGeometry);
//
//		return TRUE;
//	}
//	return FALSE;
//}

BOOL B2dIAObject::AddLine(const Point& rPos1, const Point& rPos2)
{
	INT32 nDx = rPos1.X() - rPos2.X();
	INT32 nDy = rPos1.Y() - rPos2.Y();
	UINT32 nPixelCount(0);

	if(nDx || nDy)
	{
		if(abs(nDx) > abs(nDy))
		{
			double m = (double)nDy / (double)nDx;
			double y = (double)rPos2.Y();

			if(nDx >= 0)
			{
				for(INT32 x = rPos2.X(); x < rPos1.X(); x++)
				{
					Point aPos(x, (INT32)y);
					AddLinePixel(aPos, nPixelCount++);
					y += m;
				}
			}
			else
			{
				for(INT32 x = rPos2.X(); x > rPos1.X(); x--)
				{
					Point aPos(x, (INT32)y);
					AddLinePixel(aPos, nPixelCount++);
					y -= m;
				}
			}
		}
		else
		{
			double m = (double)nDx / (double)nDy;
			double x = (double)rPos2.X();

			if(nDy >= 0)
			{
				for(INT32 y = rPos2.Y(); y < rPos1.Y(); y++)
				{
					Point aPos((INT32)x, y);
					AddLinePixel(aPos, nPixelCount++);
					x += m;
				}
			}
			else
			{
				for(INT32 y = rPos2.Y(); y > rPos1.Y(); y--)
				{
					Point aPos((INT32)x, y);
					AddLinePixel(aPos, nPixelCount++);
					x -= m;
				}
			}
		}
	}
	return TRUE;
}

BOOL B2dIAObject::AddLinePixel(const Point& rPos, UINT32 /*nPixelNum*/)
{
	AddPixel(rPos, GetBaseColor());
	return TRUE;
}

BOOL B2dIAObject::AddTriangle(const Point& rPos1, const Point& rPos2, const Point& rPos3)
{
	Point aEntTop, aEntLeft, aEntRight;

	// Sort points. Uppest to pEntTop
	if(rPos1.Y() < rPos2.Y() && rPos1.Y() < rPos3.Y())
	{
		// rPos1 is on top
		aEntTop = rPos1;

		// set Left, Right without compare (just to fill)
		aEntRight = rPos2;
		aEntLeft = rPos3;
	}
	else
	{
		if(rPos2.Y() < rPos3.Y())
		{
			// rPos2 is on top
			aEntTop = rPos2;

			// set Left, Right without compare (just to fill)
			aEntRight = rPos1;
			aEntLeft = rPos3;
		}
		else
		{
			// rPos3 is on top
			aEntTop = rPos3;

			// set Left, Right without compare (just to fill)
			aEntRight = rPos1;
			aEntLeft = rPos2;
		}
	}

	// sort left, right
	INT32 nDeltaYLeft = aEntLeft.Y() - aEntTop.Y();
	INT32 nDeltaYRight = aEntRight.Y() - aEntTop.Y();
	INT32 nXLineDelta;
	Point aOutPoint;

	if((aEntLeft.X() - aEntTop.X()) * nDeltaYRight - nDeltaYLeft * (aEntRight.X() - aEntTop.X()) > 0L)
	{
		// exchange
		aOutPoint.Y() = aEntLeft.X(); aEntLeft.X() = aEntRight.X(); aEntRight.X() = aOutPoint.Y();
		aOutPoint.Y() = aEntLeft.Y(); aEntLeft.Y() = aEntRight.Y(); aEntRight.Y() = aOutPoint.Y();

		// Deltas
		aOutPoint.Y() = nDeltaYLeft; nDeltaYLeft = nDeltaYRight; nDeltaYRight = aOutPoint.Y();
	}

	// set YStart, load interpolators
	aOutPoint.Y() = aEntTop.Y() + 1;
	BxdInterpolator aIntXPosLeft(aEntTop.X(), aEntLeft.X(), nDeltaYLeft);
	BxdInterpolator aIntXPosRight(aEntTop.X(), aEntRight.X(), nDeltaYRight);

	// one step to start
	if(nDeltaYRight)
		nDeltaYRight--;
	aIntXPosRight.Increment();

	if(nDeltaYLeft)
		nDeltaYLeft--;
	aIntXPosLeft.Increment();

	// loop 1st phase
	while(nDeltaYLeft && nDeltaYRight)
	{
		// draw from left to right
		aOutPoint.X() = aIntXPosLeft.GetLongValue();
		nXLineDelta = aIntXPosRight.GetLongValue() - aOutPoint.X();

		if(nXLineDelta > 0)
		{
			while(nXLineDelta--)
			{
				// output pixel
				AddTrianglePixel(aOutPoint);

				// next column
				aOutPoint.X()++;
			}
		}

		// next line right
		nDeltaYRight--;
		aIntXPosRight.Increment();
		nDeltaYLeft--;
		aIntXPosLeft.Increment();
		aOutPoint.Y()++;
	}

	// prepare for 2nd phase
	if(nDeltaYLeft)
	{
		// right is at end, load again with rest of left
		nDeltaYRight = nDeltaYLeft;
		aIntXPosRight.Load(aEntRight.X(), aEntLeft.X(), nDeltaYRight);
		nDeltaYRight--;
		aIntXPosRight.Increment();
	}
	else if(nDeltaYRight)
	{
		// left is at end, load again with rest of right
		aIntXPosLeft.Load(aEntLeft.X(), aEntRight.X(), nDeltaYRight);
		nDeltaYRight--;
		aIntXPosLeft.Increment();
	}

	// loop 2nd phase
	while(nDeltaYRight)
	{
		// draw from left to right
		aOutPoint.X() = aIntXPosLeft.GetLongValue();
		nXLineDelta = aIntXPosRight.GetLongValue() - aOutPoint.X();

		if(nXLineDelta > 0)
		{
			while(nXLineDelta--)
			{
				// output pixel
				AddTrianglePixel(aOutPoint);

				// next column
				aOutPoint.X()++;
			}
		}


		nDeltaYRight--;
		aIntXPosRight.Increment();
		aIntXPosLeft.Increment();
		aOutPoint.Y()++;
	}

	return TRUE;
}

BOOL B2dIAObject::AddTrianglePixel(const Point& rPos)
{
	AddPixel(rPos, GetBaseColor());
	return TRUE;
}

void B2dIAObject::FreeGeometry()
{
	if(bGeometryValid && pManager)
	{
		// tell manager about change
		InvalidateRectangle();

		// give back all used B2dIAOElements to the Manager
		while(pGeometry)
		{
			// remove from chain
			B2dIAOElement* pEntry = pGeometry;
			pGeometry = pEntry->RemFirstFromList(pGeometry);
			pManager->PutB2dIAOElement(pEntry);
		}

		// mark geometry as not valid
		bGeometryValid = FALSE;
		pGeometry = NULL;
	}
}

void B2dIAObject::Transform(const Matrix3D& rMat)
{
	B2dPositionEntity* pAct = pGeomPositions;
	while(pAct)
	{
		// transform this position
		Point3D aPnt(pAct->GetPosition());
		aPnt *= rMat;
		aPnt.Homogenize();
		pAct->SetPosition(Point(FRound(aPnt.X()), FRound(aPnt.Y())));

		// get next point
		pAct = pAct->GetNext();
	}

	// throw away geometry and BaseRect
	GeometryChange(); 
	BaseRectChange(); 
}

void B2dIAObject::SetBasePosition(Point aNew) 
{ 
	if(aNew != aBasePosition.GetPosition()) 
	{ 
		// throw away geometry
		GeometryChange(); 
		BaseRectChange(); 

		// save new position
		aBasePosition.SetPosition(aNew); 
	}
}

void B2dIAObject::SetVisible(BOOL bNew) 
{ 
	if(bNew != bVisible) 
	{
		// throw away geometry
		GeometryChange(); 

		// remember new state
		bVisible = bNew; 

		// check if animation state needs to be told to the manager
		CheckAnimationState();
	}
}

void B2dIAObject::SetBaseColor(Color aNew) 
{ 
	if(aNew != aBaseColor) 
	{
		// throw away geometry
		GeometryChange(); 

		// remember new color
		aBaseColor = aNew; 
	}
}

void B2dIAObject::SetAnimation(BOOL bNew)
{
	if(bNew != bAnimationOn)
	{
		// remember new value
		bAnimationOn = bNew;

		// check if animation state needs to be told to the manager
		CheckAnimationState();
	}
}

void B2dIAObject::CheckAnimationState()
{
	if(bVisible && bAnimationOn && !bAnimRegisteredAtManager)
	{
		// register at Manager for animation
		if(pManager)
			pManager->RegisterAnimatedObject(this);
		
		// remember new state
		bAnimRegisteredAtManager = TRUE;
	}
	else
	{
		if(bAnimRegisteredAtManager)
		{
			// free from manager
			if(pManager)
				pManager->UnregisterAnimatedObject(this);

			// remember new state
			bAnimRegisteredAtManager = FALSE;
		}
	}
}

//************************************************************
//   IAOGroup
//************************************************************

B2dIAOGroup::B2dIAOGroup()
{
	// init pointers
	pIAO = NULL;
	pIAOList = NULL;
}

B2dIAOGroup::~B2dIAOGroup()
{
	// DELETE all grouped IAOs
	Delete();
}

void B2dIAOGroup::Delete()
{
	if(pIAOList)
	{
		for(UINT32 a=0;a<pIAOList->Count();a++)
		{
			B2dIAObject* pObj = (B2dIAObject*)(pIAOList->GetObject(a));
			pObj->InvalidateRectangle();
			delete pObj;
		}
		pIAOList->Clear();
		delete pIAOList;
		pIAOList = NULL;
	}
	else if(pIAO)
	{
		pIAO->InvalidateRectangle();
		delete pIAO;
		pIAO = NULL;
	}
}

void B2dIAOGroup::Clear()
{
	if(pIAOList)
	{
		pIAOList->Clear();
		delete pIAOList;
		pIAOList = NULL;
	}
	else if(pIAO)
	{
		pIAO = NULL;
	}
}

BOOL B2dIAOGroup::InsertIAO(B2dIAObject* pNew)
{
	DBG_ASSERT(pNew, "Insert with empty element!");

	if(pIAOList)
	{
		// insert new element
		pIAOList->Insert(pNew, CONTAINER_APPEND);
	}
	else if(pIAO)
	{
		// 2nd element, create a list
		DBG_ASSERT(!pIAOList, "Want to create list, but it does exist!");
		pIAOList = new Container(64, 16, 16);
		DBG_ASSERT(pIAOList, "Could not create new Container!");

		// insert existing element
		pIAOList->Insert(pIAO);
		pIAO = NULL;

		// insert new element
		pIAOList->Insert(pNew, CONTAINER_APPEND);
	}
	else
	{
		// first element
		pIAO = pNew;
	}

	return TRUE;
}

BOOL B2dIAOGroup::RemoveIAO(B2dIAObject* pOld)
{
	DBG_ASSERT(pOld, "Remove with empty element!");
	
	if(pIAOList)
	{
		// remove from list
		B2dIAObject* pRem = (B2dIAObject*)pIAOList->Remove(pOld);
		BOOL bRetval = pRem ? TRUE : FALSE;

		if(bRetval)
		{
			if(pIAOList->Count() == 1L)
			{
				// correct handling, move last contained IAO
				// to the solo pointer
				UINT32 nNum(0L);
				pIAO = (B2dIAObject*)pIAOList->Remove(nNum);
				DBG_ASSERT(pIAO, "Remove of last element failed!");

				// destroy list, it's no longer needed
				pIAOList->Clear();
				delete pIAOList;
				pIAOList = NULL;
			}
		}

		return bRetval;
	}
	else if(pIAO)
	{
		// only one element
		if(pIAO == pOld)
		{
			pIAO = NULL;
			return TRUE;
		}
	}

	return FALSE;
}

UINT32 B2dIAOGroup::GetIAOCount() const
{
	if(pIAOList)
	{
		return pIAOList->Count();
	}
	else if(pIAO)
	{
		return 1L;
	}
	else
	{
		return 0L;
	}
}

B2dIAObject* B2dIAOGroup::GetIAObject(UINT32 nNum) const
{
	// nNum>0, access list if accessible
	if(pIAOList)
		return (B2dIAObject*)(pIAOList->GetObject(nNum));
	else if(!nNum)
		return pIAO;
	else
		return NULL;
}

BOOL B2dIAOGroup::IsHit(const Point& rPixelPos, UINT16 nTol) const
{
	if(pIAOList)
	{
		for(UINT32 a=0;a<pIAOList->Count();a++)
		{
			B2dIAObject* pTestObj = (B2dIAObject*)pIAOList->GetObject(a);
			if(pTestObj && pTestObj->IsHit(rPixelPos, nTol))
				return TRUE;
		}
	}
	else if(pIAO)
	{
		return pIAO->IsHit(rPixelPos, nTol);
	}

	return FALSE;
}

//************************************************************
//   InterActionObject Manager
//************************************************************

B2dIAOPixelProvider B2dIAOManager::aPixelProvider;
B2dIAOBitmapProvider B2dIAOManager::aBitmapProvider;
//B2dIAOBmpRefProvider B2dIAOManager::aBmpRefProvider;
B2dIAOBmpVDevProvider B2dIAOManager::aBmpVDevProvider;

B2dIAOManager::B2dIAOManager(Window* pTarget)
:	pWindow(pTarget),
	aTmpPoly(B2D_NUM_IAOPIXEL_PER_IOOPERATION),
	aAnimatedObjectList(64, 4, 4)
{
	DBG_ASSERT(pWindow, "contruction of B2dIAOManager with NULL-Pointer as Window");

	// Init pointer
	pIAObjectList = NULL;
	pIAObjectListEnd = NULL;
	pSaveList = NULL;
	nIAONumber = 0L;

	// Init BOOLs
	bVisible = TRUE;
	bChanged = FALSE;

	// mechanism for using Array-accesses for Read/WritePixel
	pTmpColor = NULL;
	nNextFree = 0;

	// Init timer for animation
	aAnimator.SetTimeout(10);
	aAnimator.SetTimeoutHdl(LINK(this, B2dIAOManager, AnimatorHdl));
	bAnimationOn = TRUE;
	bTimerIsOn = FALSE;
	nAnimatorCount = 0L;
}

B2dIAOManager::~B2dIAOManager()
{
	// need to stop timer?
	if(bTimerIsOn)
	{
		bVisible = FALSE;
		CheckTimerState();
	}

	// Remove all IAO's still remaining, do NOT delete,
	// this needs to be done by owner
	while(pIAObjectList)
	{
		// take back all B2dIAOElements
		pIAObjectList->FreeGeometry();

		// remove from here and set manager to NULL
		RemoveIAO(pIAObjectList);
	}
	nIAONumber = 0L;

	// remove all still holded save objects
	ForgetBackground();

	// Remove PixelArray if used
	if(pTmpColor)
		delete[] pTmpColor;

	// try to get rid of some memory in the ElementProviders
	aPixelProvider.TryToReleaseSomeMemory();
	aBitmapProvider.TryToReleaseSomeMemory();
//	aBmpRefProvider.TryToReleaseSomeMemory();
	aBmpVDevProvider.TryToReleaseSomeMemory();
}

IMPL_LINK_INLINE_START(B2dIAOManager, AnimatorHdl ,AutoTimer*, EMPTYARG)
{
	// increase count to make changes visible
	nAnimatorCount++;

	// call every animated object; these can do changes according to AnimatorCount
	for(void* pObj = aAnimatedObjectList.First(); pObj; pObj = aAnimatedObjectList.Next())
		((B2dIAObject*)pObj)->AnimationStep(nAnimatorCount);

	// when changes occurred, this leads to displaying them
	UpdateDisplay();
	return 0;
}
IMPL_LINK_INLINE_END(B2dIAOManager, AnimatorHdl ,AutoTimer*, EMPTYARG)

BOOL B2dIAOManager::InsertIAO(B2dIAObject* pNew)
{
	// add to the end of chain to preserve display order in paint
	DBG_ASSERT(pNew, "call to InsertIAO with NULL-Pointer");
	DBG_ASSERT(pNew->pManager == this, "call to InsertIAO to wrong Manager");

	if(pIAObjectListEnd)
	{
		// new element, add to end
		pNew->pNext = pIAObjectListEnd->pNext;
		pNew->pPrev = pIAObjectListEnd;
		pIAObjectListEnd->pNext = pNew;
		pIAObjectListEnd = pNew;
	}
	else
	{
		// first element
		pNew->pNext = pNew->pPrev = NULL;
		pIAObjectListEnd = pIAObjectList = pNew;
	}
	
	nIAONumber++;
	bChanged = TRUE;

	return TRUE;
}

BOOL B2dIAOManager::RemoveIAO(B2dIAObject* pOld)
{
	// Remove from chain
	DBG_ASSERT(pOld, "call to RemoveIAO with NULL-Pointer");
	DBG_ASSERT(pOld->pManager == this, "call to RemoveIAO to wrong Manager");
	
	if(pOld->pPrev)
		pOld->pPrev->pNext = pOld->pNext;

	if(pOld->pNext)
		pOld->pNext->pPrev = pOld->pPrev;

	if(pOld == pIAObjectList)
		pIAObjectList = pOld->pNext;

	if(pOld == pIAObjectListEnd)
		pIAObjectListEnd = pOld->pPrev;

	pOld->pManager = NULL;
	nIAONumber--;
	bChanged = TRUE;

	return TRUE;
}

void B2dIAOManager::ApplyClipRegion(const Region& rClipRegion)
{
	// remember local clip region
	if(rClipRegion != GetClipRegion())
	{
		// remember new value
		aClipRegion = rClipRegion;

		// Invalidate all geometry
		B2dIAObject* pActIAO = GetIAObjectList();
		while(pActIAO)
		{
			pActIAO->GeometryChange();
			pActIAO = pActIAO->GetNext();
		}
	}
}

void B2dIAOManager::MoveSavedElementsPixel(const Point& rDelta)
{
	B2dIAOElement* pAct = GetSave();
	while(pAct)
	{
		// move entry
		pAct->Move(rDelta);

		// next entry
		pAct = pAct->GetNext();
	}
}

void B2dIAOManager::ForgetBackground()
{
	B2dIAOElement* pAct = GetSave();
	while(pAct)
	{
		SetSave(pAct->RemFirstFromList(pAct));

		// support for VDev cached backgrounds
		if(pAct->IsBmpVDev())
		{
			VDevCacheEntry* pEntry = ((B2dIAOBmpVDev*)pAct)->GetEntry();
			if(pEntry)
			{
				GetVDevCache().Free(pEntry);
				((B2dIAOBmpVDev*)pAct)->SetEntry(0L);
			}
		}

		PutB2dIAOElement(pAct);
		pAct = GetSave();
	}
}

BOOL B2dIAOManager::SaveBackground(const Region& rClipRegion)
{
	B2dIAObject* pActIAO = GetIAObjectList();
	BOOL bRetval(FALSE);

	// local caching of to be saved pixel colors
	B2dIAOElement* pPixelList = NULL;
	UINT32 nNumPixelList = 0;

	while(pActIAO)
	{
		// only visible IAO's
		if(pActIAO->IsVisible())
		{
			// only IAO's inside clip region
			Region aClip(pActIAO->GetBaseRect());
			aClip.Intersect(rClipRegion);

			if(!aClip.IsEmpty())
			{
				B2dIAOElement* pAct = pActIAO->GetGeometry();
				while(pAct)
				{
					// really new geometry
					bRetval = TRUE;

					// save background for this element
					Point aPoint(pAct->GetPositionX(), pAct->GetPositionY());
					if(pAct->IsPixel())
					{
						// Pixel-Sized Element to save
						if(pAct->IsInside(rClipRegion))
						{
							B2dIAOPixel* pNew = GetB2dIAOPixel();
							pNew->SetPosition(aPoint);
							// add to in-between pixel list
							pPixelList = pNew->AddToList(pPixelList);
							nNumPixelList++;
						}
					}
					else
					{
						// Area Element to save
						Rectangle aRectangle(aPoint, ((B2dIAOBitmap*)pAct)->GetBitmap().GetSizePixel());
//							(pAct->IsBitmap()) ? 
//								  ((B2dIAOBitmap*)pAct)->GetBitmap().GetSizePixel()
//								: ((B2dIAOBmpRef*)pAct)->GetBitmap()->GetSizePixel());

						if(rClipRegion.IsInside(aRectangle))
						{
							// support for VDev cached backgrounds
							B2dIAOBmpVDev* pNew = GetB2dIAOBmpVDev();
							pNew->SetPosition(aPoint);
							pNew->SetEntry(GetVDevCache().Allocate(
								aPoint, aRectangle.GetSize(), *pWindow));

							// add to save list
							SetSave(pNew->AddToList(GetSave()));
						}
						else
						{
							// multiple rects to be saved
							Region aRegion(aRectangle);
							aRegion.Intersect(rClipRegion);
							RegionHandle aRegionHandle = aRegion.BeginEnumRects();
							Rectangle aClipRect;
							while(aRegion.GetEnumRects(aRegionHandle, aClipRect))
							{
								// support for VDev cached backgrounds
								B2dIAOBmpVDev* pNew = GetB2dIAOBmpVDev();
								pNew->SetPosition(aClipRect.TopLeft());
								pNew->SetEntry(GetVDevCache().Allocate(
									aClipRect.TopLeft(), aClipRect.GetSize(), *pWindow));
								
								// add to save list
								SetSave(pNew->AddToList(GetSave()));
							}
							aRegion.EndEnumRects(aRegionHandle);
						}
					}

					// Next IAOPixel
					pAct = pAct->GetNext();
				}
			}
		}

		// next IAO
		pActIAO = pActIAO->GetNext();
	}
	
	// handle in-between pixel list
	while(nNumPixelList)
	{
		// calc size of turn
		UINT16 nNumRun = (nNumPixelList > B2D_NUM_IAOPIXEL_PER_IOOPERATION) 
			? B2D_NUM_IAOPIXEL_PER_IOOPERATION : (UINT16)nNumPixelList;

		// get poly
		if(nNumRun != B2D_NUM_IAOPIXEL_PER_IOOPERATION)
			aTmpPoly = Polygon(nNumRun);

		// create poly for first nNumRun entries
		B2dIAOPixel* pAct = (B2dIAOPixel*)pPixelList;
		UINT16 a;
		for(a=0;a<nNumRun;)
		{
			aTmpPoly[a  ].X() = pAct->GetPositionX();
			aTmpPoly[a++].Y() = pAct->GetPositionY();
			pAct = (B2dIAOPixel*)pAct->GetNext();
		}

		// get colors
		Color* pSavedColors = pWindow->GetPixel(aTmpPoly);
		DBG_ASSERT(pSavedColors, "Got no colrs from Window in SaveBackground()!");

		// put colors to items and items to saved list
		for(a=0;a<nNumRun;)
		{
			B2dIAOPixel* pTmp = (B2dIAOPixel*)pPixelList;
			pPixelList = pTmp->RemFirstFromList(pPixelList);
			pTmp->SetColor(pSavedColors[a++]);
			SetSave(pTmp->AddToList(GetSave()));
		}

		// get rid of colors
		delete[] pSavedColors;

		// correct poly if necessary
		if(nNumRun != B2D_NUM_IAOPIXEL_PER_IOOPERATION)
			aTmpPoly = Polygon(B2D_NUM_IAOPIXEL_PER_IOOPERATION);

		// reduce counter
		nNumPixelList -= nNumRun;
	}

	return bRetval;
}

void B2dIAOManager::RestoreBackground(const Region& rClipRegion, const Region& rWindowRegion, BOOL bInPaint)
{
	B2dIAOElement* pAct = GetSave();
	B2dIAOElement* pPrev = NULL;

	while(pAct)
	{
		// what to do?
		BOOL bCompletelyInside = pAct->IsInside(rClipRegion);
		BOOL bCompletelyOutside = bCompletelyInside ? FALSE : pAct->IsOutside(rClipRegion);

		// remember next and NextPrev
		B2dIAOElement* pNext = pAct->GetNext();
		B2dIAOElement* pNextPrev = pAct;
		Point aPoint(pAct->GetPositionX(), pAct->GetPositionY());

		if(!bCompletelyInside && !bCompletelyOutside)
		{
			// must be a Rectangle overlapping clip region, split it
			// support for VDev cached backgrounds
			Rectangle aBaseRect(aPoint,
				((B2dIAOBmpVDev*)pAct)->GetEntry()->GetSize());
			Rectangle aClipRect;

			// bCompletelyInside=TRUE parts, always delete, restore
			// depends on !bInPaint
			if(!bInPaint)
			{
				// restore parts completely inside and forget them
				Region aRegion(aBaseRect);
				aRegion.Intersect(rClipRegion);
				RegionHandle aRegionHandle = aRegion.BeginEnumRects();
				
				while(aRegion.GetEnumRects(aRegionHandle, aClipRect))
				{
					// draw part of Bitmap
					// support for VDev cached backgrounds
					GetVDevCache().CopyPart(
						((B2dIAOBmpVDev*)pAct)->GetEntry(),
						aClipRect.TopLeft(), aClipRect.GetSize(),
						aClipRect.TopLeft() - aBaseRect.TopLeft(),
						*pWindow);
				}
				aRegion.EndEnumRects(aRegionHandle);
			}

			// remember if nextprevious is corrected by insertion
			BOOL bPrevIsCorrected(FALSE);
			
			// never restore, remember depends on not outside complete window
			if(!pAct->IsOutside(rWindowRegion))
			{
				Region aRegion(aBaseRect);
				aRegion.Exclude(rClipRegion);
				RegionHandle aRegionHandle = aRegion.BeginEnumRects();

				// create parts ompletely outside
				while(aRegion.GetEnumRects(aRegionHandle, aClipRect))
				{
					// support for VDev cached backgrounds
					VDevCacheEntry* pEntry = ((B2dIAOBmpVDev*)pAct)->GetEntry();
					B2dIAOBmpVDev* pNew = GetB2dIAOBmpVDev();
					pNew->SetPosition(aClipRect.TopLeft());
					pNew->SetEntry(GetVDevCache().Allocate(
						pEntry->GetPosition() + aClipRect.TopLeft() - aBaseRect.TopLeft(),
						aClipRect.GetSize(), GetVDevCache().GetVDev()));

					// add to list directly behind pAct
					pAct = pAct->AddToList(pNew->AddToList(pAct->GetNext()));
					if(!bPrevIsCorrected)
					{
						pNextPrev = pNew;
						bPrevIsCorrected = TRUE;
					}
				}
				aRegion.EndEnumRects(aRegionHandle);
			}
			
			// remove overlapping element from save list
			SetSave(pAct->RemFromList(GetSave(), pPrev));
			if(!bPrevIsCorrected)
				pNextPrev = pPrev;

			// support for VDev cached backgrounds
			GetVDevCache().Free(((B2dIAOBmpVDev*)pAct)->GetEntry());
			PutB2dIAOElement(pAct);
		}
		else
		{
			BOOL bCompletelyOutsideWindow = bCompletelyInside ? FALSE : pAct->IsOutside(rWindowRegion);
			BOOL bRestore = !bInPaint && bCompletelyInside;
			BOOL bDelete = bCompletelyInside || bCompletelyOutsideWindow;

			if(bRestore)
			{
				if(pAct->IsPixel())
				{
					// add pixel to collection
					PixelArrayAdd((B2dIAOPixel*)pAct);
				}
				else
				{
					// flush pixel collection
					PixelArrayFlushWrite();

					// draw Bitmap
					// support for VDev cached backgrounds
					GetVDevCache().Copy(((B2dIAOBmpVDev*)pAct)->GetEntry(), aPoint, *pWindow);
					GetVDevCache().Free(((B2dIAOBmpVDev*)pAct)->GetEntry());
				}
			}

			if(bDelete)
			{
				// remove from save list
				SetSave(pAct->RemFromList(GetSave(), pPrev));
				pNextPrev = pPrev;
				PutB2dIAOElement(pAct);
			}
		}

		// next element
		pAct = pNext;
		pPrev = pNextPrev;
	}

	// flush pixel collection
	PixelArrayFlushWrite();
}

void B2dIAOManager::Paint(const Region& rClipRegion)
{
	B2dIAObject* pActIAO = GetIAObjectList();
	while(pActIAO)
	{
		// only visible IAO's
		if(pActIAO->IsVisible())
		{
			// only IAO's inside clip region
			Region aRegion(pActIAO->GetBaseRect());
			aRegion.Intersect(rClipRegion);

			if(!aRegion.IsEmpty())
			{
				B2dIAOElement* pAct = pActIAO->GetGeometry();
				while(pAct)
				{
					if(pAct->IsPixel())
					{
						if(pAct->IsInside(rClipRegion))
						{
							// add pixel to collection
							PixelArrayAdd((B2dIAOPixel*)pAct);
						}
					}
					else
					{
						// flush pixel collection
						PixelArrayFlushWrite();

//						if(pAct->IsBitmap())
//						{
							// draw Bitmap
							Point aPoint(pAct->GetPositionX(), pAct->GetPositionY());
							pWindow->DrawBitmapEx(aPoint, ((B2dIAOBitmap*)pAct)->GetBitmap());
//						}
//						else
//						{
//							// draw BitmapReference
//							Point aPoint(pAct->GetPositionX(), pAct->GetPositionY());
//							pWindow->DrawBitmapEx(aPoint, *((B2dIAOBmpRef*)pAct)->GetBitmap());
//						}
					}

					// Next IAOPixel
					pAct = pAct->GetNext();
				}
			}
		}

		// next IAO
		pActIAO = pActIAO->GetNext();
	}

	// flush pixel collection
	PixelArrayFlushWrite();
}
	
void B2dIAOManager::PixelArrayAdd(B2dIAOPixel* pNew)
{
	if(!pTmpColor)
		pTmpColor = new Color[B2D_NUM_IAOPIXEL_PER_IOOPERATION];
	if(nNextFree == B2D_NUM_IAOPIXEL_PER_IOOPERATION)
		PixelArrayFlushWrite();
	aTmpPoly[nNextFree].X() = pNew->GetPositionX();
	aTmpPoly[nNextFree].Y() = pNew->GetPositionY();
	pTmpColor[nNextFree++] = pNew->GetColor();
}

void B2dIAOManager::PixelArrayFlushWrite()
{
	if(nNextFree)
	{
		if(nNextFree != B2D_NUM_IAOPIXEL_PER_IOOPERATION)
			aTmpPoly.SetSize(nNextFree);
		pWindow->DrawPixel(aTmpPoly, pTmpColor);
		if(nNextFree != B2D_NUM_IAOPIXEL_PER_IOOPERATION)
			aTmpPoly = Polygon(B2D_NUM_IAOPIXEL_PER_IOOPERATION);
		nNextFree = 0;
	}
}

void B2dIAOManager::ApplyDevice(OutputDevice *pOut, BOOL bInPaint)
{
	DBG_ASSERT(pOut, "NO OutDev in ApplyDevice() call");

	// calculate eventual move/scale from scrolling
	BOOL bWasMoved(FALSE);
	BOOL bWasScaled(FALSE);

	if(pOut->GetMapMode() != maLastMapMode)
	{
		if(pOut->GetMapMode().GetOrigin() != maLastMapMode.GetOrigin())
		{
			// Move was done
			bWasMoved = TRUE;
		}

		if(pOut->GetMapMode().GetScaleX() != maLastMapMode.GetScaleX()
			|| pOut->GetMapMode().GetScaleY() != maLastMapMode.GetScaleY())
		{
			// Scale was done
			bWasScaled = TRUE;

			// throw away ALL saved background
			ForgetBackground();
		}

		// if only move happened, apply to saved IAOPixels to correct
		// their positions
		if(bWasMoved && !bWasScaled)
		{
			Point aDeltaPoint(pOut->GetMapMode().GetOrigin() - maLastMapMode.GetOrigin());
			Size aDelta(aDeltaPoint.A(), aDeltaPoint.B());
			Size aDeltaSize(pOut->LogicToPixel(aDelta));
			Point aZeroPosDelta(aDeltaSize.A(), aDeltaSize.B());

			MoveSavedElementsPixel(aZeroPosDelta);
		}

		// remember new MapMode
		maLastMapMode = pOut->GetMapMode();
	}

	// propagate OutDev to single IAO's
	B2dIAObject* pAct = GetIAObjectList();
	while(pAct)
	{
		// apply evtl. changed OutDev
		pAct->ApplyDevice(pOut);

		// force geometry creation, for invalidation of regions
		if(!bInPaint)
			pAct->GetGeometry();

		// next IAO
		pAct = pAct->GetNext();
	}
}

BOOL B2dIAOManager::UpdateDisplay()
{
	BOOL bRetval(TRUE);
	BOOL bInPaint(pWindow->IsInPaint());

	if(bChanged || bInPaint)
	{
		BOOL bWasMappingEnabled(pWindow->IsMapModeEnabled());
		pWindow->EnableMapMode(FALSE);
		Region aPaintRegion(pWindow->GetPaintRegion());
		Region aWindowRegion(pWindow->GetWindowClipRegionPixel());

		// when empty, build window ClipRegion myself, use whole window
		if(aWindowRegion.IsNull())
		{
			Rectangle aWinRect(pWindow->GetDesktopRectPixel());
			aWindowRegion = Region(aWinRect);
		}

		if(bInPaint)
		{
			// combine ClipRegions
			Region aClip(aPaintRegion);
			aClip.Intersect(aWindowRegion);

			// propagate OutDev to force changes in geometry
			// mapping must be ON at this moment
			pWindow->EnableMapMode(bWasMappingEnabled);
			ApplyClipRegion(aClip);
			ApplyDevice(pWindow, TRUE);
			pWindow->EnableMapMode(FALSE);
		
			// restore as much as possible
			if(GetSave())
				RestoreBackground(aClip, aWindowRegion, TRUE);

			// save and paint new ones
			if(GetIAObjectList() && IsVisible())
			{
				if(SaveBackground(aClip))
				{
					Paint(aClip);
				}
			}
		}
		else
		{
			// build ClipRegion
			Region aClip(aWindowRegion);

			// propagate OutDev to force changes in geometry
			// mapping must be ON at this moment
			pWindow->EnableMapMode(bWasMappingEnabled);
			ApplyClipRegion(aClip);
			ApplyDevice(pWindow, FALSE);
			pWindow->EnableMapMode(FALSE);
		
			// limit save and paint to an even smaller area
			if(!aInvalidateRectangle.IsEmpty())
			{
				aClip.Intersect(aInvalidateRectangle);
			}

			// restore as much as possible
			if(GetSave())
				RestoreBackground(aClip, aWindowRegion, FALSE);

			// save and paint new ones
			if(GetIAObjectList() && IsVisible())
			{
				if(SaveBackground(aClip))
				{
					Paint(aClip);
				}
			}

			// reset invalidate region
			aInvalidateRectangle.SetEmpty();

			// reset changes, newest possible situation is shown
			bChanged = FALSE;
		}

		// enable old mapping
		pWindow->EnableMapMode(bWasMappingEnabled);
	}

	return bRetval;
}

// UpdateDisplay() in force-remove-saved-parts mode
void B2dIAOManager::ForceHide()
{
	// is there something saved at all that needs restore?
	if(GetSave())
	{
		BOOL bOldVisible = bVisible;

		bChanged = TRUE;
		bVisible = FALSE;

		UpdateDisplay();

		bVisible = bOldVisible;
		bChanged = TRUE;
	}
}

BOOL B2dIAOManager::IsVisible() const 
{ 
	return bVisible; 
}

void B2dIAOManager::SetVisible(BOOL bNew) 
{ 
	if(bNew != bVisible) 
	{ 
		// remeber new state
		bVisible = bNew; 
		bChanged = TRUE;

		// check if timer is needed
		CheckTimerState();
	}
}

void B2dIAOManager::InvalidateRectangle(const Rectangle& rRect)
{
	if(!rRect.IsEmpty())
	{
		bChanged = TRUE;
		aInvalidateRectangle.Union(rRect);
	}
}

B2dIAObject* B2dIAOManager::GetIAObject(UINT32 nNum) const
{
	B2dIAObject* pRetval = NULL;

	if(nNum < nIAONumber)
	{
		pRetval = pIAObjectList;
		for(UINT32 a=0;a<nNum;a++)
			pRetval = pRetval->GetNext();
	}

	return pRetval;
}

void B2dIAOManager::Transform(const Matrix3D& rMat)
{
	B2dIAObject* pActIAO = pIAObjectList;
	while(pActIAO)
	{
		pActIAO->Transform(rMat);
		pActIAO = pActIAO->GetNext();
	}
}

void B2dIAOManager::SetAnimation(BOOL bNew)
{
	if(bNew != bAnimationOn)
	{
		// remember new value
		bAnimationOn = bNew;

		// check if timer is needed
		CheckTimerState();
	}
}

void B2dIAOManager::RegisterAnimatedObject(B2dIAObject* pNew)
{
	// add to list
	aAnimatedObjectList.Insert(pNew);

	// check if timer is needed
	CheckTimerState();
}

void B2dIAOManager::UnregisterAnimatedObject(B2dIAObject* pOld)
{
	// remove from list
	if(aAnimatedObjectList.Remove(pOld))
	{
		// check if timer is needed
		CheckTimerState();
	}
}

void B2dIAOManager::CheckTimerState()
{
	if(bVisible && bAnimationOn && aAnimatedObjectList.Count() && !bTimerIsOn)
	{
		// start animation timer
		aAnimator.Start();

		// remember new state
		bTimerIsOn = TRUE;
	}
	else
	{
		if(bTimerIsOn)
		{
			// stop animation timer
			aAnimator.Stop();

			// remember new state
			bTimerIsOn = FALSE;
		}
	}
}


