/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: vdevcache.cxx,v $
 *
 *  $Revision: 1.4 $
 *
 *  last change: $Author: obo $ $Date: 2006/10/12 15:32:33 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

// MARKER(update_precomp.py): autogen include statement, do not remove
#include "precompiled_goodies.hxx"

#ifndef _VDEVCACHE_HXX
#include "vdevcache.hxx"
#endif

#ifndef _TOOLS_DEBUG_HXX
#include <tools/debug.hxx>
#endif

//////////////////////////////////////////////////////////////////////////////

void VDevCacheEntry::SplitHorizontal(sal_uInt32 nLength)
{
	mpLeft = new VDevCacheEntry(maPosition, Size(nLength, maSize.Height()), this);
	mpRight = new VDevCacheEntry(Point(maPosition.X() + nLength, maPosition.Y()), 
		Size(maSize.Width() - nLength, maSize.Height()), this);
}

void VDevCacheEntry::SplitVertical(sal_uInt32 nLength)
{
	mpLeft = new VDevCacheEntry(maPosition, Size(maSize.Width(), nLength), this);
	mpRight = new VDevCacheEntry(Point(maPosition.X(), maPosition.Y() + nLength),
		Size(maSize.Width(), maSize.Height() - nLength), this);
}

void VDevCacheEntry::Melt()
{
	delete mpLeft;
	mpLeft = 0L;
	delete mpRight;
	mpRight = 0L;
}

//////////////////////////////////////////////////////////////////////////////

class ImpVDCache
{
	VirtualDevice*				mpVDev;
	Size						maSize;
	VDevCacheEntry*				mpRoot;
	VDevCacheEntry*				mpFree;

	// cache resize
	void AddToFreeList(VDevCacheEntry* pNew);
	void RemoveFromFreeList(VDevCacheEntry* pOld);
	void GrowVDCache();
	VDevCacheEntry* FindCandidateInFreeList(const Size& rNeededSize);
	void TryToMelt(VDevCacheEntry* pParent);

public:
	ImpVDCache(Size aSize = Size(256, 256));
	~ImpVDCache();

	VirtualDevice& GetVDev() const { return *mpVDev; }

	VDevCacheEntry* AllocateSize(const Size& rNeededSize);
	void FreeSize(VDevCacheEntry* pOld);
};

ImpVDCache::ImpVDCache(Size aSize)
:	maSize(aSize),
	mpRoot(0L),
	mpFree(0L)
{
	// get VDev cache area itself
	mpVDev = new VirtualDevice();
	DBG_ASSERT(mpVDev, "ImpVDevCache: could not allocate new VirtualDevice (!)");
	mpVDev->SetOutputSizePixel(maSize);

	// create root
	mpRoot = new VDevCacheEntry(Point(), maSize, 0L);
	DBG_ASSERT(mpRoot, "ImpVDevCache: could not allocate new Root (!)");

	// add to free list
	AddToFreeList(mpRoot);
}

ImpVDCache::~ImpVDCache()
{
	// free VDev itself
	if(mpVDev)
		delete mpVDev;

	// delete all ImpVDCacheEntries
	if(mpRoot)
		delete mpRoot;
}

void ImpVDCache::AddToFreeList(VDevCacheEntry* pNew)
{
	pNew->SetNext(mpFree);
	pNew->SetPrev(0L);
	if(mpFree)
		mpFree->SetPrev(pNew);
	mpFree = pNew;
}

void ImpVDCache::RemoveFromFreeList(VDevCacheEntry* pOld)
{
	if(mpFree == pOld)
		mpFree = pOld->GetNext();
	if(pOld->GetPrev())
		pOld->GetPrev()->SetNext(pOld->GetNext());
	if(pOld->GetNext())
		pOld->GetNext()->SetPrev(pOld->GetPrev());
	pOld->SetPrev(0L);
	pOld->SetNext(0L);
}

void ImpVDCache::GrowVDCache()
{
	// grow horizontal or vertical?
	BOOL bGrowVertical(maSize.Width() > maSize.Height());
	Size aNewSize = bGrowVertical ? 
		Size(maSize.Width(), maSize.Height() << 1L) : 
		Size(maSize.Width() << 1L, maSize.Height());

	// get new VDev
	VirtualDevice* pNewVDev = new VirtualDevice();
	DBG_ASSERT(pNewVDev, "ImpVDevCache: could not allocate new VirtualDevice (!)");
	pNewVDev->SetOutputSizePixel(aNewSize);

	// copy old VDev to it
	pNewVDev->DrawOutDev(Point(), maSize, Point(), maSize, *mpVDev);

	// exchange VDevs
	delete mpVDev;
	mpVDev = pNewVDev;

	// create new root
	VDevCacheEntry* pNewRoot = new VDevCacheEntry(Point(), aNewSize, 0L);
	DBG_ASSERT(pNewRoot, "ImpVDevCache: could not allocate new Root (!)");
	
	// create new cache entry as second path
	VDevCacheEntry* pNewFree = new VDevCacheEntry(
		bGrowVertical ? Point(0L, maSize.Height()) : Point(maSize.Width(), 0L), 
		maSize, mpRoot);
	DBG_ASSERT(pNewRoot, "ImpVDevCache: could not allocate new Root (!)");

	// add old root and new free as pathes, change parent of old root
	pNewRoot->ImpSetLeftRight(mpRoot, pNewFree);
	mpRoot->ImpSetParent(pNewRoot);

	// set new root and size
	mpRoot = pNewRoot;
	maSize = aNewSize;

	// add new path to free list
	AddToFreeList(pNewFree);
}

VDevCacheEntry* ImpVDCache::FindCandidateInFreeList(const Size& rNeededSize)
{
	// are free parts there at all?
	if(!mpFree)
		return 0L;

	// search the list
	VDevCacheEntry* pCandidate = mpFree;
	VDevCacheEntry* pBestFit = 0L;

	while(pCandidate)
	{
		// does this one fit?
		if(pCandidate->GetSize().Width() >= rNeededSize.Width()
			&& pCandidate->GetSize().Height() >= rNeededSize.Height())
		{
			if(pBestFit)
			{
				// does it fit better than pBestFit?
				if(pCandidate->GetSize().Width() < pBestFit->GetSize().Width()
					|| pCandidate->GetSize().Height() < pBestFit->GetSize().Height())
				{
					pBestFit = pCandidate;
				}
			}
			else
				pBestFit = pCandidate;
		}

		pCandidate = pCandidate->GetNext();
	}

	return pBestFit;
}

void ImpVDCache::TryToMelt(VDevCacheEntry* pParent)
{
	if(pParent && pParent->GetLeft()->IsInFreeList() && pParent->GetRight()->IsInFreeList())
	{
		RemoveFromFreeList(pParent->GetLeft());
		RemoveFromFreeList(pParent->GetRight());
		pParent->Melt();
		AddToFreeList(pParent);

		// melt goes up
		TryToMelt(pParent->GetParent());
	}
}

VDevCacheEntry* ImpVDCache::AllocateSize(const Size& rNeededSize)
{
	VDevCacheEntry* pCandidate = 0L;

	// find a candidate in the free list
    for (;;) {
        pCandidate = FindCandidateInFreeList(rNeededSize);
        if (pCandidate != NULL) {
            break;
        }
		GrowVDCache();
    }

	// remove candidate from free list
	RemoveFromFreeList(pCandidate);

	// pCandidate is bigger in H and V, split it and add
	// free parts to the free list
	sal_uInt32 nHorizontalDiff(pCandidate->GetSize().Width() - rNeededSize.Width());
	sal_uInt32 nVerticalDiff(pCandidate->GetSize().Height() - rNeededSize.Height());
	BOOL bSplitHorizontal = (nHorizontalDiff > nVerticalDiff);

	if(bSplitHorizontal)
	{
		if(nHorizontalDiff)
		{
			pCandidate->SplitHorizontal(rNeededSize.Width());
			AddToFreeList(pCandidate->GetRight());
			pCandidate = pCandidate->GetLeft();
		}

		if(nVerticalDiff)
		{
			pCandidate->SplitVertical(rNeededSize.Height());
			AddToFreeList(pCandidate->GetRight());
			pCandidate = pCandidate->GetLeft();
		}
	}
	else
	{
		if(nVerticalDiff)
		{
			pCandidate->SplitVertical(rNeededSize.Height());
			AddToFreeList(pCandidate->GetRight());
			pCandidate = pCandidate->GetLeft();
		}

		if(nHorizontalDiff)
		{
			pCandidate->SplitHorizontal(rNeededSize.Width());
			AddToFreeList(pCandidate->GetRight());
			pCandidate = pCandidate->GetLeft();
		}
	}

	// give back cutted and exactly fitting part
	return pCandidate;
}

void ImpVDCache::FreeSize(VDevCacheEntry* pOld)
{
	// add to free list
	AddToFreeList(pOld);
	
	// try to melt at this Point
	TryToMelt(pOld->GetParent());
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

VDevCache::VDevCache(sal_uInt32 nInitialEdgeLen)
:	mpImp(new ImpVDCache(Size(nInitialEdgeLen, nInitialEdgeLen)))
{
}

VDevCache::~VDevCache()
{
	if(mpImp)
		delete mpImp;
}

VDevCacheEntry* VDevCache::Allocate(const Point& rSrcPt,
	const Size& rSrcSize, const OutputDevice& rSrcDev)
{
	VDevCacheEntry* pEntry = mpImp->AllocateSize(rSrcSize);
	DBG_ASSERT(pEntry, "VDevCache: could not allocate new region (!)");

	// copy area to allocated part
	mpImp->GetVDev().DrawOutDev(pEntry->GetPosition(), rSrcSize, 
		rSrcPt, rSrcSize, rSrcDev);
	
	// return ID for later PopArea() call
	return pEntry;
}

void VDevCache::Copy(VDevCacheEntry* pEntry, const Point& rDstPt, OutputDevice& rDstDev)
{
	if(pEntry)
	{
		rDstDev.DrawOutDev(rDstPt, pEntry->GetSize(), 
			pEntry->GetPosition(), pEntry->GetSize(), mpImp->GetVDev());
	}
}

void VDevCache::CopyPart(VDevCacheEntry* pEntry, const Point& rDstPt, 
	const Size& rDstSize, const Point& rSrcPntOffset, OutputDevice& rDstDev)
{
	if(pEntry)
	{
		rDstDev.DrawOutDev(rDstPt, rDstSize, 
			pEntry->GetPosition() + rSrcPntOffset, rDstSize, mpImp->GetVDev());
	}
}

void VDevCache::Free(VDevCacheEntry* pEntry)
{
	if(pEntry)
	{
		// free area and cleanup
		mpImp->FreeSize(pEntry);
	}
}

const VirtualDevice& VDevCache::GetVDev() const
{ 
	return mpImp->GetVDev(); 
}

//////////////////////////////////////////////////////////////////////////////
