/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: vocbitmapbuffer.cxx,v $
 *
 *  $Revision: 1.6 $
 *
 *  last change: $Author: obo $ $Date: 2006/09/17 05:39:03 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

// MARKER(update_precomp.py): autogen include statement, do not remove
#include "precompiled_svx.hxx"

#ifndef _SDR_CONTACT_VOCBITMAPBUFFER_HXX
#include <svx/sdr/contact/vocbitmapbuffer.hxx>
#endif

#ifndef _SDR_CONTACT_DISPLAYINFO_HXX
#include <svx/sdr/contact/displayinfo.hxx>
#endif

#ifndef _SV_VIRDEV_HXX
#include <vcl/virdev.hxx>
#endif

#ifndef _XOUTX_HXX
#include <xoutx.hxx>
#endif

#ifndef _SDR_CONTACT_VIEWCONTACT_HXX
#include <svx/sdr/contact/viewcontact.hxx>
#endif

#ifndef _SV_SVAPP_HXX
#include <vcl/svapp.hxx>
#endif

//////////////////////////////////////////////////////////////////////////////
// class BufferHelper

namespace sdr
{
	namespace contact
	{
		BufferHelper::BufferHelper(ViewObjectContact& rViewObjectContact)
		:	mrViewObjectContact(rViewObjectContact),
			mbBufferValid(false)
		{
		}

		BufferHelper::~BufferHelper()
		{
		}

		bool BufferHelper::ImpBufferingAllowed(DisplayInfo& rDisplayInfo) const
		{
			return (!rDisplayInfo.OutputToPrinter() 
				&& !rDisplayInfo.OutputToRecordingMetaFile()
				&& rDisplayInfo.IsBufferingAllowed());
		}

		void BufferHelper::ImpTryToCreateBuffer(DisplayInfo& rDisplayInfo)
		{
			const Rectangle aNewSizePixel(ImpCalculateClippedPixelRectangle(rDisplayInfo));

			// ensure empty bitmap
			DisposeBuffer();

			if(!aNewSizePixel.IsEmpty())
			{
				// calculate const vars
				const Point aEmptyPoint(0, 0);
				const Rectangle& aPaintRectangle(mrViewObjectContact.GetViewContact().GetPaintRectangle());
				OutputDevice* pOriginalOutDev = rDisplayInfo.GetOutputDevice();
				XOutputDevice* pOriginalExtOutDev = rDisplayInfo.GetExtendedOutputDevice();
				const Rectangle aPaintRectanglePixel(pOriginalOutDev->LogicToPixel(aPaintRectangle));
				const Size aVDevPaintSizePixel(aNewSizePixel.GetSize());
				MapMode aMapMode(pOriginalOutDev->GetMapMode());
				Point aMapModeOrigin(aPaintRectangle.TopLeft());
				
				// calculate new OffsetTopLeft in pixels
				Point aOffsetTopLeft(aNewSizePixel.TopLeft() - aPaintRectanglePixel.TopLeft());

				if(aOffsetTopLeft.X() || aOffsetTopLeft.Y())
				{
					// if there is a pixel offset, calculate the logical offset
					// for later painting
					const Point aLogicalNewSizePixelTopLeft(pOriginalOutDev->PixelToLogic(aNewSizePixel.TopLeft()));
					aOffsetTopLeft = aLogicalNewSizePixelTopLeft - aPaintRectangle.TopLeft();

					// also prepare the MapMode origin
					aMapModeOrigin += aOffsetTopLeft;
				}

				// Prepare MapMode for BufferDevice
				const Point aNegativeMapModeOffset(-aMapModeOrigin.X(), -aMapModeOrigin.Y());
				aMapMode.SetOrigin(aNegativeMapModeOffset);
 
				// prepare VirtualDevice for mask
				VirtualDevice* pBufferDevice = ImpPrepareVirtualDevice(rDisplayInfo);

				// Set OutputSize and MapMode
				pBufferDevice->SetOutputSizePixel(aVDevPaintSizePixel);
				pBufferDevice->SetMapMode(aMapMode);

				// set new values at rDisplayInfo
				pOriginalExtOutDev->SetOutDev(pBufferDevice);
				rDisplayInfo.SetOutputDevice(pBufferDevice);

				// paint in new OutDev using call to ViewObjectContact::PaintObject
				// Do not just call mrViewObjectContact.PaintObject() here, this would
				// lead to a loop (!)
				mrViewObjectContact.ViewObjectContact::PaintObject(rDisplayInfo);

				// set back to old OutDev, restore rDisplayInfo
				pOriginalExtOutDev->SetOutDev(pOriginalOutDev);
				rDisplayInfo.SetOutputDevice(pOriginalOutDev);

				// get BitmapEx
				pBufferDevice->EnableMapMode(sal_False);
				ImpCreateBufferFromVDev(*pBufferDevice);

				// delete buffer device again
				delete pBufferDevice;

				// remember MapMode of buffereded part for later comparisons
				maBufferedMapMode = pOriginalOutDev->GetMapMode();
			}
		}

		bool BufferHelper::TryToPaintObjectWithBuffer(DisplayInfo& rDisplayInfo)
		{
			bool bRetval(false);

			// first thing to do is to check if the buffer needs to be disposed
			// due to changes
			ImpCheckForChanges(rDisplayInfo);

			// check if buffering is allowed at all
			if(ImpBufferingAllowed(rDisplayInfo))
			{
				// try to create buffer
				if(!BufferIsValid())
				{
					ImpTryToCreateBuffer(rDisplayInfo);
				}

				// If buffer exists, use it now
				if(BufferIsValid())
				{
					// paint using buffer
					ImpPaintUsingBuffer(rDisplayInfo);
					bRetval = true;
				}
			}

			return bRetval;
		}

		Rectangle BufferHelper::ImpCalculateClippedPixelRectangle(DisplayInfo& rDisplayInfo) const
		{
			const OutputDevice* pOriginalOutDev = rDisplayInfo.GetOutputDevice();
			Rectangle aReturnValue(mrViewObjectContact.GetViewContact().GetPaintRectangle());
			const Point aEmptyPoint(0, 0);
			Rectangle aVisiblePixel(aEmptyPoint, pOriginalOutDev->GetOutputSizePixel());

			aReturnValue = pOriginalOutDev->LogicToPixel(aReturnValue);
			aReturnValue.Intersection(aVisiblePixel);
			aReturnValue.Justify();

			return aReturnValue;
		}

		void BufferHelper::ImpCheckForChanges(DisplayInfo& rDisplayInfo)
		{
			const OutputDevice* pOriginalOutDev = rDisplayInfo.GetOutputDevice();

			if(pOriginalOutDev)
			{
				if(BufferIsValid())
				{
					const Rectangle aNewSizePixel(ImpCalculateClippedPixelRectangle(rDisplayInfo));

					if(aNewSizePixel.GetSize() != ImpGetBufferedSizePixel())
					{
						DisposeBuffer();
					}
				}

				if(BufferIsValid())
				{
					if(pOriginalOutDev->GetMapMode() != GetBufferedMapMode())
					{
						DisposeBuffer();
					}
				}
			}
		}
	} // end of namespace contact
} // end of namespace sdr

//////////////////////////////////////////////////////////////////////////////
// class BufferHelperBitmap

namespace sdr
{
	namespace contact
	{
		Size BufferHelperBitmap::ImpGetBufferedSizePixel() const
		{
			return maBuffer.GetSizePixel();
		}

		void BufferHelperBitmap::ImpPaintUsingBuffer(DisplayInfo& rDisplayInfo) const
		{
			OutputDevice* pOriginalOutDev = rDisplayInfo.GetOutputDevice();
			const Rectangle aNewSizePixel(ImpCalculateClippedPixelRectangle(rDisplayInfo));
			const sal_Bool bWasEnabled(pOriginalOutDev->IsMapModeEnabled());

			pOriginalOutDev->EnableMapMode(sal_False);
			pOriginalOutDev->DrawBitmap(aNewSizePixel.TopLeft(), maBuffer);
			pOriginalOutDev->EnableMapMode(bWasEnabled);
		}

		VirtualDevice* BufferHelperBitmap::ImpPrepareVirtualDevice(DisplayInfo& rDisplayInfo) const
		{
			const OutputDevice* pOriginalOutDev = rDisplayInfo.GetOutputDevice();
			return new VirtualDevice(*pOriginalOutDev, 0L);
		}

		void BufferHelperBitmap::ImpCreateBufferFromVDev(VirtualDevice& rVirtualDevice)
		{
			const Point aEmptyPoint(0, 0);
			maBuffer = rVirtualDevice.GetBitmap(aEmptyPoint, rVirtualDevice.GetOutputSizePixel());

			if(!maBuffer.IsEmpty())
			{
				mbBufferValid = true;
			}
		}

		BufferHelperBitmap::BufferHelperBitmap(ViewObjectContact& rViewObjectContact)
		:	BufferHelper(rViewObjectContact)
		{
		}

		BufferHelperBitmap::~BufferHelperBitmap()
		{
		}

		void BufferHelperBitmap::DisposeBuffer()
		{
			if(BufferIsValid())
			{
				maBuffer.SetEmpty();
				mbBufferValid = false;
			}
		}

		void BufferHelperBitmap::SetBufferedData(const MapMode& rMapMode, const Bitmap& rBitmap)
		{
			maBufferedMapMode = rMapMode;
			mbBufferValid = true;
			maBuffer = rBitmap;
		}
	} // end of namespace contact
} // end of namespace sdr

//////////////////////////////////////////////////////////////////////////////
// class BufferHelperBitmapEx

namespace sdr
{
	namespace contact
	{
		Size BufferHelperBitmapEx::ImpGetBufferedSizePixel() const
		{
			return maBuffer.GetSizePixel();
		}

		void BufferHelperBitmapEx::ImpPaintUsingBuffer(DisplayInfo& rDisplayInfo) const
		{
			OutputDevice* pOriginalOutDev = rDisplayInfo.GetOutputDevice();
			const Rectangle aNewSizePixel(ImpCalculateClippedPixelRectangle(rDisplayInfo));
			const sal_Bool bWasEnabled(pOriginalOutDev->IsMapModeEnabled());

			pOriginalOutDev->EnableMapMode(sal_False);
			pOriginalOutDev->DrawBitmapEx(aNewSizePixel.TopLeft(), maBuffer);
			pOriginalOutDev->EnableMapMode(bWasEnabled);
		}

		VirtualDevice* BufferHelperBitmapEx::ImpPrepareVirtualDevice(DisplayInfo& rDisplayInfo) const
		{
			const OutputDevice* pOriginalOutDev = rDisplayInfo.GetOutputDevice();
			return new VirtualDevice(*pOriginalOutDev, 0L, 0L);
		}

		void BufferHelperBitmapEx::ImpCreateBufferFromVDev(VirtualDevice& rVirtualDevice)
		{
			const Point aEmptyPoint(0, 0);
			maBuffer = rVirtualDevice.GetBitmapEx(aEmptyPoint, rVirtualDevice.GetOutputSizePixel());

			if(!maBuffer.IsEmpty())
			{
				mbBufferValid = true;
			}
		}

		BufferHelperBitmapEx::BufferHelperBitmapEx(ViewObjectContact& rViewObjectContact)
		:	BufferHelper(rViewObjectContact)
		{
		}

		BufferHelperBitmapEx::~BufferHelperBitmapEx()
		{
		}

		void BufferHelperBitmapEx::DisposeBuffer()
		{
			if(BufferIsValid())
			{
				maBuffer.SetEmpty();
				mbBufferValid = false;
			}
		}

		void BufferHelperBitmapEx::SetBufferedData(const MapMode& rMapMode, const BitmapEx& rBitmapEx)
		{
			maBufferedMapMode = rMapMode;
			mbBufferValid = true;
			maBuffer = rBitmapEx;
		}
	} // end of namespace contact
} // end of namespace sdr

//////////////////////////////////////////////////////////////////////////////
// class VOCBitmapBuffer

namespace sdr
{
	namespace contact
	{
		VOCBitmapBuffer::VOCBitmapBuffer(ObjectContact& rObjectContact, ViewContact& rViewContact)
		:	ViewObjectContact(rObjectContact, rViewContact),
			maBuffer(*this)
		{
		}

		VOCBitmapBuffer::~VOCBitmapBuffer()
		{
		}

		void VOCBitmapBuffer::PrepareDelete()
		{
			// delete buffer
			maBuffer.DisposeBuffer();

			// call parent
			ViewObjectContact::PrepareDelete();
		}

		void VOCBitmapBuffer::PaintObject(DisplayInfo& rDisplayInfo)
		{
			if(maBuffer.TryToPaintObjectWithBuffer(rDisplayInfo))
			{
				// paint from buffer was successful, set flags
				mbIsPainted = sal_True;
				mbIsInvalidated = sal_False;
				const Rectangle& aPaintRectangle = GetViewContact().GetPaintRectangle();
				maPaintedRectangle = aPaintRectangle;
			}
			else
			{
				// paint normal, buffer not available. Call parent.
				ViewObjectContact::PaintObject(rDisplayInfo);
			}
		}

		void VOCBitmapBuffer::ActionChanged()
		{
			// call parent
			ViewObjectContact::ActionChanged();

			// delete buffer
			maBuffer.DisposeBuffer();
		}

		void VOCBitmapBuffer::ActionChildInserted(const Rectangle& rInitialRectangle)
		{
			// call parent
			ViewObjectContact::ActionChildInserted(rInitialRectangle);

			// delete buffer
			maBuffer.DisposeBuffer();
		}
	} // end of namespace contact
} // end of namespace sdr

//////////////////////////////////////////////////////////////////////////////
// class VOCBitmapExBuffer

namespace sdr
{
	namespace contact
	{
		VOCBitmapExBuffer::VOCBitmapExBuffer(ObjectContact& rObjectContact, ViewContact& rViewContact)
		:	ViewObjectContact(rObjectContact, rViewContact),
			maBuffer(*this)
		{
		}

		VOCBitmapExBuffer::~VOCBitmapExBuffer()
		{
		}

		void VOCBitmapExBuffer::PrepareDelete()
		{
			// delete buffer
			maBuffer.DisposeBuffer();

			// call parent
			ViewObjectContact::PrepareDelete();
		}

		void VOCBitmapExBuffer::PaintObject(DisplayInfo& rDisplayInfo)
		{
			if(maBuffer.TryToPaintObjectWithBuffer(rDisplayInfo))
			{
				// paint from buffer was successful, set flags
				mbIsPainted = sal_True;
				mbIsInvalidated = sal_False;
				const Rectangle& aPaintRectangle = GetViewContact().GetPaintRectangle();
				maPaintedRectangle = aPaintRectangle;
			}
			else
			{
				// paint normal, buffer not available. Call parent.
				ViewObjectContact::PaintObject(rDisplayInfo);
			}
		}

		void VOCBitmapExBuffer::ActionChanged()
		{
			// call parent
			ViewObjectContact::ActionChanged();

			// delete buffer
			maBuffer.DisposeBuffer();
		}

		void VOCBitmapExBuffer::ActionChildInserted(const Rectangle& rInitialRectangle)
		{
			// call parent
			ViewObjectContact::ActionChildInserted(rInitialRectangle);

			// delete buffer
			maBuffer.DisposeBuffer();
		}
	} // end of namespace contact
} // end of namespace sdr

//////////////////////////////////////////////////////////////////////////////

//namespace sdr
//{
//	namespace contact
//	{
//		// Calculate the clipped PixelSize (visible area) and the offset from the
//		// top left of the original object position
//		Rectangle VOCBitmapBuffer_old::CalculateClippedPixelRectangle(DisplayInfo& rDisplayInfo)
//		{
//			const OutputDevice* pOriginalOutDev = rDisplayInfo.GetOutputDevice();
//			Rectangle aReturnValue(GetViewContact().GetPaintRectangle());
//			const Point aEmptyPoint(0, 0);
//			Rectangle aVisiblePixel(aEmptyPoint, pOriginalOutDev->GetOutputSizePixel());
//
//			aReturnValue = pOriginalOutDev->LogicToPixel(aReturnValue);
//			aReturnValue.Intersection(aVisiblePixel);
//			aReturnValue.Justify();
//
//			return aReturnValue;
//		}
//
//		// Buffer Bitmap creation/deletion
//		void VOCBitmapBuffer_old::TryToCreateBuffer(const Rectangle& rNewSizePixel, DisplayInfo& rDisplayInfo)
//		{
//			// make sure to delete old bitmap if there is one
//			DisposeBuffer();
//
//			if(!rNewSizePixel.IsEmpty())
//			{
//				// calculate const vars
//				const Point aEmptyPoint(0, 0);
//				const Rectangle& aPaintRectangle = GetViewContact().GetPaintRectangle();
//				OutputDevice* pOriginalOutDev = rDisplayInfo.GetOutputDevice();
//				ExtOutputDevice* pOriginalExtOutDev = rDisplayInfo.GetExtendedOutputDevice();
//				const Rectangle aPaintRectanglePixel(pOriginalOutDev->LogicToPixel(aPaintRectangle));
//				const Size aVDevPaintSizePixel(rNewSizePixel.GetSize());
//				MapMode aMapMode(pOriginalOutDev->GetMapMode());
//				Point aMapModeOrigin(aPaintRectangle.TopLeft());
//				
//				// calculate new OffsetTopLeft in pixels
//				Point aOffsetTopLeft(rNewSizePixel.TopLeft() - aPaintRectanglePixel.TopLeft());
//
//				if(aOffsetTopLeft.X() || aOffsetTopLeft.Y())
//				{
//					// if there is a pixel offset, calculate the logical offset
//					// for later painting
//					const Point aLogicalNewSizePixelTopLeft(pOriginalOutDev->PixelToLogic(rNewSizePixel.TopLeft()));
//					aOffsetTopLeft = aLogicalNewSizePixelTopLeft - aPaintRectangle.TopLeft();
//
//					// also prepare the MapMode origin
//					aMapModeOrigin += aOffsetTopLeft;
//				}
//
//				// Prepare MapMode for BufferDevice
//				const Point aNegativeMapModeOffset(-aMapModeOrigin.X(), -aMapModeOrigin.Y());
//				aMapMode.SetOrigin(aNegativeMapModeOffset);
// 
//				// prepare VirtualDevice for mask
//				VirtualDevice* pBufferDevice(0L);
//				
//				if(DoUseAlphaChannel())
//				{
//					pBufferDevice = new VirtualDevice(*pOriginalOutDev, 0L, 0L);
//				}
//				else
//				{
//					pBufferDevice = new VirtualDevice(*pOriginalOutDev, 0L);
//				}
//
//				// Set OutputSize and MapMode
//				pBufferDevice->SetOutputSizePixel(aVDevPaintSizePixel);
//				pBufferDevice->SetMapMode(aMapMode);
//
//				// set new values at rDisplayInfo
//				pOriginalExtOutDev->SetOutDev(pBufferDevice);
//				rDisplayInfo.SetOutputDevice(pBufferDevice);
//
//				// paint in new OutDev using call to parent
//				ViewObjectContact::PaintObject(rDisplayInfo);
//
//				// set back to old OutDev, restore rDisplayInfo
//				pOriginalExtOutDev->SetOutDev(pOriginalOutDev);
//				rDisplayInfo.SetOutputDevice(pOriginalOutDev);
//
//				// get BitmapEx
//				pBufferDevice->EnableMapMode(sal_False);
//
//				if(DoUseAlphaChannel())
//				{
//					maBitmapEx = pBufferDevice->GetBitmapEx(aEmptyPoint, pBufferDevice->GetOutputSizePixel());
//				}
//				else
//				{
//					// create display bitmap
//					Bitmap aTempBitmap(pBufferDevice->GetBitmap(aEmptyPoint, pBufferDevice->GetOutputSizePixel()));
//					maBitmapEx = BitmapEx(aTempBitmap.CreateDisplayBitmap(pOriginalOutDev));
//				}
//
//				// delete buffer device again
//				delete pBufferDevice;
//
//				// set flag to valid if bitmap is OK, maybe
//				// more test will be done here
//				if(!maBitmapEx.IsEmpty())
//				{
//					mbCachedBitmapValid = sal_True;
//
//					// remember MapMode of cached part for later comparisons
//					maCachedMapMode = pOriginalOutDev->GetMapMode();
//				}
//			}
//		}
//
//		void VOCBitmapBuffer_old::DisposeBuffer()
//		{
//			if(CachedBitmapIsValid())
//			{
//				maBitmapEx.SetEmpty();
//				mbCachedBitmapValid = sal_False;
//			}
//		}
//
//		// basic constructor.
//		VOCBitmapBuffer_old::VOCBitmapBuffer_old(
//			ObjectContact& rObjectContact, ViewContact& rViewContact, bool bUseAlphaChannel)
//		:	ViewObjectContact(rObjectContact, rViewContact),
//			mbCachedBitmapValid(sal_False),
//			mbUseAlphaChannel(bUseAlphaChannel)
//		{
//		}
//
//		// The destructor.
//		VOCBitmapBuffer_old::~VOCBitmapBuffer_old()
//		{
//		}
//
//		// Prepare deletion of this object.
//		void VOCBitmapBuffer_old::PrepareDelete()
//		{
//			// delete evtl. buffered bitmap
//			DisposeBuffer();
//
//			// call parent
//			ViewObjectContact::PrepareDelete();
//		}
//
//		// Paint this object. This is before evtl. SubObjects get painted. This method
//		// needs to set the flag mbIsPainted and mbIsInvalidated and to set the
//		// maPaintedRectangle member. This information is later used for invalidates
//		// and repaints.
//		void VOCBitmapBuffer_old::PaintObject(DisplayInfo& rDisplayInfo)
//		{
//			sal_Bool bIsPainted(sal_False);
//			OutputDevice* pOriginalOutDev = rDisplayInfo.GetOutputDevice();
//
//			if(pOriginalOutDev)
//			{
//				Rectangle aNewSizePixel = CalculateClippedPixelRectangle(rDisplayInfo);
//
//				// check for changed cached pixel size, evtl. dispose buffer
//				if(CachedBitmapIsValid() 
//					&& (aNewSizePixel.GetSize() != maBitmapEx.GetSizePixel()))
//				{
//					DisposeBuffer();
//				}
//
//				// check for changed position, evtl. dispose buffer
//				if(CachedBitmapIsValid() 
//					&& (pOriginalOutDev->GetMapMode() != maCachedMapMode))
//				{
//					DisposeBuffer();
//				}
//
//				// check if buffering shall be done
//				if(!rDisplayInfo.OutputToPrinter() 
//					&& !rDisplayInfo.OutputToRecordingMetaFile()
//					&& rDisplayInfo.IsBufferingAllowed())
//				{
//					// try to create bitmap buffer
//					if(!CachedBitmapIsValid())
//					{
//						TryToCreateBuffer(aNewSizePixel, rDisplayInfo);
//					}
//
//					// If bitmap buffer exists, use it now
//					if(CachedBitmapIsValid())
//					{
//						// paint using cached bitmap
//						const sal_Bool bWasEnabled(pOriginalOutDev->IsMapModeEnabled());
//						pOriginalOutDev->EnableMapMode(sal_False);
//						pOriginalOutDev->DrawBitmapEx(aNewSizePixel.TopLeft(), maBitmapEx);
//						pOriginalOutDev->EnableMapMode(bWasEnabled);
//
//						// Set state flags. This overrides results from a contained
//						// call to PaintObject(..).
//						mbIsPainted = sal_True;
//						mbIsInvalidated = sal_False;
//						
//						// set painted rectangle
//						const Rectangle& aPaintRectangle = GetViewContact().GetPaintRectangle();
//						maPaintedRectangle = aPaintRectangle;
//
//						// remember that it's painted
//						bIsPainted = sal_True;
//					}
//				}
//			}
//
//			// when not yet painted, use the standard way
//			if(!bIsPainted)
//			{
//				// paint normal, bitmap not available. Call parent.
//				ViewObjectContact::PaintObject(rDisplayInfo);
//			}
//		}
//
//		// React on changes of the object of this ViewContact
//		void VOCBitmapBuffer_old::ActionChanged()
//		{
//			// call parent
//			ViewObjectContact::ActionChanged();
//
//			// change state
//			if(CachedBitmapIsValid())
//			{
//				DisposeBuffer();
//			}
//		}
//
//		// Take some action when new objects are inserted
//		void VOCBitmapBuffer_old::ActionChildInserted(const Rectangle& rInitialRectangle)
//		{
//			// call parent
//			ViewObjectContact::ActionChildInserted(rInitialRectangle);
//
//			// change state.
//			if(CachedBitmapIsValid())
//			{
//				DisposeBuffer();
//			}
//		}
//	} // end of namespace contact
//} // end of namespace sdr

//////////////////////////////////////////////////////////////////////////////
// eof
