/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: backgrounddownload.cxx,v $
 *
 *  $Revision: 1.10 $
 *
 *  last change: $Author: obo $ $Date: 2006/09/16 17:20:34 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

// MARKER(update_precomp.py): autogen include statement, do not remove
#include "precompiled_ucbhelper.hxx"

/**************************************************************************
								TODO
 **************************************************************************

 *************************************************************************/

#include <memory>
#ifndef _OSL_DIAGNOSE_H_
#include <osl/diagnose.h>
#endif
#ifndef _OSL_THREAD_HXX_
#include <osl/thread.hxx>
#endif
#ifndef _UCBHELPER_BACKGROUNDDOWNLOAD_HXX_
#include <ucbhelper/backgrounddownload.hxx>
#endif
#ifndef _SALHELPER_CONDITION_HXX_
#include <salhelper/condition.hxx>
#endif
#ifndef _CPPUHELPER_IMPLBASE1_HXX_
#include <cppuhelper/implbase1.hxx>
#endif
#ifndef _CPPUHELPER_IMPLBASE2_HXX_
#include <cppuhelper/implbase2.hxx>
#endif
#ifndef _COM_SUN_STAR_IO_XSEEKABLE_HPP_
#include <com/sun/star/io/XSeekable.hpp>
#endif
#ifndef _COM_SUN_STAR_IO_XINPUTSTREAM_HPP_
#include <com/sun/star/io/XInputStream_Impl.hpp>
#endif
#ifndef _COM_SUN_STAR_TASK_XINTERACTIONHANDLER_HPP_
#include <com/sun/star/task/XInteractionHandler.hpp>
#endif
#ifndef _COM_SUN_STAR_UCB_XPROGRESSHANDLER_HPP_
#include <com/sun/star/ucb/XProgressHandler.hpp>
#endif
#include "cppuhelper/weak.hxx"
#include "rtl/ustring.h"
#include "rtl/ustring.hxx"
#include <limits>
// for tmpfile
#include <stdio.h>


#define CONDITION(NAME,COND)                                                \
  friend class NAME;                                                        \
  class NAME                                                                \
     : public salhelper::Condition                                          \
  {                                                                         \
  public:                                                                   \
      NAME(osl::Mutex& rMutex, State& rState)                               \
          : salhelper::Condition(rMutex),                                   \
            m_rState(rState)                                                \
 { }                                                                        \
  protected:                                                                \
      bool applies()const {                                                 \
          return (COND) ; }                                                 \
  private:                                                                  \
      State& m_rState;                                                      \
  }

#define WAIT(COND)  \
    salhelper::ConditionWaiter aWaiter( m_pImpl -> COND )

#define SET(COND)   \
    m_pImpl->m_aState.nState = DownloadThread_Impl::COND

#define GUARD       \
    osl::MutexGuard aGuard(m_pImpl->m_aMutex)


#define WAIT_STATE(COND)  \
    salhelper::ConditionWaiter aWaiter( COND )

#define SET_STATE(COND)   \
    m_aState.nState = COND

#define GUARD_STATE       \
    osl::MutexGuard aGuard(m_aMutex)


using namespace com::sun::star::uno;
using namespace com::sun::star::lang;
using namespace com::sun::star::io;
using namespace com::sun::star::ucb;
using namespace com::sun::star::task;



namespace ucb_impl
{
    
    class DownloadThread_Impl;
    
    
    class CommandEnvironment
        : public ::cppu::WeakImplHelper1<XCommandEnvironment>
    {
    public:
        
        CommandEnvironment(
            DownloadThread_Impl* pDT,
            const Reference<XCommandEnvironment>& xEnv);
        
        Reference<XInteractionHandler> SAL_CALL
        getInteractionHandler(  
        ) 
            throw (
                RuntimeException
            );
        
        Reference<XProgressHandler> SAL_CALL
        getProgressHandler(
        )
            throw (
                RuntimeException
            );
    
    private:
    
        osl::Mutex                      m_aMutex;
        DownloadThread_Impl*            m_pDT;
        Reference<XCommandEnvironment>  m_xEnv;
        Reference<XInteractionHandler>  m_xIntAct;
        Reference<XProgressHandler>     m_xProgHand;
    };

    

    class InputStream: 
        public ucb::DataSink,
        public ::cppu::WeakImplHelper2<XInputStream,XSeekable>
    {
        friend class DownloadThread_Impl;
        
    public:
        
        InputStream(DownloadThread_Impl* pDT);
        
        virtual ~InputStream();
        
        // XInputStream
        virtual sal_Int32 SAL_CALL
        readBytes(
            Sequence< sal_Int8 > & aData,
            sal_Int32 nBytesToRead
        )
            throw( 
                NotConnectedException,
                BufferSizeExceededException,
                IOException,
                RuntimeException
            );
        
        virtual sal_Int32 SAL_CALL
        readSomeBytes(
            Sequence< sal_Int8 > & aData,
            sal_Int32 nMaxBytesToRead
        )
            throw( 
                NotConnectedException,
                BufferSizeExceededException,
                IOException,
                RuntimeException
            );
        
        virtual void SAL_CALL
        skipBytes(
            sal_Int32 nBytesToSkip
        )
            throw( 
                NotConnectedException,
                BufferSizeExceededException,
                IOException,
                RuntimeException
            );
        
        virtual sal_Int32 SAL_CALL
        available( 
            void
        )
            throw(
                NotConnectedException,
                IOException,
                RuntimeException
            );
        
        virtual void SAL_CALL
        closeInput( void )
            throw( 
                NotConnectedException,
                IOException,
                RuntimeException
            );
        
        // XSeekable
        virtual void SAL_CALL
        seek(
            sal_Int64 location
        )
            throw(
                IllegalArgumentException,
                IOException,
                RuntimeException
            );
        
        virtual sal_Int64 
        SAL_CALL 
        getPosition()
            throw( 
                IOException,
                RuntimeException
            );
        
        virtual sal_Int64 SAL_CALL
        getLength()
            throw(
                IOException,
                RuntimeException
            );
        
    private:
        
        int write(void *data,size_t size,size_t nmemb);
        
        
        /** data == 0 means skipping
         */        
        int read(void* data,size_t size,size_t nmemb,bool block = true);
        
        
        DownloadThread_Impl*         m_pImpl;
        
        size_t                       m_nLen;
        size_t                       m_nPos;
        size_t                       m_nMaxMem;
        Sequence<sal_Int8>           m_aInputBuffer;
        FILE*                        m_pFILE;
    };


    
    class  DownloadThread_Impl
        : public osl::Thread
    {
    public:
        
        DownloadThread_Impl(
            ucb::DownloadThread *pThread,
            const Reference<XCommandEnvironment >& xEnv);
        
        ~DownloadThread_Impl();
        
        
        void handle( const Reference<XInteractionRequest >& Request )
        {
            {
                salhelper::ConditionModifier aMod(m_aGET);
                m_xRequest = Request;
                SET_STATE(HANDLE_WAIT_HANDLE);
            }
            WAIT_STATE(m_aINITDONE);
        }
        
        void push( const Any& Status )
        {
            {
                salhelper::ConditionModifier aMod(m_aGET);
                m_aAny = Status;
                SET_STATE(HANDLE_WAIT_PUSH);
            }
            WAIT_STATE(m_aINITDONE);
        }
        
        void update( const Any& Status )
        {
            {
                salhelper::ConditionModifier aMod(m_aGET);
                m_aAny = Status;
                SET_STATE(HANDLE_WAIT_UPDATE);
            }
            WAIT_STATE(m_aINITDONE);
        }
        
        
        void pop()
        {
            {
                salhelper::ConditionModifier aMod(m_aGET);
                SET_STATE(HANDLE_WAIT_POP);
            }
            WAIT_STATE(m_aINITDONE);
        }
        
        void OnDownload();
        
        void OnAbort();
        
        InputStream* GetSink(size_t nMaxMem);

        Reference<XCommandEnvironment> GetEnv();
        
        
        enum THREADSTATE {
            WAIT_INITDONE,
            
            HANDLE_WAIT_PUSH,
            HANDLE_WAIT_UPDATE,
            HANDLE_WAIT_POP,
            HANDLE_WAIT_HANDLE,
            
            WAIT_DELETE,
            DELETABLE_WAIT
        };

        struct State
        {
            inline State():
                bErrorOccured(false),
                bInputStreamWritten(false),
                nState(DELETABLE_WAIT) // initially we are deletable
            {}

            bool bErrorOccured;
            bool bInputStreamWritten;
            THREADSTATE nState;
        };
        
        CONDITION(
            COND_INITDONE,
            m_rState.nState == WAIT_INITDONE);
        
        CONDITION(
            COND_GET,
            m_rState.nState == HANDLE_WAIT_PUSH    || 
            m_rState.nState == HANDLE_WAIT_UPDATE  || 
            m_rState.nState == HANDLE_WAIT_POP     || 
            m_rState.nState == HANDLE_WAIT_HANDLE  || 
            m_rState.nState == DELETABLE_WAIT      ||
            m_rState.bErrorOccured                 ||
            m_rState.bInputStreamWritten );
        
        CONDITION(
            COND_READ,
            m_rState.nState == DELETABLE_WAIT   ||
            m_rState.bErrorOccured              ||
            m_rState.bInputStreamWritten );
        
        CONDITION(
            COND_DELETE,
            m_rState.nState == WAIT_DELETE);
        
        CONDITION(
            COND_DELETABLE,
            m_rState.nState == DELETABLE_WAIT);
        
        osl::Mutex                      m_aMutex;
        ucb::DownloadThread *           m_pThread;
        Reference<XCommandEnvironment>  m_xEnv;

        State                           m_aState;
        
        COND_INITDONE                   m_aINITDONE;
        COND_GET                        m_aGET;
        COND_DELETE                     m_aDELETE;     // the object will be
        //                                             // deleted
        COND_READ                       m_aREAD;       // data are written,
        //                                             // an error occured
        //                                             // or the download ended
        COND_DELETABLE                  m_aDELETABLE;  // we have left the 
        //                                             // the work function
        
        InputStream*                    m_pSink;
        Reference<XInteractionRequest > m_xRequest;
        Any                             m_aAny;

    protected:
        
        /** overloaded methods of osl-thread class
         */
        
        void SAL_CALL run();
        
        void SAL_CALL onTerminated();

    };  // end class DownloadThread
    

    
}   // end namespace ucb_impl




namespace ucb_impl {


DownloadThread_Impl::DownloadThread_Impl(
    ucb::DownloadThread* pThread,
    const Reference<XCommandEnvironment >& xEnv
)
    : m_aMutex(),
      m_pThread(pThread),
      m_xEnv(xEnv),
      m_aINITDONE(m_aMutex, m_aState),
      m_aGET(m_aMutex, m_aState),
      m_aDELETE(m_aMutex, m_aState),
      m_aREAD(m_aMutex, m_aState),
      m_aDELETABLE(m_aMutex, m_aState)
{
    m_pSink = new InputStream(this);

    // well, that's a little bit nitpicking here, but belongs to
    // the general principle
    salhelper::ConditionModifier aMod(m_aDELETABLE);
}


DownloadThread_Impl::~DownloadThread_Impl()
{
    delete m_pThread;
}


void DownloadThread_Impl::OnDownload()
{
    {
        salhelper::ConditionModifier aMod(m_aGET);
        {
            salhelper::ConditionModifier aMod2(m_aREAD);
            m_aState.bInputStreamWritten = true;
        }
    }
}


void DownloadThread_Impl::OnAbort()
{
    salhelper::ConditionModifier aMod(m_aGET);
    {
        salhelper::ConditionModifier aMod2(m_aREAD);
        m_aState.bErrorOccured = true;
    }
}


InputStream* DownloadThread_Impl::GetSink(size_t nMaxMem) 
{
    GUARD_STATE;
    if(m_pSink->m_pFILE) fclose(m_pSink->m_pFILE),m_pSink->m_pFILE = 0;
    m_pSink->m_nLen = m_pSink->m_nPos = 0;
    m_pSink->m_nMaxMem = nMaxMem;
    m_pSink->m_aInputBuffer.realloc(0);
    m_aState.bInputStreamWritten = m_aState.bErrorOccured = false;
    m_aState.nState = WAIT_INITDONE;
    return m_pSink;
}


Reference<XCommandEnvironment> DownloadThread_Impl::GetEnv()
{
    Reference<XCommandEnvironment> xRet;
    Reference<XCommandEnvironment> xEnv;
    
    {
        GUARD_STATE;
        xEnv = m_xEnv;
    }
    
    if(xEnv.is())
        xRet = new CommandEnvironment(this,xEnv);
    
    return xRet;
}


/** osl - functions **/

void SAL_CALL DownloadThread_Impl::run()
{
    {
        WAIT_STATE(m_aINITDONE);
    }
    
    m_pThread->work();
    
    {
        salhelper::ConditionModifier aMod(m_aREAD);
        {
            salhelper::ConditionModifier aMod2(m_aDELETABLE);
            {
                salhelper::ConditionModifier aMod3(m_aGET);
                SET_STATE(DELETABLE_WAIT);
            }
        }
    }
}



void SAL_CALL DownloadThread_Impl::onTerminated()
{
}



InputStream::InputStream(DownloadThread_Impl* pDT)
    : m_pImpl(pDT),
      m_nLen(0),
      m_nPos(0),
      m_nMaxMem(1024*1024),
      m_aInputBuffer(0),
      m_pFILE(0)
{
}


InputStream::~InputStream()
{
    {
        WAIT(m_aDELETABLE);
    }
    
    {
        salhelper::ConditionModifier aMod(m_pImpl->m_aDELETE);
        SET(WAIT_DELETE);
    }
    
    if(m_pFILE) fclose(m_pFILE);
    m_pImpl->join();    
    delete m_pImpl;
}



int InputStream::write(void *data,size_t size,size_t nmemb)
{
    int ret = size*nmemb;
    {
        GUARD;
        if(m_pImpl->m_aState.bErrorOccured)
            return -1;
        
        if(m_pImpl->m_aState.nState == DownloadThread_Impl::DELETABLE_WAIT)
            // well, already closed, so simply return;
            return ret;
        
        if(m_pFILE)
        {
            fseek(m_pFILE,m_nLen,SEEK_SET);
            ret = fwrite(data,size,nmemb,m_pFILE);
            if(ret > 0)
                m_nLen += ret;
        }
        else
        {
            m_aInputBuffer.realloc(m_nLen + ret );
            rtl_copyMemory(m_aInputBuffer.getArray() + m_nLen,data,ret);
            m_nLen += ret;

            if(m_nLen > m_nMaxMem)
            {
                // switch to file now
                m_pFILE = tmpfile();
                if(m_pFILE)
                    if(fwrite(
                           (void*)m_aInputBuffer.getArray(),
                           1,m_aInputBuffer.getLength(),
                           m_pFILE)
                       != sal::static_int_cast< sal_uInt32 >(
                           m_aInputBuffer.getLength()))
                        fclose(m_pFILE),m_pFILE = 0;
                    else
                        m_aInputBuffer.realloc(0);
            }
        }
    }
    
    if(ret >=  0) {
        salhelper::ConditionModifier aMod(m_pImpl->m_aGET);
        {
            salhelper::ConditionModifier aMod2(m_pImpl->m_aREAD);
            m_pImpl->m_aState.bInputStreamWritten = true;
        }
    }
    
    return ret;
}


int InputStream::read(void *data,size_t size,size_t nmemb,bool block)
{
    bool bErrorOccured(false);
    bool bDownloadReady(false);
    size_t requested = nmemb*size;
    sal_Int8* pData = (sal_Int8*) data;
    
    while(true)
    {
        {
            WAIT(m_aREAD);
            
            if(m_pImpl->m_aState.bErrorOccured)
                bErrorOccured = m_pImpl->m_aState.bErrorOccured;
            else
            {
                bDownloadReady = 
                    m_pImpl->m_aState.nState == 
                    DownloadThread_Impl::DELETABLE_WAIT;
                // Write the data
                size_t copy = requested;
                size_t remaining = m_nLen - m_nPos;
                if( copy > remaining )
                    copy = remaining;
                
                // Read the data
                if(pData) {
                    if(m_pFILE) {
                        fseek(m_pFILE,m_nPos,SEEK_SET);
                        copy = fread(pData,1,copy,m_pFILE);
                        if(ferror(m_pFILE)) bErrorOccured = true;
                    }
                    else
                        rtl_copyMemory(
                            pData,
                            m_aInputBuffer.getConstArray() + m_nPos, 
                            copy);
                    
                    pData += copy;
                }
                // Update our stream position for next time
                m_nPos += copy;
                requested -= copy;
            }
        }
        
        if(bErrorOccured || bDownloadReady || requested == 0)
            break;
        
        if(block) {
            GUARD;
            m_pImpl->m_aState.bInputStreamWritten = false;
        }
    }
 
    if(bErrorOccured)
        return -1;
    else
        return nmemb*size - requested;
}


}  // end namespace ucb_impl


using namespace ucb_impl;
using namespace ucb;


DownloadThread::DownloadThread(const Reference<XCommandEnvironment >& xEnv)
    : m_pImpl(0),
      m_xEnv(xEnv)
{
}



DownloadThread::~DownloadThread()
{
}



bool DownloadThread::Get(
    const Reference<XActiveDataSink>& xDataSink) const
{
    m_aMutex.acquire();
    
    // can only be called successfully once;
    if(m_pImpl) {
        Reference<XInputStream> xInpStr(m_pImpl->m_pSink);
        m_aMutex.release();
        if(xDataSink.is()) xDataSink->setInputStream(xInpStr);
        return true;
    }
    
    m_pImpl = new DownloadThread_Impl(
        const_cast<DownloadThread*>(this),m_xEnv);
    
    if(!m_pImpl->create())
    {
        // give the callee the chance to handle his errors
        m_pImpl->m_pThread = 0;
        // initially the thing is deletable, but we don't want
        // to become deleted by ourself.
        delete m_pImpl->m_pSink,m_pImpl = 0;
        m_aMutex.release();
        return false;
    }
    else
    {
        while(true) {
            {
                salhelper::ConditionModifier aMod(m_pImpl->m_aINITDONE);
                SET(WAIT_INITDONE);
            }
            
            bool bErrorOccured;
            bool bInputStreamWritten;
            DownloadThread_Impl::THREADSTATE nState;
            InputStream*                     pInpStr;
            Reference<XInteractionRequest >  xRequest;
            Any                              aAny;
            
            {
                WAIT(m_aGET);
                bErrorOccured = m_pImpl->m_aState.bErrorOccured;
                bInputStreamWritten = m_pImpl->m_aState.bInputStreamWritten;
                nState = m_pImpl->m_aState.nState;
                pInpStr = m_pImpl->m_pSink;
                xRequest = m_pImpl->m_xRequest;
                aAny = m_pImpl->m_aAny;
            }
            
            if(bInputStreamWritten)
            {
                m_aMutex.release();
                if(xDataSink.is())
                    xDataSink->setInputStream(pInpStr);
                return true;
            }
            else if(bErrorOccured)
            {
                {
                    salhelper::ConditionModifier aMod(m_pImpl->m_aDELETABLE);
                    SET(DELETABLE_WAIT);
                }
                // give the callee the chance to handle his errors
                // and reset without reinit.
                m_pImpl->m_pThread = 0;
                delete m_pImpl->m_pSink,m_pImpl = 0;
                return false;
            }
            
            switch(nState) {
            case  DownloadThread_Impl::HANDLE_WAIT_UPDATE:  //Interaction
                {
                    DownloadThread_Impl* p = m_pImpl;
                    Reference<XCommandEnvironment> xEnv(m_xEnv);
                    m_aMutex.release();
                    Reference<XProgressHandler> xProgHand;
                    if(xEnv.is() &&
                       (xProgHand=(xEnv->getProgressHandler())).is())
                        xProgHand->update(aAny);
                    m_aMutex.acquire();
                    m_xEnv = xEnv;
                    m_pImpl = p;
                    break;
                }
            case  DownloadThread_Impl::HANDLE_WAIT_POP:  //Interaction
                {
                    DownloadThread_Impl* p = m_pImpl;
                    Reference<XCommandEnvironment> xEnv(m_xEnv);
                    m_aMutex.release();
                    Reference<XProgressHandler> xProgHand;
                    if(xEnv.is() && 
                       (xProgHand=(xEnv->getProgressHandler())).is())
                        xProgHand->pop();
                    m_aMutex.acquire();
                    m_xEnv = xEnv;
                    m_pImpl = p;
                    break;
                }
            case  DownloadThread_Impl::HANDLE_WAIT_PUSH:  //Interaction
                {
                    DownloadThread_Impl* p = m_pImpl;
                    Reference<XCommandEnvironment> xEnv(m_xEnv);
                    m_aMutex.release();
                    Reference<XProgressHandler> xProgHand;
                    if(xEnv.is() && 
                       (xProgHand=(xEnv->getProgressHandler())).is())
                        xProgHand->push(aAny);
                    m_aMutex.acquire();
                    m_xEnv = xEnv;
                    m_pImpl = p;
                    break;
                }
            case  DownloadThread_Impl::HANDLE_WAIT_HANDLE:  //Interaction
                {
                    DownloadThread_Impl* p = m_pImpl;
                    Reference<XCommandEnvironment> xEnv(m_xEnv);
                    m_aMutex.release();
                    Reference<XInteractionHandler> xIntAct;
                    if(xEnv.is() &&
                       (xIntAct=(xEnv->getInteractionHandler())).is())
                        xIntAct->handle(xRequest);
                    m_aMutex.acquire();
                    m_xEnv = xEnv;
                    m_pImpl = p;
                    break;
                }                
            default:  // DELETABLE_WAIT for instance
                {
                    m_aMutex.release();
                    xDataSink->setInputStream(pInpStr);
                    return true;
                }
            }
        }
    }
}



void DownloadThread::OnDownload()
{
    if(m_pImpl)
        m_pImpl->OnDownload();
    else
        return;
}


void DownloadThread::OnAbort()
{
    if(m_pImpl)
        m_pImpl->OnAbort();
    else
        return;
}


DataSink* DownloadThread::GetSink(size_t nMaxMem) const
{
    if(m_pImpl)
        return m_pImpl->GetSink(nMaxMem);
    else
        return 0;
}



Reference<XCommandEnvironment>
DownloadThread::GetEnv()
{
    if(m_pImpl)
        return m_pImpl->GetEnv();
    else
        return m_xEnv;
}


/****************************************************

// Implementation CommandEnvironment


*****************************************************/

namespace ucb_impl
{


CommandEnvironment::CommandEnvironment(
    DownloadThread_Impl* pDT,
    const Reference<XCommandEnvironment>& xEnv)
    : m_pDT(pDT),
      m_xEnv(xEnv)
{
}


class ProgressHandler
    : public ::cppu::WeakImplHelper1<XProgressHandler>
{
public:
	
    ProgressHandler(DownloadThread_Impl *pDT)
        : m_pDT(pDT)
    {
    }
    
    void SAL_CALL push( const Any& Status ) 
        throw (
            RuntimeException)
    {
        if(m_pDT)
            m_pDT->push(Status);
    }
        
    void SAL_CALL update( const Any& Status )
        throw (RuntimeException)
    {
        if(m_pDT)
            m_pDT->update(Status);
    }
	
    void SAL_CALL pop(  ) 
        throw (RuntimeException)
    {
        if(m_pDT)
            m_pDT->pop();
    }
    
	
private:
        
    DownloadThread_Impl *m_pDT;
};
    


class InteractionHandler
    : public ::cppu::WeakImplHelper1<XInteractionHandler>
{
public:
        
    InteractionHandler(DownloadThread_Impl *pDT)
        : m_pDT(pDT)
    {
    }
        
        
    void SAL_CALL 
    handle( const Reference<XInteractionRequest >& Request ) 
        throw (RuntimeException)
    {
        if(m_pDT)
            m_pDT->handle(Request);
    }
        
private:
        
    DownloadThread_Impl *m_pDT;
};
    



Reference<XInteractionHandler> SAL_CALL
CommandEnvironment::getInteractionHandler(  
) 
    throw (
        RuntimeException
    )
{
    Reference<XCommandEnvironment> xEnv;
    Reference<XInteractionHandler> xIntAct;

    {
        osl::MutexGuard aGuard(m_aMutex);
        xIntAct = m_xIntAct;
        xEnv = m_xEnv;
    }
    
    if(!xIntAct.is() && xEnv->getInteractionHandler().is()) {
        osl::MutexGuard aGuard(m_aMutex);
        xIntAct = m_xIntAct = new InteractionHandler(m_pDT);
    }
    
    return xIntAct;
}


    
Reference<XProgressHandler> SAL_CALL
CommandEnvironment::getProgressHandler(
)
    throw (
        RuntimeException
    )
{
    Reference<XCommandEnvironment> xEnv;
    Reference<XProgressHandler> xProgHand;
    
    {
        osl::MutexGuard aGuard(m_aMutex);
        xProgHand = m_xProgHand;
        xEnv = m_xEnv;
    }
    
    if(!xProgHand.is() && xEnv->getProgressHandler().is()) {
        osl::MutexGuard aGuard(m_aMutex);
        xProgHand = m_xProgHand = new ProgressHandler(m_pDT);
    }
    
    return xProgHand;
}


// XInputStream
sal_Int32 SAL_CALL
InputStream::readBytes(
    Sequence< sal_Int8 > & aData,
    sal_Int32 nBytesToRead
)
    throw( 
        NotConnectedException,
        BufferSizeExceededException,
        IOException,
        RuntimeException
    )
{
    if(nBytesToRead < 0)
        return 0;
    aData.realloc(nBytesToRead);
    int n = read(aData.getArray(),1,size_t(nBytesToRead));
    if(n < 0)
        throw IOException();
    else
        return n;
}


sal_Int32 SAL_CALL
InputStream::readSomeBytes(
    Sequence< sal_Int8 > & aData,
    sal_Int32 nBytesToRead
)
    throw( 
        NotConnectedException,
        BufferSizeExceededException,
        IOException,
        RuntimeException
    )
{
    if(nBytesToRead < 0)
        return 0;
    aData.realloc(nBytesToRead);
    int n = read(aData.getArray(),1,size_t(nBytesToRead),false);
    if(n < 0)
        throw IOException();
    else
        return n;
}



void SAL_CALL
InputStream::skipBytes(
    sal_Int32 nBytesToSkip
)
    throw( 
        NotConnectedException,
        BufferSizeExceededException,
        IOException,
        RuntimeException
    )
{
    if(nBytesToSkip < 0)
        return;
    if(read(0,1,size_t(nBytesToSkip)) < -1)
        throw IOException();
}


sal_Int32 SAL_CALL
InputStream::available( 
    void
)
    throw(
        NotConnectedException,
        IOException,
        RuntimeException
    )
{
    GUARD;
    return m_nLen-m_nPos;
}

void SAL_CALL
InputStream::closeInput( void )
    throw( 
        NotConnectedException,
        IOException,
        RuntimeException
    )
{
    salhelper::ConditionModifier aMod(m_pImpl->m_aDELETE);
    {
        salhelper::ConditionModifier aMod2(m_pImpl->m_aREAD);
        SET(DELETABLE_WAIT);
    }
}


// XSeekable
void SAL_CALL
InputStream::seek(
    sal_Int64 location
)
    throw(
        IllegalArgumentException,
        IOException,
        RuntimeException
    )
{
    if (location < 0) {
        throw IllegalArgumentException(
            rtl::OUString(
                RTL_CONSTASCII_USTRINGPARAM("seek to negative location")),
            static_cast< cppu::OWeakObject * >(this), 0);
    }
    sal_Int32 skip = 0;
    {
        GUARD;
        if(sal::static_int_cast< sal_uInt64 >(location) < m_nLen) {
            m_nPos = sal::static_int_cast< size_t >(location);
            return;
        } else if (sal::static_int_cast< sal_uInt64 >(location) - m_nPos
                   > sal::static_int_cast< sal_uInt32 >(
                       std::numeric_limits< sal_Int32 >::max()))
        {
            throw RuntimeException(
                rtl::OUString(
                    RTL_CONSTASCII_USTRINGPARAM(
                        "seek to location that causes overflow")),
                static_cast< cppu::OWeakObject * >(this));
        } else {
            skip = sal::static_int_cast< sal_Int32 >(
                sal::static_int_cast< sal_uInt64 >(location) - m_nPos);
        }
    }
    skipBytes(skip);
}

sal_Int64 
SAL_CALL 
InputStream::getPosition()
    throw( 
        IOException,
        RuntimeException
    )
{
    GUARD;
    return m_nPos;
}


sal_Int64 SAL_CALL
InputStream::getLength()
    throw(
        IOException,
        RuntimeException
    )
{
    GUARD;
//     sal_Int64 nPos = getPosition();
    
//     while(read(0,1,1024) == 1024);
//     seek(nPos);
    return m_nLen;
}

}   // end namespace ucb_impl



extern "C" int ucb::writeToDataSink(
    void *data,size_t size,size_t nmemb,void *userData)
{
    DataSink *sink =
        reinterpret_cast<DataSink*>(userData);
    
    if(sink) return sink->write(data,size,nmemb);
    else return 0;
}

