/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: ScrollPanel.hxx,v $
 *
 *  $Revision: 1.5 $
 *
 *  last change: $Author: kz $ $Date: 2006/12/12 17:54:37 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef SD_TOOLPANEL_SCROLL_PANEL_HXX
#define SD_TOOLPANEL_SCROLL_PANEL_HXX

#include "taskpane/TaskPaneTreeNode.hxx"

#include <vcl/ctrl.hxx>
#include <vcl/scrbar.hxx>
#include <memory>
#include <vector>

namespace sd { namespace toolpanel {

class TitledControl;

/** The scroll panel shows its controls one above the other.  When their
    total height is larger than the height of the scroll area then only a
    part of the controls is visible.  Scroll bars control which part that
    is.

    The scroll panel registers itself as window event listener at the
    controls and their title bars (conceptually; it really is the
    TitledControl) to track changes of the selection and focus rectangles.
    On such a change it tries to move the selected or focused part into the
    visible area.  At the moment this moving into view only works with
    valuesets and TitleBars.
*/
class ScrollPanel
    : public ::Control,
      public TreeNode
{
public:
    /** Create a new sub tool panel with the given window as its
        parent.  This will usually be a child window.
    */
    ScrollPanel (TreeNode* pParent);
    virtual ~ScrollPanel (void);

    /** Add a control to the sub panel.  An title bar is added above the
        control.
        @param rTitle
            The title that will be shown in the two title bars that
            belong to the control.
        @param nHelpId
            The help id is set at the title bar not the actual control.
        @return
            The new titled control that contains the given control and a new
            title bar as children is returned.
    */
    TitledControl* AddControl (
        ::std::auto_ptr<TreeNode> pControl, 
        const String& rTitle,
        ULONG nHelpId);

    /** Add a control to the sub panel without a title bar.
    */
    void AddControl (::std::auto_ptr<TreeNode> pControl);

    virtual void Paint (const Rectangle& rRect);

    /** Initiate a rearrangement of the controls and title bars.
    */
    virtual void Resize (void);

    virtual void RequestResize (void);

    virtual Size GetPreferredSize (void);
    virtual sal_Int32 GetPreferredWidth (sal_Int32 nHeight);
    virtual sal_Int32 GetPreferredHeight (sal_Int32 nWidth);
    virtual bool IsResizable (void);
    virtual ::Window* GetWindow (void);
    virtual sal_Int32 GetMinimumWidth (void);

    virtual void ExpandControl (
        TreeNode* pControl,
        bool bExpansionState);

    bool IsVerticalScrollBarVisible (void) const;
    bool IsHorizontalScrollBarVisible (void) const;
    ScrollBar& GetVerticalScrollBar (void);
    ScrollBar& GetHorizontalScrollBar (void);

    // ::Window
    virtual long Notify( NotifyEvent& rNEvt );

    virtual ::com::sun::star::uno::Reference<
        ::com::sun::star::accessibility::XAccessible> CreateAccessibleObject (
            const ::com::sun::star::uno::Reference<
            ::com::sun::star::accessibility::XAccessible>& rxParent);

    /** Scroll the given rectangle into the visible area.
        @param aRectangle
            The box to move into the visible area in pixel coordinates
            relative to the given window.
        @param pWindow
            This window is used to translate the given coordinates into ones
            that are relative to the scroll panel.
        
    */
    void MakeRectangleVisible (
        Rectangle& aRectangle,
        ::Window* pWindow);

protected:
    /** Initiate a rearrangement of the controls.
    */
    void ListHasChanged (void);

private:
    ::Control maScrollWindow;
    ScrollBar maVerticalScrollBar;
    ScrollBar maHorizontalScrollBar;
    ::Window maScrollBarFiller;
    ::Window maScrollWindowFiller;
    Point maScrollOffset;
    bool mbIsRearrangePending;
    bool mbIsLayoutPending;
    sal_uInt32 mnChildrenWidth;
    /// Border above top-most and below bottom-most control.
    const int mnVerticalBorder;
    /// Gap between two controls.
    const int mnVerticalGap;
    /// Border at the left and right of the controls.
    const int mnHorizontalBorder;
    /** List of horizontal stripes that is created from the gaps between
        children when they are layouted.  The stripes are painted in Paint()
        to fill the space arround the children.
    */
    typedef ::std::vector< ::std::pair<int,int> > StripeList;
    StripeList maStripeList;

    /** Calculate position, size, and visibility of the controls.
        Call this method after the list of controls, their expansion
        state, or the size of the sub panel has changed.
    */
    void Rearrange (void);

    /** Determine the minimal size that is necessary to show the controls
        one over the other.  It may be smaller than the available area.
    */
    Size GetRequiredSize (void);

    /** Place the child windows one above the other and return the size of
        the bounding box.
    */
    sal_Int32 LayoutChildren (void);

    Size SetupScrollBars (const Size& rRequiresSize);
    sal_Int32 SetupVerticalScrollBar (bool bShow, sal_Int32 nRange);
    sal_Int32 SetupHorizontalScrollBar (bool bShow, sal_Int32 nRange);

    DECL_LINK(ScrollBarHandler, ScrollBar*);
    DECL_LINK(WindowEventListener, VclSimpleEvent*);

	using Window::GetWindow;
};

} } // end of namespace ::sd::toolpanel

#endif
