/**
 * ===========================================================
 * LibRepository : a free Java content repository access layer
 * ===========================================================
 *
 * Project Info:  http://reporting.pentaho.org/librepository/
 *
 * (C) Copyright 2006-2008, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 * ------------
 * ZipReadContentItem.java
 * ------------
 */

package org.pentaho.reporting.libraries.repository.zipreader;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Date;
import java.util.zip.InflaterInputStream;
import java.util.zip.ZipEntry;

import org.pentaho.reporting.libraries.repository.ContentCreationException;
import org.pentaho.reporting.libraries.repository.ContentIOException;
import org.pentaho.reporting.libraries.repository.ContentItem;
import org.pentaho.reporting.libraries.repository.ContentLocation;
import org.pentaho.reporting.libraries.repository.LibRepositoryBoot;
import org.pentaho.reporting.libraries.repository.Repository;
import org.pentaho.reporting.libraries.base.util.IOUtils;

/**
 * Creation-Date: 17.12.2007, 12:19:20
 *
 * @author Thomas Morgner
 */
public class ZipReadContentItem implements ContentItem
{
  private String comment;
  private String name;
  private long size;
  private long time;
  private ZipReadRepository repository;
  private byte[] rawData;
  private ZipReadContentLocation parent;
  private String entryName;

  public ZipReadContentItem(final ZipReadRepository repository,
                            final ZipReadContentLocation parent,
                            final ZipEntry zipEntry,
                            final byte[] bytes)
  {
    if (repository == null)
    {
      throw new NullPointerException();
    }
    if (zipEntry == null)
    {
      throw new NullPointerException();
    }
    if (bytes == null)
    {
      throw new NullPointerException();
    }

    this.parent = parent;
    this.repository = repository;
    this.comment = zipEntry.getComment();
    this.name = zipEntry.getName();
    this.entryName = IOUtils.getInstance().getFileName(name);
    this.size = zipEntry.getSize();
    this.time = zipEntry.getTime();
    this.rawData = bytes;
  }

  public String getMimeType() throws ContentIOException
  {
    return repository.getMimeRegistry().getMimeType(this);
  }

  public OutputStream getOutputStream() throws ContentIOException, IOException
  {
    throw new ContentCreationException("This repository is read-only");
  }

  public InputStream getInputStream() throws ContentIOException, IOException
  {
    return new InflaterInputStream(new ByteArrayInputStream(rawData));
  }

  public boolean isReadable()
  {
    return true;
  }

  public boolean isWriteable()
  {
    return false;
  }

  public String getName()
  {
    return entryName;
  }

  public Object getContentId()
  {
    return name;
  }

  public Object getAttribute(String domain, String key)
  {
    if (LibRepositoryBoot.REPOSITORY_DOMAIN.equals(domain))
    {
      if (LibRepositoryBoot.SIZE_ATTRIBUTE.equals(key))
      {
        return new Long(size);
      }
      else if (LibRepositoryBoot.VERSION_ATTRIBUTE.equals(key))
      {
        return new Date(time);
      }
    }
    else if (LibRepositoryBoot.ZIP_DOMAIN.equals(domain))
    {
      if (LibRepositoryBoot.ZIP_COMMENT_ATTRIBUTE.equals(key))
      {
        return comment;
      }
    }
    return null;
  }

  public boolean setAttribute(String domain, String key, Object value)
  {
    return false;
  }

  public ContentLocation getParent()
  {
    return parent;
  }

  public Repository getRepository()
  {
    return repository;
  }

  public boolean delete()
  {
    return false;
  }
}
