/*---------------------------------------------------------------------------*
 |              PDFlib - A library for generating PDF on the fly             |
 +---------------------------------------------------------------------------+
 | Copyright (c) 1997-2005 Thomas Merz and PDFlib GmbH. All rights reserved. |
 +---------------------------------------------------------------------------+
 |                                                                           |
 |    This software is subject to the PDFlib license. It is NOT in the       |
 |    public domain. Extended versions and commercial licenses are           |
 |    available, please check http://www.pdflib.com.                         |
 |                                                                           |
 *---------------------------------------------------------------------------*/

/* $Id: pdflib_py.c,v 1.32 2006/06/01 21:16:43 rjs Exp $
 *
 * Wrapper code for the PDFlib Python binding
 *
 */

#include <string.h>
#include <stdlib.h>

#ifdef __cplusplus
extern "C" {
#endif
#define PY_SSIZE_T_CLEAN 1
#include <Python.h>
#ifdef __cplusplus
}
#endif

#if PY_VERSION_HEX < 0x02050000
typedef int Py_ssize_t;
#define PY_SSIZE_T_MAX INT_MAX
#define PY_SSIZE_T_MIN INT_MIN
#endif

/* Compilers which are not strictly ANSI conforming can set PDF_VOLATILE
 * to an empty value.
 */
#ifndef PDF_VOLATILE
#define PDF_VOLATILE    volatile
#endif

/* Definitions for Windows/Unix exporting */
#if defined(__WIN32__)
#   if defined(_MSC_VER)
#	define SWIGEXPORT(a,b) __declspec(dllexport) a b
#   else
#	if defined(__BORLANDC__)
#	    define SWIGEXPORT(a,b) a _export b
#	else
#	    define SWIGEXPORT(a,b) a b
#	endif
#   endif
#else
#   define SWIGEXPORT(a,b) a b
#endif

#ifdef SWIG_GLOBAL
#ifdef __cplusplus
#define SWIGSTATIC extern "C"
#else
#define SWIGSTATIC
#endif
#endif

#ifndef SWIGSTATIC
#define SWIGSTATIC static
#endif

typedef struct {
  char  *name;
  PyObject *(*get_attr)(void);
  int (*set_attr)(PyObject *);
} swig_globalvar;

typedef struct swig_varlinkobject {
  PyObject_HEAD
  swig_globalvar **vars;
  int    nvars;
  int    maxvars;
} swig_varlinkobject;

/* ----------------------------------------------------------------------
   swig_varlink_repr()

   Function for python repr method
   ---------------------------------------------------------------------- */

static PyObject *
swig_varlink_repr(swig_varlinkobject *v)
{
  v = v;
  return PyString_FromString("<Global variables>");
}

/* ---------------------------------------------------------------------
   swig_varlink_print()

   Print out all of the global variable names
   --------------------------------------------------------------------- */

static int
swig_varlink_print(swig_varlinkobject *v, FILE *fp, int flags)
{

  int i = 0;
  flags = flags;
  fprintf(fp,"Global variables { ");
  while (v->vars[i]) {
    fprintf(fp,"%s", v->vars[i]->name);
    i++;
    if (v->vars[i]) fprintf(fp,", ");
  }
  fprintf(fp," }\n");
  return 0;
}

/* --------------------------------------------------------------------
   swig_varlink_getattr

   This function gets the value of a variable and returns it as a
   PyObject.   In our case, we'll be looking at the datatype and
   converting into a number or string
   -------------------------------------------------------------------- */

static PyObject *
swig_varlink_getattr(swig_varlinkobject *v, char *n)
{
  int i = 0;
  char temp[128];

  while (v->vars[i]) {
    if (strcmp(v->vars[i]->name,n) == 0) {
      return (*v->vars[i]->get_attr)();
    }
    i++;
  }
  sprintf(temp,"C global variable %s not found.", n);
  PyErr_SetString(PyExc_NameError,temp);
  return NULL;
}

/* -------------------------------------------------------------------
   swig_varlink_setattr()

   This function sets the value of a variable.
   ------------------------------------------------------------------- */

static int
swig_varlink_setattr(swig_varlinkobject *v, char *n, PyObject *p)
{
  char temp[128];
  int i = 0;
  while (v->vars[i]) {
    if (strcmp(v->vars[i]->name,n) == 0) {
      return (*v->vars[i]->set_attr)(p);
    }
    i++;
  }
  sprintf(temp,"C global variable %s not found.", n);
  PyErr_SetString(PyExc_NameError,temp);
  return 1;
}

statichere PyTypeObject varlinktype = {
/*  PyObject_HEAD_INIT(&PyType_Type)  Note : This doesn't work on some machines */
  PyObject_HEAD_INIT(0)
  0,
  "varlink",                          /* Type name    */
  sizeof(swig_varlinkobject),         /* Basic size   */
  0,                                  /* Itemsize     */
  0,                                  /* Deallocator  */
  (printfunc) swig_varlink_print,     /* Print      */
  (getattrfunc) swig_varlink_getattr, /* get attr     */
  (setattrfunc) swig_varlink_setattr, /* Set attr     */
  0,                                  /* tp_compare   */
  (reprfunc) swig_varlink_repr,       /* tp_repr      */
  0,                                  /* tp_as_number */
  0,                                  /* tp_as_mapping*/
  0,                                  /* tp_hash      */
};

/* Create a variable linking object for use later */

SWIGSTATIC PyObject *
SWIG_newvarlink(void)
{
  swig_varlinkobject *result = 0;
  result = PyMem_NEW(swig_varlinkobject,1);
  varlinktype.ob_type = &PyType_Type;    /* Patch varlinktype into a PyType */
  result->ob_type = &varlinktype;
  /*  _Py_NewReference(result);  Does not seem to be necessary */
  result->nvars = 0;
  result->maxvars = 64;
  result->vars = (swig_globalvar **) malloc(64*sizeof(swig_globalvar *));
  result->vars[0] = 0;
  result->ob_refcnt = 0;
  Py_XINCREF((PyObject *) result);
  return ((PyObject*) result);
}

#ifdef PDFLIB_UNUSED
SWIGSTATIC void
SWIG_addvarlink(PyObject *p, char *name,
	   PyObject *(*get_attr)(void), int (*set_attr)(PyObject *p))
{
  swig_varlinkobject *v;
  v= (swig_varlinkobject *) p;

  if (v->nvars >= v->maxvars -1) {
    v->maxvars = 2*v->maxvars;
    v->vars = (swig_globalvar **) realloc(v->vars,v->maxvars*sizeof(swig_globalvar *));
    if (v->vars == NULL) {
      fprintf(stderr,"SWIG : Fatal error in initializing Python module.\n");
      exit(1);
    }
  }
  v->vars[v->nvars] = (swig_globalvar *) malloc(sizeof(swig_globalvar));
  v->vars[v->nvars]->name = (char *) malloc(strlen(name)+1);
  strcpy(v->vars[v->nvars]->name,name);
  v->vars[v->nvars]->get_attr = get_attr;
  v->vars[v->nvars]->set_attr = set_attr;
  v->nvars++;
  v->vars[v->nvars] = 0;
}

#endif /* PDFLIB_UNUSED */

#include <stdlib.h>

#ifdef SWIG_GLOBAL
#ifdef __cplusplus
#define SWIGSTATIC extern "C"
#else
#define SWIGSTATIC
#endif
#endif

#ifndef SWIGSTATIC
#define SWIGSTATIC static
#endif


/* SWIG pointer structure */

typedef struct SwigPtrType {
  char               *name;               /* Datatype name                  */
  int               len;                /* Length (used for optimization) */
  void               *(*cast)(void *);    /* Pointer casting function       */
  struct SwigPtrType *next;               /* Linked list pointer            */
} SwigPtrType;

/* Pointer cache structure */

typedef struct {
  int               stat;               /* Status (valid) bit             */
  SwigPtrType        *tp;                 /* Pointer to type structure      */
  char                name[256];          /* Given datatype name            */
  char                mapped[256];        /* Equivalent name                */
} SwigCacheType;

/* Some variables  */

static int SwigPtrMax  = 64;           /* Max entries that can be currently held */
                                       /* This value may be adjusted dynamically */
static int SwigPtrN    = 0;            /* Current number of entries              */
static int SwigPtrSort = 0;            /* Status flag indicating sort            */
static int SwigStart[256];             /* Starting positions of types            */

/* Pointer table */
static SwigPtrType *SwigPtrTable = 0;  /* Table containing pointer equivalences  */

/* Cached values */

#define SWIG_CACHESIZE  8
#define SWIG_CACHEMASK  0x7
static SwigCacheType SwigCache[SWIG_CACHESIZE];
static int SwigCacheIndex = 0;
static int SwigLastCache = 0;

/* Sort comparison function */
static int swigsort(const void *data1, const void *data2) {
	SwigPtrType *d1 = (SwigPtrType *) data1;
	SwigPtrType *d2 = (SwigPtrType *) data2;
	return strcmp(d1->name,d2->name);
}

/* Binary Search function */
static int swigcmp(const void *key, const void *data) {
  char *k = (char *) key;
  SwigPtrType *d = (SwigPtrType *) data;
  return strncmp(k,d->name,d->len);
}

/* Register a new datatype with the type-checker */

SWIGSTATIC
void SWIG_RegisterMapping(char *origtype, char *newtype, void *(*cast)(void *)) {

  int i;
  SwigPtrType *t = 0,*t1;

  /* Allocate the pointer table if necessary */

  if (!SwigPtrTable) {
    SwigPtrTable = (SwigPtrType *) malloc(SwigPtrMax*sizeof(SwigPtrType));
    SwigPtrN = 0;
  }
  /* Grow the table */
  if (SwigPtrN >= SwigPtrMax) {
    SwigPtrMax = 2*SwigPtrMax;
    SwigPtrTable = (SwigPtrType *) realloc((char *) SwigPtrTable,SwigPtrMax*sizeof(SwigPtrType));
  }
  for (i = 0; i < SwigPtrN; i++)
    if (strcmp(SwigPtrTable[i].name,origtype) == 0) {
      t = &SwigPtrTable[i];
      break;
    }
  if (!t) {
    t = &SwigPtrTable[SwigPtrN];
    t->name = origtype;
    t->len = strlen(t->name);
    t->cast = 0;
    t->next = 0;
    SwigPtrN++;
  }

  /* Check for existing entry */

  while (t->next) {
    if ((strcmp(t->name,newtype) == 0)) {
      if (cast) t->cast = cast;
      return;
    }
    t = t->next;
  }

  /* Now place entry (in sorted order) */

  t1 = (SwigPtrType *) malloc(sizeof(SwigPtrType));
  t1->name = newtype;
  t1->len = strlen(t1->name);
  t1->cast = cast;
  t1->next = 0;
  t->next = t1;
  SwigPtrSort = 0;
}

/* Make a pointer value string */

SWIGSTATIC
void SWIG_MakePtr(char *_c, const void *_ptr, char *type) {
  static char _hex[16] =
  {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
   'a', 'b', 'c', 'd', 'e', 'f'};
  unsigned long _p, _s;
  char _result[20], *_r;    /* Note : a 64-bit hex number = 16 digits */
  _r = _result;
  _p = (unsigned long) _ptr;
  if (_p > 0) {
    while (_p > 0) {
      _s = _p & 0xf;
      *(_r++) = _hex[_s];
      _p = _p >> 4;
    }
    *_r = '_';
    while (_r >= _result)
      *(_c++) = *(_r--);
  } else {
    strcpy (_c, "NULL");
  }
  if (_ptr)
    strcpy (_c, type);
}

/* Function for getting a pointer value */

SWIGSTATIC
char *SWIG_GetPtr(char *_c, void **ptr, char *_t)
{
  unsigned long _p;
  char temp_type[256];
  char *name;
  int i, len;
  SwigPtrType *sp,*tp;
  SwigCacheType *cache;
  int start, end;
  _p = 0;

  /* Pointer values must start with leading underscore */
  if (*_c == '_') {
      _c++;
      /* Extract hex value from pointer */
      while (*_c) {
	  if ((*_c >= '0') && (*_c <= '9'))
	    _p = (_p << 4) + (*_c - '0');
	  else if ((*_c >= 'a') && (*_c <= 'f'))
	    _p = (_p << 4) + ((*_c - 'a') + 10);
	  else
	    break;
	  _c++;
      }

      if (_t) {
	if (strcmp(_t,_c)) {
	  if (!SwigPtrSort) {
	    qsort((void *) SwigPtrTable, SwigPtrN, sizeof(SwigPtrType), swigsort);
	    for (i = 0; i < 256; i++) {
	      SwigStart[i] = SwigPtrN;
	    }
	    for (i = SwigPtrN-1; i >= 0; i--) {
	      SwigStart[(int) (SwigPtrTable[i].name[1])] = i;
	    }
	    for (i = 255; i >= 1; i--) {
	      if (SwigStart[i-1] > SwigStart[i])
		SwigStart[i-1] = SwigStart[i];
	    }
	    SwigPtrSort = 1;
	    for (i = 0; i < SWIG_CACHESIZE; i++)
	      SwigCache[i].stat = 0;
	  }

	  /* First check cache for matches.  Uses last cache value as starting point */
	  cache = &SwigCache[SwigLastCache];
	  for (i = 0; i < SWIG_CACHESIZE; i++) {
	    if (cache->stat) {
	      if (strcmp(_t,cache->name) == 0) {
		if (strcmp(_c,cache->mapped) == 0) {
		  cache->stat++;
		  *ptr = (void *) _p;
		  if (cache->tp->cast) *ptr = (*(cache->tp->cast))(*ptr);
		  return (char *) 0;
		}
	      }
	    }
	    SwigLastCache = (SwigLastCache+1) & SWIG_CACHEMASK;
	    if (!SwigLastCache) cache = SwigCache;
	    else cache++;
	  }
	  /* We have a type mismatch.  Will have to look through our type
	     mapping table to figure out whether or not we can accept this datatype */

	  start = SwigStart[(int) _t[1]];
	  end = SwigStart[(int) _t[1]+1];
	  sp = &SwigPtrTable[start];
	  while (start < end) {
	    if (swigcmp(_t,sp) == 0) break;
	    sp++;
	    start++;
	  }
	  if (start >= end) sp = 0;
	  /* Try to find a match for this */
	  if (sp) {
	    while (swigcmp(_t,sp) == 0) {
	      name = sp->name;
	      len = sp->len;
	      tp = sp->next;
	      /* Try to find entry for our given datatype */
	      while(tp) {
		if (tp->len >= 255) {
		  return _c;
		}
		strcpy(temp_type,tp->name);
		strncat(temp_type,_t+len,255-tp->len);
		if (strcmp(_c,temp_type) == 0) {

		  strcpy(SwigCache[SwigCacheIndex].mapped,_c);
		  strcpy(SwigCache[SwigCacheIndex].name,_t);
		  SwigCache[SwigCacheIndex].stat = 1;
		  SwigCache[SwigCacheIndex].tp = tp;
		  SwigCacheIndex = SwigCacheIndex & SWIG_CACHEMASK;

		  /* Get pointer value */
		  *ptr = (void *) _p;
		  if (tp->cast) *ptr = (*(tp->cast))(*ptr);
		  return (char *) 0;
		}
		tp = tp->next;
	      }
	      sp++;
	      /* Hmmm. Didn't find it this time */
	    }
	  }
	  /* Didn't find any sort of match for this data.
	     Get the pointer value and return the received type */
	  *ptr = (void *) _p;
	  return _c;
	} else {
	  /* Found a match on the first try.  Return pointer value */
	  *ptr = (void *) _p;
	  return (char *) 0;
	}
      } else {
	/* No type specified.  Good luck */
	*ptr = (void *) _p;
	return (char *) 0;
      }
  } else {
    if (strcmp (_c, "NULL") == 0) {
	*ptr = (void *) 0;
	return (char *) 0;
    }
    *ptr = (void *) 0;
    return _c;
  }
}

#include <setjmp.h>

#include "pdflib.h"

/*
 * Byte order
 * WORDS_BIGENDIAN will be set by the configure script on most platforms.
 * Only on platforms where there is no configure script we must set the
 * endianness explicitly (most importantly CodeWarrior on the Mac)
 */
#if defined(WORDS_BIGENDIAN) || defined(__POWERPC__) || defined(__MC68K__)
#define UTFCONV	"utf-16-be"
#else
#define UTFCONV	"utf-16-le"
#endif

/* Exception handling */

static PyObject *PDFlibException;

#define try	PDF_TRY(p)
#define catch	PDF_CATCH(p) { \
		PDF_throw_pyexception(p); \
		return NULL; \
		}

static void
PDF_throw_pyexception(PDF *p)
{
    char errmsg[1024];
    sprintf(errmsg, "PDFlib Error [%d] %s: %s", PDF_get_errnum(p),
			PDF_get_apiname(p), PDF_get_errmsg(p));
    PyErr_SetString(PDFlibException, errmsg); 
}

static void
PDF_WrongPDFHandle(const char *funcname)
{
    char text[128];

    sprintf(text, "Type error in argument 1 of %s. Expected _PDF_p.", funcname);
    PyErr_SetString(PyExc_TypeError, text);
}



static PyObject *_wrap_PDF_new(PyObject *self, PyObject *args)
{
    PDF *p;
    char _ptemp[128];
    char versionbuf[32];

    if(!PyArg_ParseTuple(args,":PDF_new"))
        return NULL;

    p = PDF_new();

    if (!p) {
        PyErr_SetString(PyExc_SystemError, "PDFlib error: in PDF_new()");
        return NULL;
    }

    try {
#if defined(PY_VERSION)
        sprintf(versionbuf, "Python %s", PY_VERSION);
#elif defined(PATCHLEVEL)
        sprintf(versionbuf, "Python %s", PATCHLEVEL);
#else
        sprintf(versionbuf, "Python (unknown)");
#endif
        PDF_set_parameter(p, "binding", versionbuf);
	PDF_set_parameter(p, "unicaplang", "true");
	PDF_set_parameter(p, "textformat", "auto2");
	PDF_set_parameter(p, "hypertextformat", "auto2");
	PDF_set_parameter(p, "hypertextencoding", "");
    } catch;

    SWIG_MakePtr(_ptemp, (char *) p,"_PDF_p");
    return Py_BuildValue("s",_ptemp);
}

static PyObject *_nuwrap_PDF_new(PyObject *self, PyObject *args)
{
    PDF *p;
    char _ptemp[128];
    char versionbuf[32];

    if(!PyArg_ParseTuple(args,":PDF_new"))
        return NULL;

    p = PDF_new();

    if (!p) {
        PyErr_SetString(PyExc_SystemError, "PDFlib error: in PDF_new()");
        return NULL;
    }

    try {
#if defined(PY_VERSION)
        sprintf(versionbuf, "Python %s", PY_VERSION);
#elif defined(PATCHLEVEL)
        sprintf(versionbuf, "Python %s", PATCHLEVEL);
#else
        sprintf(versionbuf, "Python (unknown)");
#endif
        PDF_set_parameter(p, "binding", versionbuf);
    } catch;

    SWIG_MakePtr(_ptemp, (char *) p,"_PDF_p");
    return Py_BuildValue("s",_ptemp);
}

static PyObject *_wrap_PDF_delete(PyObject *self, PyObject *args)
{
    PDF *p;
    char *py_p = 0;

    if (!PyArg_ParseTuple(args,"s:PDF_delete", &py_p))
        return NULL;
    if (py_p && SWIG_GetPtr(py_p, (void *) &p, "_PDF_p"))
    {
        PDF_WrongPDFHandle("PDF_delete");
        return NULL;
    }

    PDF_delete(p);

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *_nuwrap_PDF_delete(PyObject *self, PyObject *args)
{
    PDF *p;
    char *py_p = 0;

    if (!PyArg_ParseTuple(args,"s:PDF_delete", &py_p))
        return NULL;
    if (py_p && SWIG_GetPtr(py_p, (void *) &p, "_PDF_p"))
    {
        PDF_WrongPDFHandle("PDF_delete");
        return NULL;
    }

    PDF_delete(p);

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *_wrap_PDF_open_pdi(PyObject *self, PyObject *args)
{
    PDF *p;
    char *py_p = 0;
    const char *filename;
    const char *optlist;
    int reserved;
    int PDF_VOLATILE _result = 0;

    if (!PyArg_ParseTuple(args,"sssi:PDF_open_pdi", &py_p, &filename, &optlist, &reserved))
        return NULL;
    if (py_p && SWIG_GetPtr(py_p, (void *) &p, "_PDF_p"))
    {
        PDF_WrongPDFHandle("PDF_open_pdi");
        return NULL;
    }

    try { _result = (int) PDF_open_pdi(p, filename, optlist, 0);
    } catch;

    return Py_BuildValue("i", _result);
}

static PyObject *_nuwrap_PDF_open_pdi(PyObject *self, PyObject *args)
{
    PDF *p;
    char *py_p = 0;
    const char *filename;
    const char *optlist;
    int reserved;
    int PDF_VOLATILE _result = 0;

    if (!PyArg_ParseTuple(args,"sssi:PDF_open_pdi", &py_p, &filename, &optlist, &reserved))
        return NULL;
    if (py_p && SWIG_GetPtr(py_p, (void *) &p, "_PDF_p"))
    {
        PDF_WrongPDFHandle("PDF_open_pdi");
        return NULL;
    }

    try { _result = (int) PDF_open_pdi(p, filename, optlist, 0);
    } catch;

    return Py_BuildValue("i", _result);
}

static PyObject *_wrap_PDF_setpolydash(PyObject *self, PyObject *args)
{
    PDF *p;
    char *py_p = 0;
    PyObject *parray = NULL;

    PyObject *val;
    float fval, *carray;
    int PDF_VOLATILE length, i;

    if (!PyArg_ParseTuple(args,"sO:PDF_setpolydash", &py_p, &parray))
        return NULL;
    if (py_p && SWIG_GetPtr(py_p, (void *) &p, "_PDF_p"))
    {
        PDF_WrongPDFHandle("PDF_setpolydash");
        return NULL;
    }

    length = PyTuple_Size(parray);

    carray = (float *) malloc(sizeof(float) * length);
    if (carray == NULL)
    {
        PyErr_SetString(PyExc_TypeError, "Out of memory in PDF_set_polydash");
        return NULL;
    }

    for (i = 0; i < length; i++)
    {
        val = PyTuple_GetItem(parray, i);
        if (!PyArg_Parse(val, "f:PDF_setpolydash", &fval))
        {
            PyErr_SetString(PyExc_TypeError,
                "Type error in argument 2 of PDF_setpolydash. Expected float.");
            return NULL;
        }
        carray[i] = fval;
    }

    try { PDF_setpolydash(p, carray, length);
    } catch;

    free(carray);

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *_nuwrap_PDF_setpolydash(PyObject *self, PyObject *args)
{
    PDF *p;
    char *py_p = 0;
    PyObject *parray = NULL;

    PyObject *val;
    float fval, *carray;
    int PDF_VOLATILE length, i;

    if (!PyArg_ParseTuple(args,"sO:PDF_setpolydash", &py_p, &parray))
        return NULL;
    if (py_p && SWIG_GetPtr(py_p, (void *) &p, "_PDF_p"))
    {
        PDF_WrongPDFHandle("PDF_setpolydash");
        return NULL;
    }

    length = PyTuple_Size(parray);

    carray = (float *) malloc(sizeof(float) * length);
    if (carray == NULL)
    {
        PyErr_SetString(PyExc_TypeError, "Out of memory in PDF_set_polydash");
        return NULL;
    }

    for (i = 0; i < length; i++)
    {
        val = PyTuple_GetItem(parray, i);
        if (!PyArg_Parse(val, "f:PDF_setpolydash", &fval))
        {
            PyErr_SetString(PyExc_TypeError,
                "Type error in argument 2 of PDF_setpolydash. Expected float.");
            return NULL;
        }
        carray[i] = fval;
    }

    try { PDF_setpolydash(p, carray, length);
    } catch;

    free(carray);

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *_wrap_PDF_open_image(PyObject *self, PyObject *args)
{
    PDF *p;
    char *py_p = 0;
    const char *imagetype;
    const char *source;
    const char *data;
    int dummy;
    Py_ssize_t length;
    int width;
    int height;
    int components;
    int bpc;
    const char *params;
    int PDF_VOLATILE _result = 0;

    if (!PyArg_ParseTuple(args,"ssss#liiiis:PDF_open_image", &py_p, &imagetype, &source, &data, &dummy, &length, &width, &height, &components, &bpc, &params))
        return NULL;
    if (py_p && SWIG_GetPtr(py_p, (void *) &p, "_PDF_p"))
    {
        PDF_WrongPDFHandle("PDF_open_image");
        return NULL;
    }

    try { _result = (int) PDF_open_image(p, imagetype, source, data, (long)length, width, height, components, bpc, params);
    } catch;

    return Py_BuildValue("i", _result);
}

static PyObject *_nuwrap_PDF_open_image(PyObject *self, PyObject *args)
{
    PDF *p;
    char *py_p = 0;
    const char *imagetype;
    const char *source;
    const char *data;
    int dummy;
    Py_ssize_t length;
    int width;
    int height;
    int components;
    int bpc;
    const char *params;
    int PDF_VOLATILE _result = 0;

    if (!PyArg_ParseTuple(args,"ssss#liiiis:PDF_open_image", &py_p, &imagetype, &source, &data, &dummy, &length, &width, &height, &components, &bpc, &params))
        return NULL;
    if (py_p && SWIG_GetPtr(py_p, (void *) &p, "_PDF_p"))
    {
        PDF_WrongPDFHandle("PDF_open_image");
        return NULL;
    }

    try { _result = (int) PDF_open_image(p, imagetype, source, data, (long)length, width, height, components, bpc, params);
    } catch;

    return Py_BuildValue("i", _result);
}

static PyObject *_nuwrap_PDF_utf16_to_utf8(PyObject *self, PyObject *args)
{
    PDF *p;
    char *py_p = 0;
    const char *utf16string;
    Py_ssize_t utf16string_len;
    int size;
    char PDF_VOLATILE *_result = NULL;

    if (!PyArg_ParseTuple(args,"su#i:PDF_utf16_to_utf8", &py_p, &utf16string, &utf16string_len, &size))
        return NULL;
    if (py_p && SWIG_GetPtr(py_p, (void *) &p, "_PDF_p"))
    {
        PDF_WrongPDFHandle("PDF_utf16_to_utf8");
        return NULL;
    }

    try { _result = (char *) PDF_utf16_to_utf8(p, utf16string, (int)utf16string_len, &size);
    } catch;

    return Py_BuildValue("s#", _result, size);
}

static PyObject *_nuwrap_PDF_utf8_to_utf16(PyObject *self, PyObject *args)
{
    PDF *p;
    char *py_p = 0;
    const char *utf8string;
    const char *format;
    int size;
    char PDF_VOLATILE *_result = NULL;

    if (!PyArg_ParseTuple(args,"sss:PDF_utf8_to_utf16", &py_p, &utf8string, &format))
        return NULL;
    if (py_p && SWIG_GetPtr(py_p, (void *) &p, "_PDF_p"))
    {
        PDF_WrongPDFHandle("PDF_utf8_to_utf16");
        return NULL;
    }

    try { _result = (char *) PDF_utf8_to_utf16(p, utf8string, format, &size);
    } catch;

    return Py_BuildValue("u#", _result, size);
}

/* BEGIN created with wrap.pl */
#define _WRAP_CODE      1
#define PDF_UNICODE_WRAPPER 1
#include "py_wrapped.c"
#undef PDF_UNICODE_WRAPPER
#include "py_wrapped.c"
#undef _WRAP_CODE
/* END created with wrap.pl */

static PyMethodDef pdflibMethods[] =
{
    { "PDF_new", _wrap_PDF_new, 1 },
    { "PDF_delete", _wrap_PDF_delete, 1 },
    { "PDF_open_pdi", _wrap_PDF_open_pdi, 1 },
    { "PDF_setpolydash", _wrap_PDF_setpolydash, 1 },
    { "PDF_open_image", _wrap_PDF_open_image, 1 },
#define _WRAP_newXS
#define PDF_UNICODE_WRAPPER 1
#include "py_wrapped.c"
#undef PDF_UNICODE_WRAPPER
#undef _WRAP_newXS

    { NULL, NULL },
};
static PyMethodDef nupdflibMethods[] =
{
    { "PDF_new", _nuwrap_PDF_new, 1 },
    { "PDF_delete", _nuwrap_PDF_delete, 1 },
    { "PDF_open_pdi", _nuwrap_PDF_open_pdi, 1 },
    { "PDF_setpolydash", _nuwrap_PDF_setpolydash, 1 },
    { "PDF_open_image", _nuwrap_PDF_open_image, 1 },
    { "PDF_utf16_to_utf8", _nuwrap_PDF_utf16_to_utf8, 1 },
    { "PDF_utf8_to_utf16", _nuwrap_PDF_utf8_to_utf16, 1 },
#define _WRAP_newXS
#include "py_wrapped.c"
#undef _WRAP_newXS

    { NULL, NULL },
};


static PyObject *SWIG_globals;
#ifdef __cplusplus
extern "C"
#endif
SWIGEXPORT(void,initpdflib_py)() {
	 PyObject *m, *d;
	 SWIG_globals = SWIG_newvarlink();
	 m = Py_InitModule("pdflib_py", pdflibMethods);
	 d = PyModule_GetDict(m);

	 /* PDFlib exception handling */
	 PDFlibException = PyErr_NewException("pdflib_py.PDFlibException", NULL, NULL);
	 Py_INCREF(PDFlibException);
	 PyModule_AddObject(m, "PDFlibException", PDFlibException);

/*
 * These are the pointer type-equivalency mappings.
 * (Used by the SWIG pointer type-checker).
 */
	 SWIG_RegisterMapping("_signed_long","_long",0);
	 SWIG_RegisterMapping("_struct_PDF_s","_PDF",0);
	 SWIG_RegisterMapping("_long","_unsigned_long",0);
	 SWIG_RegisterMapping("_long","_signed_long",0);
	 SWIG_RegisterMapping("_PDF","_struct_PDF_s",0);
	 SWIG_RegisterMapping("_unsigned_long","_long",0);
	 SWIG_RegisterMapping("_signed_int","_int",0);
	 SWIG_RegisterMapping("_unsigned_short","_short",0);
	 SWIG_RegisterMapping("_signed_short","_short",0);
	 SWIG_RegisterMapping("_unsigned_int","_int",0);
	 SWIG_RegisterMapping("_short","_unsigned_short",0);
	 SWIG_RegisterMapping("_short","_signed_short",0);
	 SWIG_RegisterMapping("_int","_unsigned_int",0);
	 SWIG_RegisterMapping("_int","_signed_int",0);
}

#ifdef __cplusplus
extern "C"
#endif
SWIGEXPORT(void,initnupdflib_py)() {
	 PyObject *m, *d;
	 SWIG_globals = SWIG_newvarlink();
	 m = Py_InitModule("nupdflib_py", nupdflibMethods);
	 d = PyModule_GetDict(m);

	 /* PDFlib exception handling */
	 PDFlibException = PyErr_NewException("nupdflib_py.PDFlibException", NULL, NULL);
	 Py_INCREF(PDFlibException);
	 PyModule_AddObject(m, "PDFlibException", PDFlibException);

/*
 * These are the pointer type-equivalency mappings.
 * (Used by the SWIG pointer type-checker).
 */
	 SWIG_RegisterMapping("_signed_long","_long",0);
	 SWIG_RegisterMapping("_struct_PDF_s","_PDF",0);
	 SWIG_RegisterMapping("_long","_unsigned_long",0);
	 SWIG_RegisterMapping("_long","_signed_long",0);
	 SWIG_RegisterMapping("_PDF","_struct_PDF_s",0);
	 SWIG_RegisterMapping("_unsigned_long","_long",0);
	 SWIG_RegisterMapping("_signed_int","_int",0);
	 SWIG_RegisterMapping("_unsigned_short","_short",0);
	 SWIG_RegisterMapping("_signed_short","_short",0);
	 SWIG_RegisterMapping("_unsigned_int","_int",0);
	 SWIG_RegisterMapping("_short","_unsigned_short",0);
	 SWIG_RegisterMapping("_short","_signed_short",0);
	 SWIG_RegisterMapping("_int","_unsigned_int",0);
	 SWIG_RegisterMapping("_int","_signed_int",0);
}
