/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998, 1999  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_sysaudio.c,v 1.6 2000/02/09 12:16:14 hercules Exp $";
#endif

/* Allow access to a raw mixing buffer */

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <strings.h>
#ifdef __SVR4
#include <sys/audioio.h>
#else
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#endif

#include "SDL_endian.h"
#include "SDL_audio.h"
#include "SDL_audiomem.h"
#include "SDL_audiodev_c.h"
#include "SDL_lowaudio.h"

/* Open the audio device for playback, and don't block if busy */
#define OPEN_FLAGS	(O_WRONLY|O_NONBLOCK)

/* Audio driver functions */
static int DSP_OpenAudio(_THIS, SDL_AudioSpec *spec);
static void DSP_WaitAudio(_THIS);
static void DSP_PlayAudio(_THIS);
static Uint8 *DSP_GetAudioBuf(_THIS);
static void DSP_CloseAudio(_THIS);

/* Audio driver bootstrap functions */

static int Audio_Available(void)
{
	return(1);
}

static void Audio_DeleteDevice(SDL_AudioDevice *device)
{
	free(device->hidden);
	free(device);
}

static SDL_AudioDevice *Audio_CreateDevice(int devindex)
{
	SDL_AudioDevice *this;

	/* Initialize all variables that we clean on shutdown */
	this = (SDL_AudioDevice *)malloc(sizeof(SDL_AudioDevice));
	if ( this ) {
		memset(this, 0, (sizeof *this));
		this->hidden = (struct SDL_PrivateAudioData *)
				malloc((sizeof *this->hidden));
	}
	if ( (this == NULL) || (this->hidden == NULL) ) {
		SDL_OutOfMemory();
		if ( this ) {
			free(this);
		}
		return(0);
	}
	memset(this->hidden, 0, (sizeof *this->hidden));
	audio_fd = -1;

	/* Set the function pointers */
	this->OpenAudio = DSP_OpenAudio;
	this->WaitAudio = DSP_WaitAudio;
	this->PlayAudio = DSP_PlayAudio;
	this->GetAudioBuf = DSP_GetAudioBuf;
	this->CloseAudio = DSP_CloseAudio;

	this->free = Audio_DeleteDevice;

	return this;
}

AudioBootStrap AUDIO_bootstrap = {
	"dsp", Audio_Available, Audio_CreateDevice
};

#ifdef DEBUG_AUDIO
void CheckUnderflow(void)
{
#ifdef AUDIO_GETINFO
	audio_info_t info;
	int left;

	ioctl(audio_fd, AUDIO_GETINFO, &info);
	left = (written - info.play.samples);
	if ( written && (left == 0) ) {
		fprintf(stderr, "audio underflow!\n");
	}
#endif
}
#endif

void DSP_WaitAudio(_THIS)
{
#ifdef AUDIO_GETINFO
#define SLEEP_FUDGE	10*1000		/* 10 ms scheduling fudge factor */
	audio_info_t info;
	Sint32 left;

	ioctl(audio_fd, AUDIO_GETINFO, &info);
	left = (written - info.play.samples);
	if ( left > fragsize ) {
		Sint32 sleepy;

		sleepy = (((left - fragsize)*1000)/frequency);
		sleepy -= SLEEP_FUDGE;
		if ( sleepy > 0 ) {
			usleep(sleepy);
		}
	}
#else
	fd_set fdset;

	FD_ZERO(&fdset);
	FD_SET(audio_fd, &fdset);
	select(audio_fd+1, NULL, &fdset, NULL, NULL);
#endif
}

void DSP_PlayAudio(_THIS)
{
	static Uint8 snd2au(int sample);
	/* Write the audio data */
	if ( ulaw_only ) {
		/* Assuming that this->spec.freq >= 8000 Hz */
		int accum, incr, pos;
		Uint8 *aubuf;

		accum = 0;
		incr  = this->spec.freq/8;
		aubuf = ulaw_buf;
		switch (audio_fmt & 0xFF) {
			case 8: {
				Uint8 *sndbuf;

				sndbuf = mixbuf;
				for ( pos=0; pos < fragsize; ++pos ) {
					*aubuf = snd2au((0x80-*sndbuf)*64);
					accum += incr;
					while ( accum > 0 ) {
						accum -= 1000;
						sndbuf += 1;
					}
					aubuf += 1;
				}
			}
			break;
			case 16: {
				Sint16 *sndbuf;

				sndbuf = (Sint16 *)mixbuf;
				for ( pos=0; pos < fragsize; ++pos ) {
					*aubuf = snd2au(*sndbuf/4);
					accum += incr;
					while ( accum > 0 ) {
						accum -= 1000;
						sndbuf += 1;
					}
					aubuf += 1;
				}
			}
			break;
		}
#ifdef DEBUG_AUDIO
		CheckUnderflow();
#endif
		if ( write(audio_fd, ulaw_buf, fragsize) < 0 ) {
			/* Assume fatal error, for now */
			this->enabled = 0;
		}
		written += fragsize;
	} else {
#ifdef DEBUG_AUDIO
		CheckUnderflow();
#endif
		if ( write(audio_fd, mixbuf, this->spec.size) < 0 ) {
			/* Assume fatal error, for now */
			this->enabled = 0;
		}
		written += fragsize;
	}
}

Uint8 *DSP_GetAudioBuf(_THIS)
{
	return(mixbuf);
}

void DSP_CloseAudio(_THIS)
{
	if ( mixbuf != NULL ) {
		SDL_FreeAudioMem(mixbuf);
		mixbuf = NULL;
	}
	if ( ulaw_buf != NULL ) {
		free(ulaw_buf);
		ulaw_buf = NULL;
	}
	close(audio_fd);
}

int DSP_OpenAudio(_THIS, SDL_AudioSpec *spec)
{
	char audiodev[1024];
#ifdef AUDIO_GETINFO
	audio_info_t ulaw_info;
	audio_info_t play_info;
#endif

	/* Initialize our freeable variables, in case we fail*/
	audio_fd = -1;
	mixbuf = NULL;
	ulaw_buf = NULL;

	/* Determine the audio parameters from the AudioSpec */
	switch ( spec->format & 0xFF ) {

		case 8: { /* Unsigned 8 bit audio data */
			spec->format = AUDIO_U8;
		}
		break;

		case 16: { /* Signed 16 bit audio data */
			if ( SDL_BYTEORDER == SDL_BIG_ENDIAN ) {
				spec->format = AUDIO_S16MSB;
			} else {
				spec->format = AUDIO_S16LSB;
			}
		}
		break;

		default: {
			SDL_SetError("Unsupported audio format");
			return(-1);
		}
	}
	audio_fmt = spec->format;

	/* Open the audio device */
	audio_fd = SDL_OpenAudioPath(audiodev, sizeof(audiodev), OPEN_FLAGS);
	if ( audio_fd < 0 ) {
		SDL_SetError("Couldn't open %s: %s", audiodev, strerror(errno));
		return(-1);
	}

	/* Supposedly, /dev/audio is opened with U-Law settings */
	ulaw_only = 1;
#ifdef AUDIO_GETINFO
	if ( ioctl(audio_fd, AUDIO_GETINFO, &ulaw_info) < 0 ) {
		SDL_SetError("Couldn't get audio info: %s", strerror(errno));
		return(-1);
	}
	if ( (ulaw_info.play.sample_rate != 8000) ||
	     (ulaw_info.play.channels != 1) ||
	     (ulaw_info.play.precision != 8) ||
	     (ulaw_info.play.encoding != AUDIO_ENCODING_ULAW) ) {
		SDL_SetError("Audio device not in U-Law mode");
		return(-1);
	}
	memcpy(&play_info, &ulaw_info, sizeof(ulaw_info));

	/* Try to set the requested settings */
	play_info.play.sample_rate = spec->freq;
	play_info.play.channels = spec->channels;
	play_info.play.precision = (spec->format&0xFF);
	play_info.play.encoding = AUDIO_ENCODING_LINEAR;
	written = play_info.play.samples;
#ifdef DEBUG_AUDIO
  	fprintf(stderr,
	"Trying to set %s %d Hz %d bit %s encoding audio format\n",
	play_info.play.channels == 1 ? "mono" : "stereo",
	play_info.play.sample_rate, play_info.play.precision,
	play_info.play.encoding == AUDIO_ENCODING_LINEAR ? "linear" : "U-law");
#endif
	/* Apparently the dbri audio driver can't do linear 8-bit */
	if ( ((spec->format & 0xFF) != AUDIO_U8) &&
			(ioctl(audio_fd, AUDIO_SETINFO, &play_info) == 0) ) {
		if ( (play_info.play.sample_rate == spec->freq) &&
		     (play_info.play.precision == (spec->format&0xFF)) &&
		     (play_info.play.encoding == AUDIO_ENCODING_LINEAR) ) {
			ulaw_only = 0;
		} else {
			ioctl(audio_fd, AUDIO_SETINFO, &ulaw_info);
		}
	}
#endif /* AUDIO_GETINFO */

	/* We can actually convert on-the-fly to U-Law */
	if ( ulaw_only ) {
		fragsize = (spec->samples*1000)/(spec->freq/8);
		frequency = 8;
		ulaw_buf = (Uint8 *)malloc(fragsize);
		if ( ulaw_buf == NULL ) {
			SDL_OutOfMemory();
			return(-1);
		}
		spec->channels = 1;
	} else {
		fragsize = spec->samples;
		frequency = spec->freq/1000;
	}
#ifdef DEBUG_AUDIO
	fprintf(stderr, "Audio device %s U-Law only\n", 
				ulaw_only ? "is" : "is not");
#endif

	/* Update the fragment size as size in bytes */
	SDL_CalculateAudioSpec(spec);

	/* Allocate mixing buffer */
	mixbuf = (Uint8 *)SDL_AllocAudioMem(spec->size);
	if ( mixbuf == NULL ) {
		SDL_OutOfMemory();
		return(-1);
	}
	memset(mixbuf, spec->silence, spec->size);

	/* We're ready to rock and roll. :-) */
	return(0);
}

/************************************************************************/
/* This function (snd2au()) copyrighted:                                */
/************************************************************************/
/*      Copyright 1989 by Rich Gopstein and Harris Corporation          */
/*                                                                      */
/*      Permission to use, copy, modify, and distribute this software   */
/*      and its documentation for any purpose and without fee is        */
/*      hereby granted, provided that the above copyright notice        */
/*      appears in all copies and that both that copyright notice and   */
/*      this permission notice appear in supporting documentation, and  */
/*      that the name of Rich Gopstein and Harris Corporation not be    */
/*      used in advertising or publicity pertaining to distribution     */
/*      of the software without specific, written prior permission.     */
/*      Rich Gopstein and Harris Corporation make no representations    */
/*      about the suitability of this software for any purpose.  It     */
/*      provided "as is" without express or implied warranty.           */
/************************************************************************/

static Uint8 snd2au(int sample)
{

	int mask;

	if (sample < 0) {
		sample = -sample;
		mask = 0x7f;
	} else {
		mask = 0xff;
	}

	if (sample < 32) {
		sample = 0xF0 | 15 - (sample / 2);
	} else if (sample < 96) {
		sample = 0xE0 | 15 - (sample - 32) / 4;
	} else if (sample < 224) {
		sample = 0xD0 | 15 - (sample - 96) / 8;
	} else if (sample < 480) {
		sample = 0xC0 | 15 - (sample - 224) / 16;
	} else if (sample < 992) {
		sample = 0xB0 | 15 - (sample - 480) / 32;
	} else if (sample < 2016) {
		sample = 0xA0 | 15 - (sample - 992) / 64;
	} else if (sample < 4064) {
		sample = 0x90 | 15 - (sample - 2016) / 128;
	} else if (sample < 8160) {
		sample = 0x80 | 15 - (sample - 4064) /  256;
	} else {
		sample = 0x80;
	}
	return (mask & sample);
}
