/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998, 1999  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_blit.h,v 1.3 1999/12/25 16:30:50 hercules Exp $";
#endif

#ifndef _SDL_blit_h
#define _SDL_blit_h

/* The structure passed to the low level blit functions */
typedef struct {
	Uint8 *s_pixels;
	int s_width;
	int s_height;
	int s_skip;
	Uint8 *d_pixels;
	int d_width;
	int d_height;
	int d_skip;
	void *aux_data;
	SDL_PixelFormat *src;
	Uint8 *table;
	SDL_PixelFormat *dst;
} SDL_BlitInfo;

/* The type definition for the low level blit functions */
typedef void (*SDL_loblit)(SDL_BlitInfo *info);

/* This is the private info structure for software accelerated blits */
struct private_swaccel {
	SDL_loblit blit;
	void *aux_data;
};

/* Blit mapping definition */
typedef struct SDL_BlitMap {
	SDL_Surface *dst;
	int identity;
	Uint8 *table;
	SDL_blit hw_blit;
	SDL_blit sw_blit;
	struct private_hwaccel *hw_data;
	struct private_swaccel *sw_data;
} SDL_BlitMap;


/* Definitions for special global blit functions */
#include "SDL_blit_A.h"
#include "SDL_blit_K.h"
#include "SDL_blit_AK.h"

/* Functions found in SDL_blit.c */
extern int SDL_CalculateBlit(SDL_Surface *surface);

/* Functions found in SDL_blit_N.c */
extern SDL_loblit SDL_CalculateBlit0(SDL_Surface *surface, int complex);
extern SDL_loblit SDL_CalculateBlit1(SDL_Surface *surface, int complex);
extern SDL_loblit SDL_CalculateBlitN(SDL_Surface *surface, int complex);

/*
 * Useful macros for blitting routines
 */

#define FORMAT_EQUAL(A, B) \
    (((A)->BitsPerPixel == (B)->BitsPerPixel) && ((A)->Rmask == (B)->Rmask))

/* Load pixel of the specified format from a buffer and get its R-G-B values */
#define RGB_FROM_PIXEL(pixel, fmt, r, g, b)				\
{									\
	r = (((pixel&fmt->Rmask)>>fmt->Rshift)<<fmt->Rloss); 		\
	g = (((pixel&fmt->Gmask)>>fmt->Gshift)<<fmt->Gloss); 		\
	b = (((pixel&fmt->Bmask)>>fmt->Bshift)<<fmt->Bloss); 		\
}
#define DISEMBLE_RGB(buf, bpp, fmt, pixel, r, g, b)			\
{									\
	switch (bpp) {							\
		case 2:							\
			pixel = *((Uint16 *)(buf));			\
			RGB_FROM_PIXEL(pixel, fmt, r, g, b);		\
		break;							\
									\
		case 3:							\
			r = *(((Uint8 *)buf)+fmt->Rshift/8);		\
			g = *(((Uint8 *)buf)+fmt->Gshift/8);		\
			b = *(((Uint8 *)buf)+fmt->Bshift/8);		\
			pixel = (r<<fmt->Rshift)|			\
				(g<<fmt->Gshift)|			\
				(b<<fmt->Bshift);			\
		break;							\
									\
		case 4:							\
			pixel = *((Uint32 *)(buf));			\
			RGB_FROM_PIXEL(pixel, fmt, r, g, b);		\
		break;							\
	}								\
}
/* Assemble R-G-B values into a specified pixel format and store them */
#define PIXEL_FROM_RGB(pixel, fmt, r, g, b)				\
{									\
	pixel = ((r>>fmt->Rloss)<<fmt->Rshift)|				\
		((g>>fmt->Gloss)<<fmt->Gshift)|				\
		((b>>fmt->Bloss)<<fmt->Bshift);				\
}
#define ASSEMBLE_RGB(buf, bpp, fmt, r, g, b) 				\
{									\
	switch (bpp) {							\
		case 2: {						\
			Uint16 pixel;					\
									\
			PIXEL_FROM_RGB(pixel, fmt, r, g, b);		\
			*((Uint16 *)(buf)) = pixel;			\
		}							\
		break;							\
									\
		case 3: {						\
			*((buf)+fmt->Rshift/8) = r;			\
			*((buf)+fmt->Gshift/8) = g;			\
			*((buf)+fmt->Bshift/8) = b;			\
		}							\
		break;							\
									\
		case 4: {						\
			Uint32 pixel;					\
									\
			PIXEL_FROM_RGB(pixel, fmt, r, g, b);		\
			*((Uint32 *)(buf)) = pixel;			\
		}							\
		break;							\
	}								\
}

#define RGBA_FROM_PIXEL(pixel, fmt, r, g, b, a)				\
{									\
	r = (((pixel&fmt->Rmask)>>fmt->Rshift)<<fmt->Rloss); 		\
	g = (((pixel&fmt->Gmask)>>fmt->Gshift)<<fmt->Gloss); 		\
	b = (((pixel&fmt->Bmask)>>fmt->Bshift)<<fmt->Bloss); 		\
	a =  ((pixel&fmt->Amask)>>fmt->Ashift);		 		\
}
#define DISEMBLE_RGBA(buf, bpp, fmt, pixel, r, g, b, a)			\
{									\
	switch (bpp) {							\
		case 2:							\
			pixel = *((Uint16 *)(buf));			\
			RGBA_FROM_PIXEL(pixel, fmt, r, g, b, a);	\
			pixel &= ~fmt->Amask;				\
		break;							\
									\
		case 3:	/* FIXME: broken code (no alpha) */		\
			r = *(((Uint8 *)buf)+fmt->Rshift/8);		\
			g = *(((Uint8 *)buf)+fmt->Gshift/8);		\
			b = *(((Uint8 *)buf)+fmt->Bshift/8);		\
			a = 0;						\
			pixel = (r<<fmt->Rshift)|			\
				(g<<fmt->Gshift)|			\
				(b<<fmt->Bshift);			\
		break;							\
									\
		case 4:							\
			pixel = *((Uint32 *)(buf));			\
			RGBA_FROM_PIXEL(pixel, fmt, r, g, b, a);	\
			pixel &= ~fmt->Amask;				\
		break;							\
	}								\
}

#define PIXEL_FROM_RGBA(pixel, fmt, r, g, b, a)				\
{									\
	pixel = ((r>>fmt->Rloss)<<fmt->Rshift)|				\
		((g>>fmt->Gloss)<<fmt->Gshift)|				\
		((b>>fmt->Bloss)<<fmt->Bshift)|				\
		(a<<fmt->Ashift);					\
}
#define ASSEMBLE_RGBA(buf, bpp, fmt, r, g, b, a)			\
{									\
	switch (bpp) {							\
		case 2: {						\
			Uint16 pixel;					\
									\
			PIXEL_FROM_RGBA(pixel, fmt, r, g, b, a);	\
			*((Uint16 *)(buf)) = pixel;			\
		}							\
		break;							\
									\
		case 3: { /* FIXME: broken code (no alpha) */		\
			*((buf)+fmt->Rshift/8) = r;			\
			*((buf)+fmt->Gshift/8) = g;			\
			*((buf)+fmt->Bshift/8) = b;			\
		}							\
		break;							\
									\
		case 4: {						\
			Uint32 pixel;					\
									\
			PIXEL_FROM_RGBA(pixel, fmt, r, g, b, a);	\
			*((Uint32 *)(buf)) = pixel;			\
		}							\
		break;							\
	}								\
}

/* Blend the RGB values of two pixels based on a source alpha value */
#define ALPHA_BLEND(sR, sG, sB, A, Adiv, dR, dG, dB)			\
{									\
	dR = ((Uint16)sR*(Adiv-A) + (Uint16)dR*A) / Adiv;		\
	dG = ((Uint16)sG*(Adiv-A) + (Uint16)dG*A) / Adiv;		\
	dB = ((Uint16)sB*(Adiv-A) + (Uint16)dB*A) / Adiv;		\
}

/* This is a very useful loop for optimizing blitters */
#define USE_DUFFS_LOOP
#ifdef USE_DUFFS_LOOP
#define DUFFS_LOOP(pixel_copy_increment, width)				\
{ int n = (width+7)/8;							\
	switch (width % 8) {						\
	case 0: do {	pixel_copy_increment;				\
	case 7:		pixel_copy_increment;				\
	case 6:		pixel_copy_increment;				\
	case 5:		pixel_copy_increment;				\
	case 4:		pixel_copy_increment;				\
	case 3:		pixel_copy_increment;				\
	case 2:		pixel_copy_increment;				\
	case 1:		pixel_copy_increment;				\
		} while ( --n > 0 );					\
	}								\
}
#else
#define DUFFS_LOOP(pixel_copy_increment, width)				\
{ int n;								\
	for ( n=0; n < width; ++n ) {				\
		pixel_copy_increment;					\
	}								\
}
#endif

#endif /* _SDL_blit_h */
