/*
	SDL - Simple DirectMedia Layer
	Copyright (C) 1997, 1998, 1999  Sam Lantinga

	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Library General Public
	License as published by the Free Software Foundation; either
	version 2 of the License, or (at your option) any later version.

	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	Library General Public License for more details.

	You should have received a copy of the GNU Library General Public
	License along with this library; if not, write to the Free
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	Sam Lantinga
	slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_fbevents.c,v 1.6 2000/01/25 00:14:41 hercules Exp $";
#endif

/* Handle the event stream, converting X11 events into SDL events */

#include <sys/types.h>
#include <sys/time.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>

/* For parsing /proc */
#include <dirent.h>
#include <ctype.h>

#include "SDL_fbkeys.h"
#include <linux/vt.h>
#include <linux/kd.h>
#include <linux/keyboard.h>

#include "SDL.h"
#include "SDL_mutex.h"
#include "SDL_sysevents.h"
#include "SDL_sysvideo.h"
#include "SDL_events_c.h"
#include "SDL_fbvideo.h"
#include "SDL_fbevents_c.h"

#ifndef GPM_NODE_FIFO
#define GPM_NODE_FIFO	"/dev/gpmdata"
#endif

/* The translation tables from a console scancode to a SDL keysym */
#define NUM_VGAKEYMAPS	(1<<KG_CAPSSHIFT)
static Uint16 vga_keymap[NUM_VGAKEYMAPS][NR_KEYS];
static SDLKey keymap[128];
static SDL_keysym *TranslateKey(int scancode, SDL_keysym *keysym);

/* Ugh, we have to duplicate the kernel's keysym mapping code...
   Oh, it's not so bad. :-)

   FIXME: Add keyboard LED handling code
 */
static void FB_vgainitkeymaps(int fd)
{
	struct kbentry entry;
	int map, i;

	/* Load all the keysym mappings */
	for ( map=0; map<NUM_VGAKEYMAPS; ++map ) {
		memset(vga_keymap[map], 0, NR_KEYS*sizeof(Uint16));
		for ( i=0; i<NR_KEYS; ++i ) {
			entry.kb_table = map;
			entry.kb_index = i;
			if ( ioctl(fd, KDGKBENT, &entry) == 0 ) {
				/* The "Enter" key is a special case */
				if ( entry.kb_value == K_ENTER ) {
					entry.kb_value = K(KT_ASCII,13);
				}
				/* Handle numpad specially as well */
				if ( KTYP(entry.kb_value) == KT_PAD ) {
					switch ( entry.kb_value ) {
					case K_P0:
					case K_P1:
					case K_P2:
					case K_P3:
					case K_P4:
					case K_P5:
					case K_P6:
					case K_P7:
					case K_P8:
					case K_P9:
						vga_keymap[map][i]=entry.kb_value;
						vga_keymap[map][i]+= '0';
						break;
										case K_PPLUS:
						vga_keymap[map][i]=K(KT_ASCII,'+');
						break;
										case K_PMINUS:
						vga_keymap[map][i]=K(KT_ASCII,'-');
						break;
										case K_PSTAR:
						vga_keymap[map][i]=K(KT_ASCII,'*');
						break;
										case K_PSLASH:
						vga_keymap[map][i]=K(KT_ASCII,'/');
						break;
										case K_PENTER:
						vga_keymap[map][i]=K(KT_ASCII,'\r');
						break;
										case K_PCOMMA:
						vga_keymap[map][i]=K(KT_ASCII,',');
						break;
										case K_PDOT:
						vga_keymap[map][i]=K(KT_ASCII,'.');
						break;
					default:
						break;
					}
				}
				/* Do the normal key translation */
				if ( (KTYP(entry.kb_value) == KT_LATIN) ||
					 (KTYP(entry.kb_value) == KT_ASCII) ||
					 (KTYP(entry.kb_value) == KT_LETTER) ) {
					vga_keymap[map][i] = entry.kb_value;
				}
			}
		}
	}
}

void FB_CloseKeyboard(_THIS)
{
	if ( keyboard_fd >= 0 ) {
		if ( ioctl(keyboard_fd, KDSETMODE, KD_TEXT) < 0 ) {
			perror("KDSETMODE");
		}
		if ( ioctl(keyboard_fd, KDSKBMODE, saved_kbd_mode) < 0 ) {
			perror("KDSKBMODE");
		}
		tcsetattr(keyboard_fd, TCSAFLUSH, &saved_kbd_termios);
		if ( keyboard_fd > 0 ) {
			close(keyboard_fd);
		}
	}
	keyboard_fd = -1;
}

int FB_OpenKeyboard(_THIS)
{
	keyboard_fd = 0;
	if ( keyboard_fd >= 0 ) {
		struct termios keyboard_termios;

		/* Set medium-raw keyboard mode */
		if ( ioctl(keyboard_fd, KDGKBMODE, &saved_kbd_mode) < 0 ) {
			if ( keyboard_fd > 0 ) {
				close(keyboard_fd);
			}
			keyboard_fd = -1;
			return(-1);
		}
		if ( tcgetattr(keyboard_fd, &saved_kbd_termios) < 0 ) {
			if ( keyboard_fd > 0 ) {
				close(keyboard_fd);
			}
			keyboard_fd = -1;
			return(-1);
		}
		keyboard_termios = saved_kbd_termios;
		keyboard_termios.c_lflag &= ~(ICANON | ECHO | ISIG);
		keyboard_termios.c_iflag &= ~(ISTRIP | IGNCR | ICRNL | INLCR | IXOFF | IXON);
		keyboard_termios.c_cc[VMIN] = 0;
		keyboard_termios.c_cc[VTIME] = 0;
		if (tcsetattr(keyboard_fd, TCSAFLUSH, &keyboard_termios) < 0) {
			FB_CloseKeyboard(this);
			return(-1);
		}
		if ( ioctl(keyboard_fd, KDSKBMODE, K_MEDIUMRAW) < 0 ) {
			FB_CloseKeyboard(this);
			return(-1);
		}
		if ( ioctl(keyboard_fd, KDSETMODE, KD_GRAPHICS) < 0 ) {
			FB_CloseKeyboard(this);
			return(-1);
		}

		/* Set up keymap */
		FB_vgainitkeymaps(keyboard_fd);
	}
	return(keyboard_fd);
}


static enum {
	MOUSE_NONE = -1,
	MOUSE_GPM,	/* Note: GPM uses the MSC protocol */
	MOUSE_PS2,
	MOUSE_MS,
	MOUSE_BM,
	NUM_MOUSE_DRVS
} mouse_drv = MOUSE_NONE;

void FB_CloseMouse(_THIS)
{
	if ( mouse_fd >= 0 ) {
		close(mouse_fd);
	}
	mouse_fd = -1;
}

/* Returns processes listed in /proc with the desired name */
static int find_pid(DIR *proc, const char *wanted_name)
{
	struct dirent *entry;
	int pid;

	/* First scan proc for the gpm process */
	pid = 0;
	while ( (pid == 0) && ((entry=readdir(proc)) != NULL) ) {
		if ( isdigit(entry->d_name[0]) ) {
			FILE *status;
			char path[PATH_MAX];
			char name[PATH_MAX];

			sprintf(path, "/proc/%s/status", entry->d_name);
			status=fopen(path, "r");
			if ( status ) {
				name[0] = '\0';
				fscanf(status, "Name: %s", name);
				if ( strcmp(name, wanted_name) == 0 ) {
					pid = atoi(entry->d_name);
				}
				fclose(status);
			}
		}
	}
	return pid;
}

/* Returns true if /dev/gpmdata is being written to by gpm */
static int gpm_available(void)
{
	int available;
	DIR *proc;
	int pid;
	int cmdline, len, arglen;
	char path[PATH_MAX];
	char args[PATH_MAX], *arg;

	/* Don't bother looking if the fifo isn't there */
	if ( access(GPM_NODE_FIFO, F_OK) < 0 ) {
		return(0);
	}

	available = 0;
	proc = opendir("/proc");
	if ( proc ) {
		while ( (pid=find_pid(proc, "gpm")) > 0 ) {
			sprintf(path, "/proc/%d/cmdline", pid);
			cmdline = open(path, O_RDONLY, 0);
			if ( cmdline >= 0 ) {
				len = read(cmdline, args, sizeof(args));
				arg = args;
				while ( len > 0 ) {
					if ( strcmp(arg, "-R") == 0 ) {
						available = 1;
					}
					arglen = strlen(arg)+1;
					len -= arglen;
					arg += arglen;
				}
				close(cmdline);
			}
		}
		closedir(proc);
	}
	return available;
}


int FB_OpenMouse(_THIS)
{
	const char *mousedev;

	mousedev = getenv("SDL_MOUSEDEV");
	mouse_fd = -1;
	if ( mousedev == NULL ) {
		/* First try to use GPM in repeater mode */
		if ( mouse_fd < 0 ) {
			if ( gpm_available() ) {
				mouse_fd = open(GPM_NODE_FIFO, O_RDONLY, 0);
				if ( mouse_fd >= 0 ) {
#ifdef DEBUG_MOUSE
fprintf(stderr, "Using GPM mouse\n");
#endif
					mouse_drv = MOUSE_GPM;
				}
			}
		}
		/* First try to use a modern PS/2 port mouse */
		if ( mouse_fd < 0 ) {
			mouse_fd = open("/dev/psaux", O_RDONLY, 0);
			if ( mouse_fd >= 0 ) {
#ifdef DEBUG_MOUSE
fprintf(stderr, "Using PS/2 mouse\n");
#endif
				mouse_drv = MOUSE_PS2;
			}
		}
		/* Next try to use a PPC ADB port mouse */
		if ( mouse_fd < 0 ) {
			mouse_fd = open("/dev/adbmouse", O_RDONLY, 0);
			if ( mouse_fd >= 0 ) {
#ifdef DEBUG_MOUSE
fprintf(stderr, "Using ADB mouse\n");
#endif
				mouse_drv = MOUSE_BM;
			}
		}
	}
	/* Default to a serial Microsoft mouse */
	if ( mouse_fd < 0 ) {
		if ( mousedev == NULL ) {
			mousedev = "/dev/mouse";
		}
		mouse_fd = open(mousedev, O_RDONLY, 0);
		if ( mouse_fd >= 0 ) {
			struct termios mouse_termios;

			/* Set the sampling speed to 1200 baud */
			tcgetattr(mouse_fd, &mouse_termios);
			mouse_termios.c_iflag = IGNBRK | IGNPAR;
			mouse_termios.c_oflag = 0;
			mouse_termios.c_lflag = 0;
			mouse_termios.c_line = 0;
			mouse_termios.c_cc[VTIME] = 0;
			mouse_termios.c_cc[VMIN] = 1;
			mouse_termios.c_cflag = CREAD | CLOCAL | HUPCL;
			mouse_termios.c_cflag |= CS8;
			mouse_termios.c_cflag |= B1200;
			tcsetattr(mouse_fd, TCSAFLUSH, &mouse_termios);
#ifdef DEBUG_MOUSE
fprintf(stderr, "Using Microsoft mouse on %s\n", mousedev);
#endif
			mouse_drv = MOUSE_MS;
		}
	}
	if ( mouse_fd < 0 ) {
		mouse_drv = MOUSE_NONE;
	}
	return(mouse_fd);
}

static int posted = 0;

void FB_vgamousecallback(int button, int dx, int dy)
{
	if ( dx || dy ) {
		posted += SDL_PrivateMouseMotion(0, 1, dx, dy);
	}
	if ( button & 0x04 ) {  /* Left mouse button */
		if ( !(SDL_GetMouseState(NULL, NULL) & SDL_BUTTON(1)) ) {
			posted += SDL_PrivateMouseButton(SDL_PRESSED, 1, 0, 0);
		}
	} else {
		if ( (SDL_GetMouseState(NULL, NULL) & SDL_BUTTON(1)) ) {
			posted += SDL_PrivateMouseButton(SDL_RELEASED, 1, 0, 0);
		}
	}
	if ( button & 0x02 ) {  /* Middle mouse button */
		if ( !(SDL_GetMouseState(NULL, NULL) & SDL_BUTTON(2)) ) {
			posted += SDL_PrivateMouseButton(SDL_PRESSED, 2, 0, 0);
		}
	} else {
		if ( (SDL_GetMouseState(NULL, NULL) & SDL_BUTTON(2)) ) {
			posted += SDL_PrivateMouseButton(SDL_RELEASED, 2, 0, 0);
		}
	}
	if ( button & 0x01 ) {  /* Right mouse button */
		if ( !(SDL_GetMouseState(NULL, NULL) & SDL_BUTTON(3)) ) {
			posted += SDL_PrivateMouseButton(SDL_PRESSED, 3, 0, 0);
		}
	} else {
		if ( (SDL_GetMouseState(NULL, NULL) & SDL_BUTTON(3)) ) {
			posted += SDL_PrivateMouseButton(SDL_RELEASED, 3, 0, 0);
		}
	}
}

/* For now, use GPM, PS/2, and MS protocols
   Driver adapted from the SVGAlib mouse driver code (taken from gpm, etc.)
 */
static void handle_mouse(_THIS)
{
	static int start = 0;
	static unsigned char mousebuf[BUFSIZ];
	int i, nread;
	int button;
	int dx, dy;
	int packetsize;

	/* Figure out the mouse packet size */
	switch (mouse_drv) {
		case MOUSE_NONE:
			/* Ack! */
			read(mouse_fd, mousebuf, BUFSIZ);
			return;
		case MOUSE_GPM:
			packetsize = 5;
			break;
		case MOUSE_PS2:
		case MOUSE_MS:
		case MOUSE_BM:
			packetsize = 3;
			break;
	}

	/* Read as many packets as possible */
	nread = read(mouse_fd, &mousebuf[start], BUFSIZ-start);
	if ( nread < 0 ) {
		return;
	}
	nread += start;
#ifdef DEBUG_MOUSE
	fprintf(stderr, "Read %d bytes from mouse, start = %d\n", nread, start);
#endif
	for ( i=0; i<(nread-(packetsize-1)); i += packetsize ) {
		switch (mouse_drv) {
			case MOUSE_NONE:
				break;
			case MOUSE_GPM:
				/* GPM protocol has 0x80 in high byte */
				if ( (mousebuf[i] & 0xF8) != 0x80 ) {
					/* Go to next byte */
					i -= (packetsize-1);
					continue;
				}
				/* Get current mouse state */
				button = (~mousebuf[i]) & 0x07;
				dx =   (signed char)(mousebuf[i+1]) +
				       (signed char)(mousebuf[i+3]);
				dy = -((signed char)(mousebuf[i+2]) +
				       (signed char)(mousebuf[i+4]));
				break;
			case MOUSE_PS2:
				/* PS/2 protocol has nothing in high byte */
				if ( (mousebuf[i] & 0xC0) != 0 ) {
					/* Go to next byte */
					i -= (packetsize-1);
					continue;
				}
				/* Get current mouse state */
				button = (mousebuf[i] & 0x04) >> 1 | /*Middle*/
		  			 (mousebuf[i] & 0x02) >> 1 | /*Right*/
		  			 (mousebuf[i] & 0x01) << 2;  /*Left*/
		  		dx = (mousebuf[i] & 0x10) ?
		  		      mousebuf[i+1] - 256 : mousebuf[i+1];
		  		dy = (mousebuf[i] & 0x20) ?
		  		      -(mousebuf[i+2] - 256) : -mousebuf[i+2];
				break;
			case MOUSE_MS:
				/* Microsoft protocol has 0x40 in high byte */
				if ( (mousebuf[i] & 0x40) != 0x40 ) {
					/* Go to next byte */
					i -= (packetsize-1);
					continue;
				}
				/* Get current mouse state */
				button = ((mousebuf[i] & 0x20) >> 3) |
					     ((mousebuf[i] & 0x10) >> 4);
				dx = (signed char)(((mousebuf[i] & 0x03) << 6) |
				                   (mousebuf[i + 1] & 0x3F));
				dy = (signed char)(((mousebuf[i] & 0x0C) << 4) |
				                    (mousebuf[i + 2] & 0x3F));
				break;
			case MOUSE_BM:
				/* BusMouse protocol has 0xF8 in high byte */
				if ( (mousebuf[i] & 0xF8) != 0x80 ) {
					/* Go to next byte */
					i -= (packetsize-1);
					continue;
				}
				/* Get current mouse state */
				button = (~mousebuf[i]) & 0x07;
				dx =  (signed char)mousebuf[i+1];
				dy = -(signed char)mousebuf[i+2];
				break;
		}
		FB_vgamousecallback(button, dx, dy);
	}
	if ( i < nread ) {
		memcpy(mousebuf, &mousebuf[i], (nread-i));
		start = (nread-i);
	} else {
		start = 0;
	}
	return;
}

/* Handle switching to another VC, returns when our VC is back */
static void switch_vt(_THIS, unsigned short which)
{
	struct vt_stat vtstate;
	unsigned short current;
	SDL_Surface *screen;
	__u16 saved_pal[3*256];
	Uint32 screen_arealen;
	Uint8 *screen_contents;

	/* Figure out whether or not we're switching to a new console */
	if ( (ioctl(keyboard_fd, VT_GETSTATE, &vtstate) < 0) ||
	     (which == vtstate.v_active) ) {
		return;
	}
	current = vtstate.v_active;

	/* Save the contents of the screen, and go to text mode */
	SDL_mutexP(hw_lock);
	screen = SDL_VideoSurface;
	screen_arealen = (screen->h*screen->pitch);
	screen_contents = (Uint8 *)malloc(screen_arealen);
	if ( screen_contents ) {
		memcpy(screen_contents, screen->pixels, screen_arealen);
	}
	FB_SavePaletteTo(this, 256, saved_pal);
	ioctl(keyboard_fd, KDSETMODE, KD_TEXT);

	/* New console, switch to it */
	if ( ioctl(keyboard_fd, VT_ACTIVATE, which) == 0 ) {
		/* Wait for our console to be activated again */
		ioctl(keyboard_fd, VT_WAITACTIVE, which);
		while ( ioctl(keyboard_fd, VT_WAITACTIVE, current) < 0 ) {
			if ( (errno != EINTR) && (errno != EAGAIN) ) {
				/* Unknown VT error - cancel this */
				break;
			}
			SDL_Delay(500);
		}
	}

	/* Restore graphics mode and the contents of the screen */
	ioctl(keyboard_fd, KDSETMODE, KD_GRAPHICS);
	FB_RestorePaletteFrom(this, 256, saved_pal);
	if ( screen_contents ) {
		memcpy(screen->pixels, screen_contents, screen_arealen);
		free(screen_contents);
	}
	SDL_mutexV(hw_lock);
}

static void handle_keyboard(_THIS)
{
	unsigned char keybuf[BUFSIZ];
	int i, nread;
	int pressed;
	int scancode;
	SDL_keysym keysym;

	nread = read(keyboard_fd, keybuf, BUFSIZ);
	for ( i=0; i<nread; ++i ) {
		scancode = keybuf[i] & 0x7F;
		if ( keybuf[i] & 0x80 ) {
			pressed = SDL_RELEASED;
		} else {
			pressed = SDL_PRESSED;
		}
		TranslateKey(scancode, &keysym);
		/* Handle Alt-FN for vt switch */
		switch (keysym.sym) {
		    case SDLK_F1:
		    case SDLK_F2:
		    case SDLK_F3:
		    case SDLK_F4:
		    case SDLK_F5:
		    case SDLK_F6:
		    case SDLK_F7:
		    case SDLK_F8:
		    case SDLK_F9:
		    case SDLK_F10:
		    case SDLK_F11:
		    case SDLK_F12:
			if ( SDL_GetModState() & KMOD_ALT ) {
				switch_vt(this, (keysym.sym-SDLK_F1)+1);
				break;
			}
			/* Fall through to normal processing */
		    default:
			posted += SDL_PrivateKeyboard(pressed, &keysym);
			break;
		}
	}
}

void FB_PumpEvents(_THIS)
{
	fd_set fdset;
	int max_fd;
	static struct timeval zero;

	do {
		posted = 0;

		FD_ZERO(&fdset);
		max_fd = 0;
		if ( keyboard_fd >= 0 ) {
			FD_SET(keyboard_fd, &fdset);
			if ( max_fd < keyboard_fd ) {
				max_fd = keyboard_fd;
			}
		}
		if ( mouse_fd >= 0 ) {
			FD_SET(mouse_fd, &fdset);
			if ( max_fd < mouse_fd ) {
				max_fd = mouse_fd;
			}
		}
		if ( select(max_fd+1, &fdset, NULL, NULL, &zero) > 0 ) {
			if ( keyboard_fd >= 0 ) {
				if ( FD_ISSET(keyboard_fd, &fdset) ) {
					handle_keyboard(this);
				}
			}
			if ( mouse_fd >= 0 ) {
				if ( FD_ISSET(mouse_fd, &fdset) ) {
					handle_mouse(this);
				}
			}
		}
	} while ( posted );
}

void FB_InitOSKeymap(_THIS)
{
	int i;

	/* Initialize the BeOS key translation table */
	for ( i=0; i<SDL_TABLESIZE(keymap); ++i )
		keymap[i] = SDLK_UNKNOWN;

	keymap[SCANCODE_ESCAPE] = SDLK_ESCAPE;
	keymap[SCANCODE_1] = SDLK_1;
	keymap[SCANCODE_2] = SDLK_2;
	keymap[SCANCODE_3] = SDLK_3;
	keymap[SCANCODE_4] = SDLK_4;
	keymap[SCANCODE_5] = SDLK_5;
	keymap[SCANCODE_6] = SDLK_6;
	keymap[SCANCODE_7] = SDLK_7;
	keymap[SCANCODE_8] = SDLK_8;
	keymap[SCANCODE_9] = SDLK_9;
	keymap[SCANCODE_0] = SDLK_0;
	keymap[SCANCODE_MINUS] = SDLK_MINUS;
	keymap[SCANCODE_EQUAL] = SDLK_EQUALS;
	keymap[SCANCODE_BACKSPACE] = SDLK_BACKSPACE;
	keymap[SCANCODE_TAB] = SDLK_TAB;
	keymap[SCANCODE_Q] = SDLK_q;
	keymap[SCANCODE_W] = SDLK_w;
	keymap[SCANCODE_E] = SDLK_e;
	keymap[SCANCODE_R] = SDLK_r;
	keymap[SCANCODE_T] = SDLK_t;
	keymap[SCANCODE_Y] = SDLK_y;
	keymap[SCANCODE_U] = SDLK_u;
	keymap[SCANCODE_I] = SDLK_i;
	keymap[SCANCODE_O] = SDLK_o;
	keymap[SCANCODE_P] = SDLK_p;
	keymap[SCANCODE_BRACKET_LEFT] = SDLK_LEFTBRACKET;
	keymap[SCANCODE_BRACKET_RIGHT] = SDLK_RIGHTBRACKET;
	keymap[SCANCODE_ENTER] = SDLK_RETURN;
	keymap[SCANCODE_LEFTCONTROL] = SDLK_LCTRL;
	keymap[SCANCODE_A] = SDLK_a;
	keymap[SCANCODE_S] = SDLK_s;
	keymap[SCANCODE_D] = SDLK_d;
	keymap[SCANCODE_F] = SDLK_f;
	keymap[SCANCODE_G] = SDLK_g;
	keymap[SCANCODE_H] = SDLK_h;
	keymap[SCANCODE_J] = SDLK_j;
	keymap[SCANCODE_K] = SDLK_k;
	keymap[SCANCODE_L] = SDLK_l;
	keymap[SCANCODE_SEMICOLON] = SDLK_SEMICOLON;
	keymap[SCANCODE_APOSTROPHE] = SDLK_QUOTE;
	keymap[SCANCODE_GRAVE] = SDLK_BACKQUOTE;
	keymap[SCANCODE_LEFTSHIFT] = SDLK_LSHIFT;
	keymap[SCANCODE_BACKSLASH] = SDLK_BACKSLASH;
	keymap[SCANCODE_Z] = SDLK_z;
	keymap[SCANCODE_X] = SDLK_x;
	keymap[SCANCODE_C] = SDLK_c;
	keymap[SCANCODE_V] = SDLK_v;
	keymap[SCANCODE_B] = SDLK_b;
	keymap[SCANCODE_N] = SDLK_n;
	keymap[SCANCODE_M] = SDLK_m;
	keymap[SCANCODE_COMMA] = SDLK_COMMA;
	keymap[SCANCODE_PERIOD] = SDLK_PERIOD;
	keymap[SCANCODE_SLASH] = SDLK_SLASH;
	keymap[SCANCODE_RIGHTSHIFT] = SDLK_RSHIFT;
	keymap[SCANCODE_KEYPADMULTIPLY] = SDLK_KP_MULTIPLY;
	keymap[SCANCODE_LEFTALT] = SDLK_LALT;
	keymap[SCANCODE_SPACE] = SDLK_SPACE;
	keymap[SCANCODE_CAPSLOCK] = SDLK_CAPSLOCK;
	keymap[SCANCODE_F1] = SDLK_F1;
	keymap[SCANCODE_F2] = SDLK_F2;
	keymap[SCANCODE_F3] = SDLK_F3;
	keymap[SCANCODE_F4] = SDLK_F4;
	keymap[SCANCODE_F5] = SDLK_F5;
	keymap[SCANCODE_F6] = SDLK_F6;
	keymap[SCANCODE_F7] = SDLK_F7;
	keymap[SCANCODE_F8] = SDLK_F8;
	keymap[SCANCODE_F9] = SDLK_F9;
	keymap[SCANCODE_F10] = SDLK_F10;
	keymap[SCANCODE_NUMLOCK] = SDLK_NUMLOCK;
	keymap[SCANCODE_SCROLLLOCK] = SDLK_SCROLLOCK;
	keymap[SCANCODE_KEYPAD7] = SDLK_KP7;
	keymap[SCANCODE_CURSORUPLEFT] = SDLK_KP7;
	keymap[SCANCODE_KEYPAD8] = SDLK_KP8;
	keymap[SCANCODE_CURSORUP] = SDLK_KP8;
	keymap[SCANCODE_KEYPAD9] = SDLK_KP9;
	keymap[SCANCODE_CURSORUPRIGHT] = SDLK_KP9;
	keymap[SCANCODE_KEYPADMINUS] = SDLK_KP_MINUS;
	keymap[SCANCODE_KEYPAD4] = SDLK_KP4;
	keymap[SCANCODE_CURSORLEFT] = SDLK_KP4;
	keymap[SCANCODE_KEYPAD5] = SDLK_KP5;
	keymap[SCANCODE_KEYPAD6] = SDLK_KP6;
	keymap[SCANCODE_CURSORRIGHT] = SDLK_KP6;
	keymap[SCANCODE_KEYPADPLUS] = SDLK_KP_PLUS;
	keymap[SCANCODE_KEYPAD1] = SDLK_KP1;
	keymap[SCANCODE_CURSORDOWNLEFT] = SDLK_KP1;
	keymap[SCANCODE_KEYPAD2] = SDLK_KP2;
	keymap[SCANCODE_CURSORDOWN] = SDLK_KP2;
	keymap[SCANCODE_KEYPAD3] = SDLK_KP3;
	keymap[SCANCODE_CURSORDOWNRIGHT] = SDLK_KP3;
	keymap[SCANCODE_KEYPAD0] = SDLK_KP0;
	keymap[SCANCODE_KEYPADPERIOD] = SDLK_KP_PERIOD;
	keymap[SCANCODE_LESS] = SDLK_LESS;
	keymap[SCANCODE_F11] = SDLK_F11;
	keymap[SCANCODE_F12] = SDLK_F12;
	keymap[SCANCODE_KEYPADENTER] = SDLK_KP_ENTER;
	keymap[SCANCODE_RIGHTCONTROL] = SDLK_RCTRL;
	keymap[SCANCODE_CONTROL] = SDLK_RCTRL;
	keymap[SCANCODE_KEYPADDIVIDE] = SDLK_KP_DIVIDE;
	keymap[SCANCODE_PRINTSCREEN] = SDLK_PRINT;
	keymap[SCANCODE_RIGHTALT] = SDLK_RALT;
	keymap[SCANCODE_BREAK] = SDLK_BREAK;
	keymap[SCANCODE_BREAK_ALTERNATIVE] = SDLK_UNKNOWN;
	keymap[SCANCODE_HOME] = SDLK_HOME;
	keymap[SCANCODE_CURSORBLOCKUP] = SDLK_UP;
	keymap[SCANCODE_PAGEUP] = SDLK_PAGEUP;
	keymap[SCANCODE_CURSORBLOCKLEFT] = SDLK_LEFT;
	keymap[SCANCODE_CURSORBLOCKRIGHT] = SDLK_RIGHT;
	keymap[SCANCODE_END] = SDLK_END;
	keymap[SCANCODE_CURSORBLOCKDOWN] = SDLK_DOWN;
	keymap[SCANCODE_PAGEDOWN] = SDLK_PAGEDOWN;
	keymap[SCANCODE_INSERT] = SDLK_INSERT;
	keymap[SCANCODE_REMOVE] = SDLK_DELETE;
	keymap[119] = SDLK_PAUSE;
	keymap[SCANCODE_RIGHTWIN] = SDLK_RSUPER;
	keymap[SCANCODE_LEFTWIN] = SDLK_LSUPER;
	keymap[127] = SDLK_MENU;
}

static SDL_keysym *TranslateKey(int scancode, SDL_keysym *keysym)
{
	/* Set the keysym information */
	keysym->scancode = scancode;
	keysym->sym = keymap[scancode];
	keysym->mod = KMOD_NONE;

	/* If UNICODE is on, get the UNICODE value for the key */
	keysym->unicode = 0;
	if ( SDL_TranslateUNICODE ) {
		int map;
		SDLMod modstate;

		modstate = SDL_GetModState();
		map = 0;
		if ( modstate & KMOD_SHIFT ) {
			map |= (1<<KG_SHIFT);
		}
		if ( modstate & KMOD_CTRL ) {
			map |= (1<<KG_CTRL);
		}
		if ( modstate & KMOD_ALT ) {
			map |= (1<<KG_ALT);
		}
		if ( modstate & KMOD_MODE ) {
			map |= (1<<KG_ALTGR);
		}
		if ( KTYP(vga_keymap[map][scancode]) == KT_LETTER ) {
			if ( modstate & KMOD_CAPS ) {
				map ^= (1<<KG_SHIFT);
			}
		}
		if ( KTYP(vga_keymap[map][scancode]) == KT_PAD ) {
			if ( modstate & KMOD_NUM ) {
				keysym->unicode=KVAL(vga_keymap[map][scancode]);
			}
		} else {
			keysym->unicode = KVAL(vga_keymap[map][scancode]);
		}
	}
	return(keysym);
}
