/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998, 1999  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_x11image.c,v 1.2 2000/01/16 14:34:22 hercules Exp $";
#endif

#include <stdlib.h>

#include "SDL_error.h"
#include "SDL_endian.h"
#include "SDL_x11image_c.h"

/* Shared memory information */
extern int XShmQueryExtension(Display *dpy);	/* Not in X11 headers */

/* Shared memory error handler routine */
static int shm_error;
static int (*X_handler)(Display *, XErrorEvent *) = NULL;
static int shm_errhandler(Display *d, XErrorEvent *e)
{
        if ( e->error_code == BadAccess ) {
        	++shm_error;
        	return(0);
        } else
		return(X_handler(d,e));
}

/* Various screen update functions available */
static void X11_NormalUpdate(_THIS, int numrects, SDL_Rect *rects);
static void X11_MITSHMUpdate(_THIS, int numrects, SDL_Rect *rects);

int X11_SetupImage(_THIS, SDL_Surface *screen)
{
	/* Allocate shared memory if possible */
	if ( use_mitshm ) {
		shminfo.shmid = shmget(IPC_PRIVATE, screen->h*screen->pitch,
								IPC_CREAT|0777);
		if ( shminfo.shmid >= 0 ) {
			shminfo.shmaddr = (char *)shmat(shminfo.shmid, 0, 0);
			shminfo.readOnly = False;
			if ( shminfo.shmaddr != (char *)-1 ) {
				shm_error = False;
				X_handler = XSetErrorHandler(shm_errhandler);
				XShmAttach(SDL_Display, &shminfo);
				XSync(SDL_Display, True);
				XSetErrorHandler(X_handler);
				if ( shm_error == True )
					shmdt(shminfo.shmaddr);
			} else {
				shm_error = True;
			}
			shmctl(shminfo.shmid, IPC_RMID, NULL);
		} else {
			shm_error = True;
		}
		if ( shm_error == True )
			use_mitshm = 0;
	}
	if ( use_mitshm ) {
		screen->pixels = shminfo.shmaddr;
	} else {
		screen->pixels = malloc(screen->h*screen->pitch);
	}
	if ( screen->pixels == NULL ) {
		SDL_OutOfMemory();
		return(-1);
	}

	if ( use_mitshm ) {
		SDL_Ximage = XShmCreateImage(SDL_Display, SDL_Visual,
			DefaultDepth(SDL_Display, SDL_Screen),
			ZPixmap, shminfo.shmaddr, &shminfo, 
						screen->w, screen->h);
	} else {
		SDL_Ximage = XCreateImage(SDL_Display, SDL_Visual,
			DefaultDepth(SDL_Display, SDL_Screen),
			ZPixmap, 0, (char *)screen->pixels, 
			screen->w, screen->h,
			((screen->format)->BytesPerPixel == 3) ? 32 :
				(screen->format)->BytesPerPixel*8, 0);
	}
	if ( SDL_Ximage == NULL ) {
		SDL_SetError("Couldn't create XImage");
		if ( use_mitshm ) {
			XShmDetach(SDL_Display, &shminfo);
			XSync(SDL_Display, False);
			shmdt(shminfo.shmaddr);
			screen->pixels = NULL;
		}
		return(-1);
	}

	/* Determine what blit function to use */
	if ( use_mitshm ) {
		this->UpdateRects = X11_MITSHMUpdate;
	} else {
		this->UpdateRects = X11_NormalUpdate;
	}
	return(0);
}

void X11_DestroyImage(_THIS, SDL_Surface *screen)
{
	if ( SDL_Ximage ) {
		XDestroyImage(SDL_Ximage);
		if ( use_mitshm ) {
			XShmDetach(SDL_Display, &shminfo);
			XSync(SDL_Display, False);
			shmdt(shminfo.shmaddr);
		}
		SDL_Ximage = NULL;
	}
	if ( screen ) {
		screen->pixels = NULL;
	}
}

int X11_ResizeImage(_THIS, SDL_Surface *screen, Uint32 flags)
{
	int retval;

	X11_DestroyImage(this, screen);
	retval = X11_SetupImage(this, screen);
	return(retval);
}

/* We don't actually allow hardware surfaces other than the main one */
int X11_AllocHWSurface(_THIS, SDL_Surface *surface)
{
	return(-1);
}
void X11_FreeHWSurface(_THIS, SDL_Surface *surface)
{
	return;
}

int X11_LockHWSurface(_THIS, SDL_Surface *surface)
{
	return(0);
}
void X11_UnlockHWSurface(_THIS, SDL_Surface *surface)
{
	return;
}

int X11_FlipHWSurface(_THIS, SDL_Surface *surface)
{
	return(0);
}

/* Byte-swap the pixels in the display image */
static void X11_SwapPixels(SDL_Surface *screen, int numrects, SDL_Rect *rects)
{
	int i;
	int x, minx, maxx;
	int y, miny, maxy;

	switch (screen->format->BytesPerPixel) {
	    case 2: {
		Uint16 *spot;
		for ( i=0; i<numrects; ++i ) {
			minx = rects[i].x;
			maxx = rects[i].x+rects[i].w;
			miny = rects[i].y;
			maxy = rects[i].y+rects[i].h;
			for ( y=miny; y<maxy; ++y ) {
				spot = (Uint16 *) ((Uint8 *)screen->pixels +
						y * screen->pitch + minx * 2);
				for ( x=minx; x<maxx; ++x, ++spot ) {
					*spot = SDL_Swap16(*spot);
				}
			}
		}
	    }
	    break;

	    case 4: {
		Uint32 *spot;
		for ( i=0; i<numrects; ++i ) {
			minx = rects[i].x;
			maxx = rects[i].x+rects[i].w;
			miny = rects[i].y;
			maxy = rects[i].y+rects[i].h;
			for ( y=miny; y<maxy; ++y ) {
				spot = (Uint32 *) ((Uint8 *)screen->pixels +
						y * screen->pitch + minx * 4);
				for ( x=minx; x<maxx; ++x, ++spot ) {
					*spot = SDL_Swap32(*spot);
				}
			}
		}
	    }
	    break;

	    default:
		/* should never get here */
		break;
	}
}

static void X11_NormalUpdate(_THIS, int numrects, SDL_Rect *rects)
{
	int i;

	/* Check for endian-swapped X server, swap if necessary (VERY slow!) */
	if ( swap_pixels &&
	     ((this->screen->format->BytesPerPixel%2) == 0) ) {
		X11_SwapPixels(this->screen, numrects, rects);
		for ( i=0; i<numrects; ++i ) {
			XPutImage(GFX_Display, SDL_Window, SDL_GC, SDL_Ximage,
				rects[i].x, rects[i].y,
				rects[i].x, rects[i].y, rects[i].w, rects[i].h);
		}
		X11_SwapPixels(this->screen, numrects, rects);
	} else {
		for ( i=0; i<numrects; ++i ) {
			XPutImage(GFX_Display, SDL_Window, SDL_GC, SDL_Ximage,
				rects[i].x, rects[i].y,
				rects[i].x, rects[i].y, rects[i].w, rects[i].h);
		}
	}
	XSync(GFX_Display, False);
}

static void X11_MITSHMUpdate(_THIS, int numrects, SDL_Rect *rects)
{
	int i;

	for ( i=0; i<numrects; ++i ) {
		XShmPutImage(GFX_Display, SDL_Window, SDL_GC, SDL_Ximage,
				rects[i].x, rects[i].y,
				rects[i].x, rects[i].y, rects[i].w, rects[i].h,
									False);
	}
	XSync(GFX_Display, False);
}

