/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998, 1999  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_x11video.c,v 1.26 2000/02/22 07:01:49 hercules Exp $";
#endif

/* X11 based SDL video driver implementation.
   Note:  This implementation does not currently need X11 thread locking,
          since the event thread uses a separate X connection and any
          additional locking necessary is handled internally.  However,
          if full locking is neccessary, take a look at XInitThreads().
*/

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/ioctl.h>
#ifdef MTRR_SUPPORT
#include <asm/mtrr.h>
#include <sys/fcntl.h>
#endif

#include "SDL.h"
#include "SDL_error.h"
#include "SDL_timer.h"
#include "SDL_video.h"
#include "SDL_mouse.h"
#include "SDL_endian.h"
#include "SDL_sysvideo.h"
#include "SDL_pixels_c.h"
#include "SDL_events_c.h"
#include "SDL_x11wm_c.h"
#include "SDL_x11mouse_c.h"
#include "SDL_x11events_c.h"
#include "SDL_x11modes_c.h"
#include "SDL_x11image_c.h"
#include "blank_cursor.h"

/* Initialization/Query functions */
static int X11_VideoInit(_THIS, SDL_PixelFormat *vformat);
static SDL_Surface *X11_SetVideoMode(_THIS, SDL_Surface *current, int width, int height, int bpp, Uint32 flags);
static int X11_ToggleFullScreen(_THIS, int on);
static void X11_UpdateMouse(_THIS);
static int X11_SetColors(_THIS, int firstcolor, int ncolors);
static void X11_VideoQuit(_THIS);

/* X11 driver bootstrap functions */

static int X11_Available(void)
{
	Display *display;

	display = XOpenDisplay(NULL);
	if ( display != NULL ) {
		XCloseDisplay(display);
	}
	return(display != NULL);
}

static void X11_DeleteDevice(SDL_VideoDevice *device)
{
	free(device->hidden);
	free(device);
}

static SDL_VideoDevice *X11_CreateDevice(int devindex)
{
	SDL_VideoDevice *device;

	/* Initialize all variables that we clean on shutdown */
	device = (SDL_VideoDevice *)malloc(sizeof(SDL_VideoDevice));
	if ( device ) {
		memset(device, 0, (sizeof *device));
		device->hidden = (struct SDL_PrivateVideoData *)
				malloc((sizeof *device->hidden));
	}
	if ( (device == NULL) || (device->hidden == NULL) ) {
		SDL_OutOfMemory();
		if ( device ) {
			free(device);
		}
		return(0);
	}
	memset(device->hidden, 0, (sizeof *device->hidden));

	/* Set the driver flags */
	device->handles_any_size = 1;

	/* Set the function pointers */
	device->VideoInit = X11_VideoInit;
	device->ListModes = X11_ListModes;
	device->SetVideoMode = X11_SetVideoMode;
	device->ToggleFullScreen = X11_ToggleFullScreen;
	device->UpdateMouse = X11_UpdateMouse;
	device->SetColors = X11_SetColors;
	device->UpdateRects = NULL;
	device->VideoQuit = X11_VideoQuit;
	device->AllocHWSurface = X11_AllocHWSurface;
	device->CheckHWBlit = NULL;
	device->FillHWRect = NULL;
	device->SetHWColorKey = NULL;
	device->SetHWAlpha = NULL;
	device->LockHWSurface = X11_LockHWSurface;
	device->UnlockHWSurface = X11_UnlockHWSurface;
	device->FlipHWSurface = X11_FlipHWSurface;
	device->FreeHWSurface = X11_FreeHWSurface;
	device->SetIcon = X11_SetIcon;
	device->SetCaption = X11_SetCaption;
	device->IconifyWindow = X11_IconifyWindow;
	device->GrabInput = X11_GrabInput;
	device->GetWMInfo = X11_GetWMInfo;
	device->FreeWMCursor = X11_FreeWMCursor;
	device->CreateWMCursor = X11_CreateWMCursor;
	device->ShowWMCursor = X11_ShowWMCursor;
	device->WarpWMCursor = X11_WarpWMCursor;
	device->CheckMouseMode = X11_CheckMouseMode;
	device->InitOSKeymap = X11_InitOSKeymap;
	device->PumpEvents = X11_PumpEvents;

	device->free = X11_DeleteDevice;

	return device;
}

VideoBootStrap X11_bootstrap = {
	"x11", X11_Available, X11_CreateDevice
};

/* Shared memory information */
extern int XShmQueryExtension(Display *dpy);	/* Not in X11 headers */

/* Shared memory error handler routine */
static int shm_error;
static int (*X_handler)(Display *, XErrorEvent *) = NULL;
static int shm_errhandler(Display *d, XErrorEvent *e)
{
        if ( e->error_code == BadAccess ) {
        	++shm_error;
        	return(0);
        } else
		return(X_handler(d,e));
}

/* X11 I/O error handler routine */
static int (*XIO_handler)(Display *) = NULL;
static int xio_errhandler(Display *d)
{
	/* Ack!  Lost X11 connection! */

	/* We will crash if we try to clean up our display */
	if ( current_video->hidden->Ximage ) {
		SDL_VideoSurface->pixels = NULL;
	}
	current_video->hidden->X11_Display = NULL;

	/* Continue with the standard X11 error handler */
	return(XIO_handler(d));
}

static int X11_VideoInit(_THIS, SDL_PixelFormat *vformat)
{
	char *display;
	XGCValues gcv;
	XSetWindowAttributes xattr;

	/* Open the X11 display */
	display = NULL;		/* Get it from DISPLAY environment variable */
	if ( (strncmp(XDisplayName(display), ":", 1) == 0) ||
	     (strncmp(XDisplayName(display), "unix:", 5) == 0) ) {
		local_X11 = 1;
	} else {
		local_X11 = 0;
	}
	SDL_Display = XOpenDisplay(display);
	if ( SDL_Display == NULL ) {
		SDL_SetError("Couldn't open X11 display");
		return(-1);
	}

	/* Create an alternate X display for graphics updates -- allows us
	   to do graphics updates in a separate thread from event handling.
	   Thread-safe X11 doesn't seem to exist.
	 */
	GFX_Display = XOpenDisplay(display);
	if ( GFX_Display == NULL ) {
		SDL_SetError("Couldn't open X11 display");
		return(-1);
	}

	/* Set the error handler if we lose the X display */
	XIO_handler = XSetIOErrorHandler(xio_errhandler);

	/* Check for MIT shared memory extension */
	use_mitshm = 0;
	if ( local_X11 ) {
		use_mitshm = XShmQueryExtension(SDL_Display);
	}

	/* See whether or not we need to swap pixels */
	swap_pixels = 0;
	if ( SDL_BYTEORDER == SDL_LIL_ENDIAN ) {
		if ( XImageByteOrder(SDL_Display) == MSBFirst ) {
			swap_pixels = 1;
		}
	} else {
		if ( XImageByteOrder(SDL_Display) == LSBFirst ) {
			swap_pixels = 1;
		}
	}

	/* Determine the default screen depth */
	{
		XPixmapFormatValues *pix_format;
		int i, num_formats;

		vformat->BitsPerPixel = DefaultDepth(SDL_Display, SDL_Screen);
		pix_format = XListPixmapFormats(SDL_Display, &num_formats);
		if ( pix_format == NULL ) {
			SDL_SetError("Couldn't determine screen formats");
			return(-1);
		}
		for ( i=0; i<num_formats; ++i ) {
			if ( vformat->BitsPerPixel == pix_format[i].depth )
				break;
		}
		if ( i != num_formats )
			vformat->BitsPerPixel = pix_format[i].bits_per_pixel;
		XFree((char *)pix_format);
	}
	if ( vformat->BitsPerPixel < 8 ) {
		SDL_SetError("Color depth of at least 8 bit required");
		return(-1);
	}
	if ( vformat->BitsPerPixel > 8 ) {
		vformat->Rmask = SDL_Visual->red_mask;
	  	vformat->Gmask = SDL_Visual->green_mask;
	  	vformat->Bmask = SDL_Visual->blue_mask;
	}

	/* Get the available video modes */
	X11_GetVideoModes(this);

	/* See if we have been passed a window to use */
	SDL_windowid = getenv("SDL_WINDOWID");

	if ( SDL_windowid ) {
		FSwindow = 0;
		WMwindow = atol(SDL_windowid);
	} else {
		/* Create the window for fullscreen mode display */
		xattr.override_redirect = True;
		xattr.background_pixel = BlackPixel(SDL_Display, SDL_Screen);
		xattr.colormap = XDefaultColormap(SDL_Display, SDL_Screen);
		FSwindow = XCreateWindow(SDL_Display, SDL_Root, 0, 0, 32, 32, 0,
				CopyFromParent, CopyFromParent, CopyFromParent,
				CWOverrideRedirect | CWBackPixel | CWColormap,
				&xattr);
		XSelectInput(SDL_Display, FSwindow, StructureNotifyMask);

		/* Tell KDE to keep the fullscreen window on top */
		{ XEvent ev;
	          long mask;

			memset(&ev, 0, sizeof(ev));
			ev.xclient.type = ClientMessage;
			ev.xclient.window = SDL_Root;
			ev.xclient.message_type = XInternAtom(SDL_Display,
						"KWM_KEEP_ON_TOP", False);
			ev.xclient.format = 32;
			ev.xclient.data.l[0] = FSwindow;
			ev.xclient.data.l[1] = CurrentTime;
			mask = SubstructureRedirectMask;
			XSendEvent(SDL_Display, SDL_Root, False, mask, &ev);
		}

		/* Create the window for windowed management */
		WMwindow = XCreateSimpleWindow(SDL_Display, SDL_Root,
					0, 0, 32, 32, 0,
					BlackPixel(SDL_Display, SDL_Screen),
					BlackPixel(SDL_Display, SDL_Screen));
		/* Set the input hints so we get keyboard input */
		{
			XWMHints *hints;
			hints = XAllocWMHints();
			if ( hints != NULL ) {
				hints->input = True;
				hints->flags = InputHint;
				XSetWMHints(SDL_Display, WMwindow, hints);
				XFree(hints);
			}
		}
		XSelectInput(SDL_Display, WMwindow, (FocusChangeMask 
				| KeyPressMask | KeyReleaseMask
				| PropertyChangeMask | StructureNotifyMask));

		/* Set the class hints so we can get an icon (AfterStep) */
		{
			XClassHint *classhints;
			classhints = XAllocClassHint();
			if ( classhints != NULL ) {
				classhints->res_name = "SDL_App";
				classhints->res_class = "SDL_App";
				XSetClassHint(SDL_Display,WMwindow,classhints);
				XFree(classhints);
			}
		}

		/* Allow the window to be deleted by the window manager */
		WM_DELETE_WINDOW = XInternAtom(SDL_Display,
						"WM_DELETE_WINDOW", False);
		XSetWMProtocols(SDL_Display, WMwindow, &WM_DELETE_WINDOW, 1);
	}

	/* Create the blank cursor */
	SDL_BlankCursor = this->CreateWMCursor(this, blank_cdata, blank_cmask,
					BLANK_CWIDTH, BLANK_CHEIGHT,
						BLANK_CHOTX, BLANK_CHOTY);
	SDL_DisplayColormap = DefaultColormap(SDL_Display, SDL_Screen);

	/* Fill in some window manager capabilities */
	this->info.wm_available = 1;

	/* We're done! */
	XFlush(SDL_Display);
	return(0);
}

static void X11_DestroyWindow(_THIS, SDL_Surface *screen)
{
	if ( ! SDL_windowid ) {
		/* Hide the managed window */
		if ( WMwindow ) {
			XUnmapWindow(SDL_Display, WMwindow);
		}
		if ( screen && (screen->flags & SDL_FULLSCREEN) ) {
			screen->flags &= ~SDL_FULLSCREEN;
			X11_LeaveFullScreen(this);
		}

		/* Destroy the output window */
		if ( SDL_Window ) {
			XDestroyWindow(SDL_Display, SDL_Window);
		}

		/* Free the colormap entries */
		if ( SDL_XPixels ) {
			int numcolors;
			unsigned long pixel;

			numcolors = this->screen->format->palette->ncolors;
			for ( pixel=0; pixel<numcolors; ++pixel ) {
				while ( SDL_XPixels[pixel] > 0 ) {
					XFreeColors(GFX_Display,
						SDL_DisplayColormap,&pixel,1,0);
					--SDL_XPixels[pixel];
				}
			}
			free(SDL_XPixels);
			SDL_XPixels = NULL;
		} 
	}
}

static void X11_SetSizeHints(_THIS, int w, int h)
{
	XSizeHints *hints;

	hints = XAllocSizeHints();
	if ( hints ) {
		hints->min_width = hints->max_width = w;
		hints->min_height = hints->max_height = h;
		hints->flags = PMaxSize | PMinSize;
		XSetWMNormalHints(SDL_Display, WMwindow, hints);
		XFree(hints);
	}
}

static int X11_CreateWindow(_THIS,
			SDL_Surface *screen, int w, int h, Uint32 flags)
{
	XGCValues gcv;

	/* If a window is already present, destroy it and start fresh */
	if ( SDL_Window ) {
		X11_DestroyWindow(this, screen);
	}

	/* Resize the window manager window */
	if ( ! SDL_windowid ) {
		X11_SetSizeHints(this, w, h);
		current_w = w;
		current_h = h;
		XResizeWindow(SDL_Display, WMwindow, w, h);
	}

	/* Create (or use) the X11 display window */
	if ( SDL_windowid ) {
		SDL_Window = atol(SDL_windowid);
	} else {
		SDL_Window = XCreateSimpleWindow(SDL_Display, WMwindow, 0, 0,
					w, h, 0,
					BlackPixel(SDL_Display, SDL_Screen),
					BlackPixel(SDL_Display, SDL_Screen));
		XSelectInput(SDL_Display, SDL_Window,
					( EnterWindowMask | LeaveWindowMask
					| ButtonPressMask | ButtonReleaseMask
					| PointerMotionMask | ExposureMask ));
	}

	/* Create the graphics context */
	gcv.graphics_exposures = False;
	SDL_GC = XCreateGC(SDL_Display, SDL_Window, GCGraphicsExposures, &gcv);
	if ( ! SDL_GC ) {
		SDL_SetError("Couldn't create graphics context");
		return(-1);
	}

	/* If the surface is palettized, create the colormap */
	/* FIXME: Use visual info */
	if ( screen->format->palette != NULL ) {
		int     ncolors;

		/* Get the default display colormap (used by icons) */
		SDL_XColorMap = SDL_DisplayColormap;

		/* Allocate the pixel flags */
		ncolors = (1<<screen->format->BitsPerPixel);
		SDL_XPixels = (int *)malloc(ncolors*sizeof(int));
		if ( SDL_XPixels == NULL ) {
			SDL_OutOfMemory();
			return(-1);
		}
		memset(SDL_XPixels, 0, ncolors*sizeof(*SDL_XPixels));

		if ( flags & SDL_HWPALETTE ) {
			screen->flags |= SDL_HWPALETTE;
			if ( SDL_PrivateColormap == 0 ) {
				SDL_PrivateColormap = XCreateColormap(SDL_Display, SDL_Window, SDL_Visual, AllocAll);
			}
			SDL_XColorMap = SDL_PrivateColormap;
		} else {
			/* Free any currently allocated hidden colormap */
			if ( SDL_PrivateColormap != 0 ) {
				XFreeColormap(SDL_Display, SDL_PrivateColormap);
				SDL_PrivateColormap = 0;
			}
			SDL_XColorMap = SDL_DisplayColormap;
		}
		XSetWindowColormap(SDL_Display, FSwindow, SDL_XColorMap);
		XSetWindowColormap(SDL_Display, WMwindow, SDL_XColorMap);
		XSetWindowColormap(SDL_Display, SDL_Window, SDL_XColorMap);
	}

	/* Cache the window in the server, when possible */
	{
		unsigned long mask;
		Screen *xscreen;
		XSetWindowAttributes a;

		xscreen = ScreenOfDisplay(SDL_Display, SDL_Screen);
		if ( DoesBackingStore(xscreen) != NotUseful ) {
			mask = CWBackingStore;
			a.backing_store = DoesBackingStore(xscreen);
		} else {
			mask = CWSaveUnder;
			a.save_under = DoesSaveUnders(xscreen);
		}
		XChangeWindowAttributes(SDL_Display, SDL_Window, mask, &a);
	}

	/* Map them both and go fullscreen, if requested */
	if ( ! SDL_windowid ) {
		XMapWindow(SDL_Display, SDL_Window);
		XMapWindow(SDL_Display, WMwindow);
		if ( flags & SDL_FULLSCREEN ) {
			screen->flags |= SDL_FULLSCREEN;
			X11_WaitMapped(this, WMwindow);
#ifdef GRAB_FULLSCREEN
			X11_EnterFullScreen(this);
#else
			X11_QueueEnterFullScreen(this);
#endif
		} else {
			screen->flags &= ~SDL_FULLSCREEN;
		}
	}
	return(0);
}

static int X11_ResizeWindow(_THIS,
			SDL_Surface *screen, int w, int h, Uint32 flags)
{
	if ( ! SDL_windowid ) {
		/* Resize the window manager window */
		X11_SetSizeHints(this, w, h);
		current_w = w;
		current_h = h;
		XResizeWindow(SDL_Display, WMwindow, w, h);

		/* Resize the fullscreen and display windows */
		if ( flags & SDL_FULLSCREEN ) {
			if ( screen->flags & SDL_FULLSCREEN ) {
				X11_ResizeFullScreen(this);
			} else {
				screen->flags |= SDL_FULLSCREEN;
#ifdef GRAB_FULLSCREEN
				X11_EnterFullScreen(this);
#else
				X11_QueueEnterFullScreen(this);
#endif
			}
		} else {
			if ( screen->flags & SDL_FULLSCREEN ) {
				screen->flags &= ~SDL_FULLSCREEN;
				X11_LeaveFullScreen(this);
			}
		}
		XResizeWindow(SDL_Display, SDL_Window, w, h);
	}
	return(0);
}

SDL_Surface *X11_SetVideoMode(_THIS, SDL_Surface *current,
				int width, int height, int bpp, Uint32 flags)
{
	/* Lock the event thread, in multi-threading environments */
	SDL_Lock_EventThread();

	/* Check the combination of flags we were passed */
	if ( flags & SDL_FULLSCREEN ) {
		/* Clear fullscreen flag if not supported */
		if ( SDL_windowid ) {
			flags &= ~SDL_FULLSCREEN;
		}
	}

	/* Set up the X11 window */
	if ( SDL_Window ) {
		X11_ResizeWindow(this, current, width, height, flags);
	} else {
		X11_CreateWindow(this, current, width, height, flags);
	}

	/* Set up the new mode framebuffer */
	if ( (current->w != width) || (current->h != height) ) {
		current->w = width;
		current->h = height;
		current->pitch = SDL_CalculatePitch(current);
		X11_ResizeImage(this, current, flags);
	}

	/* Release the event thread */
	XSync(SDL_Display, False);
	SDL_Unlock_EventThread();

	/* We're done! */
	return(current);
}

static int X11_ToggleFullScreen(_THIS, int on)
{
	Uint32 event_thread;

	/* Don't switch if we don't own the window */
	if ( SDL_windowid ) {
		return(0);
	}

	/* Don't lock if we are the event thread */
	event_thread = SDL_EventThreadID();
	if ( event_thread && (SDL_ThreadID() == event_thread) ) {
		event_thread = 0;
	}
	if ( event_thread ) {
		SDL_Lock_EventThread();
	}
	if ( on ) {
#ifdef GRAB_FULLSCREEN
		this->screen->flags |= SDL_FULLSCREEN;
		X11_EnterFullScreen(this);
#else
		X11_QueueEnterFullScreen(this);
#endif
	} else {
		this->screen->flags &= ~SDL_FULLSCREEN;
		X11_LeaveFullScreen(this);
	}
	XSync(SDL_Display, False);
	if ( event_thread ) {
		SDL_Unlock_EventThread();
	}
	SDL_ResetKeyboard();
	return(1);
}

/* Update the current mouse state and position */
static void X11_UpdateMouse(_THIS)
{
	Window u1; int u2; unsigned int u3;
	Window current_win;
	int x, y;
	unsigned int mask;

	/* Lock the event thread, in multi-threading environments */
	SDL_Lock_EventThread();
	if ( XQueryPointer(SDL_Display, SDL_Window, &u1, &current_win,
	                   &u2, &u2, &x, &y, &mask) ) {
		if ( (x >= 0) && (x < SDL_VideoSurface->w) &&
		     (y >= 0) && (y < SDL_VideoSurface->h) ) {
			SDL_PrivateAppActive(1, SDL_APPMOUSEFOCUS);
			SDL_PrivateMouseMotion(0, 0, x, y);
		} else {
			SDL_PrivateAppActive(0, SDL_APPMOUSEFOCUS);
		}
	}
	SDL_Unlock_EventThread();
}

int X11_SetColors(_THIS, int firstcolor, int ncolors)
{
	SDL_Color *colors;
	int      alloct_all;
	int      i;
	XColor  *xcmap;

	/* Check to make sure we have a colormap allocated */
	if ( SDL_XPixels == NULL ) {
		return(0);
	}
	alloct_all = 1;
	colors = this->screen->format->palette->colors;
	ncolors = this->screen->format->palette->ncolors;
#ifdef linux /* alloca() is faster */
	xcmap = (XColor *)alloca(ncolors*sizeof(*xcmap));
#else
	xcmap = (XColor *)malloc(ncolors*sizeof(*xcmap));
#endif
	if ( xcmap == NULL ) {
		return(0);
	}

	/* It's easy if we have a hidden colormap */
	if ( (this->screen->flags & SDL_HWPALETTE) == SDL_HWPALETTE ) {
		for ( i=0; i<ncolors; ++i ) {
			xcmap[i].pixel = i;
			xcmap[i].red   = (colors[i].r<<8)|colors[i].r;
			xcmap[i].green = (colors[i].g<<8)|colors[i].g;
			xcmap[i].blue  = (colors[i].b<<8)|colors[i].b;
			xcmap[i].flags = (DoRed|DoGreen|DoBlue);
		}
		XStoreColors(GFX_Display, SDL_XColorMap, xcmap, ncolors);
		XFlush(GFX_Display);
	} else {
		unsigned long pixel;
		XColor        wanted;
		XColor        color;

		/* Free existing allocated colors */
		for ( pixel=0; pixel<ncolors; ++pixel ) {
			while ( SDL_XPixels[pixel] > 0 ) {
				XFreeColors(GFX_Display, SDL_XColorMap, &pixel,
									1, 0);
				--SDL_XPixels[pixel];
			}
		}

		/* Try to allocate all the colors */
		for ( i=0; i<ncolors; ++i ) {
			color.pixel = i;
			color.red   = (colors[i].r<<8);
			color.green = (colors[i].g<<8);
			color.blue  = (colors[i].b<<8);
			color.flags = (DoRed|DoGreen|DoBlue);
			if ( alloct_all ) {
				memcpy(&wanted, &color, sizeof(wanted));
				if (XAllocColor(GFX_Display,
						SDL_XColorMap, &color)) {
					++SDL_XPixels[color.pixel];
				}
				if ( memcmp(&color, &wanted, sizeof(wanted)) )
					alloct_all = 0;
			} else {
				if (XAllocColor(GFX_Display,
						SDL_XColorMap, &color)) {
					++SDL_XPixels[color.pixel];
				}
			}
		}
		if ( ! alloct_all ) {
			/* Copy the colors from the colormap to our palette */
			for ( i=0; i<ncolors; ++i ) {
				xcmap[i].pixel = i;
				xcmap[i].flags = (DoRed|DoGreen|DoBlue);
			}
			XQueryColors(GFX_Display,SDL_XColorMap,xcmap,ncolors);
			for ( i=0; i<ncolors; ++i ) {
				colors[i].r = (xcmap[i].red>>8);
				colors[i].g = (xcmap[i].green>>8);
				colors[i].b = (xcmap[i].blue>>8);
			}
		}
	}
#ifdef linux
	/* alloca() cleans up on function return */
#else
	free(xcmap);
#endif
	return(alloct_all);
}

/* Note:  If we are terminated, this could be called in the middle of
   another SDL video routine -- notably UpdateRects.
*/
void X11_VideoQuit(_THIS)
{
	/* Shutdown everything that's still up */
	/* The event thread should be done, so we can touch SDL_Display */
	if ( SDL_Display != NULL ) {

		/* Start shutting down the windows */
		X11_DestroyImage(this, this->screen);
		X11_DestroyWindow(this, this->screen);
		X11_FreeVideoModes(this);
		if ( SDL_PrivateColormap != 0 ) {
			XFreeColormap(SDL_Display, SDL_PrivateColormap);
			SDL_PrivateColormap = 0;
		}
		if ( SDL_iconcolors ) {
			unsigned long pixel;
			int numcolors =
				((this->screen->format)->palette)->ncolors;
			for ( pixel=0; pixel<numcolors; ++pixel ) {
				while ( SDL_iconcolors[pixel] > 0 ) {
					XFreeColors(SDL_Display,
						SDL_DisplayColormap,&pixel,1,0);
					--SDL_iconcolors[pixel];
				}
			}
			free(SDL_iconcolors);
			SDL_iconcolors = NULL;
		} 
		/* Free that blank cursor */
		if ( SDL_BlankCursor != NULL ) {
			this->FreeWMCursor(this, SDL_BlankCursor);
			SDL_BlankCursor = NULL;
		}

		/* Close the X11 graphics connection */
		if ( GFX_Display != NULL ) {
			XCloseDisplay(GFX_Display);
			GFX_Display = NULL;
		}

		/* Close the X11 display connection */
		XCloseDisplay(SDL_Display);
		SDL_Display = NULL;

		/* Reset the IO error handler */
		if ( XIO_handler ) {
			XSetIOErrorHandler(XIO_handler);
		}
	}
	if ( this->screen && (this->screen->flags & SDL_HWSURFACE) ) {
		/* Direct screen access, no memory buffer */
		this->screen->pixels = NULL;
	}
}
