// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#ifdef __GNUG__
#pragma implementation
#endif
#include "EasyParser.H"
#ifdef STDIO
#include "StdioStorage.H"
#endif
#include "PosixStorage.H"
#include "macros.H"
#ifdef MULTI_BYTE
#include "UTF8CodingSystem.H"
#include "UCS2CodingSystem.H"
#include "UnicodeCodingSystem.H"
#include "UJISCodingSystem.H"
#include "SJISCodingSystem.H"
#endif
#include "IdentityCodingSystem.H"
#include "Vector.H"
#include "Portable.H"

#include <iostream.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#ifdef MULTI_BYTE
static UnivCharsetDesc::Range range = { 0, 65536, 0 };
#else
static UnivCharsetDesc::Range range = { 0, 256, 0 };
#endif

#ifdef MULTI_BYTE
static UTF8CodingSystem utf8CodingSystem;
static UCS2CodingSystem ucs2CodingSystem;
static UnicodeCodingSystem unicodeCodingSystem;
static UJISCodingSystem ujisCodingSystem;
static SJISCodingSystem sjisCodingSystem;
#endif
static IdentityCodingSystem identityCodingSystem;

static struct {
  const char *name;
  const CodingSystem *cs;
} codingSystems[] = {
#ifdef MULTI_BYTE
  { "UTF8", &utf8CodingSystem },
  { "UCS2", &ucs2CodingSystem },
  { "UNICODE", &unicodeCodingSystem },
  { "UJIS", &ujisCodingSystem },
  { "SJIS", &sjisCodingSystem },
#endif
  { "IDENTITY", &identityCodingSystem },
};

EasyParser::EasyParser(const char *codingName)
: storageManager_(0),
  fdStorageManager_(0),
  parser_(0),
  systemCharset_(&range, 1),
  MessageReporter(&errorStream_)
{
  codingSystem_ = 0;
  if (codingName && strlen(codingName) < 50) {
    char buf[50];
    strcpy(buf, codingName);
    for (char *p = buf; *p; p++)
      *p = toupper((unsigned char)*p);
    for (int i = 0; i < SIZEOF(codingSystems); i++)
      if (strcmp(buf, codingSystems[i].name) == 0) {
	codingSystem_ = codingSystems[i].cs;
	break;
      }
  }
  if (!codingSystem_)
    codingSystem_ = &identityCodingSystem;
  errorStream_.open(cerr.rdbuf(), codingSystem_);
}

EasyParser::~EasyParser()
{
  delete parser_;
  delete portable_;
  delete storageManager_;
  delete fdStorageManager_;
}

#ifdef __GNUG__
typedef Vector<CString> Dummy_Vector_CString;
#endif

void EasyParser::init(int nFiles, char *const *files,
		      Boolean allLinkTypesActivated)
{
  Vector<CString> filenames(nFiles == 0 ? 1 : nFiles);
  int i;
  for (i = 0; i < nFiles; i++)
    filenames[i] = codingSystem_->convertIn(strcmp(files[i], "-") == 0
					    ? "FD:0"
					    : files[i]);
  if (nFiles == 0)
    filenames[0] = codingSystem_->convertIn("FD:0");
  storageManager_ = new PosixStorageManager("FILE",
					    systemCharset_,
					    codingSystem_,
					    5);
  portable_ = new Portable(storageManager_, codingSystem_);
  fdStorageManager_ = new PosixFdStorageManager("FD", systemCharset_);
  portable_->registerStorageManager(fdStorageManager_);
  for (i = 0; i < SIZEOF(codingSystems); i++)
    portable_->registerCodingSystem(codingSystems[i].name,
				    codingSystems[i].cs);
  parser_ = new SgmlParser(portable_->entityManager(),
			   portable_->openDocumentEntity(filenames.pointer(),
							 filenames.length(),
							 systemCharset_,
							 *this),
			   systemCharset_,
			   options);
  for (i = 0; i < catalogSysids_.length(); i++)
    // filenames specified on command-line must exist
    portable_->addCatalogSysid(codingSystem_->convertIn(catalogSysids_[i]), 1);
  const char *e = getenv("SGML_CATALOG_FILES");
  if (!e)
    e = "CATALOG";
  // FIXME Should probably convert before splitting.
  String<char> str(e, strlen(e) + 1);
  char *start = str.pointer();
  char *p = start;
  for (;;) {
    if (*p == '\0') {
      portable_->addCatalogSysid(codingSystem_->convertIn(start));
      break;
    }
    if (*p == ':') {
      *p++ = '\0';
      portable_->addCatalogSysid(codingSystem_->convertIn(start));
      start = p;
    }
    else
      p++;
  }
  for (i = 0; i < includes_.length(); i++)
    parser_->addInclude(codingSystem_->convertIn(includes_[i]));
  for (i = 0; i < activeLinkTypes_.length(); i++) 
    parser_->activateLinkType(codingSystem_->convertIn(activeLinkTypes_[i]));
  if (allLinkTypesActivated)
    parser_->allLinkTypesActivated();
}

const char *EasyParser::messageText(const char *source, int number)
{
  static struct {
    const char *source;
    const char *(*getText)(int);
  } messageSourceTable[] = {
    { SgmlParser::messageSource, SgmlParser::messageText },
    { Portable::messageSource, Portable::messageText },
    { PosixStorageManager::messageSource, PosixStorageManager::messageText },
    { PosixFdStorageManager::messageSource, PosixFdStorageManager::messageText },
#ifdef STDIO
    { StdioStorageManager::messageSource, StdioStorageManager::messageText },
#endif
  };
  for (int i = 0; i < SIZEOF(messageSourceTable); i++)
    if (source == messageSourceTable[i].source)
      return messageSourceTable[i].getText(number);
  return 0;
}

void EasyParser::addCatalogSysid(const char *s)
{
  catalogSysids_.grow() = s;
}

void EasyParser::addInclude(const char *s)
{
  includes_.grow() = s;
}

void EasyParser::activateLinkType(const char *s)
{
  activeLinkTypes_.grow() = s;
}
