// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#include "MessageReporter.H"
#include "Portable.H"
#include "SgmlParser.H"
#include "macros.H"
#include "rtti.H"
#include "MessageArg.H"
#include "ErrnoMessageArg.H"
#include "FilenameMessageArg.H"
#include "Event.H"

#include <string.h>
extern "C" {
  char *strerror(int);
}

MessageReporter::MessageReporter(OutputCharStream *errsp)
: os_(errsp), options_(0)
{
}

void MessageReporter::addOption(Option option)
{
  options_ |= option;
}

void MessageReporter::setCurrentLocation(const Location &loc)
{
  currentLocation_ = loc;
}

void MessageReporter::inputContextMessage(const char *source,
					  const Location &loc,
					  unsigned flags, int number,
					  const MessageArg **args,
					  int nArgs)
{
  Vector<Owner<MessageArg> > argv(nArgs);
  for (int i = 0; i < nArgs; i++)
    argv[i] = args[i]->copy();
  MessageEvent::Type type
    = MessageEvent::Type(flags & InputContext::severityMask);
  Location l;
  if (loc.origin().isNull()) {
    l = currentLocation_;
    if (flags & InputContext::parentLocation) {
      if (!l.origin().isNull())
	l = l.origin()->parent();
    }
  }
  else if (!(flags & InputContext::nullLocation))
    l = loc;
  MessageEvent event(source, type, l, number, argv, Location());
  reportMessage(event);
}

void MessageReporter::reportMessage(const MessageEvent &message)
{
  Offset off;
  const ExternalInfo *externalInfo = locationHeader(message.location(), off);
  printLocation(externalInfo, off);
  char c = '\0';
  switch (message.type()) {
  case MessageEvent::info:
    c = 'I';
    break;
  case MessageEvent::warning:
    c = 'W';
    break;
  case MessageEvent::quantityError:
    c = 'Q';
    break;
  case MessageEvent::error:
    c = 'E';
    break;
  }
  os() << c << ": ";
  const char *text = messageText(message.source(), message.number());
  if (!text)
    text = "(invalid message)";
  while (*text != '\0') {
    if (*text == '%') {
      text++;
      if (*text == '\0')
	break;
      if (*text >= '1' && *text <= '9') {
	const MessageArg *arg = message.arg(*text - '1');
	arg->append(*this);
      }
      else if (*text == 'L') {
	os() << '\n';
	Offset off;
	const ExternalInfo *externalInfo = locationHeader(message.auxLocation(),
							  off);
	if (!printLocation(externalInfo, off))
	  os() << "(unknown location):";
	os() << ' ';
      }
      else
	os() << *text;
      text++;
    }
    else {
      os() << *text;
      text++;
    }
  }
  os() << '\n';
  if ((options_ & openElements) && message.nOpenElements() > 0) {
    printLocation(externalInfo, off);
    os() << " open elements:";
    unsigned nOpenElements = message.nOpenElements();
    for (unsigned i = 0;; i++) {
      if (i > 0
	  && (i == nOpenElements || message.openElementInfo(i).included)) {
	// describe last match in previous open element
	const OpenElementInfo &prevInfo(message.openElementInfo(i - 1));
	if (prevInfo.matchType.length() != 0) {
	  os() << " (" << prevInfo.matchType;
	  if (prevInfo.matchIndex != 0)
	    os() << '[' << (unsigned long)prevInfo.matchIndex << ']';
	  os() << ')';
	}
      }
      if (i == nOpenElements)
	break;
      const OpenElementInfo &e(message.openElementInfo(i));
      os() << ' ' << e.gi;
      if (i > 0 && !e.included) {
	unsigned long n = message.openElementInfo(i - 1).matchIndex;
	if (n != 0)
	  os() << '[' << n << ']';
      }
    }
    os() << '\n';
  }
  os().flush();
}

const ExternalInfo *MessageReporter::locationHeader(const Location &loc,
						    Offset &off)
{
  const Origin *origin = loc.origin().pointer();
  Index index = loc.index();
  if (!(options_ & openEntities)) {
    while (origin) {
      const EntityOrigin *entityOrigin = origin->asEntityOrigin();
      if (entityOrigin) {
	const ExternalInfo *externalInfo = entityOrigin->externalInfo();
	if (externalInfo) {
	  NamedCharRef ref;
	  while (entityOrigin->lookupIndex(index, off, ref)
		 == EntityOrigin::replacementCharIndex)
	    index = ref.refStartIndex();
	  return externalInfo;
	}
      }
      const Location &loc = origin->parent();
      index = loc.index() + origin->refLength();
      origin = loc.origin().pointer();
    }
  }
  else {
    Boolean doneHeader = 0;
    while (origin) {
      const EntityOrigin *entityOrigin = origin->asEntityOrigin();
      if (entityOrigin) {
	if (!doneHeader) {
	  if (!entityOrigin->entity().isNull()) {
	    Offset parentOff;
	    Location parentLoc = entityOrigin->parent();
	    parentLoc += entityOrigin->refLength();
	    const ExternalInfo *parentInfo = locationHeader(parentLoc, parentOff);
	    if (parentInfo) {
	      os() << "In entity " << entityOrigin->entity()->name()
		<< " included from ";
	      if (!printLocation(parentInfo, parentOff))
		os() << "(unknown location):";
	      os() << '\n';
	    }
	  }
	  doneHeader = 1;
	}
	NamedCharRef ref;
	while (entityOrigin->lookupIndex(index, off, ref)
	       == EntityOrigin::replacementCharIndex)
	  index = ref.refStartIndex();
	const ExternalInfo *externalInfo = entityOrigin->externalInfo();
	if (externalInfo)
	  return externalInfo;
	const InternalEntity *internal
	  = entityOrigin->entity()->asInternalEntity();
	if (!internal)
	  break;
	Location loc = internal->text().charLocation(off);
	index = loc.index();
	origin = loc.origin().pointer();
      }
      else {
	const Location &loc = origin->parent();
	index = loc.index() + origin->refLength();
	origin = loc.origin().pointer();
      }
    }
  }
  return 0;
}

Boolean MessageReporter::printLocation(const ExternalInfo *externalInfo,
				       Offset off)
{
  if (!externalInfo)
    return 0;
  const StorageObjectSpec *sos;
  unsigned long lineno;
  unsigned long colno;
  unsigned long byteIndex;
  if (!Portable::externalize(externalInfo, off, sos, lineno, colno, byteIndex))
    return 0;
  if (strcmp(sos->storageType, "FILE") != 0)
    os() << sos->storageType << ':';
  os() << sos->id;
  os() << ':' << lineno << ':' << colno << ':';
#if 0
  if (byteIndex != size_t(-1))
    os () << byteIndex << ':';
#endif
  return 1;
}

void MessageReporter::appendNumber(unsigned long n)
{
  os() << n;
}

void MessageReporter::appendOrdinal(unsigned long n)
{
  os() << n;
  switch (n % 10) {
  case 1:
    os() << "st";
    break;
  case 2:
    os() << "nd";
    break;
  case 3:
    os() << "rd";
    break;
  default:
    os() << "th";
    break;
  }
}

void MessageReporter::appendChars(const Char *p, size_t n)
{
  os().put('`').write(p, n).put('\'');
}

void MessageReporter::appendEntityManagerArg(const EntityManagerMessageArg *p)
{
  const FilenameMessageArg *fa = ptr_cast(FilenameMessageArg, p);
  if (fa) {
    os() << '`' << fa->filename() << '\'';
    return;
  }
  const ErrnoMessageArg *ea = ptr_cast(ErrnoMessageArg, p);
  if (ea) {
    os() << strerror(ea->errnum());
    return;
  }
  os() << "(unknown argument type)";
}

void MessageReporter::appendFragment(int n)
{
  const char *s = SgmlParser::fragmentText(n);
  if (s)
    os() << s;
  else
    os() << "(unknown fragment " << n << ")";
}
