// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#include "OutputCharStream.H"
#include "CodingSystem.H"
#include "macros.H"
#include <iostream.h>
#include <stdio.h>

OutputCharStream::OutputCharStream()
: ptr_(0), end_(0)
{
}

OutputCharStream::~OutputCharStream()
{
}

OutputCharStream &OutputCharStream::write(const Char *s, size_t n)
{
  for (;;) {
    size_t spare = end_ - ptr_;
    if (n <= spare) {
      memcpy(ptr_, s, n*sizeof(Char));
      ptr_ += n;
      break;
    }
    if (spare > 0) {
      memcpy(ptr_, s, spare*sizeof(Char));
      ptr_ += spare;
      s += spare;
      n -= spare;
    }
    n--;
    flushBuf(*s++);
  }
  return *this;
}

OutputCharStream &OutputCharStream::operator<<(const char *s)
{
  while (*s)
    put(*s++);
  return *this;
}

// FIXME Avoid stdio

OutputCharStream &OutputCharStream::operator<<(unsigned long n)
{
  char buf[sizeof(unsigned long)*3 + 1];
  sprintf(buf, "%lu", n);
  return *this << buf;
}

OutputCharStream &OutputCharStream::operator<<(int n)
{
  char buf[sizeof(int)*3 + 2];
  sprintf(buf, "%d", n);
  return *this << buf;
}


IosOutputCharStream::IosOutputCharStream()
: buf_(0), byteStream_(0), encoder_(0)
{
}

IosOutputCharStream::IosOutputCharStream(streambuf *byteStream,
					 const OutputCodingSystem *codingSystem)
: buf_(0),
  byteStream_(byteStream),
  encoder_(codingSystem->makeEncoder())
{
  allocBuf(codingSystem->fixedBytesPerChar());
}

IosOutputCharStream::~IosOutputCharStream()
{
  if (byteStream_)
    flush();
  delete [] buf_;
  delete encoder_;
}

void IosOutputCharStream::open(streambuf *byteStream,
			       const OutputCodingSystem *codingSystem)
{
  if (byteStream_)
    flush();
  byteStream_ = byteStream;
  delete encoder_;
  encoder_ = codingSystem->makeEncoder();
  delete [] buf_;
  buf_ = 0;
  ptr_ = end_ = buf_;
  allocBuf(codingSystem->fixedBytesPerChar());
}

void IosOutputCharStream::flush()
{
  if (ptr_ > buf_) {
    encoder_->output(buf_, ptr_ - buf_, byteStream_);
    ptr_ = buf_;
  }
  byteStream_->sync();
}

void IosOutputCharStream::flushBuf(Char c)
{
  ASSERT(buf_ != 0);
  encoder_->output(buf_, ptr_ - buf_, byteStream_);
  ptr_ = buf_;
  *ptr_++ = c;
}

void IosOutputCharStream::allocBuf(int bytesPerChar)
{
  const int blockSize = 1024;
  size_t bufSize = bytesPerChar ? blockSize/bytesPerChar : blockSize;
  ptr_ = buf_ = new Char[bufSize];
  end_ = buf_ + bufSize;
}

StrOutputCharStream::StrOutputCharStream()
{
  sync(0);
}

void StrOutputCharStream::extractString(CString &str)
{
  str_.setLength(ptr_ - str_.pointer());
  str_.moveTo(str);
  sync(0);
}

void StrOutputCharStream::flushBuf(Char c)
{
  size_t usedLength = ptr_ - str_.pointer();
  size_t oldLength = str_.length();
  str_.setLength(oldLength ? 2*oldLength : 10);
  sync(usedLength);
  *ptr_++ = c;
}

void StrOutputCharStream::flush()
{
}

void StrOutputCharStream::sync(size_t length)
{
  ptr_ = str_.pointer() + length;
  end_ = str_.pointer() + str_.length();
}
