// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#ifdef __GNUG__
#pragma implementation
#endif

#include "RastEventHandler.H"
#include "SgmlParser.H"
#include "Entity.H"
#include "Notation.H"
#include "Attribute.H"
#include "GrowableVectorD.H"
#include "Vector.H"
#include "InputContext.H"

#include <stdlib.h>
#include <string.h>

// This is based on ISO/IEC 13673, Intermediate Editor's Draft, 1994/8/29.

const char nl = '\n';

const RastPrintable RastEventHandler::printable;

RastPrintable::RastPrintable()
{
  static const char s[] =
    " !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~";
  int i;
  for (i = 0; i < sizeof(v_); i++)
    v_[i] = 0;
  for (i = 0; s[i] != '\0'; i++)
    v_[(unsigned char)s[i]] = 32 + i;
}

// inline
void RastEventHandler::flushLine(LineType type)
{
  if (lineLength_ > 0) {
    os() << char(type) << nl;
    lineLength_ = 0;
  }
}

RastSubdocState::RastSubdocState()
{
  init(0, 0);
}

RastSubdocState::RastSubdocState(SgmlParser *parser, RastEventHandler *rast)
{
  init(parser, rast);
}

void RastSubdocState::init(SgmlParser *parser, RastEventHandler *rast)
{
  parser_ = parser;
  hadActiveLpdOrDtd_ = 0;
  activeLinkTypes_.clear();
  hadDocumentElement_ = 0;
  linkProcess_.clear();
  linkProcess_.setHandler(rast);
  haveLinkProcess_ = 0;
  simpleLinkEvents_.clear();
  parseSubdocQueue_.clear();
  linkRuleQueue_.clear();
  for (int i = 0; i < nAttributeType; i++)
    attributeSortOrder_[i].clear();
}

void RastSubdocState::moveTo(RastSubdocState &to)
{
  to.parser_ = parser_;
  to.hadActiveLpdOrDtd_ = hadActiveLpdOrDtd_;
  activeLinkTypes_.moveTo(to.activeLinkTypes_);
  to.hadDocumentElement_ = hadDocumentElement_;
  linkProcess_.moveTo(to.linkProcess_);
  simpleLinkEvents_.moveTo(to.simpleLinkEvents_);
  parseSubdocQueue_.moveTo(to.parseSubdocQueue_);
  linkRuleQueue_.moveTo(to.linkRuleQueue_);
  for (int i = 0; i < nAttributeType; i++)
    attributeSortOrder_[i].moveTo(to.attributeSortOrder_[i]);
}

RastEventHandler::RastEventHandler(SgmlParser *parser, InputContext *ic)
: lineLength_(0),
  os_(0),
  errorFlags_(0),
  RastSubdocState(parser, this),
  ic_(ic)
{
  RastSubdocState::init(parser, this);
}

void RastEventHandler::message(MessageEvent *event)
{
  if (event->isError())
    errorFlags_ |= sgmlError;
  delete event;
}

void RastEventHandler::end()
{
  if (errorFlags_) {
    truncateOutput();
    os() << ((errorFlags_ & piError) 
	     ? "#RAST-PI-ERROR"
	     : "#ERROR")
         << nl;
  }
}

void RastEventHandler::truncateOutput()
{
  // This must be handled by derived classes to get conforming output.
}

void RastEventHandler::sgmlDecl(SgmlDeclEvent *event)
{
  rastParseSubdocYesString_ = event->sd().execToDoc("rast-parse-subdoc:yes");
  rastParseSubdocNoString_ = event->sd().execToDoc("rast-parse-subdoc:no");
  rastActiveLpdString_ = event->sd().execToDoc("rast-active-lpd:");
  rastLinkRuleString_ = event->sd().execToDoc("rast-link-rule:");
  delete event;
}

void RastEventHandler::startElement(StartElementEvent *event)
{
  flushLine(dataLine);
  if (!hadDocumentElement_) {
    if (activeLinkTypes_.length() > 0) {
      activeLinks();
      simpleLinkInfo();
    }
    hadDocumentElement_ = 1;
  }
  os() << '[' << event->name();
  Boolean hadNewline;
  if (event->attributes().length() > 0) {
    hadNewline = 1;
    os() << nl;
    attributeInfo(event->attributes(), dtdAttribute);
  }
  else
    hadNewline = 0;
  if (haveLinkProcess_) {
    const AttributeList *linkAttributes;
    const ResultElementSpec *resultElementSpec;
    linkProcess_.startElement(event->elementType(),
			      event->attributes(),
			      event->location(),
			      *this,
			      linkAttributes,
			      resultElementSpec);
    if (linkProcess_.nImpliedLinkRules() > 0) {
      if (!hadNewline) {
	os() << nl;
	hadNewline = 1;
      }
      os() << "#LINK-SET-INFO" << nl;
      impliedSourceLinkRules();
    }
    if (linkAttributes) {
      if (!hadNewline) {
	os() << nl;
	hadNewline = 1;
      }
      os() << "#LINK-RULE" << nl;
      attributeInfo(*linkAttributes, linkAttribute);
      if (linkProcess_.explicit()) {
	os() << "#RESULT=";
	if (resultElementSpec && resultElementSpec->elementType) {
	  os() << resultElementSpec->elementType->name() << nl;
	  attributeInfo(resultElementSpec->attributeList, resultAttribute);
	}
	else
	  os() << "#IMPLIED" << nl;
      }
    }
    else
      hadNewline = 0;
  }
  os() << ']' << nl;
  delete event;
}

void RastEventHandler::activeLinks()
{
  for (size_t i = 0; i < activeLinkTypes_.length(); i++) {
    os() << "#ACTIVE-LINK=" << activeLinkTypes_[i] << nl;
    Boolean found = 0;
    if (haveLinkProcess_ && linkProcess_.name() == activeLinkTypes_[i]) {
      found = 1;
      if (linkProcess_.nImpliedLinkRules() > 0) {
	os() << "#INITIAL" << nl;
	impliedSourceLinkRules();
      }
    }
    for (size_t j = 0; j < simpleLinkEvents_.length(); j++)
      if (simpleLinkEvents_[j]->name() == activeLinkTypes_[i]) {
	if (found) {
	  errorFlags_ |= piError;
   	  break;
	}
	found = 1;
      }
    if (!found)
      errorFlags_ |= piError;
    os() << "#END-ACTIVE-LINK" << nl;
  }
}

void RastEventHandler::simpleLinkInfo()
{
  for (size_t i = 0; i < activeLinkTypes_.length(); i++) {
    for (size_t j = 0; j < simpleLinkEvents_.length(); j++) {
      SimpleLinkEvent *e = simpleLinkEvents_[j].pointer();
      if (e->name() == activeLinkTypes_[i]) {
	os() << "#SIMPLE-LINK=" << e->name() << nl;
	if (e->attributes().length() > 0)
	  attributeInfo(e->attributes(), simpleAttribute);
	os() << "#END-SIMPLE-LINK" << nl;
	break;
      }
    }
  }
}

void RastEventHandler::impliedSourceLinkRules()
{
  size_t n = linkProcess_.nImpliedLinkRules();
  Vector<size_t> sortOrder(n);
  size_t i;
  for (i = 0; i < n; i++)
    sortOrder[i] = i;
  for (i = 1; i < n; i++) {
    size_t tem = sortOrder[i];
    const CString &name
      = linkProcess_.impliedLinkRule(tem).elementType->name();
    size_t j;
    for (j = i; j > 0; j--) {
      if (lexCmp(linkProcess_.impliedLinkRule(j - 1).elementType->name(),
		 name) <= 0)
	break;
      sortOrder[j] = sortOrder[j - 1];
    }
    sortOrder[j] = tem;
  }
  for (i = 0; i < n; i++) {
    const ResultElementSpec &result
      = linkProcess_.impliedLinkRule(sortOrder[i]);
    os() << '[' << result.elementType->name();
    if (result.attributeList.length() > 0) {
      os() << nl;
      attributeInfo(result.attributeList, resultAttribute);
    }
    os() << ']' << nl;
  }
}

void RastEventHandler::endElement(EndElementEvent *event)
{
  if (haveLinkProcess_)
    linkProcess_.endElement();
  flushLine(dataLine);
  os() << "[/" << event->name() << ']' << nl;
  delete event;
}

void RastEventHandler::data(DataEvent *event)
{
  lines(dataLine, event->data(), event->dataLength());
  delete event;
}

void RastEventHandler::pi(PiEvent *event)
{
  flushLine(dataLine);
  os() << "[?";
  size_t dataLength = event->dataLength();
  if (dataLength > 0) {
    const Char *data = event->data();
    if (dataLength >= 4
	&& memcmp(data,
		  rastParseSubdocYesString_.pointer(),
		  4*sizeof(Char)) == 0
	&& !interpretRastPi(data, dataLength))
      errorFlags_ |= piError;
    os() << nl;
    lines(dataLine, event->data(), dataLength);
    flushLine(dataLine);
  }
  os() << ']' << nl;
  delete event;
}

inline
Boolean equal(const Char *s1, size_t n1, const CString &s2)
{
  return (n1 == s2.length()
	  && (n1 == 0
	      || memcmp(s1, s2.pointer(), n1*sizeof(Char)) == 0));
}

// Is s2 a prefix of s1 of length n1?

inline
Boolean prefix(const Char *s1, size_t n1, const CString &s2)
{
  return (n1 >= s2.length()
	  && (n1 == 0
	      || memcmp(s1, s2.pointer(), s2.length()*sizeof(Char)) == 0));
}

Boolean RastEventHandler::interpretRastPi(const Char *data,
					  size_t dataLength)
{
  if (equal(data, dataLength, rastParseSubdocNoString_)) {
    queueParseSubdoc(0);
    return 1;
  }
  if (equal(data, dataLength, rastParseSubdocYesString_)) {
    queueParseSubdoc(1);
    return 1;
  }
  if (prefix(data, dataLength, rastActiveLpdString_)) {
    if (hadActiveLpdOrDtd_)
      return 1;
    hadActiveLpdOrDtd_ = 1;
    const Char *p = data + rastActiveLpdString_.length();
    size_t n = dataLength - rastActiveLpdString_.length();
    CString name;
    for (;;) {
      if (n == 0 || *p == ',') {
	if (name.length() == 0)
	  return 0;
	name.moveTo(activeLinkTypes_.grow());
	name.clear();
	if (n == 0)
	  break;
      }
      else
	name += *p;
      p++;
      n--;
    }
    for (size_t i = 0; i < activeLinkTypes_.length(); i++)
      parser_->activateLinkType(activeLinkTypes_[i]);
    return 1;
  }
  if (prefix(data, dataLength, rastLinkRuleString_)) {
    linkRuleQueue_.grow().set(data + rastLinkRuleString_.length(),
			      dataLength - rastLinkRuleString_.length());
    return 1;
  }
  return 0;
}

void RastEventHandler::sdataEntity(SdataEntityEvent *event)
{
  flushLine(dataLine);
  os() << "#SDATA-TEXT" << nl;
  lines(markupLine, event->data(), event->dataLength());
  flushLine(markupLine);
  os() << "#END-SDATA" << nl;
  delete event;
}

void RastEventHandler::externalDataEntity(ExternalDataEntityEvent *event)
{
  const ExternalDataEntity *entity = event->entity();
  if (!entity)
    return;
  flushLine(dataLine);
  os() << "[&" << entity->name() << nl;
  externalEntityInfo(entity, dtdAttribute);
  os() << ']' << nl;
  delete event;
}

void RastEventHandler::externalEntityInfo(const ExternalDataEntity *entity,
					  AttributeType attributeType)
{
  char c;
  switch (entity->dataType()) {
  case Entity::cdata:
    c = 'C';
    break;
  case Entity::sdata:
    c = 'S';
    break;
  case Entity::ndata:
    c = 'N';
    break;
  default:
    return;
  }
  os() << '#' << c << "DATA-EXTERNAL" << nl;
  externalIdInfo(entity->externalId());
  os() << "#NOTATION=" << entity->notation()->name() << nl;
  externalIdInfo(entity->notation()->externalId());
  attributeInfo(entity->attributes(),
		(attributeType == resultAttribute
		 ? resultAttribute
		 : dtdAttribute));
}

void RastEventHandler::subdocEntity(SubdocEntityEvent *event)
{
  const SubdocEntity *entity = event->entity();
  if (!entity)
    return;
  flushLine(dataLine);
  os() << "[&" << entity->name() << nl;
  subdocEntityInfo(entity, event->entityOrigin());
  os() << ']' << nl;
  delete event;
}

void RastEventHandler::subdocEntityInfo(const SubdocEntity *entity,
					const ResourcePointer<EntityOrigin> &entityOrigin)
{
  os() << "#SUBDOC" << nl;
  externalIdInfo(entity->externalId());
  if (parseNextSubdoc()) {
    // FIXME subdocuments in entity attributes shouldn't count against
    // SUBDOC quantity limit.
    os() << "#PARSED-SUBDOCUMENT" << nl;
    SgmlParser parser(*parser_, *entity, entityOrigin);
    RastSubdocState oldSubdocState;
    RastSubdocState::moveTo(oldSubdocState);
    RastSubdocState::init(&parser, this);
    parser.parseAll(*this);
    oldSubdocState.moveTo(*this);
  }
}

void RastEventHandler::queueParseSubdoc(Boolean parseSubdoc)
{
  parseSubdocQueue_.grow() = parseSubdoc;
}

Boolean RastEventHandler::parseNextSubdoc()
{
  if (parseSubdocQueue_.length() == 0)
    return 0;
  Boolean result = parseSubdocQueue_[0];
  if (parseSubdocQueue_.length() > 1) {
    for (size_t i = 1; i < parseSubdocQueue_.length(); i++)
      parseSubdocQueue_[i - 1] = parseSubdocQueue_[i];
  }
  parseSubdocQueue_.setLength(parseSubdocQueue_.length() - 1);
  return result;
}


void RastEventHandler::externalIdInfo(const ExternalId &id)
{
  const CString *systemId = id.systemIdString();
  const CString *publicId = id.publicIdString();
  if (publicId) {
    os() << "#PUBLIC" << nl;
    if (publicId->length() == 0)
      os() << "#EMPTY" << nl;
    else {
      lines(markupLine, publicId->pointer(), publicId->length());
      flushLine(markupLine);
    }
  }
  if (systemId || !publicId) {
    os() << "#SYSTEM" << nl;
    if (!systemId)
      os() << "#NONE" << nl;
    else if (systemId->length() == 0)
      os() << "#EMPTY" << nl;
    else {
      lines(markupLine, systemId->pointer(), systemId->length());
      flushLine(markupLine);
    }
  }
}

void RastEventHandler::lines(LineType type, const Char *p, size_t length)
{
  // This needs to be fast.
  while (length != 0) {
    if (printable(*p)) {
      size_t lim;
      switch (lineLength_) {
      case maxLineLength:
	os() << char(type) << nl;
	lineLength_ = 0;
	// fall through
      case 0:
	os() << char(type);
	lim = maxLineLength;
	break;
      default:
	lim = maxLineLength - lineLength_;
	break;
      }
      if (lim > length)
	lim = length;
      size_t n = lim;
      for (;;) {
	os().put(*p);
	p++;
	if (--n == 0)
	  break;
	if (!printable(*p)) {
	  lim -= n;
	  break;
	}
      }
      length -= lim;
      lineLength_ += lim;
    }
    else {
      // *p is an unprintable character print it
      flushLine(type);
      switch (*p) {
      case RS:
	os() << "#RS" << nl;
	break;
      case RE:
	os() << "#RE" << nl;
	break;
      case TAB:
	os() << "#TAB" << nl;
	break;
      default:
	os() << '#' << (unsigned long)*p << nl;
	break;
      }
      p++;
      length--;
    }
  }
}

int RastEventHandler::lexCmp(const CString &s1, const CString &s2)
{
  const Char *p1 = &s1[0];
  size_t n1 = s1.length();
  const Char *p2 = &s2[0];
  size_t n2 = s2.length();
  for (;;) {
    if (n1 == 0)
      return n2 == 0 ? 0 : -1;
    if (n2 == 0)
      return 1;
    if (*p1 != *p2) {
      // printable characters precede non-printable characters;
      // printable characters are in ASCII order
      // non-printable characters are in document character set order
      int a1 = printable(*p1);
      int a2 = printable(*p2);
      if (a1 == 0) {
	if (a2 == 0)
	  return *p1 < *p2 ? -1 : 1;
	else
	  return 1;
      }
      else if (a2 == 0)
	return -1;
      else
	return a1 - a2;
    }
    p1++;
    p2++;
    n1--;
    n2--;
  }
}

void RastEventHandler::attributeInfo(const AttributeList &attributes,
				     AttributeType attributeType)
{
  size_t length = attributes.length();
  if (length == 0)
    return;
  size_t defIndex = attributes.defIndex();
  if (defIndex >= attributeSortOrder_[attributeType].length())
    attributeSortOrder_[attributeType].setLength(defIndex + 1);
  Vector<size_t> &sortOrder(attributeSortOrder_[attributeType][defIndex]);
  if (sortOrder.length() == 0
      || attributeType == simpleAttribute) {
    sortOrder.init(length);
    for (size_t i = 0; i < length; i++)
      sortOrder[i] = i;
    // insertion sort
    for (i = 1; i < length; i++) {
      size_t tem = sortOrder[i];
      for (size_t j = i; j > 0; j--) {
	if (lexCmp(attributes.name(sortOrder[j - 1]),
		   attributes.name(tem)) <= 0)
	  break;
	sortOrder[j] = sortOrder[j - 1];
      }
      sortOrder[j] = tem;
    }
  }
  for (size_t j = 0; j < length; j++) {
    // Don't use sortOrder because attributeSortOrder_ may be grown
    // because of data attributes.
    size_t i = attributeSortOrder_[attributeType][defIndex][j];
    os() << attributes.name(i) << '=' << nl;
    const Text *text;
    const CString *string;
    const AttributeValue *value = attributes.value(i);
    if (value) {
      switch (value->info(text, string)) {
      case AttributeValue::implied:
	os() << "#IMPLIED" << nl;
	break;
      case AttributeValue::tokenized:
	lines(markupLine, string->pointer(), string->length());
	flushLine(markupLine);
	break;
      case AttributeValue::cdata:
	{
	  EsisTextIter iter(*text);
	  EsisTextIter::Type type;
	  const Char *p;
	  size_t length;
	  while (iter.next(type, p, length))
	    if (type == EsisTextIter::data)
	      lines(markupLine, p, length);
	    else {
	      flushLine(markupLine);
	      os() << "#SDATA-TEXT" << nl;
	      lines(markupLine, p, length);
	      flushLine(markupLine);
	      os() << "#END-SDATA" << nl;
	    }
	  flushLine(markupLine);
	}
	break;
      }
    }
    const AttributeSemantics *semantics = attributes.semantics(i);
    if (semantics) {
      ConstResourcePointer<Notation> notation
	= semantics->notation();
      if (!notation.isNull())
	externalIdInfo(notation->externalId());
      size_t nEntities = semantics->nEntities();
      for (size_t i = 0; i < nEntities; i++) {
	ConstResourcePointer<Entity> entity
	  = semantics->entity(i);
	if (!entity.isNull()) {
	  const ExternalDataEntity *externalDataEntity
	    = entity->asExternalDataEntity();
	  if (externalDataEntity)
	    externalEntityInfo(externalDataEntity,
			       (attributeType == resultAttribute
				? resultAttribute
				: dtdAttribute));
	  else {
	    const SubdocEntity *subdocEntity = entity->asSubdocEntity();
	    if (subdocEntity) {
	      ResourcePointer<EntityOrigin> entityOrigin
		= new EntityOrigin(entity,
				   ((TokenizedAttributeValue *)value)->tokenLocation(i),
				   0);
	      subdocEntityInfo(subdocEntity, entityOrigin);
	    }
	    else {
	      const InternalEntity *internalEntity = entity->asInternalEntity();
	      if (internalEntity)
		internalEntityInfo(internalEntity);
	    }
	  }
	}
	os() << "#END-ENTITY" << nl;
      }
    }
  }
}

void RastEventHandler::internalEntityInfo(const InternalEntity *entity)
{
  if (!entity)
    return;
  os() << '#'
       << char(entity->dataType() == Entity::cdata ? 'C' : 'S')
       << "DATA-INTERNAL" << nl;
  const CString &str(entity->string());
  lines(markupLine, str.pointer(), str.length());
  flushLine(markupLine);
}

void RastEventHandler::simpleLink(SimpleLinkEvent *event)
{
  simpleLinkEvents_.grow() =  event;
}

void RastEventHandler::complexLink(ComplexLinkEvent *event)
{
  linkProcess_.init(event->lpd(), event->activeLpds());
  haveLinkProcess_ = 1;
  delete event;
}

void RastEventHandler::uselink(UselinkEvent *event)
{
  linkProcess_.uselink(event->linkSet(),
		       event->restore(),
		       event->lpd().pointer());
  delete event;
}

void RastEventHandler::inputContextMessage(const char *source,
					   const Location &loc,
					   unsigned flags, int number,
					   const MessageArg **args,
					   int nArgs)
{
  unsigned severity = (flags & InputContext::severityMask);
  if (severity != InputContext::info && severity != InputContext::warning)
    errorFlags_ |= sgmlError;
  ic_->inputContextMessage(source, loc, flags, number, args, nArgs);
}

RastLinkProcess::RastLinkProcess()
: rast_(0)
{
}

void RastLinkProcess::setHandler(RastEventHandler *rast)
{
  rast_ = rast;
}

// Always return 1. 0 means not ready.

Boolean RastLinkProcess::selectLinkRule(const AttributeList **linkAttributes,
					size_t nLinkAttributes,
					size_t &selected)
{
  if (rast_->linkRuleQueue_.length() > 0) {
    CString pi;
    GrowableVectorD<CString> &queue = rast_->linkRuleQueue_;
    queue[0].moveTo(pi);
    for (size_t i = 1; i < queue.length(); i++)
      queue[i].moveTo(queue[i - 1]);
    queue.setLength(queue.length() - 1);
    if (!selectLinkRulePi(pi, linkAttributes, nLinkAttributes, selected)) {
      rast_->errorFlags_ |= RastEventHandler::piError;
      selected = 0;
    }
  }
  else {
    // FIXME catch exceptions and restore os_
    OutputCharStream *origOs = rast_->os_;
    StrOutputCharStream strOs;
    rast_->os_ = &strOs;
    selected = 0;
    rast_->attributeInfo(*linkAttributes[0], RastSubdocState::linkAttribute);
    CString firstStr;
    strOs.extractString(firstStr);
    for (size_t i = 1; i < nLinkAttributes; i++) {
      rast_->attributeInfo(*linkAttributes[i], RastSubdocState::linkAttribute);
      CString str;
      strOs.extractString(str);
      if (rast_->lexCmp(str, firstStr) < 0) {
	str.moveTo(firstStr);
	selected = i;
      }
    }
    rast_->os_ = origOs;
  }
  return 1;
}

// Return zero for failure (RAST-PI-ERROR).

Boolean RastLinkProcess::selectLinkRulePi(const CString &str,
					  const AttributeList **linkAttributes,
					  size_t nLinkAttributes,
					  size_t &selected)
{
  GrowableVectorD<CString> names;
  GrowableVectorD<CString> values;
  if (!parseNamesValues(str, names, values))
    return 0;
  Boolean haveSelection = 0;
  size_t i;
  for (i = 0; i < nLinkAttributes; i++) {
    const AttributeList &a = *linkAttributes[i];
    size_t n = a.length();
    Boolean matchPi = 1;
    // This finds a link rule whose attributes include all those
    // specified in the PI.
    for (size_t j = 0; j < names.length() && matchPi; j++) {
      Boolean matchNameValue = 0;
      for (size_t k = 0; k < n; k++) {
	if (names[j] == a.name(k)) {
	  const Text *textp;
	  const CString *strp;
	  switch (a.value(k)->info(textp, strp)) {
	  default:
	    break;
	  case AttributeValue::cdata:
	    // What if it contains SDATA entities?
	    matchNameValue = (values[j] == textp->string());
	    break;
	  case AttributeValue::tokenized:
	    matchNameValue = (values[j] == *strp);
	    break;
	  }
	  break;
	}
      }
      if (!matchNameValue)
	matchPi = 0;
    }
    if (matchPi) {
      if(haveSelection)
	return 0;
      haveSelection = 1;
      selected = i;
    }
  }
  return 1;
}

Boolean RastLinkProcess::parseNamesValues(const CString &str,
					  GrowableVectorD<CString> &names,
					  GrowableVectorD<CString> &values)
{
  // Should use SGML declaration to get these.
  const Char RE = '\r';
  const Char RS = '\n';
  const Char equals = '=';
  const Char lit = '"';
  const Char lita = '\'';
  const Char *p = str.pointer();
  size_t n = str.length();
  if (n-- == 0 || *p++ != RE || n-- == 0 || *p++ != RS)
    return 0;
  for (;;) {
    const Char *start = p;
    for (;;) {
      if (n == 0)
	return 0;
      if (*p == equals)
	break;
      p++, n--;
    }
    names.grow().set(start, p - start);
    p++, n--;			// skip equals
    if (n-- == 0)
      return 0;
    if (*p != lit && *p != lita)
      return 0;
    Char delim = *p++;
    start = p;
    // 13673 says we should interpret character references.
    // This is ridiculous.
    for (;;) {
      if (n == 0)
	return 0;
      if (*p == delim)
	break;
      p++, n--;
    }
    values.grow().set(start, p - start);
    p++, n--;			// skip delim
    if (n-- == 0 || *p++ != RE || n-- == 0 || *p++ != RS)
      return 0;
    if (n == 0)
      break;
  }
  return 1;
}


void RastLinkProcess::moveTo(RastLinkProcess &to)
{
  LinkProcess::moveTo(to);
  to.rast_ = rast_;
}
