// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#include "Catalog.H"
#include "CharsetInfo.H"
#include "MessageArg.H"
#include "ExternalId.H"

// FIXME canonicalize ISO owner identifier of publicId by changing hyphen to
// colon etc.

Catalog::Catalog()
: catalogNumber_(0)
{
}

void Catalog::endCatalog()
{
  catalogNumber_++;
}

Boolean Catalog::lookupEntity(const PublicId *publicId,
			      const CString &name, DeclType declType,
			      const SubstTable<Char> *substTable,
			      const CString *&systemId,
			      Location &specLoc) const
{
  const CatalogEntry *entry = 0;
  if (publicId)
    entry = publicIds_.lookup(publicId->string());
  if (!entry || entry->catalogNumber > 0) {
    const CatalogEntry *entityEntry = 0; 
    if (!substTable)
      entityEntry = names_[declType].lookup(name);
    else {
      HashTableIter<CString,CatalogEntry> iter(names_[declType]);
      const CString *key;
      const CatalogEntry *value;
      CString buffer;
      while (iter.next(key, value)) {
	buffer = *key;
	for (size_t i = 0; i < buffer.length(); i++)
	  substTable->subst(buffer[i]);
	if (buffer == name) {
	  entityEntry = value;
	  break;
	}
      }
    }
    // match for public id has priority over match for entity in same
    // catalog
    if (!entry || (entityEntry
		   && entityEntry->catalogNumber < entry->catalogNumber))
      entry = entityEntry;
  }
  if (!entry)
    return false;
  systemId = &entry->to;
  specLoc = entry->loc;
  return true;
}

void Catalog::addPublicId(CString &publicId, CString &systemId,
			  const Location &loc)
{
  CatalogEntry entry;
  entry.loc = loc;
  entry.catalogNumber = catalogNumber_;
  systemId.moveTo(entry.to);
  publicIds_.insert(publicId, entry, false);
}

void Catalog::addName(CString &name, DeclType declType, CString &systemId,
		      const Location &loc)
{
  CatalogEntry entry;
  entry.loc = loc;
  entry.catalogNumber = catalogNumber_;
  systemId.moveTo(entry.to);
  names_[declType].insert(name, entry, false);
}

CatalogParser::CatalogParser(const CharsetInfo &charset)
: categoryTable_(data),
  entityKey_(charset.execToDesc("ENTITY")),
  publicKey_(charset.execToDesc("PUBLIC")),
  doctypeKey_(charset.execToDesc("DOCTYPE")),
  linktypeKey_(charset.execToDesc("LINKTYPE"))
{
  static const char lcletters[] = "abcdefghijklmnopqrstuvwxyz";
  static const char ucletters[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
  // minimum data other than lcletter, ucletter
  static const char minChars[] = "0123456789-.'()+,/:=?";
  static const char sChars[] = " \n\r\t";
  categoryTable_.setChar(0, nul);
  const char *p;
  const char *q;
  for (p = lcletters, q = ucletters; *p; p++, q++) {
    Char lc = charset.execToDesc(*p);
    Char uc = charset.execToDesc(*q);
    substTable_.addSubst(lc, uc);
    categoryTable_.setChar(lc, min);
    categoryTable_.setChar(uc, min);
  }
  for (p = sChars; *p; p++)
    categoryTable_.setChar(charset.execToDesc(*p), s);
  for (p = minChars; *p; p++)
    categoryTable_.setChar(charset.execToDesc(*p), min);
  categoryTable_.setChar(charset.execToDesc('\''), lit);
  categoryTable_.setChar(charset.execToDesc('"'), lit);
  minus_ = charset.execToDesc('-');
  categoryTable_.setChar(minus_, minus);
  tab_ = charset.execToDesc('\t');
  re_ = charset.execToDesc('\r');
  rs_ = charset.execToDesc('\n');
  space_ = charset.execToDesc(' ');
  categoryTable_.setEe(eof);
}

const char *CatalogParser::messageText(int n)
{
  static const char *const text[] = {
    "name expected",
    "literal expected",
    "name or literal expected",
    "nul character",
    "not a minimum data character",
    "end of entity in comment",
    "end of entity in literal",
  };
  return n < 0 || n >= sizeof(text)/sizeof(text[0]) ? 0 : text[n];
}

void CatalogParser::parseCatalog(Catalog *catalog,
				 InputSource *in,
				 InputContext &ic)
{
  catalog_ = catalog;
  in_ = in;
  ic_ = &ic;
  Boolean recovering = false;
  for (;;) {
    Param parm = parseParam();
    if (parm == nameParam) {
      upcase(param_);
      recovering = false;
      if (param_ == publicKey_)
	parsePublic();
      else if (param_ == entityKey_)
	parseNameMap(Catalog::entityDecl);
      else if (param_ == doctypeKey_)
	parseNameMap(Catalog::doctypeDecl);
      else if (param_ == linktypeKey_)
	parseNameMap(Catalog::linktypeDecl);
      else
	recovering = true;
    }
    else if (parm == eofParam)
      break;
    else if (!recovering) {
      recovering = true;
      error(nameExpectedError);
    }
  }
  catalog->endCatalog();
}

void CatalogParser::parsePublic()
{
  if (parseParam(minimumLiteral) != literalParam) {
    error(literalExpectedError);
    return;
  }
  CString publicId;
  param_.moveTo(publicId);
  if (!parseArg())
    return;
  catalog_->addPublicId(publicId, param_, paramLoc_);
}

void CatalogParser::parseNameMap(Catalog::DeclType declType)
{
  if (!parseArg())
    return;
  CString name;
  param_.moveTo(name);
  if (!parseArg())
    return;
  catalog_->addName(name, declType, param_, paramLoc_);
}

Boolean CatalogParser::parseArg()
{
  Param parm = parseParam();
  if (parm != nameParam && parm != literalParam) {
    error(nameOrLiteralExpectedError);
    return false;
  }
  return true;
}

CatalogParser::Param CatalogParser::parseParam(unsigned flags)
{
  for (;;) {
    Xchar c = get();
    switch (categoryTable_[c]) {
    case eof:
      return eofParam;
    case lit:
      parseLiteral(c, flags);
      return literalParam;
    case s:
      break;
    case nul:
      error(nulError);
      break;
    case minus:
      c = get();
      if (c == minus_) {
	skipComment();
	break;
      }
      unget();
      // fall through
    default:
      parseName();
      return nameParam;
    }
  }
}

void CatalogParser::skipComment()
{
  for (;;) {
    Xchar c = get();
    if (c == minus_) {
      c = get();
      if (c == minus_)
	break;
    }
    if (c == InputSource::eE) {
      error(eofInCommentError);
      break;
    }
  }
}

void CatalogParser::parseLiteral(Char delim, unsigned flags)
{
  paramLoc_ = in_->currentLocation();
  enum { no, yesBegin, yesMiddle } skipping = yesBegin;
  param_.clear();
  for (;;) {
    Xchar c = get();
    if (c == InputSource::eE) {
      error(eofInLiteralError);
      break;
    }
    if (Char(c) == delim)
      break;
    if (flags & minimumLiteral) {
      if (!isMinimumData(c))
	error(minimumDataError);
      if (c == rs_)
	;
      else if (c == space_ || c == re_) {
	if (skipping == no) {
	  param_ += space_;
	  skipping = yesMiddle;
	}
      }
      else {
	skipping = no;
	param_ += Char(c);
      }
    }
    else
      param_ += Char(c);
  }
  if (skipping == yesMiddle)
    param_.setLength(param_.length() - 1);
}

void CatalogParser::parseName()
{
  paramLoc_ = in_->currentLocation();
  for (size_t length = 1;; length++) {
    Xchar c = in_->tokenChar(inputContext());
    int cat = categoryTable_[c];
    if (cat == eof || cat == s)
      break;
    // FIXME maybe check for LIT or LITA
    if (cat == nul)
      error(nulError);
  }
  in_->endToken(length);
  param_.set(in_->currentTokenStart(), in_->currentTokenLength());
}

void CatalogParser::upcase(CString &str)
{
  for (size_t i = 0; i < str.length(); i++)
    substTable_.subst(str[i]);
}

void CatalogParser::error(ErrorCode ec)
{
  message(Portable::messageSource, InputContext::error, errorNumberBase + ec);
}
 
void CatalogParser::error(ErrorCode ec, const MessageArg &arg)
{
  message(Portable::messageSource, InputContext::error, errorNumberBase + ec,
	  arg);
}

void CatalogParser::inputContextMessage(const char *source,
					const Location &loc,
					unsigned flags, int number,
					const MessageArg **args,
					int nArgs)
{
  ic_->inputContextMessage(source,
			   loc.origin().isNull() 
			   ? in_->currentLocation()
			   : loc, flags, number, args, nArgs);
}
