// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#ifndef Catalog_INCLUDED
#define Catalog_INCLUDED 1

#include "InputContext.H"
#include "Portable.H"
#include "CString.H"
#include "types.H"
#include "GrowableVectorD.H"
#include "HashTable.H"
#include "InputSource.H"
#include "Boolean.H"
#include "SubstTable.H"

class CharsetInfo;
class MessageArg;
class PublicId;

struct CatalogEntry {
  CString to;
  Location loc;
  size_t catalogNumber;
  void destruct() { this->CatalogEntry::~CatalogEntry(); }
};

class Catalog {
public:
  enum DeclType { entityDecl, doctypeDecl, linktypeDecl };
  enum { nDeclType = 3 };
  Catalog();
  Catalog(const Catalog &);	// undefined
  void operator=(const Catalog &); // undefined
  Boolean lookupEntity(const PublicId *,
		       const CString &name, DeclType,
		       const SubstTable<Char> *substTable,
		       const CString *&systemId,
		       Location &specLoc) const;
  void addPublicId(CString &publicID, CString &systemId, const Location &);
  void addName(CString &name, DeclType, CString &systemId, const Location &);
  void endCatalog();
private:
  HashTable<CString,CatalogEntry> publicIds_;
  HashTable<CString,CatalogEntry> names_[nDeclType];
  size_t catalogNumber_;
};

class CatalogParser : private InputContext {
public:
  CatalogParser(const CharsetInfo &);
  void parseCatalog(Catalog *, InputSource *, InputContext &ic);
  enum { errorNumberBase = 100 };
  static const char *messageText(int);
private:
  enum ErrorCode {
    nameExpectedError,
    literalExpectedError,
    nameOrLiteralExpectedError,
    nulError,
    minimumDataError,
    eofInCommentError,
    eofInLiteralError
  };
public:
  // Since it's a return type, it has to be public to keep some 
  // (broken) compilers happy.
  enum Param {
    eofParam,
    literalParam,
    nameParam,
    percentParam
  };
private:
  enum {
    data,
    eof,
    nul,
    lit,
    minus,
    s,
    min				// other minimum data characters
  };
  enum { minimumLiteral = 01 };

  InputContext &inputContext() { return *this; }
  void inputContextMessage(const char *source,
			   const Location &,
			   unsigned flags, int number,
			   const MessageArg **args,
			   int nArgs);
  void parsePublic();
  void parseNameMap(Catalog::DeclType declType);
  Param parseParam(unsigned flags = 0);
  Boolean parseArg();
  void parseLiteral(Char delim, unsigned flags);
  void parseName();
  void skipComment();
  void upcase(CString &);
  void error(ErrorCode);
  void error(ErrorCode, const MessageArg &);
  Boolean isMinimumData(Xchar c) {
    int cat = categoryTable_[c];
    return (cat == min || (cat == s && c != tab_) || cat == minus);
  }
  Xchar get() { return in_->get(inputContext()); }
  void unget() { in_->ungetToken(); }
  InputContext *ic_;
  InputSource *in_;
  Catalog *catalog_;
  CString param_;
  Location paramLoc_;
  Char minus_;
  Char tab_;
  Char rs_;
  Char re_;
  Char space_;
  CString publicKey_;
  CString entityKey_;
  CString doctypeKey_;
  CString linktypeKey_;
  XcharMap<unsigned char> categoryTable_;
  SubstTable<Char> substTable_;
};

#endif /* not Catalog_INCLUDED */
