// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#include "Catalog.H"
#include "Portable.H"
#include "Location.H"
#include "String.H"
#include "rtti.H"
#include "macros.H"
#include "CString.H"
#include "Entity.H"
#include "ExternalId.H"
#include "InputContext.H"
#include "String.H"
#include "Vector.H"
#include "setAll.H"
#include "EntityManager.H"
#include "Boolean.H"
#include "types.H"
#include "OffsetOrderedList.H"
#include "types.H"
#include "InputSource.H"
#include "String.H"
#include "StorageManager.H"
#include "GrowableVectorD.H"
#include "DOwner.H"
#include "MessageArg.H"
#include "CharsetInfo.H"
#include "CodingSystem.H"
#include "StorageObjectInfo.H"
#include "RegisteredCodingSystem.H"

#include <stddef.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

extern "C" {
  void *memmove(void *, const void *, size_t);
}

class PortableInputSource;

class PortableEntityManager : public EntityManager {
public:
  PortableEntityManager(StorageManager *defaultStorageManager,
			const InputCodingSystem *defaultCodingSystem);
  PortableEntityManager(const PortableEntityManager &);	// undefined
  void operator=(const PortableEntityManager &); // undefined
  void registerStorageManager(StorageManager *);
  void registerCodingSystem(const char *, const InputCodingSystem *);
  InputSource *openExternal(const ExternalEntity &,
			    EntityOrigin *,
			    const CharsetInfo &,
			    const SubstTable<Char> *,
			    const CString &,
			    Boolean mayRewind,
			    InputContext &);
  InputSource *openDocumentEntity(const CString *sysids,
				  size_t nSysids,
				  const UnivCharsetDesc &charsetDesc,
				  InputContext &ic);
  StorageManager *lookupStorageType(const CString &, const CharsetInfo &);
  StorageManager *lookupStorageType(const char *);
  
  const InputCodingSystem *lookupCodingSystem(const CString &,
					      const CharsetInfo &,
					      const char *&);
  const InputCodingSystem *lookupCodingSystem(const char *);
  void addCatalogSysid(const CString &sysid, Boolean mustExist);
  Boolean getExternalEntityStorage(const ExternalEntity &entity,
				   const CharsetInfo &docCharset,
				   const SubstTable<Char> *subst,
				   const CString &pero,
				   InputContext &ic,
				   GrowableVectorD<StorageObjectSpec> &specs);
private:
  static const StorageObjectSpec *defStorageObject(const Location &);
  void loadCatalog(const CharsetInfo &, InputContext &);
  Boolean parseSystemId(const Char *,
			size_t,
			const CharsetInfo &docCharset,
			InputContext &ic,
			GrowableVectorD<StorageObjectSpec> &specs);
  Boolean parseConcat(const Char *s,
		      size_t n,
		      const CharsetInfo &docCharset,
		      InputContext &ic,
		      GrowableVectorD<StorageObjectSpec> &specs);
  PortableInputSource *
    makeInputSource(GrowableVectorD<StorageObjectSpec> &specs,
		    EntityOrigin *origin,
		    Boolean mayRewind,
		    InputContext &ic);
  Boolean convertId(CString &id,
		    const CharsetInfo &fromCharset,
		    const CharsetInfo &toCharset,
		    InputContext &ic);
  static Boolean matchKey(const CString &type, const char *s,
			  const CharsetInfo &docCharset);
  Boolean loadedCatalog_;
  Catalog catalog_;
  GrowableVectorD<CString> catalogSysids_;
  GrowableVector<PackedBoolean> catalogSysidMustExist_;
  GrowableVector<StorageManager *> storageManagers_;
  GrowableVector<RegisteredCodingSystem> codingSystems_;
  StorageManager *defaultStorageManager_;
  const InputCodingSystem *defaultCodingSystem_;
};

class PortableExternalInfo : public ExternalInfo {
  RTTI_CLASS
public:
  PortableExternalInfo(GrowableVectorD<StorageObjectSpec> &specs);
  const StorageObjectSpec &spec(size_t i) const;
  size_t nSpecs() const;
  void noteRS(Offset);
  void noteFileEnd(Offset);
  void setDecoder(size_t i, Decoder *);
  Boolean convertOffset(Offset,
			const StorageObjectSpec *&,
			unsigned long &lineno,
			unsigned long &colno,
			unsigned long &byteIndex) const;
private:
  GrowableVectorD<StorageObjectSpec> specs_;
  Vector<size_t> fileIndex_;
  Vector<Offset> fileEndOffset_;
  // the number of RSs preceding line 1 of file i,
  // or -1 if this hasn't been computed yet.
  Vector<size_t> fileLine1RS_;
  // does the file start with an RS_
  Vector<size_t> fileStartsWithRS_;
  Vector<DOwner<Decoder> > decoders_;
  size_t currentFile_;
  // list of inserted RSs
  OffsetOrderedList rsList_;
};

class PortableInputSource : public InputSource {
public:
  PortableInputSource(GrowableVectorD<StorageObjectSpec> &specs,
		      Vector<StorageObjectInfo> &sov,
		      EntityOrigin *origin,
		      Boolean mayRewind);
  void pushCharRef(Char, const NamedCharRef &);
  ~PortableInputSource();
private:
  Xchar fill(InputContext &);
  Boolean rewind(InputContext &);
  void willNotRewind();

  void init();
  void noteRS();
  void reallocateBuffer(size_t size);
  void insertChar(Char);
  static const Char *nextNewline(const Char *start, const Char *end);

  PortableExternalInfo *info_;
  Char *buf_;
  const Char *bufLim_;
  Offset bufLimOffset_;
  size_t bufSize_;
  size_t readSize_;
  Vector<StorageObjectInfo> sov_;
  StorageObject *so_;
  size_t soIndex_;
  Boolean insertRS_;
  Decoder *decoder_;
  int minBytesPerChar_;
  const char *leftOver_;
  size_t nLeftOver_;
  Boolean mayRewind_;
};

const Char RS = '\n';
const Char RE = '\r';
const char lineEnd = '\n';

const char Portable::messageSource[] = "Portable";

const int cannotGenerateSystemIdMessage = 1;
const int emptySystemIdMessage = 2;

Portable::Portable(StorageManager *defaultStorageManager,
		   const InputCodingSystem *defaultCodingSystem)
{
  em_ = new PortableEntityManager(defaultStorageManager,
				  defaultCodingSystem);
}

Portable::~Portable()
{
  delete em_;
}

EntityManager *Portable::entityManager()
{
  return em_;
}

InputSource *Portable::openDocumentEntity(const CString *sysids,
					  size_t nSysids,
					  const UnivCharsetDesc &charset,
					  InputContext &ic)
{
  return em_->openDocumentEntity(sysids, nSysids, charset, ic);
}

void Portable::registerStorageManager(StorageManager *sm)
{
  em_->registerStorageManager(sm);
}

void Portable::registerCodingSystem(const char *s,
				    const InputCodingSystem *ics)
{
  em_->registerCodingSystem(s, ics);
}


void Portable::addCatalogSysid(const CString &str, Boolean mustExist)
{
  em_->addCatalogSysid(str, mustExist);
}

Boolean Portable::getExternalEntityStorage(const ExternalEntity &entity,
					   const CharsetInfo &docCharset,
					   const SubstTable<Char> *subst,
					   const CString &pero,
					   InputContext &ic,
					   GrowableVectorD<StorageObjectSpec> &specs)
{
  return em_->getExternalEntityStorage(entity, docCharset, subst, pero,
				       ic, specs);
}

Boolean Portable::externalize(const ExternalInfo *info,
			      Offset off,
			      const StorageObjectSpec *&sos,
			      unsigned long &lineno,
			      unsigned long &colno,
			      unsigned long &byteIndex)
{
  if (!info)
    return false;
  const PortableExternalInfo *p = ptr_cast(PortableExternalInfo, info);
  if (!p)
    return false;
  return p->convertOffset(off, sos, lineno, colno, byteIndex);
}

const char *Portable::messageText(int n)
{
  if (n >= CatalogParser::errorNumberBase)
    return CatalogParser::messageText(n - CatalogParser::errorNumberBase);
  static const char *const text[] = {
    "cannot generate system identifier for entity %1",
    "empty system identifier"
  };
  return n - 1 < 0 || n - 1 >= sizeof(text)/sizeof(text[0]) ? 0 : text[n - 1];
}

PortableEntityManager::PortableEntityManager(StorageManager *defaultStorageManager,
					     const InputCodingSystem *defaultCodingSystem)
: defaultStorageManager_(defaultStorageManager),
  defaultCodingSystem_(defaultCodingSystem),
  loadedCatalog_(false)
{
}

InputSource *
PortableEntityManager::openDocumentEntity(const CString *sysids,
					  size_t nSysids,
					  const UnivCharsetDesc &charsetDesc,
					  InputContext &ic)
{
  CharsetInfo charsetInfo(charsetDesc);
  GrowableVectorD<StorageObjectSpec> specs;
  for (int i = 0; i < nSysids; i++)
    if (!parseSystemId(sysids[i].pointer(),
		       sysids[i].length(),
		       charsetInfo,
		       ic,
		       specs))
      return 0;
  return makeInputSource(specs, new EntityOrigin(0, Location(), 0), 1, ic);
}

PortableInputSource *
PortableEntityManager::makeInputSource(GrowableVectorD<StorageObjectSpec> &specs,
				       EntityOrigin *origin,
				       Boolean mayRewind,
				       InputContext &ic)
{
  Vector<StorageObjectInfo> info;
  info.init(specs.length());

  for (size_t i = 0; i < specs.length(); i++) {
    StorageManager *sm = lookupStorageType(specs[i].storageType);

    info[i].storageObject = sm->makeStorageObject(specs[i].id, ic);
    info[i].codingSystem = lookupCodingSystem(specs[i].codingSystem);
    if (!info[i].codingSystem)
      info[i].codingSystem = defaultCodingSystem_;
  }
  return new PortableInputSource(specs, info, origin, mayRewind);
}


InputSource *
PortableEntityManager::openExternal(const ExternalEntity &entity,
				    EntityOrigin *origin,
				    const CharsetInfo &docCharset,
				    const SubstTable<Char> *subst,
				    const CString &pero,
				    Boolean mayRewind,
				    InputContext &ic)
{
  GrowableVectorD<StorageObjectSpec> specs;
  
  if (!getExternalEntityStorage(entity, docCharset, subst, pero, ic, specs))
    return 0;
  return makeInputSource(specs, origin, mayRewind, ic);
}

Boolean
PortableEntityManager::getExternalEntityStorage(const ExternalEntity &entity,
						const CharsetInfo &docCharset,
						const SubstTable<Char> *subst,
						const CString &pero,
						InputContext &ic,
						GrowableVectorD<StorageObjectSpec> &specs)
{
  const ExternalId &id = entity.externalId();
  const CString *str = id.systemIdString();
  Location defLocation;
  if (str)
    defLocation = entity.defLocation();
  else {
    if (!loadedCatalog_)
      loadCatalog(docCharset, ic);
    Catalog::DeclType declType;
    Boolean usePname = false;
    CString pname;
    switch (entity.declType()) {
    case Entity::generalEntity:
      declType = Catalog::entityDecl;
      break;
    case Entity::parameterEntity:
      usePname = true;
      pname = pero;
      pname += entity.name();
      declType = Catalog::entityDecl;
      break;
    case Entity::doctype:
      declType = Catalog::doctypeDecl;
      break;
    case Entity::linktype:
      declType = Catalog::linktypeDecl;
      break;
    default:
      CANNOT_HAPPEN();
    }
    if (!catalog_.lookupEntity(id.publicId(),
			       usePname ? pname : entity.name(),
			       declType, subst,
			       str, defLocation)) {
      // FIXME perhaps use name (and subst) to construct filename.
      ic.message(Portable::messageSource, InputContext::error,
		 cannotGenerateSystemIdMessage,
		 StringMessageArg(entity.name()));
      return false;
    }
  }
  if (!parseSystemId(str->pointer(), str->length(), docCharset, ic, specs))
    return false;
  const StorageObjectSpec *defSpec = defStorageObject(defLocation);
  if (defSpec) {
    StorageManager *sm = lookupStorageType(defSpec->storageType);
    for (size_t i = 0; i < specs.length(); i++) {
      if (specs[i].storageType = defSpec->storageType)
	sm->mergeId(specs[i].id, defSpec->id);
      if (specs[i].codingSystem == 0)
	specs[i].codingSystem = defSpec->codingSystem;
    }
  }

  return true;
}

const StorageObjectSpec *
PortableEntityManager::defStorageObject(const Location &location)
{
  Offset off;
  const ExternalInfo *info;
  Location loc(location);
  for (;;) {
    if (loc.origin().isNull())
      return 0;
    const EntityOrigin *entityOrigin = loc.origin()->asEntityOrigin();
    if (entityOrigin) {
      NamedCharRef ref;
      Index index = loc.index();
      while (entityOrigin->lookupIndex(index, off, ref)
	     == EntityOrigin::replacementCharIndex)
	index = ref.refStartIndex();
      info = entityOrigin->externalInfo();
      if (info)
	break;
      const InternalEntity *internal = entityOrigin->entity()->asInternalEntity();
      if (!internal)
	return 0;
      loc = internal->text().charLocation(off);
    }
    else
      loc = loc.origin()->parent();
  }
  const StorageObjectSpec *sos;
  unsigned long dummy;
  if (!Portable::externalize(info, off, sos, dummy, dummy, dummy))
    return 0;
  return sos;
}


// Append onto specs

Boolean PortableEntityManager::parseSystemId(const Char *s,
					     size_t n,
					     const CharsetInfo &docCharset,
					     InputContext &ic,
					     GrowableVectorD<StorageObjectSpec> &specs)
{
  Char colon = docCharset.execToDesc(':');
  for (size_t i = 0; i < n; i++)
    if (s[i] == colon) {
      CString key(s, i);
      if (matchKey(key, "CONCAT", docCharset))
	return parseConcat(s + i + 1, n - i - 1, docCharset, ic, specs);
      StorageManager *sm = lookupStorageType(key, docCharset);
      if (sm) {
	StorageObjectSpec &spec = specs.grow();
	spec.storageType = sm->type();
	spec.id.set(s + i + 1, n - i - 1);
	if (!convertId(spec.id, docCharset, sm->idCharset(), ic))
	  return 0;
	return 1;
      }
      const char *codingSystemName;
      const InputCodingSystem *codingSystem
	= lookupCodingSystem(key, docCharset, codingSystemName);
      if (codingSystem) {
	size_t startLength = specs.length();
	if (!parseSystemId(s + i + 1, n - i - 1, docCharset, ic, specs))
	  return 0;
	for (size_t j = startLength; j < specs.length(); j++)
	  if (!specs[j].codingSystem)
	    specs[j].codingSystem = codingSystemName;
	return 1;
      }
      break;
    }
  StorageObjectSpec &spec = specs.grow();
  spec.id.set(s, n);
  if (!convertId(spec.id, docCharset, defaultStorageManager_->idCharset(), ic))
    return 0;
  spec.storageType = defaultStorageManager_->type();
  return 1;
}

Boolean PortableEntityManager::convertId(CString &id,
					 const CharsetInfo &fromCharset,
					 const CharsetInfo &toCharset,
					 InputContext &ic)
{
  for (size_t i = 0; i < id.length(); i++) {
    Char &c = id[i];
    UnivChar univ;
    WideChar wide;
    ISet<WideChar> wideSet;
    if (!fromCharset.descToUniv(c, univ)
	|| toCharset.univToDesc(univ, wide, wideSet) != 1
	|| wide > Char(-1))
      return 0;			// FIXME give error
    c = Char(wide);
  }
  return 1;
}


Boolean PortableEntityManager::parseConcat(const Char *s,
					   size_t n,
					   const CharsetInfo &docCharset,
					   InputContext &ic,
					   GrowableVectorD<StorageObjectSpec> &specs)
{
  if (n == 0) {
    // FIXME should use location of system id for error.
    ic.message(Portable::messageSource, InputContext::error,
	       emptySystemIdMessage);
    return false;
  }
  Char plus = docCharset.execToDesc('+');
  size_t start = 0;
  for (size_t i = 0; i < n; i++)
    if (s[i] == plus) {
      if (!parseSystemId(s + start, i - start, docCharset, ic, specs))
	return 0;
      start = i + 1;
    }
  return parseSystemId(s + start, n - start, docCharset, ic, specs);
}

void PortableEntityManager::addCatalogSysid(const CString &str,
					    Boolean mustExist)
{
  catalogSysids_.grow() = str;
  catalogSysidMustExist_.grow() = mustExist;
}

void PortableEntityManager::loadCatalog(const CharsetInfo &docCharset,
					InputContext &ic)
{
  class NullInputContext : public InputContext {
    void inputContextMessage(const char *source,
			     const Location &,
			     unsigned flags, int number,
			     const MessageArg **args,
			     int nArgs) {}
  };

  NullInputContext nullInputContext;
  loadedCatalog_ = true;
  CatalogParser parser(docCharset);
  // There are unfortunately semantics associated with the way
  // the catalog divides into files, so we have to parse
  // each storage object separately.
  for (int i = 0; i < catalogSysids_.length(); i++) {
    GrowableVectorD<StorageObjectSpec> specs;
    InputContext *useIc = catalogSysidMustExist_[i] ? &ic : &nullInputContext;
    if (parseSystemId(catalogSysids_[i].pointer(),
		      catalogSysids_[i].length(),
		      docCharset,
		      *useIc,
		      specs)) {
      for (size_t j = 0; j < specs.length(); j++) {
	if (!catalogSysidMustExist_[i]) {
	  StorageManager *sm = lookupStorageType(specs[j].storageType);
	  StorageObject *obj = sm->makeStorageObject(specs[j].id,
						     nullInputContext);
	  size_t dummy;
	  if (!obj || !obj->open(nullInputContext, 0, dummy))
	    continue;
	}
	GrowableVectorD<StorageObjectSpec> thisSpec(1);
	thisSpec[0] = specs[j];
	InputSource *in = makeInputSource(thisSpec,
					  new EntityOrigin(0, Location(), 0),
					  0,
					  ic);
	parser.parseCatalog(&catalog_, in, ic);
	delete in;
      }
    }
  }
}

StorageManager *
PortableEntityManager::lookupStorageType(const CString &type,
					 const CharsetInfo &docCharset)
{
  if (type.length() == 0)
    return 0;
  if (matchKey(type, defaultStorageManager_->type(), docCharset))
    return defaultStorageManager_;
  for (int i = 0; i < storageManagers_.length(); i++)
    if (matchKey(type, storageManagers_[i]->type(), docCharset))
      return storageManagers_[i];
  return 0;
}

StorageManager *PortableEntityManager::lookupStorageType(const char *type)
{
  if (type == defaultStorageManager_->type())
    return defaultStorageManager_;
  for (int i = 0; i < storageManagers_.length(); i++)
    if (type == storageManagers_[i]->type())
      return storageManagers_[i];
  return 0;
}

const InputCodingSystem *
PortableEntityManager::lookupCodingSystem(const CString &type,
					  const CharsetInfo &docCharset,
					  const char *&name)
{
  for (size_t i = 0; i < codingSystems_.length(); i++)
    if (matchKey(type, codingSystems_[i].name, docCharset)) {
      name = codingSystems_[i].name;
      return codingSystems_[i].ics;
    }
  return 0;
}

const InputCodingSystem *
PortableEntityManager::lookupCodingSystem(const char *s)
{
  for (size_t i = 0; i < codingSystems_.length(); i++)
    if (s == codingSystems_[i].name)
      return codingSystems_[i].ics;
  return 0;
}

Boolean
PortableEntityManager::matchKey(const CString &type,
				const char *s,
				const CharsetInfo &docCharset)
{
  if (strlen(s) != type.length())
    return false;
  for (int i = 0; i < type.length(); i++)
    if (docCharset.execToDesc(toupper(s[i])) != type[i]
	&& docCharset.execToDesc(tolower(s[i])) != type[i])
      return false;
  return true;
}

void PortableEntityManager::registerStorageManager(StorageManager *sm)
{
  storageManagers_.grow() = sm;
}

void PortableEntityManager::registerCodingSystem(const char *name,
						 const InputCodingSystem *ics)
{
  RegisteredCodingSystem &rcs = codingSystems_.grow();
  rcs.name = name;
  rcs.ics = ics;
}

PortableInputSource::PortableInputSource(GrowableVectorD<StorageObjectSpec> &specs,
					 Vector<StorageObjectInfo> &sov,
					 EntityOrigin *origin,
					 Boolean mayRewind)
: InputSource(origin, 0, 0),
  mayRewind_(mayRewind)
{
  init();
  sov.moveTo(sov_);
  info_ = new PortableExternalInfo(specs);
  origin->setExternalInfo(info_);
}

void PortableInputSource::init()
{
  so_ = 0;
  buf_ = 0;
  bufSize_ = 0;
  bufLim_ = 0;
  bufLimOffset_ = 0;
  insertRS_ = true;
  soIndex_ = 0;
  leftOver_ = 0;
  nLeftOver_ = 0;  
}

PortableInputSource::~PortableInputSource()
{
  if (buf_)
    delete [] buf_;
}

Boolean PortableInputSource::rewind(InputContext &ic)
{
  reset(0, 0);
  if (buf_)
    delete [] buf_;
  // reset makes a new EntityOrigin
  GrowableVectorD<StorageObjectSpec> specs(info_->nSpecs());
  size_t i;
  for (i = 0; i < specs.length(); i++)
    specs[i] = info_->spec(i);
  info_ = new PortableExternalInfo(specs);
  entityOrigin()->setExternalInfo(info_);
  so_ = 0;
  for (i = 0; i < soIndex_; i++) {
    if (sov_[i].storageObject && !sov_[i].storageObject->rewind(ic))
      return 0;
  }
  init();
  return 1;
}

void PortableInputSource::willNotRewind()
{
  for (size_t i = 0; i < soIndex_; i++)
    if (sov_[i].storageObject)
      sov_[i].storageObject->willNotRewind();
  mayRewind_ = 0;
}

// Round up N so that it is a power of TO.
// TO must be a power of 2.

inline
size_t roundUp(size_t n, size_t to)
{
  return (n + (to - 1)) & ~(to - 1);
}

inline
void PortableInputSource::noteRS()
{
  info_->noteRS(bufLimOffset_ - (bufLim_ - cur()));
}

Xchar PortableInputSource::fill(InputContext &ic)
{
  ASSERT(cur() == end());
  while (end() >= bufLim_) {
    // need more data
    while (so_ == 0) {
      if (soIndex_ >= sov_.length())
	return eE;
      if (soIndex_ > 0)
	info_->noteFileEnd(bufLimOffset_ - (bufLim_ - end()));
      so_ = sov_[soIndex_].storageObject.pointer();
      if (so_) {
	decoder_ = sov_[soIndex_].codingSystem->makeDecoder();
	info_->setDecoder(soIndex_, decoder_);
	minBytesPerChar_ = sov_[soIndex_].codingSystem->minBytesPerChar();
	soIndex_++;
	// so_ might be NULL
	if (so_->open(ic, mayRewind_, readSize_))
	  break;
	so_ = 0;
      }
      else
	soIndex_++;
    }

    size_t keepSize = end() - start();
    const size_t align = sizeof(int)/sizeof(Char);
    size_t readSizeChars = (readSize_ + (sizeof(Char) - 1))/sizeof(Char);
    readSizeChars = roundUp(readSizeChars, align);
    size_t neededSize;		// in Chars
    size_t startOffset;
    // compute neededSize and readSize
    if (nLeftOver_ == 0 && minBytesPerChar_ >= sizeof(Char)) {
      // In this case we want to do decoding in place.
      // FIXME It might be a win on some systems (Irix?) to arrange that the
      // read buffer is on a page boundary.

      if (keepSize >= size_t(-1)/sizeof(Char) - (align - 1) - insertRS_)
	abort();			// FIXME throw an exception
      
      // Now size_t(-1)/sizeof(Char) - (align - 1) - insertRS_ - keepSize > 0
      if (readSizeChars
	  > size_t(-1)/sizeof(Char) - (align - 1) - insertRS_ - keepSize)
	abort();
      neededSize = roundUp(readSizeChars + keepSize + insertRS_, align);
      startOffset = neededSize - readSizeChars - insertRS_ - keepSize;
    }
    else {
      // Needs to be room for everything before decoding.
      neededSize = (keepSize + insertRS_ + readSizeChars
		    + (nLeftOver_ + sizeof(Char) - 1)/sizeof(Char));
      // Also must be room for everything after decoding.
      size_t neededSize2
	= (keepSize + insertRS_
	   // all the converted characters
	   + (nLeftOver_ + readSize_)/minBytesPerChar_
	   // enough Chars to contain left over bytes
	   + ((readSize_ % minBytesPerChar_ + sizeof(Char) - 1)
	      / sizeof(Char)));
      if (neededSize2 > neededSize)
	neededSize = neededSize2;
      neededSize = roundUp(neededSize, align);
      if (neededSize > size_t(-1)/sizeof(Char))
	abort();
      startOffset = 0;
    }
    if (bufSize_ < neededSize)
      reallocateBuffer(neededSize);
    Char *newStart = buf_ + startOffset;
    if (newStart != start() && keepSize > 0)
      memmove(newStart, start(), keepSize*sizeof(Char));
    char *bytesStart = (char *)(buf_ + bufSize_ - readSizeChars) - nLeftOver_;
    if (nLeftOver_ > 0 && leftOver_ != bytesStart)
      memmove(bytesStart, leftOver_, nLeftOver_);
    moveStart(newStart);
    bufLim_ = end();

    size_t nread;
    if (so_->read((char *)(buf_ + bufSize_ - readSizeChars), readSize_,
		  ic, nread)) {
      size_t nChars = decoder_->decode((Char *)end() + insertRS_,
				       bytesStart,
				       nLeftOver_ + nread,
				       &leftOver_);
      nLeftOver_ = bytesStart + nLeftOver_ + nread - leftOver_;
      if (insertRS_) {
	noteRS();
	*(Char *)end() = RS;
	advanceEnd(end() + 1);
	insertRS_ = false;
	bufLim_ += 1;
	bufLimOffset_ += 1;
      }
      bufLim_ += nChars;
      bufLimOffset_ += nChars;
      break;
    }
    else
      so_ = 0;
  }
  ASSERT(end() < bufLim_);
  if (insertRS_) {
    noteRS();
    insertChar(RS);
    insertRS_ = false;
    bufLimOffset_ += 1;
  }
   
  Char *e = (Char *)nextNewline(end(), bufLim_);
  if (e) {
    advanceEnd(e + 1);
    *e = RE;
    insertRS_ = true;
  }
  else
    advanceEnd(bufLim_);

  return nextChar();
}

const Char *PortableInputSource::nextNewline(const Char *start,
					     const Char *end)
{
  for (; start < end; start++)
    if (*start == '\n')
      return start;
  return 0;
}

void PortableInputSource::pushCharRef(Char ch, const NamedCharRef &ref)
{
  ASSERT(cur() == start());
  noteCharRef(startIndex() + (cur() - start()), ref);
  insertChar(ch);
}

void PortableInputSource::insertChar(Char ch)
{
  if (start() > buf_) {
    if (cur() > start())
      memmove((Char *)start() - 1, start(), (cur() - start())*sizeof(Char));
    moveLeft();
    *(Char *)cur() = ch;
  }
  else {
    // must have start == buf
    if (bufLim_ - buf_
	== bufSize_ - (nLeftOver_ + sizeof(Char) - 1)/sizeof(Char)) {
      if (bufSize_ == size_t(-1))
	abort();		// FIXME throw an exception
      reallocateBuffer(bufSize_ + 1);
    }
    else if (nLeftOver_ > 0 && ((char *)(bufLim_ + 1) > leftOver_)) {
      char *s = (char *)(buf_ + bufSize_) - nLeftOver_;
      memmove(s, leftOver_, nLeftOver_);
      leftOver_ = s;
    }
    if (cur() < bufLim_)
      memmove((Char *)cur() + 1, cur(), (bufLim_ - cur())*sizeof(Char));
    *(Char *)cur() = ch;
    advanceEnd(end() + 1);
    bufLim_ += 1;
  }
}

void PortableInputSource::reallocateBuffer(size_t newSize)
{
  Char *newBuf = new Char[newSize];
  
  memcpy(newBuf, buf_, bufSize_*sizeof(Char));
  bufSize_ = newSize;
  changeBuffer(newBuf, buf_);
  bufLim_ = newBuf + (bufLim_ - buf_);
  if (nLeftOver_ > 0) {
    char *s = (char *)(newBuf + bufSize_) - nLeftOver_;
    memmove(s,
	    newBuf + (leftOver_ - (char *)buf_),
	    nLeftOver_);
    leftOver_ = s;
  }
  delete [] buf_;
  buf_ = newBuf;
}

RTTI_DEF1(PortableExternalInfo, ExternalInfo)

PortableExternalInfo::PortableExternalInfo(GrowableVectorD<StorageObjectSpec> &specs)
: currentFile_(0)
{
  specs.moveTo(specs_);
  fileEndOffset_.init(specs_.length());
  setAll(fileEndOffset_.pointer(), fileEndOffset_.length(), Offset(-1));
  fileLine1RS_.init(specs_.length());
  setAll(fileLine1RS_.pointer(), fileLine1RS_.length(), size_t(0));
  decoders_.init(specs_.length());
  fileStartsWithRS_.init(specs_.length());
  setAll(fileStartsWithRS_.pointer(), fileStartsWithRS_.length(), size_t(0));
}

void PortableExternalInfo::setDecoder(size_t i, Decoder *decoder)
{
  decoders_[i] = decoder;
}

void PortableExternalInfo::noteRS(Offset offset)
{
  rsList_.append(offset);
  if (offset == (currentFile_ == 0 ? 0 : fileEndOffset_[currentFile_ - 1]))
    fileStartsWithRS_[currentFile_] += 1;
}

void PortableExternalInfo::noteFileEnd(Offset offset)
{
  ASSERT(currentFile_ < fileEndOffset_.length());
  // The last entry in fileEndOffset_ must be -1.
  if (currentFile_ < fileEndOffset_.length() - 1) {
    fileEndOffset_[currentFile_++] = offset;
    fileLine1RS_[currentFile_] = rsList_.length();
  }
}

Boolean PortableExternalInfo::convertOffset(Offset off,
					    const StorageObjectSpec *&spec,
					    unsigned long &lineno,
					    unsigned long &colno,
					    unsigned long &byteIndex) const
{
  spec = 0;
  if (off == -1 || fileEndOffset_.length() == 0)
    return false;
  // the last element of fileEndOffset_ is Offset(-1), so this will
  // terminate
  for (int i = 0; off >= fileEndOffset_[i]; i++)
    ;
  spec = &specs_[i];
  size_t line1RS = fileLine1RS_[i];
  Offset fileStartOffset = i == 0 ? 0 : fileEndOffset_[i - 1];
  // line1RS is now the number of RSs that are before or on the current line.
  size_t j;
  Offset colStart;
  byteIndex = off - fileStartOffset;
  if (rsList_.findPreceding(off, j, colStart)) {
    byteIndex -= j + 1 - line1RS;
    j++;
    colStart++;
  }
  else {
    j = 0;
    colStart = 0;
  }
  if (!decoders_[i] || !decoders_[i]->convertOffset(byteIndex))
    byteIndex = (unsigned long)-1;
  // j is now the number of RSs that are before or on the current line
  // colStart is the offset of the first column
  lineno = j - line1RS + 1 - fileStartsWithRS_[i];
  // the offset of the first column
  if (colStart < fileStartOffset)
    colStart = fileStartOffset;
  // the RS that starts a line will be in column 0;
  // the first real character of a line will be column 1
  colno = 1 + off - colStart;
  return true;
}

const StorageObjectSpec &PortableExternalInfo::spec(size_t i) const
{
  return specs_[i];
}

size_t PortableExternalInfo::nSpecs() const
{
  return specs_.length();
}


StorageObjectSpec::StorageObjectSpec()
: storageType(0), codingSystem(0)
{
}

