// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#ifdef __GNUG__
#pragma implementation
#endif

#include "PosixStorage.H"
#include "StorageManager.H"
#include "DescriptorManager.H"
#include "MessageArg.H"
#include "FilenameMessageArg.H"
#include "ErrnoMessageArg.H"
#include "InputContext.H"
#include "CString.H"
#include "String.H"
#include "CharsetInfo.H"
#include "CodingSystem.H"
#include "macros.H"

#include <sys/types.h>
#include <unistd.h>
#ifndef __GNUG__
#include <osfcn.h>
#endif
#include <fcntl.h>
#include <sys/stat.h>
#include <errno.h>
#include <stddef.h>

#if defined(sun) || defined(__sun)
// struct stat has st_blksize member
#define STAT_BLKSIZE 1
#endif

class PosixBaseStorageObject : public StorageObject {
public:
  PosixBaseStorageObject(int fd);
protected:
  size_t getBlockSize() const;
  enum { defaultBlockSize = 8192 };
  int fd_;
  PackedBoolean mayRewind_;
  PackedBoolean eof_;

  void rewindPrepare();
  void saveBytes(const char *, size_t);
  Boolean readSaved(char *, size_t, size_t &);
  Boolean rewind(InputContext &);
  void willNotRewind();
  virtual Boolean seek(off_t, InputContext &) = 0;
  virtual void close();
private:
  PackedBoolean savingBytes_;
  PackedBoolean readingSaved_;
  off_t startOffset_;
  PackedBoolean canSeek_;
  String<char> savedBytes_;
  size_t nBytesRead_;
};

inline
void PosixBaseStorageObject::saveBytes(const char *s, size_t n)
{
  if (savingBytes_)
    savedBytes_.append(s, n);
}


PosixBaseStorageObject::PosixBaseStorageObject(int fd)
: fd_(fd), mayRewind_(0), eof_(0), readingSaved_(0)
{
}

void PosixBaseStorageObject::rewindPrepare()
{
  if (!mayRewind_)
    return;
  struct stat sb;
  if (fstat(fd_, &sb) < 0 || !S_ISREG(sb.st_mode)
      || (startOffset_ = lseek(fd_, off_t(0), SEEK_CUR)) < 0) {
    canSeek_ = 0;
    savingBytes_ = 1;
  }
  else
    canSeek_ = 1;
}

Boolean PosixBaseStorageObject::rewind(InputContext &ic)
{
  ASSERT(mayRewind_);
  if (canSeek_) {
    eof_ = 0;
    return seek(startOffset_, ic);
  }
  else {
    readingSaved_ = 1;
    nBytesRead_ = 0;
    return 1;
  }
}

void PosixBaseStorageObject::willNotRewind()
{
  mayRewind_ = 0;
  savingBytes_ = 0;
  if (!readingSaved_)
    savedBytes_.clear();
  if (eof_ && fd_ >= 0) {
    close();
    fd_ = -1;
  }
}

Boolean PosixBaseStorageObject::readSaved(char *buf, size_t bufSize,
					  size_t &nread)
{
  if (!readingSaved_)
    return 0;
  if (nBytesRead_ >= savedBytes_.length()) {
    if (!mayRewind_)
      savedBytes_.clear();
    readingSaved_ = 0;
    return 0;
  }
  nread = savedBytes_.length() - nBytesRead_;
  if (nread > bufSize)
    nread = bufSize;
  memcpy(buf, savedBytes_.pointer() + nBytesRead_, nread);
  nBytesRead_ += nread;
  return 1;
}

void PosixBaseStorageObject::close()
{
}

class PosixStorageObject : public PosixBaseStorageObject, private DescriptorUser {
public:
  PosixStorageObject(const CString &,
		     const OutputCodingSystem *,
		     DescriptorManager *);
  ~PosixStorageObject();
  Boolean open(InputContext &, Boolean mayRewind, size_t &blockSize);
  Boolean read(char *buf, size_t bufSize, InputContext &ic, size_t &nread);
  Boolean suspend();
  Boolean seek(off_t, InputContext &);
  void close();
  enum SystemCall {
    noSystemCall,
    readSystemCall,
    openSystemCall,
    closeSystemCall,
    lseekSystemCall
  };
  enum {
    invalidFilenameError = lseekSystemCall + 1
    };
private:
  void resume(InputContext &);

  PackedBoolean suspended_;
  off_t suspendPos_;
  SystemCall suspendFailedSystemCall_;
  int suspendErrno_;
  CString filename_;
  String<char> filenameBytes_;

  void systemError(InputContext &, SystemCall, int);
};

static int xclose(int);
const char PosixStorageManager::messageSource[] = "PosixStorageManager";

PosixStorageManager::PosixStorageManager(const char *type,
					 const UnivCharsetDesc &filenameCharset,
					 const OutputCodingSystem *filenameCodingSystem,
					 int maxFDs)
: StorageManager(filenameCharset),
  type_(type),
  filenameCodingSystem_(filenameCodingSystem),
  descriptorManager_(maxFDs)
{
}

const char *PosixStorageManager::type() const
{
  return type_;
}

const char *PosixStorageManager::messageText(int n)
{
  static const char *const text[] = {
    "error reading %1 (%2)",
    "cannot open %1 (%2)",
    "error closing %1 (%2)",
    "error seeking on %1 (%2)",
    "invalid filename %1"
  };
  return n - 1 < 0 || n - 1 >= sizeof(text)/sizeof(text[0]) ? 0 : text[n - 1];
}

// These need to be written for each filesystem syntax.
// fixme should use idCharset.

static
Boolean isAbsolute(const Char *s, size_t len)
{
  return len > 0 && *s == '/';
}

static
const Char *basename(const Char *s, size_t len)
{
  for (const Char *p = s; len > 0; p++, len--)
    if (*p == '/')
      s = p + 1;
  return s;
}

void PosixStorageManager::mergeId(CString &id, const CString &specId)
{
  if (id.length() == 0
      || isAbsolute(id.pointer(), id.length()))
    return;
  const Char *dirend = basename(specId.pointer(), specId.length());
  if (dirend > specId.pointer()) {
    CString merged(specId.pointer(), dirend - specId.pointer());
    merged += id;
    merged.moveTo(id);
  }
}


StorageObject *
PosixStorageManager::makeStorageObject(const CString &str, InputContext &ic)
{
  if (str.length() == 0) {
    ic.message(PosixStorageManager::messageSource,
	       InputContext::error,
	       PosixStorageObject::invalidFilenameError,
	       StringMessageArg(str));
    return 0;
  }
  return new PosixStorageObject(str, filenameCodingSystem_,
				&descriptorManager_);
}

PosixStorageObject::PosixStorageObject(const CString &filename,
				       const OutputCodingSystem *filenameCodingSystem,
				       DescriptorManager *manager)
: DescriptorUser(manager),
  PosixBaseStorageObject(-1),
  suspended_(0),
  filename_(filename),
  filenameBytes_(filenameCodingSystem->convertOut(filename))
{
}

PosixStorageObject::~PosixStorageObject()
{
  if (fd_ >= 0) {
    (void)xclose(fd_);
    releaseD();
  }
}

Boolean PosixStorageObject::seek(off_t off, InputContext &ic)
{
  if (lseek(fd_, off, SEEK_SET) < 0) {
    fd_ = -1;
    systemError(ic, lseekSystemCall, errno);
    return 0;
  }
  else
    return 1;
}

Boolean PosixStorageObject::open(InputContext &ic, Boolean mayRewind,
				 size_t &blockSize)
{
  mayRewind_ = mayRewind;
  rewindPrepare();
  if (fd_ < 0) {
    acquireD();
    do {
      fd_ = ::open(filenameBytes_.pointer(), O_RDONLY);
    } while (fd_ < 0 && errno == EINTR);
    if (fd_ < 0) {
      releaseD();
      systemError(ic, openSystemCall, errno);
      return 0;
    }
  }
  blockSize = getBlockSize();
  return 1;
}

Boolean PosixStorageObject::read(char *buf, size_t bufSize, InputContext &ic,
				 size_t &nread)
{
  if (readSaved(buf, bufSize, nread))
    return 1;
  if (suspended_)
    resume(ic);
  if (fd_ < 0 || eof_)
    return 0;
  long n;
  do {
    n = ::read(fd_, buf, bufSize);
  } while (n < 0 && errno == EINTR);
  if (n > 0) {
    nread = size_t(n);
    saveBytes(buf, nread);
    return 1;
  }
  if (n < 0) {
    int saveErrno = errno;
    releaseD();
    (void)xclose(fd_);
    systemError(ic, readSystemCall, saveErrno);
    fd_ = -1;
  }
  else {
    eof_ = 1;
    // n == 0, so end of file
    if (!mayRewind_) {
      releaseD();
      if (xclose(fd_) < 0)
	systemError(ic, closeSystemCall, errno);
      fd_ = -1;
    }
    
  }
  return 0;
}

void PosixStorageObject::close()
{
  releaseD();
  (void)xclose(fd_);
}

Boolean PosixStorageObject::suspend()
{
  if (fd_ < 0 || suspended_)
    return 0;
  struct stat sb;
  if (fstat(fd_, &sb) < 0 || !S_ISREG(sb.st_mode))
    return 0;
  suspendFailedSystemCall_ = noSystemCall;
  suspendPos_ = lseek(fd_, 0, SEEK_CUR);
  if (suspendPos_ == (off_t)-1) {
    suspendFailedSystemCall_ = lseekSystemCall;
    suspendErrno_ = errno;
  }
  if (xclose(fd_) < 0 && !suspendFailedSystemCall_) {
    suspendFailedSystemCall_ = closeSystemCall;
    suspendErrno_ = errno;
  }
  fd_ = -1;
  suspended_ = 1;
  releaseD();
  return 1;
}

void PosixStorageObject::resume(InputContext &ic)
{
  ASSERT(suspended_);
  if (suspendFailedSystemCall_ != noSystemCall) {
    systemError(ic, suspendFailedSystemCall_, suspendErrno_);
    suspended_ = 0;
    return;
  }
  acquireD();
  // suspended_ must be 1 until after acquireD() is called,
  // so that we don't try to suspend this one before it is resumed.
  suspended_ = 0;
  do {
    fd_ = ::open(filenameBytes_.pointer(), O_RDONLY);
  } while (fd_ < 0 && errno == EINTR);
  if (fd_ < 0) {
    releaseD();
    systemError(ic, openSystemCall, errno);
    return;
  }
  if (lseek(fd_, suspendPos_, SEEK_SET) < 0) {
    systemError(ic, lseekSystemCall, errno);
    (void)xclose(fd_);
    fd_ = -1;
    releaseD();
  }
}

#ifdef STAT_BLKSIZE

size_t PosixBaseStorageObject::getBlockSize() const
{
  struct stat sb;
  long sz;
  if (fstat(fd_, &sb) < 0)
    return defaultBlockSize;
  if (!S_ISREG(sb.st_mode))
    return defaultBlockSize;
  if (sb.st_blksize > size_t(-1))
    sz = size_t(-1);
  else
    sz = sb.st_blksize;
  return sz;
}

#else /* not STAT_BLKSIZE */

size_t PosixBaseStorageObject::getBlockSize(int fd) const
{
  return defaultBlockSize;
}

#endif /* not STAT_BLKSIZE */

void PosixStorageObject::systemError(InputContext &ic,
				     SystemCall systemCall,
				     int err)
{
  ic.message(PosixStorageManager::messageSource,
	     InputContext::error|InputContext::parentLocation,
	     systemCall,
	     FilenameMessageArg(filename_),
	     ErrnoMessageArg(err));
}

static
int xclose(int fd)
{
  int ret;
  do {
    ret = close(fd);
  } while (ret < 0 && errno == EINTR);
  return ret;
}


class PosixFdStorageObject : public PosixBaseStorageObject {
public:
  PosixFdStorageObject(int);
  Boolean open(InputContext &, Boolean mayRewind, size_t &blockSize);
  Boolean read(char *buf, size_t bufSize, InputContext &ic, size_t &nread);
  Boolean seek(off_t, InputContext &);
  enum {
    noError,
    readError,
    invalidNumberError,
    lseekError
  };
private:
  int origFd_;
};

const char PosixFdStorageManager::messageSource[] = "PosixFdStorageManager";

const char *PosixFdStorageManager::messageText(int n)
{
  static const char *const text[] = {
    "error reading file descriptor %1 (%2)",
    "%1 is not a valid file descriptor number",
    "error seeking on file descriptor %1 (%2)",
  };
  return n - 1 < 0 || n - 1 >= sizeof(text)/sizeof(text[0]) ? 0 : text[n - 1];
}

PosixFdStorageManager::PosixFdStorageManager(const char *type,
					     const UnivCharsetDesc &idCharset)
: StorageManager(idCharset), type_(type)
{
}

StorageObject *PosixFdStorageManager::makeStorageObject(const CString &id,
							InputContext &ic)
{
  int n = 0;
  for (int i = 0; i < id.length(); i++) {
    UnivChar ch;
    if (!idCharset().descToUniv(id[i], ch))
      break;
    if (ch < UnivCharsetDesc::zero || ch > UnivCharsetDesc::zero + 9)
      break;
    int digit = ch - UnivCharsetDesc::zero;
    // Allow the division to be done at compile-time.
    if (n > INT_MAX/10)
      break;
    n *= 10;
    if (n > INT_MAX - digit)
      break;
    n += digit;
  }
  if (i < id.length() || i == 0) {
    ic.message(PosixFdStorageManager::messageSource,
	       InputContext::error,
	       PosixFdStorageObject::invalidNumberError,
	       StringMessageArg(id));
    return 0;
  }
  return new PosixFdStorageObject(n);
}

PosixFdStorageObject::PosixFdStorageObject(int fd)
: PosixBaseStorageObject(fd), origFd_(fd)
{
}

const char *PosixFdStorageManager::type() const
{
  return type_;
}

Boolean PosixFdStorageObject::open(InputContext &, Boolean mayRewind,
				   size_t &blockSize)
{
  mayRewind_ = mayRewind;
  rewindPrepare();
  // Could check access mode with fcntl(fd_, F_GETFL).
  blockSize = getBlockSize();
  return 1;
}

Boolean PosixFdStorageObject::read(char *buf, size_t bufSize, InputContext &ic,
				   size_t &nread)
{
  if (readSaved(buf, bufSize, nread))
    return 1;
  if (fd_ < 0 || eof_)
    return 0;
  long n;
  do {
    n = ::read(fd_, buf, bufSize);
  } while (n < 0 && errno == EINTR);
  if (n > 0) {
    nread = size_t(n);
    saveBytes(buf, nread);
    return 1;
  }
  if (n < 0) {
    ic.message(PosixFdStorageManager::messageSource,
	       InputContext::error|InputContext::parentLocation,
	       readError,
	       NumberMessageArg(fd_),
	       ErrnoMessageArg(errno));
    fd_ = -1;
  }
  else
    eof_ = 1;
  return 0;
}

Boolean PosixFdStorageObject::seek(off_t off, InputContext &ic)
{
  if (lseek(fd_, off, SEEK_SET) < 0) {
    ic.message(PosixFdStorageManager::messageSource,
	       InputContext::error|InputContext::parentLocation,
	       lseekError,
	       NumberMessageArg(fd_),
	       ErrnoMessageArg(errno));
    return 0;
  }
  else
    return 1;
}
