// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#ifdef __GNUG__
#pragma implementation
#endif

#include "StdioStorage.H"
#include "InputContext.H"
#include "types.H"
#include "FilenameMessageArg.H"
#include "ErrnoMessageArg.H"
#include "String.H"
#include "CString.H"
#include "CodingSystem.H"

#include <stdio.h>
#include <string.h>
#include <stddef.h>
#include <errno.h>

#ifndef SEEK_SET
#define SEEK_SET 0
#endif

class StdioStorageObject : public StorageObject {
public:
  StdioStorageObject(const CString &filename,
		     const OutputCodingSystem *filenameCodingSystem);
  ~StdioStorageObject();
  Boolean open(InputContext &, Boolean mayRewind, size_t &blockSize);
  Boolean read(char *buf, size_t bufSize, InputContext &ic, size_t &nread);
  Boolean rewind(InputContext &ic);
private:
  enum ErrorIndex {
    invalidErrorIndex,
    fopenFailed,
    readError,
    seekError
  };
  void error(InputContext &ic, ErrorIndex i, int err);

  FILE *fp_;
  CString filename_;
  String<char> filenameBytes_;
};

const char StdioStorageManager::messageSource[] = "StdioStorageManager";

const char *StdioStorageManager::messageText(int n)
{
  static const char *const text[] = {
    "cannot open %1 (%2)",
    "error reading %1 (%2)",
    "error seeking %1 (%2)",
  };
  return n - 1 < 0 || n - 1 >= sizeof(text)/sizeof(text[0]) ? 0 : text[n - 1];
}

StdioStorageManager::StdioStorageManager(const char *type,
					 const UnivCharsetDesc &filenameCharset,
					 const OutputCodingSystem *filenameCodingSystem)
: StorageManager(filenameCharset),
  type_(type),
  filenameCodingSystem_(filenameCodingSystem)
{
}

StorageObject *StdioStorageManager::makeStorageObject(const CString &str,
						      InputContext &)
{
  return new StdioStorageObject(str, filenameCodingSystem_);
}

const char *StdioStorageManager::type() const
{
  return type_;
}

StdioStorageObject::StdioStorageObject(const CString &filename,
				       const OutputCodingSystem *filenameCodingSystem)
: fp_(0),
  filename_(filename),
  filenameBytes_(filenameCodingSystem->convertOut(filename))
{
}

StdioStorageObject::~StdioStorageObject()
{
  if (fp_) {
    fclose(fp_);
    fp_ = 0;
  }
}

Boolean StdioStorageObject::rewind(InputContext &ic)
{
  if (fp_) {
    errno = 0;
    if (fseek(fp_, 0L, SEEK_SET) < 0) {
      error(ic, seekError, errno);
      return 0;
    }
    return 1;
  }
  return 1;
}

Boolean StdioStorageObject::open(InputContext &ic, Boolean,
				 size_t &blockSize)
{
  if (!fp_) {
    errno = 0;
    fp_ = fopen(filenameBytes_.pointer(), "r");
    if (!fp_) {
      error(ic, fopenFailed, errno);
      return 0;
    }
  }
  blockSize = BUFSIZ;
  return 1;
}

Boolean StdioStorageObject::read(char *buf, size_t bufSize, InputContext &ic,
				 size_t &nread)
{
  if (!fp_)
    return 0;
  errno = 0;
  size_t n = 0;
  FILE *fp = fp_;
  while (n < bufSize) {
    int c = getc(fp);
    if (c == EOF) {
      if (ferror(fp)) {
	error(ic, readError, errno);
	(void)fclose(fp);
	return 0;
      }
      fclose(fp);
      fp_ = 0;
      break;
    }
    buf[n++] = c;
  }
  nread = n;
  return n > 0;
}

void StdioStorageObject::error(InputContext &ic,
			       ErrorIndex i,
			       int err)
{
  ic.message(StdioStorageManager::messageSource,
	     InputContext::error|InputContext::parentLocation,
	     i,
	     FilenameMessageArg(filename_),
	     ErrnoMessageArg(err));
}
