// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

// This uses a big endian byte order irrespective of host byte order.
// Nothing special is done with FEFF/FFFE.

#ifdef MULTI_BYTE

#include "UCS2CodingSystem.H"
#include "macros.H"

#include <iostream.h>

class UCS2Decoder : public Decoder {
public:
  size_t decode(Char *to, const char *from, size_t fromLen,
		const char **rest);
  Boolean convertOffset(unsigned long &offset) const;
};

class UCS2Encoder : public Encoder {
public:
  UCS2Encoder();
  ~UCS2Encoder();
  void output(Char *, size_t, streambuf *);
  void output(const Char *, size_t, streambuf *);
private:
  void allocBuf(size_t);
  char *buf_;
  size_t bufSize_;
};

Decoder *UCS2CodingSystem::makeDecoder() const
{
  return new UCS2Decoder;
}

Encoder *UCS2CodingSystem::makeEncoder() const
{
  return new UCS2Encoder;
}

int UCS2CodingSystem::minBytesPerChar() const
{
  return 2;
}

size_t UCS2Decoder::decode(Char *to, const char *from, size_t fromLen,
			   const char **rest)
{
#ifdef BIG_ENDIAN
  if (sizeof(Char) == 2 && from == (char *)to) {
    *rest = from + (fromLen & ~1);
    return fromLen/2;
  }
#endif
  fromLen &= ~1;
  *rest = from + fromLen;
  for (size_t n = fromLen; n > 0; n -= 2) {
    *to++ = ((unsigned char)from[0] << 8) + (unsigned char)from[1];
    from += 2;
  }
  return fromLen/2;
}

Boolean UCS2Decoder::convertOffset(unsigned long &n) const
{
  n *= 2;
  return true;
}

UCS2Encoder::UCS2Encoder()
: buf_(0), bufSize_(0)
{
}

UCS2Encoder::~UCS2Encoder()
{
  delete [] buf_;
}

void UCS2Encoder::allocBuf(size_t n)
{
  if (bufSize_ < n) {
    delete [] buf_;
    buf_ = new char[bufSize_ = n];
  }
}

// FIXME handle errors from streambuf::sputn

void UCS2Encoder::output(Char *s, size_t n, streambuf *sb)
{
#ifdef BIG_ENDIAN
  if (sizeof(Char) == 2) {
    sb->sputn((char *)s, n*2);
    return;
  }
#endif
  ASSERT(sizeof(Char) >= 2);
  char *p = (char *)s;
  for (size_t i = 0; i < n; i++) {
    *p++ = (s[i] >> 8) & 0xff;
    *p++ = s[i] & 0xff;
  }
  sb->sputn((char *)s, n*2);
}

void UCS2Encoder::output(const Char *s, size_t n, streambuf *sb)
{
#ifdef BIG_ENDIAN
  if (sizeof(Char) == 2) {
    sb->sputn((char *)s, n*2);
    return;
  }
#endif
  allocBuf(n*2);
  for (size_t i = 0; i < n; i++) {
    buf_[i*2] = (s[i] >> 8) & 0xff;
    buf_[i*2 + 1] = s[i] & 0xff;
  }
  sb->sputn(buf_, n*2);
}

#else /* not MULTI_BYTE */

#ifndef __GNUG__
static char non_empty_translation_unit;	// sigh
#endif

#endif /* not MULTI_BYTE */
