// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

// This uses a big endian byte order irrespective of host byte order.
// Nothing special is done with FEFF/FFFE.

#ifdef MULTI_BYTE

#include "UnicodeCodingSystem.H"
#include "macros.H"

#include <stddef.h>
#include <string.h>
extern "C" {
  void *memmove(void *, const void *, size_t);
}
#include <iostream.h>

const unsigned short byteOrderMark = 0xfeff;
const unsigned short swappedByteOrderMark = 0xfffe;

class UnicodeDecoder : public Decoder {
public:
  UnicodeDecoder();
  size_t decode(Char *to, const char *from, size_t fromLen,
		const char **rest);
  Boolean convertOffset(unsigned long &offset) const;
private:
  PackedBoolean hadFirstChar_;
  PackedBoolean hadByteOrderMark_;
  PackedBoolean swapBytes_;
};

class UnicodeEncoder : public Encoder {
public:
  UnicodeEncoder(Boolean outputByteOrderMark);
  ~UnicodeEncoder();
  void output(Char *, size_t, streambuf *);
  void output(const Char *, size_t, streambuf *);
private:
  void allocBuf(size_t);
  unsigned short *buf_;
  size_t bufSize_;
  PackedBoolean outputByteOrderMark_;
};

Decoder *UnicodeCodingSystem::makeDecoder() const
{
  return new UnicodeDecoder;
}

Encoder *UnicodeCodingSystem::makeEncoder() const
{
  return new UnicodeEncoder(0);
}

Encoder *FileUnicodeCodingSystem::makeEncoder() const
{
  return new UnicodeEncoder(1);
}

int UnicodeCodingSystem::minBytesPerChar() const
{
  return 2;
}

UnicodeDecoder::UnicodeDecoder()
: hadByteOrderMark_(0), hadFirstChar_(0), swapBytes_(0)
{
}


size_t UnicodeDecoder::decode(Char *to, const char *from, size_t fromLen,
			      const char **rest)
{
  union U {
    unsigned short word;
    char bytes[2];
  };
    
  if (!hadFirstChar_) {
    hadFirstChar_ = 1;
    if (fromLen < 2) {
      *rest = from;
      return 0;
    }
    U u;
    u.bytes[0] = from[0];
    u.bytes[1] = from[1];
    if (u.word == byteOrderMark) {
      hadByteOrderMark_ = 1;
      from += 2;
      fromLen -= 2;
    }
    else if (u.word == swappedByteOrderMark) {
      hadByteOrderMark_ = 1;
      from += 2;
      fromLen -= 2;
      swapBytes_ = 1;
    }
  }
  fromLen &= ~1;
  *rest = from + fromLen;
  if (sizeof(Char) == 2) {
    if (!swapBytes_) {
      if (from != (char *)to)
	memmove(to, from, fromLen);
      return fromLen/2;
    }
  }
  if (swapBytes_) {
    for (size_t n = fromLen; n > 0; n -= 2) {
      U u;
      u.bytes[1] = *from++;
      u.bytes[0] = *from++;
      *to++ = u.word;
    }
  }
  else  {
    for (size_t n = fromLen; n > 0; n -= 2) {
      U u;
      u.bytes[0] = *from++;
      u.bytes[1] = *from++;
      *to++ = u.word;
    }
  }
  return fromLen/2;
}

Boolean UnicodeDecoder::convertOffset(unsigned long &n) const
{
  n *= 2;
  if (hadByteOrderMark_)
    n += 1;
  return true;
}

UnicodeEncoder::UnicodeEncoder(Boolean outputByteOrderMark)
: buf_(0), bufSize_(0), outputByteOrderMark_(outputByteOrderMark)
{
}

UnicodeEncoder::~UnicodeEncoder()
{
  delete [] buf_;
}

void UnicodeEncoder::allocBuf(size_t n)
{
  if (bufSize_ < n) {
    delete [] buf_;
    buf_ = new unsigned short[bufSize_ = n];
  }
}

// FIXME handle errors from streambuf::sputn

void UnicodeEncoder::output(Char *s, size_t n, streambuf *sb)
{
  if (outputByteOrderMark_) {
    const unsigned short n = 0xfeff;
    sb->sputn((char *)&n, 2);
    outputByteOrderMark_ = 0;
  }
  if (sizeof(Char) == 2) {
    sb->sputn((char *)s, n*2);
    return;
  }
  ASSERT(sizeof(Char) >= 2);
  unsigned short *p = (unsigned short *)s;
  for (size_t i = 0; i < n; i++)
    p[i] = s[i] & 0xffff;
  sb->sputn((char *)s, n*2);
}

void UnicodeEncoder::output(const Char *s, size_t n, streambuf *sb)
{
  if (outputByteOrderMark_) {
    const unsigned short n = byteOrderMark;
    sb->sputn((char *)&n, 2);
    outputByteOrderMark_ = 0;
  }
  if (sizeof(Char) == 2) {
    sb->sputn((char *)s, n*2);
    return;
  }
  allocBuf(n);
  for (size_t i = 0; i < n; i++)
    buf_[i] = s[i] & 0xffff;
  sb->sputn((char *)buf_, n*2);
}

#else /* not MULTI_BYTE */

#ifndef __GNUG__
static char non_empty_translation_unit;	// sigh
#endif

#endif /* not MULTI_BYTE */
