// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#include <stddef.h>
#include <string.h>
#include "new.H"

template<class T>
GrowableVector<T>::GrowableVector()
: alloc_(0)
{
}

template<class T>
GrowableVector<T>::GrowableVector(size_t n)
{
  alloc_ = length_ = n;
  if (n > 0) {
    ptr_ = (T *)::operator new(length_*sizeof(T));
    for (size_t i = 0; i < n; i++)
      (void)new (ptr_ + i) T;
  }
  else
    ptr_ = 0;
}


template<class T>
GrowableVector<T>::~GrowableVector()
{
  if (ptr_)
    ::operator delete((void *)ptr_);
}

template<class T>
GrowableVector<T>::GrowableVector(const GrowableVector<T> &vec)
{
  alloc_ = length_ = vec.length_;
  if (length_ > 0) {
    ptr_ = (T *)::operator new(alloc_*sizeof(T));
    memcpy(ptr_, vec.ptr_, length_*sizeof(T));
  }
  else
    ptr_ = 0;
}

template<class T>
GrowableVector<T> &GrowableVector<T>::operator=(const GrowableVector<T> &vec)
{
  setLength(vec.length_);
  if (length_ > 0)
    memcpy(ptr_, vec.ptr_, length_*sizeof(T));
  return *this;
}

template<class T>
T &GrowableVector<T>::grow()
{
  if (length_ >= alloc_) {
    alloc_ = alloc_ == 0 ? 8 : alloc_*2;
    void *p = ::operator new(alloc_*sizeof(T));
    if (ptr_) {
      memcpy(p, ptr_, length_*sizeof(T));
      ::operator delete((void *)ptr_);
    }
    ptr_ = (T *)p;
  }
  (void)new (ptr_ + length_) T;
  return ptr_[length_++];
}

template<class T>
void GrowableVector<T>::setLength(size_t length)
{
  if (length < length_)
    length_ = length;
  else if (length > length_) {
    if (length > alloc_) {
      alloc_ *= 2;
      if (length > alloc_)
	alloc_ += length;
      void *p = ::operator new(alloc_*sizeof(T));
      if (ptr_) {
	memcpy(p, ptr_, length_*sizeof(T));
	::operator delete((void *)ptr_);
      }
      ptr_ = (T *)p;
    }
    for (size_t i = length_; i < length; i++)
      (void)new (ptr_ + i) T;
    length_ = length;
  }
}

template<class T>
void GrowableVector<T>::clear()
{
  if (alloc_ > 0) {
    ::operator delete((void *)ptr_);
    alloc_ = 0;
    ptr_ = 0;
    length_ = 0;
  }
}

template<class T>
void GrowableVector<T>::append(const GrowableVector<T> &v)
{
  size_t oldLength = length_;
  setLength(oldLength + v.length_);
  for (size_t i = 0; i < v.length_; i++)
    ptr_[oldLength + i] = v.ptr_[i];
}

template<class T>
void GrowableVector<T>::moveTo(GrowableVector<T> &to)
{
  to.clear();
  to.ptr_ = ptr_;
  to.length_ = length_;
  to.alloc_ = alloc_;
  ptr_ = 0;
  length_ = 0;
  alloc_ = 0;
}
