// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#include <string.h>
#include <stddef.h>

template<class T>
String<T>::String(const T *ptr, size_t length)
: length_(length), alloc_(length)
{
  ptr_ = new T[length];
  memcpy(ptr_, ptr, length*sizeof(T));
}

template<class T>
String<T>::String()
: ptr_(0), length_(0), alloc_(0)
{
}

template<class T>
String<T>::String(const String<T> &s)
: length_(s.length_), alloc_(s.length_)
{
  ptr_ = new T[length_];
  memcpy(ptr_, s.ptr_, length_*sizeof(T));
}

template<class T>
String<T> &String<T>::operator=(const String<T> &s)
{
  if (&s != this) {
    if (s.length_ > alloc_) {
      if (ptr_)
	delete [] ptr_;
      ptr_ = new T[alloc_ = s.length_];
    }
    memcpy(ptr_, s.ptr_, s.length_*sizeof(T));
    length_ = s.length_;
  }
  return *this;
}

template<class T>
void String<T>::insert(size_t i, const String<T> &s)
{
  if (length_ + s.length_ > alloc_)
    grow(s.length_);
  for (size_t n = length_ - i; n > 0; n--)
    ptr_[i + n - 1 + s.length_] = ptr_[i + n - 1];
  length_ += s.length_;
  memcpy(ptr_ + i, s.ptr_, s.length_*sizeof(T));
}

template<class T>
void String<T>::clear()
{
  if (alloc_ > 0) {
    delete [] ptr_;
    ptr_ = 0;
    length_ = 0;
    alloc_ = 0;
  }
}

template<class T>
void String<T>::append(const T *p, size_t length)
{
  if (length_ + length > alloc_)
    grow(length);
  memcpy(ptr_ + length_, p, length*sizeof(T));
  length_ += length;
}

template<class T>
void String<T>::grow(size_t n)
{
  if (alloc_ < n)
    alloc_ += n + 16;
  else
    alloc_ += alloc_;
  T *s = new T[alloc_];
  memcpy(s, ptr_, length_*sizeof(T));
  delete [] ptr_;
  ptr_ = s;
}

template<class T>
void String<T>::moveTo(String<T> &to)
{
  if (to.ptr_)
    delete [] to.ptr_;
  to.ptr_ = ptr_;
  to.length_ = length_;
  to.alloc_ = alloc_;
  ptr_ = 0;
  length_ = 0;
  alloc_ = 0;
}

template<class T>
void String<T>::set(const T *p, size_t n)
{
  if (alloc_ < n) {
    if (ptr_)
      delete [] ptr_;
    ptr_ = new T[alloc_ = n];
  }
  length_ = n;
  for(T *to = ptr_; n > 0; n--, to++, p++)
    *to = *p;
}

template<class T>
void String<T>::init(size_t n)
{
  if (alloc_ < n) {
    if (ptr_)
      delete [] ptr_;
    ptr_ = new T[alloc_ = n];
  }
  length_ = n;
}

template<class T>
void String<T>::setLength(size_t n)
{
  if (alloc_ < n) {
    T *oldPtr_ = ptr_;
    ptr_ = new T[alloc_ = n];
    if (length_ > 0) {
      memcpy(ptr_, oldPtr_, length_*sizeof(T));
      delete [] oldPtr_;
    }
  }
  length_ = n;
}
