// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#include "SgmlsEventHandler.H"
#include "SgmlParser.H"
#include "Entity.H"
#include "Notation.H"
#include "Attribute.H"
#include "Portable.H"
#include "MessageReporter.H"
#include "macros.H"

const char dataCode = '-';
const char piCode = '?';
const char conformingCode = 'C';
const char appinfoCode = '#';
const char startElementCode = '(';
const char endElementCode = ')';
const char referenceEntityCode = '&';
const char attributeCode = 'A';
const char dataAttributeCode = 'D';
const char linkAttributeCode = 'a';
const char defineNotationCode = 'N';
const char defineExternalEntityCode = 'E';
const char defineInternalEntityCode = 'I';
const char defineSubdocEntityCode = 'S';
const char pubidCode = 'p';
const char sysidCode = 's';
const char startSubdocCode = '{';
const char endSubdocCode = '}';
const char fileCode = 'f';
const char locationCode = 'L';

const char nl = '\n';
const char space = ' ';
const Char re = '\r';

inline
void SgmlsEventHandler::startData()
{
  if (!haveData_) {
    os() << dataCode;
    haveData_ = 1;
  }
}

inline
void SgmlsEventHandler::flushData()
{
  if (haveData_) {
    os() << nl;
    haveData_ = 0;
  }
}

inline
void SgmlsEventHandler::outputLocation(const Location &loc)
{
  if (outputLocations_)
    outputLocation1(loc);
}

SgmlsEventHandler::SgmlsEventHandler(const SgmlParser *parser,
				     Portable *portable,
				     OutputCharStream *os,
				     MessageReporter *reporter,
				     Boolean outputLocations)
: SgmlsSubdocState(parser), portable_(portable), os_(os), reporter_(reporter),
  outputLocations_(outputLocations), 
  haveData_(0), lastSos_(0), errorCount_(0)
{
}

void SgmlsEventHandler::endDocument()
{
  flushData();
  if (!errorCount_)
    os() << conformingCode << nl;
}

void SgmlsEventHandler::message(MessageEvent *event)
{
  if (event->isError())
    errorCount_++;
  reporter_->reportMessage(*event);
  delete event;
}

void SgmlsEventHandler::appinfo(AppinfoEvent *event)
{
  const CString *str;
  if (event->literal(str)) {
    outputLocation(event->location());
    flushData();
    os() << appinfoCode;
    outputString(*str);
    os() << nl;
  }
  delete event;
}

void SgmlsEventHandler::simpleLink(SimpleLinkEvent *event)
{
  flushData();
  attributes(event->attributes(), linkAttributeCode, &event->name());
  delete event;
}

void SgmlsEventHandler::complexLink(ComplexLinkEvent *event)
{
  linkProcess_.init(event->lpd(), event->activeLpds());
  haveLinkProcess_ = 1;
  flushData();
  delete event;
}

void SgmlsEventHandler::uselink(UselinkEvent *event)
{
  linkProcess_.uselink(event->linkSet(),
		       event->restore(),
		       event->lpd().pointer());
  delete event;
}

void SgmlsEventHandler::sgmlDecl(SgmlDeclEvent *event)
{
  sd_ = event->sdPointer();
  syntax_ = event->instanceSyntaxPointer(); // FIXME which syntax?
  delete event;
}

void SgmlsEventHandler::data(DataEvent *event)
{
  outputLocation(event->location());
  startData();
  outputString(event->data(), event->dataLength());
  delete event;
}

void SgmlsEventHandler::sdataEntity(SdataEntityEvent *event)
{
  outputLocation(event->location());
  startData();
  os() << "\\|";
  outputString(event->data(), event->dataLength());
  os() << "\\|";
  delete event;
}

void SgmlsEventHandler::pi(PiEvent *event)
{
  outputLocation(event->location());
  flushData();
  os() << piCode;
  outputString(event->data(), event->dataLength());
  os() << nl;
  delete event;
}

void SgmlsEventHandler::startElement(StartElementEvent *event)
{
  flushData();
  if (haveLinkProcess_) {
    const AttributeList *linkAttributes;
    const ResultElementSpec *resultElementSpec;
    linkProcess_.startElement(event->elementType(),
			      event->attributes(),
			      event->location(),
			      *this, // InputContext &
			      linkAttributes,
			      resultElementSpec);
    if (linkAttributes)
      attributes(*linkAttributes, linkAttributeCode, &linkProcess_.name());
  }
  attributes(event->attributes(), attributeCode, 0);
  outputLocation(event->location());
  os() << startElementCode << event->name() << nl;
  delete event;
}

void SgmlsEventHandler::attributes(const AttributeList &attributes,
				   char code,
				   const CString *ownerName)
{
  size_t nAttributes = attributes.length();
  for (size_t i = 0; i < nAttributes; i++) {
    const Text *text;
    const CString *string;
    const AttributeValue *value = attributes.value(i);
    if (value) {
      switch (value->info(text, string)) {
      case AttributeValue::implied:
	startAttribute(attributes.name(i), code, ownerName);
	os() << "IMPLIED" << nl;
	break;
      case AttributeValue::tokenized:
	{
	  const char *typeString = "TOKEN";
	  const AttributeSemantics *semantics = attributes.semantics(i);
	  if (semantics) {
	    ConstResourcePointer<Notation> notation
	      = semantics->notation();
	    if (!notation.isNull()) {
	      defineNotation(notation.pointer());
	      typeString = "NOTATION";
	    }
	    else {
	      size_t nEntities = semantics->nEntities();
	      if (nEntities) {
		typeString = "ENTITY";
		for (size_t i = 0; i < nEntities; i++)
		  defineEntity(semantics->entity(i).pointer());
	      }
	    }
	  }
	  startAttribute(attributes.name(i), code, ownerName);
	  os() << typeString << space << *string << nl;
	}
	break;
      case AttributeValue::cdata:
	{
	  startAttribute(attributes.name(i), code, ownerName);
	  os() << "CDATA ";
	  EsisTextIter iter(*text);
	  EsisTextIter::Type type;
	  const Char *p;
	  size_t length;
	  while (iter.next(type, p, length))
	    if (type == EsisTextIter::data)
	      outputString(p, length);
	    else {
	      os() << "\\|";
	      outputString(p, length);
	      os() << "\\|";
	    }
	  os() << nl;
	}
	break;
      }
    }
  }
}

void SgmlsEventHandler::startAttribute(const CString &name,
				       char code,
				       const CString *ownerName)
{
  os() << code;
  if (ownerName)
    os() << *ownerName << space;
  os() << name << space;
}

void SgmlsEventHandler::endElement(EndElementEvent *event)
{
  flushData();
  if (haveLinkProcess_)
    linkProcess_.endElement();
  os() << endElementCode << event->name() << nl;
  delete event;
}

void SgmlsEventHandler::externalDataEntity(ExternalDataEntityEvent *event)
{
  outputLocation(event->entityOrigin()->parent());
  flushData();
  defineExternalDataEntity(event->entity());
  os() << referenceEntityCode << event->entity()->name() << nl;
  delete event;
}

void SgmlsEventHandler::subdocEntity(SubdocEntityEvent *event)
{
  outputLocation(event->entityOrigin()->parent());
  flushData();
  const SubdocEntity *entity = event->entity();
  defineSubdocEntity(entity);
  os() << startSubdocCode << entity->name() << nl;
  SgmlParser parser(*parser_, *entity, event->entityOrigin());
  SgmlsSubdocState oldState;
  SgmlsSubdocState::moveTo(oldState);
  SgmlsSubdocState::init(&parser);
  parser.parseAll(*this);
  oldState.moveTo(*this);
  os() << endSubdocCode << entity->name() << nl;
  delete event;
}

void SgmlsEventHandler::defineEntity(const Entity *entity)
{
  const ExternalDataEntity *externalDataEntity
    = entity->asExternalDataEntity();
  if (externalDataEntity)
    defineExternalDataEntity(externalDataEntity);
  else {
    const InternalEntity *internalEntity = entity->asInternalEntity();
    if (internalEntity)
      defineInternalEntity(internalEntity);
    else {
    const SubdocEntity *subdocEntity = entity->asSubdocEntity();
    if (subdocEntity)
      defineSubdocEntity(subdocEntity);
    }
  }
}

void SgmlsEventHandler::defineExternalDataEntity(const ExternalDataEntity *entity)
{
  if (markEntity(entity))
    return;
  const Notation *notation = entity->notation();
  defineNotation(notation);
  externalId(entity->externalId());
  externalEntityFilenames(entity);
  const char *typeString;
  switch (entity->dataType()) {
  case Entity::cdata:
    typeString = "CDATA";
    break;
  case Entity::sdata:
    typeString = "SDATA";
    break;
  case Entity::ndata:
    typeString = "NDATA";
    break;
  default:
    CANNOT_HAPPEN();
  }
  os() << defineExternalEntityCode << entity->name()
       << space << typeString
       << space << notation->name()
       << nl;
  attributes(entity->attributes(), dataAttributeCode, &entity->name());
}

void SgmlsEventHandler::defineSubdocEntity(const SubdocEntity *entity)
{
  if (markEntity(entity))
    return;
  externalId(entity->externalId());
  externalEntityFilenames(entity);
  os() << defineSubdocEntityCode << entity->name() << nl;
}

void SgmlsEventHandler::defineInternalEntity(const InternalEntity *entity)
{
  if (markEntity(entity))
    return;
  os() << defineInternalEntityCode << entity->name() << space
       << (entity->dataType() == Entity::sdata ? "SDATA" : "CDATA") << space;
  outputString(entity->string());
  os() << nl;
}

void SgmlsEventHandler::defineNotation(const Notation *notation)
{
  if (markNotation(notation))
    return;
  externalId(notation->externalId());
  os() << defineNotationCode << notation->name() << nl;
}

void SgmlsEventHandler::externalId(const ExternalId &id)
{
  const CString *str = id.publicIdString();
  if (str) {
    os() << pubidCode;
    outputString(*str);
    os() << nl;
  }
  str = id.systemIdString();
  if (str) {
    os() << sysidCode;
    outputString(*str);
    os() << nl;
  }
}

Boolean SgmlsEventHandler::markEntity(const Entity *entity)
{
  return definedEntities_.add(entity->name());
}

Boolean SgmlsEventHandler::markNotation(const Notation *notation)
{
  return definedNotations_.add(notation->name());
}

void SgmlsEventHandler::outputString(const Char *p, size_t n)
{
  for (; n > 0; p++, n--) {
    switch (*p) {
    case '\\':			// FIXME we're punning Chars and chars
      os() << "\\\\";
      break;
    case re:
      os() << "\\n";
      if (outputLocations_ && haveData_)
	lastLineno_++;
      break;
    default:
      // FIXME not clear what to do here given possibility of wide characters
      if (*p < 040) {
	static const char digits[] = "0123456789";
	os() << "\\0" << digits[*p / 8] << digits[*p % 8];
      }
      else
	os().put(*p);
      break;
    }
  }
}

void SgmlsEventHandler::externalEntityFilenames(const ExternalEntity *entity)
{
  GrowableVectorD<StorageObjectSpec> storageSpecs;
  if (!portable_->getExternalEntityStorage(*entity,
					   sd_->docCharset(),
					   entity->substTable(*syntax_),
					   syntax_->delimGeneral(Syntax::dPERO),
					   *reporter_,
					   storageSpecs))
    return;
  size_t i;
  for (i = 0; i < storageSpecs.length(); i++) {
    os() << fileCode;
    if (storageSpecs[i].codingSystem)
      os() << storageSpecs[i].codingSystem;
    if (strcmp(storageSpecs[i].storageType, "FILE") != 0)
      os() << storageSpecs[i].storageType;
    outputString(storageSpecs[i].id);
    os() << nl;
  }
}

void SgmlsEventHandler::outputLocation1(const Location &loc)
{
  const Origin *origin = loc.origin().pointer();
  const EntityOrigin *entityOrigin;
  const ExternalInfo *info;
  Index index = loc.index();
  for (;;) {
    if (!origin)
      return;
    entityOrigin = origin->asEntityOrigin();
    if (entityOrigin) {
      info = entityOrigin->externalInfo();
      if (info)
	break;
    }
    const Location &loc = origin->parent();
    index = loc.index();
    origin = loc.origin().pointer();
  }
  NamedCharRef ref;
  Offset off;
  while (entityOrigin->lookupIndex(index, off, ref)
	 == EntityOrigin::replacementCharIndex)
    index = ref.refStartIndex();
  const StorageObjectSpec *sos;
  unsigned long lineno;
  unsigned long dummy;
  if (!Portable::externalize(info, off, sos, lineno, dummy, dummy))
    return;
  if (sos == lastSos_) {
    if (lineno == lastLineno_)
      return;
    flushData();
    os() << locationCode << lineno << nl;
    lastLineno_ = lineno;
  }
  else {
    flushData();
    os() << locationCode << lineno << space;
    outputString(sos->id);
    os() << nl;
    lastLineno_ = lineno;
    lastSos_ = sos;
  }
}

void SgmlsEventHandler::inputContextMessage(const char *source,
					    const Location &loc,
					    unsigned flags, int number,
					    const MessageArg **args,
					    int nArgs)
{
  unsigned severity = (flags & InputContext::severityMask);
  if (severity != InputContext::info && severity != InputContext::warning)
    errorCount_++;
  reporter_->inputContextMessage(source, loc, flags, number, args, nArgs);
}

SgmlsSubdocState::SgmlsSubdocState()
: haveLinkProcess_(0), parser_(0)
{
}

SgmlsSubdocState::SgmlsSubdocState(const SgmlParser *parser)
: haveLinkProcess_(0), parser_(parser)
{
}

void SgmlsSubdocState::init(const SgmlParser *parser)
{
  parser_ = parser;
  definedNotations_.clear();
  definedEntities_.clear();
  haveLinkProcess_ = 0;
  linkProcess_.clear();
}

void SgmlsSubdocState::moveTo(SgmlsSubdocState &to)
{
  to.parser_ = parser_;
  to.haveLinkProcess_ = haveLinkProcess_;
  linkProcess_.moveTo(to.linkProcess_);
  definedNotations_.moveTo(to.definedNotations_);
  definedEntities_.moveTo(to.definedEntities_);
}
