// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#include "SgmlParser.H"
#include "Event.H"
#include "MessageEventHandler.H"
#include "SgmlsEventHandler.H"
#include "OutputCharStream.H"
#include "EasyParser.H"
#include "Boolean.H"
#include "programName.H"
#include "getopt.h"
#include "macros.H"

#include <iostream.h>
#include <stdlib.h>

static void usage();
static int parseProlog(EasyParser &easy, unsigned flags);
static int validate(EasyParser &easy, unsigned flags);
static int outputSgmls(EasyParser &easy, streambuf *, Boolean outputLocations);

int main(int argc, char **argv)
{
  programName = argv[0];
  int opt;
  Boolean suppressOutput = 0;
  EasyParser easy(getenv("NSGMLS_CODE"));
  Boolean prologOnly = 0;
  Boolean lFlag = 0;

  while ((opt = getopt(argc, argv, "a:c:degi:lm:prsuv")) >= 0) {
    switch (opt) {
    case 'a':
      // activate link
      easy.activateLinkType(optarg);
      break;
    case 'c':
      // output RACT file
      cerr << programName << ": -c option not implemented\n";
      break;
    case 'd':
      // warn about duplicate entity declarations
      easy.options.warnDuplicateEntity = 1;
      break;
    case 'e':
      // describe open entities in error messages
      easy.addOption(MessageReporter::openEntities);
      break;
    case 'g':
      // show gis of open elements in error messages
      easy.addOption(MessageReporter::openElements);
      break;
    case 'i':
      // pretend that optarg is defined as INCLUDE
      easy.addInclude(optarg);
      break;
    case 'l':
      // output L commands
      lFlag = 1;
      break;
    case 'm':
      easy.addCatalogSysid(optarg);
      break;
    case 'p':
      prologOnly = 1;
      break;
    case 'r':
      // warn about defaulted entity reference
      easy.options.warnDefaultEntityReference = 1;
      break;
    case 's':
      suppressOutput = 1;
      break;
    case 'u':
      // warn about undefined elements
      easy.options.warnUndefinedElement = 1;
      break;
    case 'v':
      // print the version number
      {
	extern const char *versionString;
	cerr << "nsgmls version " << versionString << '\n';
	cerr.flush();
      }
      break;
    case '?':
      usage();
    default:
      CANNOT_HAPPEN();
    }
  }
#ifdef __GNUG__
  // cout is a performance disaster in libg++ unless we do this.
  ios::sync_with_stdio(0);
#endif
  // could have option to parse all declared subdocuments
  easy.init(argc - optind, argv + optind, 1);
  if (prologOnly)
    return parseProlog(easy, 0);
  else if (suppressOutput)
    return validate(easy, MessageEventHandler::parseSubdocs);
  else 
    return outputSgmls(easy, cout.rdbuf(), lFlag);
}

static
void usage()
{
  cerr << "usage: "
       << programName
       << " [-deglprsuv] [-a linktype] [-i entity] [-m catalog_file] filename...\n";
  exit(1);
}

static
int parseProlog(EasyParser &easy, unsigned flags)
{
  SgmlParser &parser = easy.parser();
  MessageEventHandler handler(&parser, &easy, flags);
  Event *event;
  while ((event = parser.nextEvent()) != 0
	 && event->type() != Event::endProlog)
    event->handle(handler);
  return handler.errorCount() > 0;
}

static
int validate(EasyParser &easy, unsigned flags)
{
  SgmlParser &parser = easy.parser();
  MessageEventHandler handler(&parser, &easy, flags);
  parser.parseAll(handler);
  return handler.errorCount() > 0;
}

static
int outputSgmls(EasyParser &easy, streambuf *stream, Boolean lFlag)
{
  IosOutputCharStream os(stream, easy.codingSystem());
  SgmlParser &parser = easy.parser();
  SgmlsEventHandler handler(&parser, &easy.portable(), &os, &easy, lFlag);
  parser.parseAll(handler);
  handler.endDocument();
  return handler.errorCount() > 0;
}
