// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#ifdef __GNUG__
#pragma implementation
#endif
#include "Attribute.H"
#include "ParserState.H"
#include "MessageArg.H"
#include "macros.H"

DeclaredValue::DeclaredValue()
{
}

DeclaredValue::~DeclaredValue()
{
}

AttributeValue *DeclaredValue::makeValueFromToken(CString &, Text &text,
						  ParserState &parser,
						  const CString &name,
						  unsigned &specLength) const
{
  return makeValue(text, parser, name, specLength);
}

AttributeSemantics *DeclaredValue::makeSemantics(const TokenizedAttributeValue &,
						 ParserState &,
						 const CString &,
						 unsigned &,
						 unsigned &) const
{
  return 0;
}

Boolean DeclaredValue::containsToken(const CString &) const
{
  return 0;
}

Boolean DeclaredValue::isNotation() const
{
  return 0;
}

Boolean DeclaredValue::isId() const
{
  return 0;
}

Boolean DeclaredValue::isIdref() const
{
  return 0;
}

size_t DeclaredValue::getTokens(const CString *&) const
{
  return 0;
}


CdataDeclaredValue::CdataDeclaredValue()
{
}

Boolean CdataDeclaredValue::tokenized() const
{
  return 0;
}

AttributeValue *CdataDeclaredValue::makeValue(Text &text, ParserState &parser,
					      const CString &,
					      unsigned &specLength) const
{
  size_t normalizedLength = text.length() + (parser.syntax().normsep()
					     * (1 + text.nDataEntities()));
  specLength += normalizedLength;
  size_t litlen = parser.syntax().litlen();
  if (normalizedLength > litlen)
    parser.message(Messages::normalizedAttributeValueLength,
		   NumberMessageArg(litlen),
		   NumberMessageArg(normalizedLength));
  return new CdataAttributeValue(text);
}

void CdataDeclaredValue::buildDesc(AttributeDefinitionDesc &desc) const
{
  desc.declaredValue = AttributeDefinitionDesc::cdata;
}

TokenizedDeclaredValue::TokenizedDeclaredValue(TokenType type,
					       Boolean isList)
: type_(type), isList_(isList)
{
  switch (type) {
  case name:
  case entityName:
    initialCategories_ = Syntax::nameStartCategory;
    subsequentCategories_ = (Syntax::nameStartCategory|Syntax::digitCategory
			     | Syntax::otherNameCategory);
    break;
  case number:
    initialCategories_ = Syntax::digitCategory;
    subsequentCategories_ = Syntax::digitCategory;
    break;
  case nameToken:
    initialCategories_ = (Syntax::nameStartCategory|Syntax::digitCategory
			  | Syntax::otherNameCategory);
    subsequentCategories_ = initialCategories_;
    break;
  case numberToken:
    initialCategories_ = Syntax::digitCategory;
    subsequentCategories_ = (Syntax::nameStartCategory|Syntax::digitCategory
			     | Syntax::otherNameCategory);
    break;
  }
}

Boolean TokenizedDeclaredValue::tokenized() const
{
  return 1;
}

AttributeValue *TokenizedDeclaredValue::makeValue(Text &text,
						  ParserState &parser,
						  const CString &str,
						  unsigned &specLength) const
{
  return makeTokenizedValue(text, parser, str, specLength);
}

TokenizedAttributeValue *
TokenizedDeclaredValue::makeTokenizedValue(Text &text,
					   ParserState &parser,
					   const CString &,
					   unsigned &specLength) const
{
  const SubstTable<Char> &subst = *(type_ == entityName
				    ? parser.syntax().entitySubstTable()
				    : parser.syntax().generalSubstTable());
  GrowableVector<size_t> spaceIndex;
  CString value(text.string());
  Char space = parser.syntax().space();
  size_t i = 0;
  size_t length = value.length();

  for (;;) {
    if (i >= length) {
      parser.message(Messages::attributeValueSyntax);
      return 0;
    }
    if (!(parser.syntax().charCategory(value[i]) & initialCategories_)) {
      parser.message(text.charLocation(i),
		     Messages::attributeValueSyntax);
      return 0;
    }
    value[i] = subst[value[i]];
    size_t startIndex = i;
    for (++i;
	 i < length
	 && (parser.syntax().charCategory(value[i]) & subsequentCategories_);
	 i++)
      value[i] = subst[value[i]];
    if (i - startIndex > parser.syntax().namelen())
      parser.message(text.charLocation(i),
		     Messages::nameTokenLength,
		     NumberMessageArg(parser.syntax().namelen()));
    if (i == length)
      break;
    if (value[i] == space) {
      if (!isList_) {
	parser.message(text.charLocation(i),
		       Messages::attributeValueSyntax);
	return 0;
      }
      spaceIndex.grow() = i;
      i++;
    }
    else {
      parser.message(text.charLocation(i),
		     Messages::attributeValueSyntax);
      return 0;
    }
  }
  size_t normsep = parser.syntax().normsep();
  size_t litlen = parser.syntax().litlen();
  size_t normalizedLength = normsep + length;
  // should we count CDATA and SDATA entities here?
  if (isList_) {
    normalizedLength += 1;
    // length is now the number of characters in each token in the list
    // + 1 for each token in the list; so add normsep - 1 for each
    // token in the list.
    if (normsep > 0)
      normalizedLength += (normsep - 1)*(spaceIndex.length() + 1);
    else
      normalizedLength -= spaceIndex.length() + 1;
  }
  specLength += normalizedLength;
  // A length error will already have been given if
  // length > litlen - normsep.
  if (litlen >= normsep && length <= litlen - normsep
      && normalizedLength > litlen)
    parser.message(Messages::normalizedAttributeValueLength,
		   NumberMessageArg(litlen),
		   NumberMessageArg(normalizedLength));
  return new TokenizedAttributeValue(value, text, spaceIndex);
}

void TokenizedDeclaredValue::buildDesc(AttributeDefinitionDesc &desc) const
{
  desc.declaredValue = AttributeDefinitionDesc::DeclaredValue(
    type_ - name + (isList_
		    ? AttributeDefinitionDesc::names
		    : AttributeDefinitionDesc::name));
}

GroupDeclaredValue::GroupDeclaredValue(TokenType type,
				       Vector<CString> &vec)
: TokenizedDeclaredValue(type, 0)
{
  vec.moveTo(allowedValues_);
}

void GroupDeclaredValue::buildDesc(AttributeDefinitionDesc &desc) const
{
  desc.allowedValues.init(allowedValues_.length());
  for (size_t i = 0; i < allowedValues_.length(); i++)
    desc.allowedValues[i] = allowedValues_[i];
}

AttributeValue *GroupDeclaredValue::makeValue(Text &text,
					      ParserState &parser,
					      const CString &str,
					      unsigned &specLength) const
{
  TokenizedAttributeValue *val = makeTokenizedValue(text, parser, str,
						    specLength);
  if (!val)
    return 0;
  for (size_t i = 0; i < allowedValues_.length(); i++)
    if (val->string() == allowedValues_[i])
      return val;
  parser.message(Messages::attributeValueNotInGroup,
		 StringMessageArg(val->string()));
  return 0;
}

AttributeValue *GroupDeclaredValue::makeValueFromToken(CString &token,
						       Text &text,
						       ParserState &parser,
						       const CString &,
						       unsigned &specLength)
     const
{
  size_t litlen = parser.syntax().litlen();
  size_t normsep = parser.syntax().normsep();
  if (normsep > litlen || token.length() >  litlen - normsep)
    parser.message(Messages::normalizedAttributeValueLength,
		   NumberMessageArg(litlen),
		   NumberMessageArg(token.length() + normsep));
  specLength += token.length() + normsep;
  return new TokenizedAttributeValue(token, text, Vector<size_t>());
}

Boolean GroupDeclaredValue::containsToken(const CString &token) const
{
  for (size_t i = 0; i < allowedValues_.length(); i++)
    if (allowedValues_[i] == token)
      return 1;
  return 0;
}

size_t GroupDeclaredValue::getTokens(const CString *&ptr) const
{
  ptr = allowedValues_.pointer();
  return allowedValues_.length();
}

NameTokenGroupDeclaredValue::NameTokenGroupDeclaredValue(Vector<CString> &vec)
: GroupDeclaredValue(nameToken, vec)
{
}

void NameTokenGroupDeclaredValue::buildDesc(AttributeDefinitionDesc &desc) const
{
  GroupDeclaredValue::buildDesc(desc);
  desc.declaredValue = AttributeDefinitionDesc::nameTokenGroup;
}

NotationDeclaredValue::NotationDeclaredValue(Vector<CString> &vec)
: GroupDeclaredValue(name, vec)
{
}

Boolean NotationDeclaredValue::isNotation() const
{
  return 1;
}

AttributeSemantics *
NotationDeclaredValue::makeSemantics(const TokenizedAttributeValue &value,
				     ParserState &parser,
				     const CString &,
				     unsigned &,
				     unsigned &) const
{
  ConstResourcePointer<Notation> notation;
  if (parser.haveCurrentDtd())
    notation = parser.currentDtd().lookupNotation(value.string());
  else if (parser.resultAttributeSpecMode()) {
    const Dtd *resultDtd = parser.defComplexLpd().resultDtd().pointer();
    if (!resultDtd)
      return 0;
    notation = resultDtd->lookupNotation(value.string());
  }
  else
    return 0;			// link attribute
  if (notation.isNull()) {
    parser.message(Messages::invalidNotationAttribute,
		   StringMessageArg(value.string()));
    return 0;
  }
  else
    return new NotationAttributeSemantics(notation);
}

void NotationDeclaredValue::buildDesc(AttributeDefinitionDesc &desc) const
{
  GroupDeclaredValue::buildDesc(desc);
  desc.declaredValue = AttributeDefinitionDesc::notation;
}

EntityDeclaredValue::EntityDeclaredValue(Boolean isList)
: TokenizedDeclaredValue(entityName, isList)
{
}

AttributeSemantics *
EntityDeclaredValue::makeSemantics(const TokenizedAttributeValue &value,
				   ParserState &parser,
				   const CString &,
				   unsigned &,
				   unsigned &nEntityNames) const
{
  Boolean valid = 1;
  size_t nTokens = value.nTokens();
  nEntityNames += nTokens;
  Vector<ConstResourcePointer<Entity> > entities(nTokens);
  for (size_t i = 0; i < nTokens; i++) {
    CString str(value.token(i));
    Boolean defaulted;
    entities[i] = parser.lookupEntity(0, str, 0, defaulted);
    if (entities[i].isNull()) {
      parser.message(Messages::invalidEntityAttribute,
		     StringMessageArg(str));
      valid = 0;
    }
    else if (!entities[i]->isDataOrSubdoc()) {
      parser.message(Messages::notDataOrSubdocEntity,
		     StringMessageArg(str));
      valid = 0;
    }
    else if (defaulted && parser.options().warnDefaultEntityReference)
      parser.message(Messages::defaultEntityInAttribute,
		     StringMessageArg(str));
  }
  if (valid)
    return new EntityAttributeSemantics(entities);
  else
    return 0;
}

IdDeclaredValue::IdDeclaredValue()
: TokenizedDeclaredValue(name, 0)
{
}

Boolean IdDeclaredValue::isId() const
{
  return 1;
}

AttributeSemantics *
IdDeclaredValue::makeSemantics(const TokenizedAttributeValue &value,
			       ParserState &parser,
			       const CString &,
			       unsigned &,
			       unsigned &) const
{
  if (!parser.inInstance())
    return 0;
  Id *id = parser.lookupCreateId(value.string());
  if (id->defined())
    parser.message(value.tokenLocation(0),
		   Messages::duplicateId, StringMessageArg(value.string()),
		   id->defLocation());
  else
    id->define(value.tokenLocation(0));
  return 0;
}

void IdDeclaredValue::buildDesc(AttributeDefinitionDesc &desc) const
{
  desc.declaredValue = AttributeDefinitionDesc::id;
}

IdrefDeclaredValue::IdrefDeclaredValue(Boolean isList)
: TokenizedDeclaredValue(name, isList)
{
}

AttributeSemantics *
IdrefDeclaredValue::makeSemantics(const TokenizedAttributeValue &value,
				  ParserState &parser,
				  const CString &,
				  unsigned &nIdrefs,
				  unsigned &) const
{
  if (!parser.inInstance())
    return 0;
  size_t nTokens = value.nTokens();
  nIdrefs += nTokens;
  for (size_t i = 0; i < nTokens; i++) {
    Id *id = parser.lookupCreateId(value.token(i));
    if (!id->referenced())
      id->reference(value.tokenLocation(i));
  }
  return 0;
}

Boolean IdrefDeclaredValue::isIdref() const
{
  return 1;
}

void IdrefDeclaredValue::buildDesc(AttributeDefinitionDesc &desc) const
{
  TokenizedDeclaredValue::buildDesc(desc);
  if (desc.declaredValue == AttributeDefinitionDesc::name)
    desc.declaredValue = AttributeDefinitionDesc::idref;
  else
    desc.declaredValue = AttributeDefinitionDesc::idrefs;
}


AttributeDefinition::AttributeDefinition(const CString &name,
					 DeclaredValue *value)
: name_(name), declaredValue_(value)
{
}

AttributeDefinition::~AttributeDefinition()
{
}

AttributeValue *AttributeDefinition::makeValue(Text &text,
					       ParserState &parser,
					       unsigned &specLength) const
{
  return declaredValue_->makeValue(text, parser, name_, specLength);
}

AttributeValue *
AttributeDefinition::makeValueFromToken(CString &token,
					Text &text,
					ParserState &parser,
					unsigned &specLength) const
{
  return declaredValue_->makeValueFromToken(token, text, parser, name_,
					    specLength);
}

void AttributeDefinition::getDesc(AttributeDefinitionDesc &desc) const
{
  desc.allowedValues.clear();
  desc.defaultValue.clear();
  desc.currentIndex = 0;
  buildDesc(desc);
  declaredValue_->buildDesc(desc);
}

Boolean AttributeDefinition::isConref() const
{
  return 0;
}

Boolean AttributeDefinition::isCurrent() const
{
  return 0;
}

Boolean AttributeDefinition::isFixed() const
{
  return 0;
}

RequiredAttributeDefinition::RequiredAttributeDefinition(const CString &name,
							 DeclaredValue *value)
: AttributeDefinition(name, value)
{
}

ConstResourcePointer<AttributeValue>
RequiredAttributeDefinition::makeMissingValue(ParserState &parser) const
{
  parser.message(Messages::requiredAttributeMissing,
		 StringMessageArg(name()));
  return 0;
}

void RequiredAttributeDefinition::buildDesc(AttributeDefinitionDesc &desc) const
{
  desc.defaultValueType = AttributeDefinitionDesc::required;
}

CurrentAttributeDefinition::CurrentAttributeDefinition(const CString &name, DeclaredValue *value, size_t index)
: AttributeDefinition(name, value), currentIndex_(index)
{
}

ConstResourcePointer<AttributeValue>
CurrentAttributeDefinition::makeMissingValue(ParserState &parser) const
{
  if (parser.inInstance()
      && (parser.sd().omittag() || parser.sd().shorttag())) {
    ConstResourcePointer<AttributeValue> currentValue
      = parser.currentAttribute(currentIndex_);
    if (currentValue.isNull())
      parser.message(Messages::currentAttributeMissing,
		     StringMessageArg(name()));
    return currentValue;
  }
  else {
    parser.message(Messages::attributeMissing, StringMessageArg(name()));
    return 0;
  }
}

AttributeValue *
CurrentAttributeDefinition::makeValue(Text &text, ParserState &parser,
				      unsigned &specLength) const
{
  AttributeValue *value = AttributeDefinition::makeValue(text, parser,
							 specLength);
  if (parser.inInstance())
    parser.currentAttribute(currentIndex_) = value;
  return value;
}

void CurrentAttributeDefinition::buildDesc(AttributeDefinitionDesc &desc) const
{
  desc.defaultValueType = AttributeDefinitionDesc::current;
  desc.currentIndex = currentIndex_;
}

Boolean CurrentAttributeDefinition::isCurrent() const
{
  return 1;
}

ImpliedAttributeDefinition::ImpliedAttributeDefinition(const CString &name,
						       DeclaredValue *value)
: AttributeDefinition(name, value)
{
}

ConstResourcePointer<AttributeValue>
ImpliedAttributeDefinition::makeMissingValue(ParserState &parser) const
{
  return parser.makeImpliedAttributeValue();
}

void ImpliedAttributeDefinition::buildDesc(AttributeDefinitionDesc &desc) const
{
  desc.defaultValueType = AttributeDefinitionDesc::implied;
}

ConrefAttributeDefinition::ConrefAttributeDefinition(const CString &name,
						     DeclaredValue *value)
: ImpliedAttributeDefinition(name, value)
{
}

Boolean ConrefAttributeDefinition::isConref() const
{
  return 1;
}

void ConrefAttributeDefinition::buildDesc(AttributeDefinitionDesc &desc) const
{
  desc.defaultValueType = AttributeDefinitionDesc::conref;
}

DefaultAttributeDefinition:: DefaultAttributeDefinition(const CString &name,
							DeclaredValue *declaredValue,
							AttributeValue *defaultValue)
: AttributeDefinition(name, declaredValue),
  value_(defaultValue)
{
}

ConstResourcePointer<AttributeValue>
DefaultAttributeDefinition::makeMissingValue(ParserState &parser) const
{
  if (parser.sd().omittag() || parser.sd().shorttag())
    return value_;
  else {
    parser.message(Messages::attributeMissing,
		   StringMessageArg(name()));
    return 0;
  }
}

void DefaultAttributeDefinition::buildDesc(AttributeDefinitionDesc &desc) const
{
  desc.defaultValueType = AttributeDefinitionDesc::defaulted;
  desc.defaultValue = value_;
}

FixedAttributeDefinition:: FixedAttributeDefinition(const CString &name,
						    DeclaredValue *declaredValue,
						    AttributeValue *defaultValue)
: DefaultAttributeDefinition(name, declaredValue, defaultValue)
{
}

Boolean FixedAttributeDefinition::isFixed() const
{
  return 1;
}

AttributeValue *FixedAttributeDefinition::makeValue(Text &text,
						    ParserState &parser,
						    unsigned &specLength) const
{
  AttributeValue *value = AttributeDefinition::makeValue(text, parser,
							 specLength);
  const AttributeValue *fixedValue = defaultValue();
  if (value && fixedValue) {
    const Text *text;
    const CString *str;
    const Text *fixedText;
    const CString *fixedStr;
    switch (value->info(text, str)) {
    case AttributeValue::implied:
      CANNOT_HAPPEN();
    case AttributeValue::cdata:
      if (fixedValue->info(fixedText, fixedStr) == AttributeValue::cdata) {
	if (!text->fixedEqual(*fixedText))
	  parser.message(Messages::notFixedValue, StringMessageArg(name()));
      }
      break;
    case AttributeValue::tokenized:
      if (fixedValue->info(fixedText, fixedStr) == AttributeValue::tokenized) {
	if (*str != *fixedStr)
	  parser.message(Messages::notFixedValue, StringMessageArg(name()));
      }
      break;
    }
  }
  return value;
}

void FixedAttributeDefinition::buildDesc(AttributeDefinitionDesc &desc) const
{
  // get the fixed value
  DefaultAttributeDefinition::buildDesc(desc);
  desc.defaultValueType = AttributeDefinitionDesc::fixed;
}

AttributeDefinitionList::AttributeDefinitionList(VectorBase<Owner<AttributeDefinition> > &vec, size_t index)
: defs_(vec.length()), index_(index)
{
  for (size_t i = 0; i < defs_.length(); i++)
    vec[i].moveTo(defs_[i]);
}

Boolean AttributeDefinitionList::tokenIndex(const CString &token, unsigned &index) const
{
  for (size_t i = 0; i < defs_.length(); i++)
    if (defs_[i]->containsToken(token)) {
      index = i;
      return 1;
    }
  return 0;
}

Boolean AttributeDefinitionList::attributeIndex(const CString &name,
						unsigned &index) const
{
  for (size_t i = 0; i < defs_.length(); i++)
    if (defs_[i]->name() == name) {
      index = i;
      return 1;
    }
  return 0;
}

AttributeSemantics::AttributeSemantics()
{
}

AttributeSemantics::~AttributeSemantics()
{
}

size_t AttributeSemantics::nEntities() const
{
  return 0;
}

ConstResourcePointer<Entity> AttributeSemantics::entity(size_t) const
{
  return 0;
}

ConstResourcePointer<Notation> AttributeSemantics::notation() const
{
  return 0;
}


NotationAttributeSemantics::NotationAttributeSemantics(const ConstResourcePointer<Notation> &notation)
: notation_(notation)
{
}

ConstResourcePointer<Notation> NotationAttributeSemantics::notation() const
{
  return notation_;
}

AttributeSemantics *NotationAttributeSemantics::copy() const
{
  return new NotationAttributeSemantics(*this);
}

EntityAttributeSemantics::EntityAttributeSemantics(Vector<ConstResourcePointer<Entity> > &entity)
{
  entity.moveTo(entity_);
}

size_t EntityAttributeSemantics::nEntities() const
{
  return entity_.length();
}

ConstResourcePointer<Entity> EntityAttributeSemantics::entity(size_t i) const
{
  return entity_[i];
}

AttributeSemantics *EntityAttributeSemantics::copy() const
{
  return new EntityAttributeSemantics(*this);
}

AttributeValue::AttributeValue()
{
}

AttributeValue::~AttributeValue()
{
}

AttributeSemantics *AttributeValue::makeSemantics(const DeclaredValue *,
						  ParserState &,
						  const CString &,
						  unsigned &,
						  unsigned &) const
{
  return 0;
}

ImpliedAttributeValue::ImpliedAttributeValue()
{
}

AttributeValue::Type ImpliedAttributeValue::info(const Text *&,
						 const CString *&) const
{
  return implied;
}

TokenizedAttributeValue::TokenizedAttributeValue(CString &value,
						 Text &text,
						 const VectorBase<size_t> &spaceIndex)
: spaceIndex_(spaceIndex.length())
{
  value.moveTo(value_);
  text.moveTo(origText_);
  for (size_t i = 0; i < spaceIndex.length(); i++)
    spaceIndex_[i] = spaceIndex[i];
}

AttributeValue::Type TokenizedAttributeValue::info(const Text *&,
						   const CString *&string) const
{
  string = &value_;
  return tokenized;
}

AttributeSemantics *
TokenizedAttributeValue::makeSemantics(const DeclaredValue *value,
				       ParserState &parser,
				       const CString &name,
				       unsigned &nIdrefs,
				       unsigned &nEntityNames) const
{
  return value->makeSemantics(*this, parser, name, nIdrefs, nEntityNames);
}

CString TokenizedAttributeValue::token(size_t i) const
{
  size_t startIndex = i == 0 ? 0 : spaceIndex_[i - 1] + 1;
  size_t endIndex = i == spaceIndex_.length() ? value_.length() : spaceIndex_[i];
  return CString(&value_[startIndex], endIndex - startIndex);
}

Location TokenizedAttributeValue::tokenLocation(size_t i) const
{
  return origText_.charLocation(i == 0 ? 0 : spaceIndex_[i - 1] + 1);
}

CdataAttributeValue::CdataAttributeValue(Text &text)
{
  text.moveTo(text_);
}

AttributeValue::Type CdataAttributeValue::info(const Text *&text,
					       const CString *&) const
{
  text = &text_;
  return cdata;
}

AttributeSpec::AttributeSpec(unsigned specIndex, const CString &origName)
: specIndex_(specIndex), origName_(origName)
{
}

AttributeSpec::AttributeSpec(unsigned specIndex)
: specIndex_(specIndex)
{
}

AttributeSpec *AttributeSpec::copy() const
{
  return new AttributeSpec(*this);
}

Attribute::Attribute()
{
}

void Attribute::clear()
{
  spec_.clear();
  value_.clear();
  semantics_.clear();
}

AttributeList::AttributeList(const ConstResourcePointer<AttributeDefinitionList> &def)
: def_(def), vec_(def.isNull() ? 0 : def->length()), nSpec_(0), conref_(0),
  notation_(0), nIdrefs_(0), nEntityNames_(0), idIndex_(size_t(-1))
{
}

AttributeList::AttributeList()
: nSpec_(0), conref_(0)
{
}

void AttributeList::init(const ConstResourcePointer<AttributeDefinitionList> &def)
{
  def_ = def;
  nSpec_ = 0;
  conref_ = 0;
  notation_ = 0;
  nIdrefs_ = 0;
  nEntityNames_ = 0;
  idIndex_ = size_t(-1);
  if (def_.isNull())
    vec_.setLength(0);
  else {
    size_t newLength = def_->length();
    size_t clearLim = vec_.length();
    if (clearLim > newLength)
      clearLim = newLength;
    vec_.setLength(newLength);
    for (size_t i = 0; i < clearLim; i++)
      vec_[i].clear();
  }
}

void AttributeList::moveTo(AttributeList &to)
{
  vec_.moveTo(to.vec_);
  def_.moveTo(to.def_);
  to.nSpec_ = nSpec_;
  to.conref_ = conref_;
}

void AttributeList::finish(ParserState &parser)
{
  for (size_t i = 0; i < vec_.length(); i++)
    if (!vec_[i].specified()) {
      ConstResourcePointer<AttributeValue> value
	= def(i)->makeMissingValue(parser);
      vec_[i].setValue(value);
      if (!value.isNull())
	vec_[i].setSemantics(def(i)->makeSemantics(value.pointer(),
						   parser,
						   nIdrefs_,
						   nEntityNames_));
    }
  if (nIdrefs_ > parser.syntax().grpcnt())
    parser.message(Messages::idrefGrpcnt,
		   NumberMessageArg(parser.syntax().grpcnt()));
  if (nEntityNames_ > parser.syntax().grpcnt())
    parser.message(Messages::entityNameGrpcnt,
		   NumberMessageArg(parser.syntax().grpcnt()));
  if (notation_ && conref_)
    parser.message(Messages::conrefNotation);
}

void AttributeList::setSpec(unsigned i, const CString &origName, ParserState &parser)
{
  if (vec_[i].specified())
    parser.message(Messages::duplicateAttributeSpec,
		   StringMessageArg(def(i)->name()));
  else
    vec_[i].setSpec(new AttributeSpec(nSpec_++, origName));
}

void AttributeList::setSpec(unsigned i, ParserState &parser)
{
  setSpec(i, CString(), parser);
}

void AttributeList::setValue(unsigned i, Text &text, ParserState &parser,
			     unsigned &specLength)
{
  AttributeValue *value = def(i)->makeValue(text, parser, specLength);
  if (def(i)->isConref())
    conref_ = 1;
  if (def(i)->isNotation())
    notation_ = 1;
  if (def(i)->isId())
    idIndex_ = i;
  vec_[i].setValue(value);
  if (value)
    vec_[i].setSemantics(def(i)->makeSemantics(value, parser,
					       nIdrefs_, nEntityNames_));
}

void AttributeList::setValue(unsigned i, CString &token, Text &text,
			     ParserState &parser, unsigned &specLength)
{
  AttributeValue *value = def(i)->makeValueFromToken(token, text, parser,
						     specLength);
  if (def(i)->isConref())
    conref_ = 1;
  vec_[i].setValue(value);
  if (value)
    vec_[i].setSemantics(def(i)->makeSemantics(value, parser,
					       nIdrefs_, nEntityNames_));
}

Boolean AttributeList::getId(const CString *&str) const
{
  if (idIndex_ == size_t(-1))
    return 0;
  str = &((const TokenizedAttributeValue *)vec_[idIndex_].value())->string();
  return 1;
}
