// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#ifndef Attribute_INCLUDED
#define Attribute_INCLUDED 1
#ifdef __GNUG__
#pragma interface
#endif

#include <stddef.h>
#include "Resource.H"
#include "Owner.H"
#include "CString.H"
#include "Vector.H"
#include "CopyVector.H"
#include "CopyGrowableVectorD.H"
#include "CopyOwner.H"
#include "Boolean.H"
#include "Text.H"
#include "ResourcePointer.H"

class Entity;
class Notation;
class ParserState;
class DeclaredValue;
class AttributeValue;
class TokenizedAttributeValue;
class AttributeSemantics;
class AttributeSpec;

class AttributeDefinitionDesc {
public:
  AttributeDefinitionDesc() { }
  enum DeclaredValue {
    cdata,
    name,
    number,
    nmtoken,
    nutoken,
    entity,
    idref,
    names,
    numbers,
    nmtokens,
    nutokens,
    entities,
    idrefs,
    id,
    notation,
    nameTokenGroup
  };
  DeclaredValue declaredValue;
  enum DefaultValueType {
    required,
    current,
    implied,
    conref,
    defaulted,
    fixed
  };
  DefaultValueType defaultValueType;
  ConstResourcePointer<AttributeValue> defaultValue;
  Vector<CString> allowedValues;
  // Attribute definitions whose default value type is current and
  // which have the same currentIndex share current values.
  size_t currentIndex;
private:
  AttributeDefinitionDesc(const AttributeDefinitionDesc &); // undefined
  void operator=(const AttributeDefinitionDesc &);	    // undefined
};

class DeclaredValue {
public:
  DeclaredValue();
  virtual ~DeclaredValue();
  // This performs syntactic checking on the value.
  virtual AttributeValue *makeValue(Text &, ParserState &,
				    const CString &name,
				    unsigned &specLength) const = 0;
  // This is used to avoid unnecessary syntactic checking in the
  // case where the attribute name and vi have been omitted.
  virtual AttributeValue *makeValueFromToken(CString &, Text &,
					     ParserState &,
					     const CString &name,
					     unsigned &specLength) const;
  // This performs semantic checking on the value.
  virtual AttributeSemantics *makeSemantics(const TokenizedAttributeValue &,
					    ParserState &,
					    const CString &,
					    unsigned &nIdrefs,
					    unsigned &nEntityNames) const;
  virtual Boolean containsToken(const CString &) const;
  virtual Boolean tokenized() const = 0;
  virtual Boolean isNotation() const;
  virtual Boolean isId() const;
  virtual Boolean isIdref() const;
  virtual size_t getTokens(const CString *&) const;
  virtual void buildDesc(AttributeDefinitionDesc &) const = 0;
};

class CdataDeclaredValue : public DeclaredValue {
public:
  CdataDeclaredValue();
  Boolean tokenized() const;
  AttributeValue *makeValue(Text &, ParserState &, const CString &,
			    unsigned &) const;
  void buildDesc(AttributeDefinitionDesc &) const;
};

class TokenizedDeclaredValue : public DeclaredValue {
public:
  // must be in same order as AttributeDefinitionDesc
  enum TokenType {
    name,
    number,
    nameToken,
    numberToken,
    entityName
  };
  TokenizedDeclaredValue(TokenType type, Boolean isList);
  AttributeValue *makeValue(Text &, ParserState &, const CString &,
			    unsigned &) const;
  TokenizedAttributeValue *makeTokenizedValue(Text &, ParserState &,
					      const CString &, unsigned &) const;
  Boolean tokenized() const;
  void buildDesc(AttributeDefinitionDesc &) const;
private:
  TokenType type_;
  Boolean isList_;
  unsigned initialCategories_;
  unsigned subsequentCategories_;
};

class GroupDeclaredValue : public TokenizedDeclaredValue {
public:
  GroupDeclaredValue(TokenType, Vector<CString> &);
  Boolean containsToken(const CString &) const;
  AttributeValue *makeValue(Text &, ParserState &, const CString &,
			    unsigned &) const;
  AttributeValue *makeValueFromToken(CString &, Text &,
				     ParserState &,
				     const CString &name,
				     unsigned &) const;
  size_t getTokens(const CString *&) const;
  void buildDesc(AttributeDefinitionDesc &) const;
private:
  GroupDeclaredValue(const GroupDeclaredValue &); // undefined
  void operator=(const GroupDeclaredValue &);	  // undefined
  Vector<CString> allowedValues_;
};

class NameTokenGroupDeclaredValue : public GroupDeclaredValue {
public:
  NameTokenGroupDeclaredValue(Vector<CString> &);
  void buildDesc(AttributeDefinitionDesc &) const;
private:
  NameTokenGroupDeclaredValue(const NameTokenGroupDeclaredValue &); // undefined
  void operator=(const NameTokenGroupDeclaredValue &);		    // undefined
};

class NotationDeclaredValue : public GroupDeclaredValue {
public:
  NotationDeclaredValue(Vector<CString> &);
  AttributeSemantics *makeSemantics(const TokenizedAttributeValue &,
				    ParserState &,
				    const CString &,
				    unsigned &nIdrefs,
				    unsigned &nEntityNames) const;
  Boolean isNotation() const;
  void buildDesc(AttributeDefinitionDesc &) const;
private:
  NotationDeclaredValue(const NotationDeclaredValue &); // undefined
  void operator=(const NotationDeclaredValue &);	// undefined
};

class EntityDeclaredValue : public TokenizedDeclaredValue {
public:
  EntityDeclaredValue(Boolean isList);
  AttributeSemantics *makeSemantics(const TokenizedAttributeValue &,
				    ParserState &,
				    const CString &,
				    unsigned &nIdrefs,
				    unsigned &nEntityNames) const;
};

class IdDeclaredValue : public TokenizedDeclaredValue {
public:
  IdDeclaredValue();
  AttributeSemantics *makeSemantics(const TokenizedAttributeValue &,
				    ParserState &,
				    const CString &,
				    unsigned &nIdrefs,
				    unsigned &nEntityNames) const;
  Boolean isId() const;
  void buildDesc(AttributeDefinitionDesc &) const;
};

class IdrefDeclaredValue : public TokenizedDeclaredValue {
public:
  IdrefDeclaredValue(Boolean isList);
  AttributeSemantics *makeSemantics(const TokenizedAttributeValue &,
				    ParserState &,
				    const CString &,
				    unsigned &nIdrefs,
				    unsigned &nEntityNames) const;
  Boolean isIdref() const;
  void buildDesc(AttributeDefinitionDesc &) const;
};

class AttributeDefinition {
public:
  AttributeDefinition(const CString &, DeclaredValue *);
  AttributeDefinition(const AttributeDefinition &); // undefined
  void operator=(const AttributeDefinition &);	    // undefined
  virtual ~AttributeDefinition();
  virtual ConstResourcePointer<AttributeValue>
    makeMissingValue(ParserState &) const = 0;
  virtual AttributeValue *makeValue(Text &, ParserState &, unsigned &) const;
  virtual AttributeValue *makeValueFromToken(CString &, Text &, 
					     ParserState &,
					     unsigned &) const;
  
  virtual Boolean isConref() const;
  virtual Boolean isCurrent() const;
  virtual Boolean isFixed() const;
  AttributeSemantics *makeSemantics(const AttributeValue *,
				    ParserState &,
				    unsigned &nIdrefs,
				    unsigned &nEntityNames) const;
  Boolean tokenized() const;
  const CString &name() const;
  Boolean containsToken(const CString &) const;
  Boolean isNotation() const;
  Boolean isId() const;
  Boolean isIdref() const;
  void getDesc(AttributeDefinitionDesc &) const;
private:
  virtual void buildDesc(AttributeDefinitionDesc &) const = 0;
  CString name_;
  Owner<DeclaredValue> declaredValue_;
};

class RequiredAttributeDefinition : public AttributeDefinition {
public:
  RequiredAttributeDefinition(const CString &, DeclaredValue *);
  RequiredAttributeDefinition(const RequiredAttributeDefinition &); // undefined
  void operator=(const RequiredAttributeDefinition &);		    // undefined
  ConstResourcePointer<AttributeValue> makeMissingValue(ParserState &) const;
  void buildDesc(AttributeDefinitionDesc &) const;
};

class CurrentAttributeDefinition : public AttributeDefinition {
public:
  CurrentAttributeDefinition(const CString &, DeclaredValue *, size_t index);
  CurrentAttributeDefinition(const CurrentAttributeDefinition &); // undefined
  void operator=(const CurrentAttributeDefinition &);		  // undefined
  ConstResourcePointer<AttributeValue> makeMissingValue(ParserState &) const;
  AttributeValue *makeValue(Text &, ParserState &, unsigned &) const;
  void buildDesc(AttributeDefinitionDesc &) const;
  Boolean isCurrent() const;
private:
  size_t currentIndex_;
};

class ImpliedAttributeDefinition : public AttributeDefinition {
public:
  ImpliedAttributeDefinition(const CString &, DeclaredValue *);
  ImpliedAttributeDefinition(const ImpliedAttributeDefinition &); // undefined
  void operator=(const ImpliedAttributeDefinition &);		  // undefined
  ConstResourcePointer<AttributeValue> makeMissingValue(ParserState &) const;
  void buildDesc(AttributeDefinitionDesc &) const;
};

class ConrefAttributeDefinition : public ImpliedAttributeDefinition {
public:
  ConrefAttributeDefinition(const CString &, DeclaredValue *);
  ConrefAttributeDefinition(const ConrefAttributeDefinition &); // undefined
  void operator=(const ConrefAttributeDefinition &);		// undefined
  Boolean isConref() const;
  void buildDesc(AttributeDefinitionDesc &) const;
};

class DefaultAttributeDefinition : public AttributeDefinition {
public:
  DefaultAttributeDefinition(const CString &, DeclaredValue *,
			     AttributeValue *);
  DefaultAttributeDefinition(const DefaultAttributeDefinition &); // undefined
  void operator=(const DefaultAttributeDefinition &);		  // undefined
  ConstResourcePointer<AttributeValue> makeMissingValue(ParserState &) const;
  void buildDesc(AttributeDefinitionDesc &) const;
protected:
  const AttributeValue *defaultValue() const;
private:
  ConstResourcePointer<AttributeValue> value_;
};

class FixedAttributeDefinition : public DefaultAttributeDefinition {
public:
  FixedAttributeDefinition(const CString &, DeclaredValue *,
			   AttributeValue *);
  FixedAttributeDefinition(const FixedAttributeDefinition &); // undefined
  void operator=(const FixedAttributeDefinition &);	      // undefined
  // check that it's equal to the default
  AttributeValue *makeValue(Text &, ParserState &, unsigned &) const;
  void buildDesc(AttributeDefinitionDesc &) const;
  Boolean isFixed() const;
};

class AttributeDefinitionList : public Resource {
public:
  AttributeDefinitionList(VectorBase<Owner<AttributeDefinition> > &,
			  size_t);
  AttributeDefinitionList(const AttributeDefinitionList &); // undefined
  void operator=(const AttributeDefinitionList &);	    // undefined
  size_t length() const;
  const AttributeDefinition *def(size_t) const;
  Boolean tokenIndex(const CString &, unsigned &) const;
  Boolean attributeIndex(const CString &, unsigned &) const;
  size_t index() const;
private:
  Vector<Owner<AttributeDefinition> > defs_;
  size_t index_;
};

class AttributeSemantics {
public:
  AttributeSemantics();
  virtual ~AttributeSemantics();
  virtual size_t nEntities() const;
  virtual ConstResourcePointer<Entity> entity(size_t) const;
  virtual ConstResourcePointer<Notation> notation() const;
  virtual AttributeSemantics *copy() const = 0;
};

class EntityAttributeSemantics  : public AttributeSemantics {
public:
  EntityAttributeSemantics(Vector<ConstResourcePointer<Entity> > &);
  size_t nEntities() const;
  ConstResourcePointer<Entity> entity(size_t) const;
  AttributeSemantics *copy() const;
private:
  CopyVector<ConstResourcePointer<Entity> > entity_;
};

class NotationAttributeSemantics : public AttributeSemantics {
public:
  NotationAttributeSemantics(const ConstResourcePointer<Notation> &);
  ConstResourcePointer<Notation> notation() const;
  AttributeSemantics *copy() const;
private:
  ConstResourcePointer<Notation> notation_;
};

class AttributeValue : public Resource {
public:
  enum Type {
    implied,
    cdata,
    tokenized
    };
  AttributeValue();
  virtual ~AttributeValue();
  virtual AttributeSemantics *makeSemantics(const DeclaredValue *,
					    ParserState &,
					    const CString &,
					    unsigned &,
					    unsigned &) const;
  virtual Type info(const Text *&, const CString *&) const = 0;
};

class ImpliedAttributeValue : public AttributeValue {
public:
  ImpliedAttributeValue();
  Type info(const Text *&, const CString *&) const;
};

class CdataAttributeValue : public AttributeValue {
public:
  CdataAttributeValue(Text &);
  Type info(const Text *&, const CString *&) const;
private:
  Text text_;
};

class TokenizedAttributeValue : public AttributeValue {
public:
  TokenizedAttributeValue(CString &, Text &, const VectorBase<size_t> &);
  size_t nTokens() const;
  AttributeSemantics *makeSemantics(const DeclaredValue *,
				    ParserState &,
				    const CString &,
				    unsigned &,
				    unsigned &) const;
  Type info(const Text *&, const CString *&) const;
  const CString &string() const;
  CString token(size_t) const;
  Location tokenLocation(size_t) const;
private:
  TokenizedAttributeValue(const TokenizedAttributeValue &); // undefined
  void operator=(const TokenizedAttributeValue &);	    // undefined
  CString value_;		// after substitution
  Text origText_;		// before substitution
  // index into value of each space
  // length is number of tokens - 1
  Vector<size_t> spaceIndex_;
};

class AttributeSpec {
public:
  AttributeSpec(unsigned);
  AttributeSpec(unsigned, const CString &);
  AttributeSpec *copy() const;
private:
  unsigned specIndex_;
  CString origName_;		// empty if name omitted
};

class Attribute {
public:
  Attribute();
  Boolean specified() const;
  const AttributeSpec *spec() const;
  const AttributeValue *value() const;
  const ConstResourcePointer<AttributeValue> Attribute::valuePointer() const;
  const AttributeSemantics *semantics() const;
  void setSpec(AttributeSpec *);
  void setValue(const ConstResourcePointer<AttributeValue> &);
  void setSemantics(AttributeSemantics *);
  void clear();
  void destruct() { this->Attribute::~Attribute(); }
private:
  CopyOwner<AttributeSpec> spec_;
  ConstResourcePointer<AttributeValue> value_;
  CopyOwner<AttributeSemantics> semantics_;
};

class AttributeList  {
public:
  AttributeList();
  AttributeList(const ConstResourcePointer<AttributeDefinitionList> &);
  void init(const ConstResourcePointer<AttributeDefinitionList> &);
  // was a conref attribute specified?
  Boolean conref() const;
  size_t length() const;
  const CString &name(unsigned) const;
  const AttributeValue *value(unsigned) const;
  const ConstResourcePointer<AttributeValue> valuePointer(unsigned) const;
  const AttributeSemantics *semantics(unsigned) const;
  Boolean tokenized(unsigned index) const;
  Boolean tokenIndex(const CString &, unsigned &) const;
  Boolean attributeIndex(const CString &, unsigned &) const;
  void finish(ParserState &);
  void setValue(unsigned index, Text &, ParserState &, unsigned &specLength);
  void setValue(unsigned index, CString &, Text &, ParserState &,
		unsigned &specLength);
  void setSpec(unsigned index, const CString &, ParserState &);
  void setSpec(unsigned index, ParserState &);
  void moveTo(AttributeList &);
  size_t nSpec() const;
  size_t defIndex() const;
  // is the attribute #current
  Boolean current(unsigned) const;
  Boolean specified(unsigned) const;
  Boolean id(unsigned) const;
  Boolean idref(unsigned) const;
  Boolean getId(const CString *&) const;
private:
  const AttributeDefinition *def(size_t) const;
  PackedBoolean conref_;
  PackedBoolean notation_;
  unsigned nIdrefs_;
  unsigned nEntityNames_;
  size_t nSpec_;
  size_t idIndex_;		// -1 if no id
  CopyGrowableVectorD<Attribute> vec_;
  ConstResourcePointer<AttributeDefinitionList> def_;
};

inline
Boolean AttributeDefinition::tokenized() const
{
  return declaredValue_->tokenized();
}

inline
Boolean AttributeDefinition::isNotation() const
{
  return declaredValue_->isNotation();
}

inline
Boolean AttributeDefinition::isId() const
{
  return declaredValue_->isId();
}

inline
Boolean AttributeDefinition::isIdref() const
{
  return declaredValue_->isIdref();
}

inline
AttributeSemantics *
AttributeDefinition::makeSemantics(const AttributeValue *value,
				   ParserState &parser,
				   unsigned &nIdrefs,
				   unsigned &nEntityNames) const
{
  return value->makeSemantics(declaredValue_.pointer(), parser, name_,
			      nIdrefs, nEntityNames);
}

inline
Boolean AttributeDefinition::containsToken(const CString &token) const
{
  return declaredValue_->containsToken(token);
}

inline
const CString &AttributeDefinition::name() const
{
  return name_;
}

inline
size_t AttributeDefinitionList::length() const
{
  return defs_.length();
}

inline
size_t AttributeDefinitionList::index() const
{
  return index_;
}

inline
const AttributeDefinition *AttributeDefinitionList::def(size_t i) const
{
  return defs_[i].pointer();
}

inline
size_t TokenizedAttributeValue::nTokens() const
{
  return spaceIndex_.length() + 1;
}

inline
const CString &TokenizedAttributeValue::string() const
{
  return value_;
}

inline
Boolean Attribute::specified() const
{
  return spec_ != 0;
}

inline
const AttributeSpec *Attribute::spec() const
{
  return spec_.pointer();
}

inline
const AttributeValue *Attribute::value() const
{
  return value_.pointer();
}

inline
const ConstResourcePointer<AttributeValue> Attribute::valuePointer() const
{
  return value_;
}

inline
const AttributeSemantics *Attribute::semantics() const
{
  return semantics_.pointer();
}

inline
void Attribute::setSpec(AttributeSpec *spec)
{
  spec_ = spec;
}

inline
void Attribute::setValue(const ConstResourcePointer<AttributeValue> &value)
{
  value_ = value;
}

inline
void Attribute::setSemantics(AttributeSemantics *semantics)
{
  semantics_ = semantics;
}

inline
size_t AttributeList::length() const
{
  return vec_.length();
}

inline
const AttributeDefinition *AttributeList::def(size_t i) const
{
  return def_->def(i);
}

inline
Boolean AttributeList::tokenized(unsigned i) const
{
  return def(i)->tokenized();
}

inline
Boolean AttributeList::tokenIndex(const CString &name, unsigned &index) const
{
  return !def_.isNull() && def_->tokenIndex(name, index);
}

inline
Boolean AttributeList::attributeIndex(const CString &name, unsigned &index) const
{
  return !def_.isNull() && def_->attributeIndex(name, index);
}

inline
const CString &AttributeList::name(unsigned i) const
{
  return def(i)->name();
}

inline
const AttributeValue *AttributeList::value(unsigned i) const
{
  return vec_[i].value();
}

inline
const ConstResourcePointer<AttributeValue> AttributeList::valuePointer(unsigned i)
     const
{
  return vec_[i].valuePointer();
}

inline
const AttributeSemantics *AttributeList::semantics(unsigned i) const
{
  return vec_[i].semantics();
}

inline
size_t AttributeList::nSpec() const
{
  return nSpec_;
}

inline
Boolean AttributeList::conref() const
{
  return conref_;
}

inline
size_t AttributeList::defIndex() const
{
  return def_.isNull() ? -1 : def_->index();
}

inline
Boolean AttributeList::current(unsigned i) const
{
  return def(i)->isCurrent();
}

inline
const AttributeValue *DefaultAttributeDefinition::defaultValue() const
{
  return value_.pointer();
}

inline
Boolean AttributeList::id(unsigned i) const
{
  return def(i)->isId();
}

inline
Boolean AttributeList::idref(unsigned i) const
{
  return def(i)->isIdref();
}

inline
Boolean AttributeList::specified(unsigned i) const
{
  return vec_[i].specified();
}

#endif /* not Attribute_INCLUDED */
