// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#ifdef __GNUG__
#pragma implementation
#endif
#include "CharsetDecl.H"
#include "macros.H"
#include "ISet.H"

CharsetDeclRange::CharsetDeclRange()
{
}

CharsetDeclRange::CharsetDeclRange(WideChar descMin, Number count,
				   WideChar baseMin)
: descMin_(descMin),
  count_(count),
  type_(number),
  baseMin_(baseMin)
{
}

CharsetDeclRange::CharsetDeclRange(WideChar descMin, Number count)
: descMin_(descMin),
  count_(count),
  type_(unused)
     
{
}

CharsetDeclRange::CharsetDeclRange(WideChar descMin, Number count,
				   const CString &str)
: descMin_(descMin),
  count_(count),
  type_(string),
  str_(str)
{
}

void CharsetDeclRange::moveTo(CharsetDeclRange &to)
{
  CString temp;
  str_.moveTo(temp);
  to = *this;
  temp.moveTo(to.str_);
}

void CharsetDeclRange::rangeDeclared(WideChar min, Number count,
				     ISet<WideChar> &declared) const
{
  if (count > 0 && min + count > descMin_ && min < descMin_ + count_) {
    WideChar commMin = (descMin_ > min) ? descMin_ : min;
    WideChar commMax = min + ((min + count < descMin_ + count_
			       ? count
			       : descMin_ + count_ - min) - 1);
    ASSERT(commMin <= commMax);
    declared.addRange(commMin, commMax);
  }
}

void CharsetDeclRange::declaredSet(ISet<WideChar> &set) const
{
  if (count_ > 0)
    set.addRange(descMin_, descMin_ + (count_ - 1));
}

void CharsetDeclRange::usedSet(ISet<Char> &set) const
{
  if (type_ != unused && count_ > 0 && descMin_ <= Char(-1)) {
    Char max;
    if (Char(-1) - descMin_ < count_ - 1)
      max = Char(-1);
    else
      max = Char(descMin_ + (count_ - 1));
    set.addRange(Char(descMin_), max);
  }
}

void CharsetDeclRange::stringToChar(const CString &str, ISet<WideChar> &to)
     const
{
  if (type_ == string && str_ == str && count_ > 0)
    to.addRange(descMin_, descMin_ + (count_ - 1));
}

void CharsetDeclRange::numberToChar(Number n, ISet<WideChar> &to)
     const
{
  if (type_ == number && n >= baseMin_ && n - baseMin_ < count_)
    to.add(descMin_ + (n - baseMin_));
}

Boolean CharsetDeclRange::getCharInfo(WideChar fromChar,
				      CharsetDeclRange::Type &type,
				      Number &n,
				      CString &str) const
{
  if (fromChar >= descMin_ && fromChar - descMin_ < count_) {
    type = type_;
    if (type == number)
      n = baseMin_ + (fromChar - descMin_);
    else if (type == string)
      str = str_;
    return 1;
  }
  else
    return 0;
}

CharsetDeclSection::CharsetDeclSection()
{
}

CharsetDeclSection::CharsetDeclSection(PublicId &id)
{
  id.moveTo(baseset_);
}

void CharsetDeclSection::moveTo(CharsetDeclSection &to)
{
  baseset_.moveTo(to.baseset_);
  ranges_.moveTo(to.ranges_);
}

void CharsetDeclSection::addRange(CharsetDeclRange &range)
{
  range.moveTo(ranges_.grow());
}

void CharsetDeclSection::rangeDeclared(WideChar min, Number count,
				       ISet<WideChar> &declared) const
{
  for (size_t i = 0; i < ranges_.length(); i++)
    ranges_[i].rangeDeclared(min, count, declared);
}

void CharsetDeclSection::declaredSet(ISet<WideChar> &set) const
{
  for (size_t i = 0; i < ranges_.length(); i++)
    ranges_[i].declaredSet(set);
}

void CharsetDeclSection::usedSet(ISet<Char> &set) const
{
  for (size_t i = 0; i < ranges_.length(); i++)
    ranges_[i].usedSet(set);
}


void CharsetDeclSection::stringToChar(const CString &str, ISet<WideChar> &to)
     const
{
  for (size_t i = 0; i < ranges_.length(); i++)
    ranges_[i].stringToChar(str, to);
}

void CharsetDeclSection::numberToChar(const PublicId *id, Number n,
				      ISet<WideChar> &to) const
{
  PublicId::OwnerType ownerType;
  CString seq1, seq2;
  if (id->string() == baseset_.string()
      // Assume that 2 ISO character sets are the same if
      // their designating sequences are the same.
      || (id->getOwnerType(ownerType)
	  && ownerType == PublicId::ISO
	  && baseset_.getOwnerType(ownerType)
	  && ownerType == PublicId::ISO
	  && id->getDesignatingSequence(seq1)
	  && baseset_.getDesignatingSequence(seq2)
	  && seq1 == seq2)) {
    for (size_t i = 0; i < ranges_.length(); i++)
      ranges_[i].numberToChar(n, to);
  }
}

Boolean CharsetDeclSection::getCharInfo(WideChar fromChar,
					const PublicId *&id,
					CharsetDeclRange::Type &type,
					Number &n,
					CString &str) const
{
  for (size_t i = 0; i < ranges_.length(); i++)
    if (ranges_[i].getCharInfo(fromChar, type, n, str)) {
      id = &baseset_;
      return 1;
    }
  return 0;
}

CharsetDecl::CharsetDecl()
{
}

void CharsetDecl::addSection(PublicId &id)
{
  CharsetDeclSection(id).moveTo(sections_.grow());
}


void CharsetDecl::moveTo(CharsetDecl &to)
{
  sections_.moveTo(to.sections_);
}

void CharsetDecl::clear()
{
  sections_.clear();
}

void CharsetDecl::addRange(WideChar min, Number count, WideChar baseMin)
{
  CharsetDeclRange range(min, count, baseMin);
  sections_.last().addRange(range);
}

void CharsetDecl::addRange(WideChar min, Number count)
{
  CharsetDeclRange range(min, count);
  sections_.last().addRange(range);
}

void CharsetDecl::addRange(WideChar min, Number count, const CString &str)
{
  CharsetDeclRange range(min, count, str);
  sections_.last().addRange(range);
}

void CharsetDecl::rangeDeclared(WideChar min, Number count,
				ISet<WideChar> &declared) const
{
  for (size_t i = 0; i < sections_.length(); i++)
    sections_[i].rangeDeclared(min, count, declared);
}

void CharsetDecl::declaredSet(ISet<WideChar> &set) const
{
  for (size_t i = 0; i < sections_.length(); i++)
    sections_[i].declaredSet(set);
}

void CharsetDecl::usedSet(ISet<Char> &set) const
{
  for (size_t i = 0; i < sections_.length(); i++)
    sections_[i].usedSet(set);
}

Boolean CharsetDecl::getCharInfo(WideChar fromChar,
				 const PublicId *&id,
				 CharsetDeclRange::Type &type,
				 Number &n,
				 CString &str) const
{
  for (size_t i = 0; i < sections_.length(); i++)
    if (sections_[i].getCharInfo(fromChar, id, type, n, str))
      return 1;
  return 0;
}

void CharsetDecl::stringToChar(const CString &str, ISet<WideChar> &to) const
{
  for (size_t i = 0; i < sections_.length(); i++)
    sections_[i].stringToChar(str, to);
}

void CharsetDecl::numberToChar(const PublicId *id, Number n,
			       ISet<WideChar> &to) const
{
  for (size_t i = 0; i < sections_.length(); i++)
    sections_[i].numberToChar(id, n, to);
}
