// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#ifndef Dtd_INCLUDED
#define Dtd_INCLUDED 1
#ifdef __GNUG__
#pragma interface
#endif

#include "NamedTable.H"
#include "NamedResourceTable.H"
#include "ElementType.H"
#include "Notation.H"
#include "Entity.H"
#include "ShortReferenceMap.H"
#include "Resource.H"
#include "CString.H"
#include "StringResource.H"
#include "Boolean.H"

class EntitySet : public Resource {
public:
  typedef ConstNamedResourceTableIter<Entity> ConstEntityIter;
  typedef NamedResourceTableIter<Entity> EntityIter;
  EntitySet();
  ConstResourcePointer<Entity> lookupEntity(Boolean isParameter,
					    const CString &) const;
  ResourcePointer<Entity> lookupEntity(Boolean isParameter, const CString &);
  ResourcePointer<Entity> insertEntity(const ResourcePointer<Entity> &);
  ConstEntityIter generalEntityIter() const;
  EntityIter generalEntityIter();

  ConstResourcePointer<Entity> defaultEntity() const;
  void setDefaultEntity(const ResourcePointer<Entity> &);
private:
  EntitySet(const EntitySet &);	// undefined
  void operator=(const EntitySet &); // undefined

  NamedResourceTable<Entity> generalEntityTable_;
  NamedResourceTable<Entity> parameterEntityTable_;
  ConstResourcePointer<Entity> defaultEntity_;
};

class Dtd : public EntitySet {
public:
  typedef NamedTableIter<ElementType> ElementTypeIter;
  typedef ConstNamedTableIter<ElementType> ConstElementTypeIter;
  typedef NamedTableIter<ShortReferenceMap> ShortReferenceMapIter;
  typedef ConstNamedResourceTableIter<Notation> ConstNotationIter;
  Dtd(const CString &name, size_t nDelimShortref);

  const ElementType *lookupElementType(const CString &) const;
  ElementType *lookupElementType(const CString &);
  ElementType *insertElementType(ElementType *);
  size_t nElementTypeIndex() const;
  ConstElementTypeIter elementTypeIter() const;
  ElementTypeIter elementTypeIter();

  const RankStem *lookupRankStem(const CString &) const;
  RankStem *lookupRankStem(const CString &);
  RankStem *insertRankStem(RankStem *);
  size_t nRankStem() const;

  const ShortReferenceMap *lookupShortReferenceMap(const CString &) const;
  ShortReferenceMap *lookupShortReferenceMap(const CString &);
  ShortReferenceMap *insertShortReferenceMap(ShortReferenceMap *);
  ShortReferenceMapIter shortReferenceMapIter();

  ConstResourcePointer<Notation> lookupNotation(const CString &) const;
  ResourcePointer<Notation> lookupNotation(const CString &);
  ResourcePointer<Notation> insertNotation(const ResourcePointer<Notation> &);
  ConstNotationIter notationIter() const;

  const ConstResourcePointer<StringResource<Char> > &namePointer() const;
  const CString &name() const;

  size_t allocCurrentAttributeIndex();
  size_t nCurrentAttribute() const;
  size_t allocElementDefinitionIndex();
  size_t nElementDefinition() const;
  size_t allocAttributeDefinitionListIndex();
  size_t nAttributeDefinitionList() const;
  const ElementType *documentElementType() const;
  void noteShortrefUsed(size_t i);
  const PackedBoolean *usedShortrefs() const;
private:
  Dtd(const Dtd &);		// undefined
  void operator=(const Dtd &);	// undefined
  ConstResourcePointer<StringResource<Char> > name_;
  NamedTable<ElementType> elementTypeTable_;
  NamedTable<RankStem> rankStemTable_;
  NamedTable<ShortReferenceMap> shortReferenceMapTable_;
  NamedResourceTable<Notation> notationTable_;
  size_t nCurrentAttribute_;
  size_t nElementDefinition_;
  size_t nAttributeDefinitionList_;
  ElementType *documentElementType_;
  Vector<PackedBoolean> shortrefUsed_;
};

inline
ConstResourcePointer<Entity> EntitySet::lookupEntity(Boolean isParameter,
					       const CString &name) const
{
  return (isParameter
	  ? &parameterEntityTable_
	  : &generalEntityTable_)->clookup(name);
}

inline
ResourcePointer<Entity> EntitySet::insertEntity(const ResourcePointer<Entity> &entity)
{
  return (entity->declType() == Entity::parameterEntity
	  ? &parameterEntityTable_
	  : &generalEntityTable_)->insert(entity);
}

inline
EntitySet::ConstEntityIter EntitySet::generalEntityIter() const
{
  return ConstEntityIter(generalEntityTable_);
}

inline
EntitySet::EntityIter EntitySet::generalEntityIter()
{
  return EntityIter(generalEntityTable_);
}

inline
ConstResourcePointer<Entity> EntitySet::defaultEntity() const
{
  return defaultEntity_;
}

inline
void EntitySet::setDefaultEntity(const ResourcePointer<Entity> &entity)
{
  defaultEntity_ = entity.pointer();
}

inline
const ConstResourcePointer<StringResource<Char> > &Dtd::namePointer() const
{
  return name_;
}

inline
const CString &Dtd::name() const
{
  return *name_;
}

inline
size_t Dtd::allocCurrentAttributeIndex()
{
  return nCurrentAttribute_++;
}

inline
size_t Dtd::nCurrentAttribute() const
{
  return nCurrentAttribute_;
}

inline
size_t Dtd::allocElementDefinitionIndex()
{
  return nElementDefinition_++;
}

inline
size_t Dtd::nElementDefinition() const
{
  return nElementDefinition_;
}

inline
size_t Dtd::allocAttributeDefinitionListIndex()
{
  return nAttributeDefinitionList_++;
}

inline
size_t Dtd::nAttributeDefinitionList() const
{
  return nAttributeDefinitionList_;
}

inline
const ElementType *Dtd::lookupElementType(const CString &name) const
{
  return elementTypeTable_.lookup(name);
}

inline
ElementType *Dtd::lookupElementType(const CString &name)
{
  return elementTypeTable_.lookup(name);
}

inline
ElementType *Dtd::insertElementType(ElementType *e)
{
  return elementTypeTable_.insert(e);
}

inline
Dtd::ElementTypeIter Dtd::elementTypeIter()
{
  return ElementTypeIter(elementTypeTable_);
}

inline
Dtd::ConstElementTypeIter Dtd::elementTypeIter() const
{
  return ConstElementTypeIter(elementTypeTable_);
}

inline
size_t Dtd::nElementTypeIndex() const
{
  // index 0 is reserved for #pcdata
  return 1 + elementTypeTable_.count();
}

inline
const RankStem *Dtd::lookupRankStem(const CString &name) const
{
  return rankStemTable_.lookup(name);
}

inline
RankStem *Dtd::lookupRankStem(const CString &name)
{
  return rankStemTable_.lookup(name);
}

inline
RankStem *Dtd::insertRankStem(RankStem *e)
{
  return rankStemTable_.insert(e);
}

inline
size_t Dtd::nRankStem() const
{
  return rankStemTable_.count();
}

inline
ConstResourcePointer<Notation> Dtd::lookupNotation(const CString &name) const
{
  return notationTable_.clookup(name);
}

inline
ResourcePointer<Notation> Dtd::lookupNotation(const CString &name)
{
  return notationTable_.lookup(name);
}

inline
ResourcePointer<Notation> Dtd::insertNotation(const ResourcePointer<Notation> &nt)
{
  return notationTable_.insert(nt);
}

inline
Dtd::ConstNotationIter Dtd::notationIter() const
{
  return ConstNotationIter(notationTable_);
}

inline
const ElementType *Dtd::documentElementType() const
{
  return documentElementType_;
}

inline
const ShortReferenceMap *Dtd::lookupShortReferenceMap(const CString &name) const
{
  return shortReferenceMapTable_.lookup(name);
}

inline
ShortReferenceMap *Dtd::lookupShortReferenceMap(const CString &name)
{
  return shortReferenceMapTable_.lookup(name);
}

inline
ShortReferenceMap *Dtd::insertShortReferenceMap(ShortReferenceMap *map)
{
  return shortReferenceMapTable_.insert(map);
}

inline
Dtd::ShortReferenceMapIter Dtd::shortReferenceMapIter()
{
  return ShortReferenceMapIter(shortReferenceMapTable_);
}

inline
const PackedBoolean *Dtd::usedShortrefs() const
{
  return shortrefUsed_.pointer();
}

inline
void Dtd::noteShortrefUsed(size_t i)
{
  shortrefUsed_[i] = 1;
}

#endif /* not Dtd_INCLUDED */
